/**************************************************************************
 * $Id: DeviceDlg.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "SamplinInput.h"
#include <samplin.h>
#include <fcntl.h>
#include <qmessagebox.h>
#include <qfile.h>
#include "DeviceDlg.h"

#include "DeviceDlg.moc"

DeviceDlg::DeviceDlg
(
	QWidget* parent,
	const char* name
)
	:
	DeviceDlgData( parent, name )
{
   setCaption("Device configuration");
   button_ok->setDefault(TRUE);
   
   devices=new QList<SamplinDevice>;
   devices->setAutoDelete(TRUE);
   d=NULL;
   
   dtype = new TypeDlg(tab, "Path");
   dopt = new OptionsDlg(tab, "Opt");
   dser = new SerialDlg(tab, "Ser");
   dgpib = new GPIBDlg(tab, "GPIB");
   
   tab->addTab(dtype, "Path");
   tab->addTab(dopt,"Common options");
   tab->addTab(dser, "Serial options");
   tab->addTab(dgpib,"GPIB options");
   tab->setTabEnabled("Ser",FALSE);
   tab->setTabEnabled("GPIB",FALSE);

   connect(dopt, SIGNAL(typeSignal(int)),this,SLOT(typeSlot(int)));
   
}


DeviceDlg::~DeviceDlg()
{
}

void DeviceDlg::ok()
{
   if(save()==TRUE)
     accept();
   else QMessageBox::warning(this,"Error","Can't write to file "+filename);
}

void DeviceDlg::cancel()
{
   reject();
}


bool DeviceDlg::setFile(const char *str)
{
   SamplinDevice *p;
   
   if(str!=NULL){
      
      filename=str;

      if(QFile::exists(filename)==FALSE)
	QMessageBox::information(this,"Info","Device configuration file\n"+filename+"\ndoes not exist. Creating new file.");
      else{
	 if(load()==TRUE){
	    if(devices!=NULL){
	       list->clear();
	       for(p=devices->first();p!=NULL;p=devices->next())
		 list->inSort(p->getName());
	    }
	 }
	 else{
	    QMessageBox::warning(this,"Error","Can't read file "+filename);
	    return FALSE;
	 }
      }
   }
   return TRUE;
}

int DeviceDlg::load(void)
{
   int ret;
   
   QFile f(filename);
   if((ret=f.open( IO_ReadOnly))==TRUE){
      QDataStream s(&f);
      devices->clear();
      s >> *devices;
      if(f.status()!=IO_Ok)ret=FALSE;
      f.close();
      if(f.status()!=IO_Ok)ret=FALSE;
   }
   return ret;
}

int DeviceDlg::save()
{
   int ret;
   
   QFile f(filename);
   if(f.exists()==TRUE)rename(filename,filename+"~");
   
   if((ret=f.open(IO_WriteOnly))==TRUE){
      QDataStream s(&f);
      loadBack();
      s << *devices;
      if(f.status()!=IO_Ok)ret=FALSE;
      f.close();
      if(f.status()!=IO_Ok)ret=FALSE;
   }
   return ret;
}



int DeviceDlg::findItem(const char *item)
{
   uint i;
   const char *str;
   
   i=0;
   while(i<list->count()){
      str=list->text(i);
      if(!strcmp(item,str) && strlen(str)==strlen(item)){
	 return i;
      }
      ++i;
   }
   
   return -1;
}


SamplinDevice *DeviceDlg::findDevice(int index)
{
   
      return(findDevice(list->text(index)));
   
}

SamplinDevice *DeviceDlg::findDevice(const char *str)
{
      SamplinDevice *p;
   
      if(str==NULL)return NULL;
   
      for(p=devices->first();p!=NULL;p=devices->next()){
	       if(strlen(str)==strlen(p->getName()) && !strcmp(str,p->getName()))
	           return p;
      }
   
      return NULL;
}

void DeviceDlg::loadBack(void)
{
   int rd,wr,i;
   
   if(d!=NULL){
      d->setPath(dtype->getDevice());
      d->setHost(dtype->getHost());
      d->setTitle(dtype->getTitle());
	 
      d->setType(dopt->getType());
      d->setEndian(dopt->getEndian());
      d->setDelay(atoi(dopt->getDelay()));
      //      d->setTimeout(atoi(dopt->getTimeout()));
      d->setNTimeout(atoi(dopt->getNTimeout()));
      rd=dopt->getRead();
      wr=dopt->getWrite();
      if(rd && wr)d->setFlags(O_RDWR);
      if(rd && !wr)d->setFlags(O_RDONLY);
      if(!rd && wr)d->setFlags(O_WRONLY);
      if(!rd && !wr)d->setFlags(O_RDONLY);
      
      d->setSpeed(dser->getBaud());
      d->setDatabits(dser->getData());
      d->setStopbits(dser->getStop());
      d->setParity(dser->getParity());
      
      d->setAddress(atoi(dgpib->getAddress()));
      sscanf(dopt->getEos(),"0x%x",&i);
      d->setEos(i);
      d->setMaster(dgpib->getMaster());
   }
}

void DeviceDlg::highlightedSlot(const char *str)
{
   char s[64];
   int flg;
   
   loadBack();
   
   d=findDevice(str);
   
   if(d!=NULL){
//      emit infoSignal(d->getTitle());
      dtype->setDevice(d->getPath());
      dtype->setHost(d->getHost());
      dtype->setTitle(d->getTitle());
      
      dopt->setType(d->getType());
      dopt->setEndian(d->getEndian());
      sprintf(s, "%i",d->getDelay());
      dopt->setDelay(s);
      //      sprintf(s, "%i",d->getTimeout());
      //      dopt->setTimeout(s);      
      sprintf(s, "%i",d->getNTimeout());
      dopt->setNTimeout(s);
      
      flg=d->getFlags();
      switch(flg){
       case O_RDONLY:
	 dopt->setRead(TRUE);
	 dopt->setWrite(FALSE);
	 break;
       case O_WRONLY:
	 dopt->setRead(FALSE);
	 dopt->setWrite(TRUE);
	 break;
       case O_RDWR:
	 dopt->setRead(TRUE);
	 dopt->setWrite(TRUE);
	 break;
      }
      
      dser->setBaud(d->getSpeed());
      dser->setData(d->getDatabits());
      dser->setStop(d->getStopbits());
      dser->setParity(d->getParity());
      
      sprintf(s, "%i",d->getAddress());
      dgpib->setAddress(s);
      if(d->getEos() & 0xff00)sprintf(s, "0x%04x",d->getEos());
      else sprintf(s, "0x%02x",d->getEos());
      dopt->setEos(s);
      dgpib->setMaster(d->getMaster());
      
      setEnabled(d->getType());
   }
}

void DeviceDlg::deleteSlot()
{
      SamplinDevice *p;
      int item;
   
      item=list->currentItem();
      if(item==-1)return;
   
      p=findDevice(item);
      if(p==NULL)return;
   
      d=NULL;
      list->setCurrentItem(MIN(item+1,(int)(list->count()-1)));
      list->removeItem(item);
      devices->removeRef(p);
}

void DeviceDlg::newSlot()
{
   SamplinInput *inp;
   SamplinDevice *p;
   QString name="";
   
   inp = new SamplinInput(this,"input");
   inp->setTitle("New device name");
   inp->show();
   if(inp->result()==QDialog::Accepted)
     name=inp->getText();
   delete inp;
   if(strlen(name)==0)return;
   if(findDevice(name)!=NULL){
      QMessageBox::warning(this,"SampLin", "Could not create device, name already exists.");
      return;
   }
   
   p = new SamplinDevice();
   p->setName(name);
   devices->insert(devices->count(),p);
   list->inSort(name);
   list->setCurrentItem(findItem(name));
   list->setTopItem(findItem(name));
}

void DeviceDlg::typeSlot(int index)
{
      setEnabled(index);
}

void DeviceDlg::setEnabled(int type)
{
   switch(type)
     {
      case DEVICE_SERIAL:
	tab->setTabEnabled("Ser",TRUE);
	tab->setTabEnabled("GPIB",FALSE);
	break;
      case DEVICE_GPIB:
	tab->setTabEnabled("GPIB",TRUE);
	tab->setTabEnabled("Ser",FALSE);
	break;
      case DEVICE_VOID:
      default:
	tab->setTabEnabled("Ser",FALSE);
	tab->setTabEnabled("GPIB",FALSE);
	break;
     }
   
}

void DeviceDlg::copySlot()
{
   SamplinInput *inp;
   SamplinDevice *p;
   QString name="";
   int index=list->currentItem();
   if(index==-1)return;
   
   inp = new SamplinInput(this,"input");
   inp->setTitle("Copy to new device");
   inp->show();
   if(inp->result()==QDialog::Accepted)
     name=inp->getText();
   delete inp;
   if(strlen(name)==0)return;
   if(findDevice(name)!=NULL){
      QMessageBox::warning(this,"SampLin", "Could not create device, name already exists.");
      return;
   }
   
   p = new SamplinDevice();
   *p = *findDevice(list->text(index));
   p->setName(name);

   devices->insert(devices->count(),p);
   list->inSort(name);
   list->setCurrentItem(findItem(name));
   list->setTopItem(findItem(name));


}

void DeviceDlg::renameSlot()
{
   SamplinInput *inp;
   SamplinDevice *p;
   QString name="";
   int index=list->currentItem();
   if(index==-1)return;
    
   inp = new SamplinInput(this,"input");
   inp->setTitle("Rename device");
   inp->setText(list->text(index));
   inp->show();
   if(inp->result()==QDialog::Accepted){
      name=inp->getText();
      if(strlen(name)==0)return;
      if((p=findDevice(name))!=NULL){
	 QMessageBox::warning(this,"SampLin", "Could not rename device, name already exists.");
	 return;
      }
      p=findDevice(list->text(index));
      p->setName(name);
      list->removeItem(index);
      list->inSort(name);
      list->setCurrentItem(findItem(name));
      list->setTopItem(findItem(name));
   }
   delete inp;

}
