/**************************************************************************
 * $Id: devices.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include <rserial.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <ibSampLin.h>
#include "devices.h"
//#include "main.h"
#define RETRY_READ 30000
#define FAIL_MSG ""

SamplinDevice::SamplinDevice( void )
{
   type = DEVICE_VOID;
   endian = ENDIAN_LITTLE;
   delay = DEF_DELAY;
   handle = 0;
   rhandle = NULL;
   flags = O_RDWR;
   serial_speed=8;
   serial_stop=0;
   serial_data=0;
   serial_parity=0;
   timeout=10;
   ntimeout=10;
   
   gpib_address=0;
   eos=0x0d;
   gpib_master=FALSE;
   gpib_dcl=FALSE;
   gpib_llo=FALSE;
}


SamplinDevice::~SamplinDevice()
{
   
}

SamplinDevice &SamplinDevice::operator=(SamplinDevice &d)
{
   
   name=d.getName();
   title=d.getTitle();
   path=d.getPath();
   host=d.getHost();
//   handle=d.getHandle();
   type=d.getType();
   endian=d.getEndian();
   delay=d.getDelay();
   flags=d.getFlags();
   serial_data=d.getDatabits();
   serial_stop=d.getStopbits();
   serial_parity=d.getParity();
   serial_speed=d.getSpeed();
   ntimeout=d.getNTimeout();
   gpib_address=d.getAddress();
   eos=d.getEos();
   gpib_master=d.getMaster();

   return(*this);
}

bool streq(const char *s1, const char *s2)
{
   if(!strcmp(s1,s2) && (strlen(s1)==strlen(s2)))return TRUE;
      else return FALSE;
   
}

bool operator!=(SamplinDevice &d1, SamplinDevice &d2)
{
   int ret=FALSE;

   if(!streq(d1.getName(),d2.getName()) || !streq(d1.getTitle(),d2.getTitle()))ret=TRUE;
   if(!streq(d1.getPath(),d2.getPath()))ret=TRUE;
   if(!streq(d1.getHost(),d2.getHost()))ret=TRUE;
   if((d1.getType()!=d2.getType()) || (d1.getEndian()!=d2.getEndian()))ret=TRUE;
   if((d1.getDelay()!=d2.getDelay()) || (d1.getFlags()!=d2.getFlags()))ret=TRUE;
   if((d1.getDatabits()!=d2.getDatabits()) || (d1.getStopbits()!=d2.getStopbits()))ret=TRUE;
   if((d1.getParity()!=d2.getParity()) || (d1.getSpeed()!=d2.getSpeed()))ret=TRUE;
   if((d1.getNTimeout()!=d2.getNTimeout()) || (d1.getAddress()!=d2.getAddress()))ret=TRUE;
   if((d1.getEos()!=d2.getEos()) || (d1.getMaster()!=d2.getMaster()))ret=TRUE;
   
   return(ret);
}

bool operator==(SamplinDevice &d1, SamplinDevice &d2)
{
   bool ret=FALSE;
   if(streq(d1.getName(),d2.getName()) && streq(d1.getTitle(),d2.getTitle()) &&
      streq(d1.getPath(),d2.getPath()) && streq(d1.getHost(),d2.getHost()) &&
      (d1.getType()==d2.getType()) && (d1.getEndian()==d2.getEndian()) &&
      (d1.getDelay()==d2.getDelay()) && (d1.getFlags()==d2.getFlags()) &&
      (d1.getDatabits()==d2.getDatabits()) && (d1.getStopbits()==d2.getStopbits()) &&
      (d1.getParity()==d2.getParity()) && (d1.getSpeed()==d2.getSpeed()) &&
      (d1.getNTimeout()==d2.getNTimeout()) && (d1.getAddress()==d2.getAddress()) &&
      (d1.getEos()==d2.getEos()) && (d2.getMaster()==d2.getMaster())
      )ret=TRUE;
   
   return(ret);
}

int SamplinDevice::Write(const char *data, const int len, const int binary)
{
   int stat,ret,n;
   serial_request req, *result;
   char *mybuff;
   
   ret=0;n=len;
   if(n==0)n=strlen(data);
   if(handle==0)return(-1);

   switch(type){

    case DEVICE_GPIB:
      if(host.isEmpty()){
//	 stat = ibwrt(handle,data,n);
	 stat=ibFunc(handle,IBEOS,eos);	 
	 if(stat & ERR)ret=-2;
	 else {
	    stat=ibFunc(handle,gpib_master ? IBWRT:DVWRT,gpib_address,data,n);	 
	    if(stat & ERR)ret=-2;
	 }
      }
      else{
	 stat=ibFuncRemote(rhandle,handle,IBEOS,eos,0,0);
	 if(stat == ERR){
	    ret=-3;
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	 }
	 else if(stat & ERR)ret=-2;
	 else{
	    stat=ibFuncRemote(rhandle,handle,gpib_master ? IBWRT:DVWRT,gpib_address,data,n);
	    if(stat == ERR){
	       ret=-3;
	       errlog=clnt_sperror(rhandle,FAIL_MSG);
	    }
	    else if(stat & ERR)ret=-2;
	 }
	    
      }
      if(ret==0)usleep(delay);
      break;

    case DEVICE_SERIAL:
    case DEVICE_VOID:
      mybuff=(char *)calloc(n+4+2,sizeof(char));
      memset(mybuff,0,n+4+2);
      memcpy(mybuff,data,n);
      if(!binary){
//	 *(mybuff+n)='\r'; !!! EOS
	 if(eos & 0xff00){
	    *(mybuff+n)=(eos & 0xff00)>>8;
	    *(mybuff+n+1)=(eos & 0xff);
	    n+=2;
	 }
	 else {
	    *(mybuff+n)=eos;
	    ++n;
	 }
      }
      
      if(host.isEmpty()){
	 ret=write(handle,mybuff,n);
	 if(ret==-1)--ret;
      }
      else{
//	 unsigned x1,x2;
//	 x1=mybuff[0];
//	 x2=mybuff[1];
//	 printf("Writing: %u, %u\n",x1,x2);
	 memset(&req,0,sizeof(req));
	 req.device="";
	 req.client="";
	 req.request=REQ_WRITE;
	 req.handle=handle;
	 req.buffer.buffer_val=mybuff;
	 req.buffer.buffer_len=n;
	 result=rserial_dorequest_1(&req,rhandle,ntimeout);
	 ret=0;
	 if(result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    ret=-3;
	 }
	 else if(result->ret==-1)ret=-2;

      }
      if(delay!=0)usleep(delay);
      free(mybuff);
      break;
   }
   return(ret);
}

int SamplinDevice::Flush(void)
{
   return 0;  
}
int SamplinDevice::Seek(long pos, int whence)
{
   return 0;
}

// support for long *arg added, warning highly explosive !
// default len=sizeof(long) to minimize danger

int SamplinDevice::Ioctl(int request, char *data, const int len)
{
   int ret;
   serial_request req, *result;
//   char mybuff[sizeof(long)+4];
   
//   *(long*)mybuff=arg;
   ret=0;
   if(handle==0)return(-1);

   switch(type){
    case DEVICE_GPIB:
      return(-1);
      break;
    case DEVICE_SERIAL:
    case DEVICE_VOID:
      if(host.isEmpty()){
	 ret=ioctl(handle,request,/*mybuff*/ data);
	 if(ret==-1)--ret;
      }
      else{
//	 printf("Ioctl, arg=%li\n",arg);
	 memset(&req,0,sizeof(req));
	 req.device="";
	 req.client="";
	 req.request=REQ_IOCTL;
	 req.handle=handle;
	 req.arg=request;
	 req.buffer.buffer_val=data; //mybuff;
	 req.buffer.buffer_len=len; //sizeof(long);
	 result=rserial_dorequest_1(&req,rhandle,ntimeout);
	 if(result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    ret=-3;
	 }
	 else{
	    ret=0;
	    memcpy(data,result->buffer.buffer_val,result->buffer.buffer_len);
	    free(result->buffer.buffer_val);
	 }
      }
      break;
   }
   return(ret);
}

int SamplinDevice::Read(char *data, const int len, const int binary)
{
   int stat,ret;
   serial_request req, *result;
   int i, retry_count;
   char chout;
   int eos_flag,eos1,eos2;
   
   eos1=(eos & 0xff);
   eos2=(eos & 0xff00)>>8;
   
   ret=0;

   if(handle==0)return(-1);

   switch(type){

    case DEVICE_GPIB:
      if(host.isEmpty()){
//	 stat = ibrd(handle,data,len);
//	 if(stat & ERR)ret=-2;
	 stat=ibFunc(handle,IBEOS,eos);
	 if(stat & ERR)ret=-2;	 
	 else{
	    stat=ibFunc(handle,gpib_master ? IBRD:DVRD,gpib_address,data,len);
	    if(stat & ERR)ret=-2;	 
	 }
      }
      else{
	 stat=ibFuncRemote(rhandle,handle,IBEOS,eos,0,0);
	 if(stat == ERR){
	    ret=-3;
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	 }
	 else if(stat & ERR)ret=-2;
	 else{
	    stat=ibFuncRemote(rhandle,handle,gpib_master ? IBRD:DVRD,gpib_address,data,len);
	    if(stat == ERR){
	       ret=-3;
	       errlog=clnt_sperror(rhandle,FAIL_MSG);
	    }
	    else if(stat & ERR)ret=-2;
	 }
      }
      if(ret==0)usleep(delay);
      break;

    case DEVICE_VOID:
    case DEVICE_SERIAL:
      if(host.isEmpty()){
	 memset(data,0,len);
	 i=0;

	 /*	 if(type==DEVICE_VOID){
	    ret=read(handle, data, len);
	    if(ret!=-1)i=ret;
	 }
	  else{*/
	 eos_flag=0;
	 do{
	    retry_count=0;
	    do{
	       ret=read(handle, &chout, sizeof(chout));
	       ++retry_count;
	    }while(ret==-1 && errno==11 && retry_count<RETRY_READ);
	    
	    if(ret!=-1 && (chout!=eos1 /*'\n'*/ || binary )){
	       *(data+i)=chout;
	       ++i;
	    }

	    if(i>0)
	      if(eos1==chout && (eos2 == *(data+i-1) || !eos2))
		eos_flag=1;
	    
	 }while(ret!=0 && ret!=-1 && ((!eos_flag && chout!=0)/*'\n'*/ || binary) && i<len);

	 if(!binary && eos_flag && eos2){
	    *(data+i-1)=0;
	 }

	 if(ret!=-1)ret=i;
	 else ret=-2;
      }
      else{
	 memset(&req,0,sizeof(req));
	 req.device="";
	 req.client="";
	 req.count=len;
	 req.ret=eos;
	 //	 printf("req len=%i\n",len);
	 if(type==DEVICE_VOID)req.arg=0;
	 else req.arg=RETRY_READ;
	 
	 if(binary==0)req.request=REQ_READ_STR;
	 else req.request=REQ_READ_BIN;
	 req.handle=handle;
	 
	 result=rserial_dorequest_1(&req,rhandle,ntimeout);
	 if(result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    ret=-3;
	 }
	 else{
	    memset(data,0,len+1);
	    if(result->buffer.buffer_len>0){
	       memcpy(data,result->buffer.buffer_val,result->buffer.buffer_len);
	       free(result->buffer.buffer_val);
	    }
	    ret=result->count;
	 }
      }
      
      break;
   }
   return(ret);
}


int SamplinDevice::Ready( void )
{
   int ret=0;
   int maxfd;
   serial_request req, *result;
   fd_set readfs;
   struct timeval timeout;
   
   if(handle==0)return(-1);
   switch(type){
    case DEVICE_GPIB:
      break;
    case DEVICE_VOID:
    case DEVICE_SERIAL:
      if(host.isEmpty()){
	 maxfd=handle+1;
	 FD_ZERO(&readfs);
	 FD_SET(handle,&readfs);
	 timeout.tv_usec=0;
	 timeout.tv_sec=0;
	 ret=select(maxfd, &readfs, NULL,NULL,&timeout);
      }
      else{
	 memset(&req,0,sizeof(req));
	 req.device="";
	 req.client="";
	 req.request=REQ_TEST;
	 req.handle=handle;
	 result=rserial_dorequest_1(&req,rhandle,ntimeout);
	 if(result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    ret=-3;
	 }
	 else{
	    ret=result->ret;
	 }
      }
   
   }   
   return(ret);
}

int SamplinDevice::Close( void )
{
   int *result;
   int ret=0;
   serial_request req;

//   printf("close handle %d\n",handle);
   
   if(handle==0)return(-1);
   switch(type){
    case DEVICE_GPIB:
      if(!host.isEmpty()){
	 if(rhandle==NULL){
	    handle=0;
	    return -1;
	 }
	 ret=ibCloseRemote(rhandle,handle,gpib_address);
	 if(ret&ERR){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    ret=-3;
	 }
	 else{
	    ret=0;
	    rhandle=NULL;
	    handle=0;
	 }
      }
      else{
	 ret=ibClose(handle, gpib_address);
	 if(ret==0)handle=0;
      }
      break;
    case DEVICE_VOID:
    case DEVICE_SERIAL:
      if(host.isEmpty()){
	 ret=close(handle);
      }
      else{
	 memset(&req,0,sizeof(req));
	 req.device=path.data();
	 req.client="";
	 req.request=REQ_CLOSE;
	 req.handle=handle;
	 result=rserial_gethandle_1(&req,rhandle,ntimeout);
	 if(result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    ret=-3;
	 }
	 else{
	    clnt_destroy(rhandle);
	    ret=*result;
	    rhandle=NULL;
	 }
      }
      if(ret==0)handle=0;
      break;
   }   
   
   return(ret);
}

int SamplinDevice::Open( void )
{
   struct termios options;
   int *result;
   serial_request req, *req_result;
   int ret=handle;
   char tmp, *mybuff;
   char  hostname[MAXNETNAMELEN];
   QString tmphst;
   
   speed_t speed=0;
   tcflag_t dbits=0, flgs=0;
   
   if(handle!=0)return(ret);

   
   switch(type){

    case DEVICE_GPIB:
      if(host.isEmpty()){
	 handle=ibOpen(path.data(),gpib_address);
	 if(handle<0){
	    handle=0;
	    return(-1);
	 }
	 ret=handle;
	 if(ibFunc(handle,DVRSP,gpib_address,&tmp) & ERR){
	    ret=-1;
	    Close();
	    handle=0;
	 }
	 //	 ibFunc(handle,DVCLR,gpib_address);
      }
      else{
	 rhandle=ibCreateRemote(host.data());
	 if(rhandle==NULL){
	    errlog=clnt_spcreateerror(host.data());
	    return(-3);	       
	 }
	 handle=ibOpenRemote(rhandle,path.data(),gpib_address);
	 if(handle&ERR){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    handle=0;
	    return(-3);
	 }
	 ret=handle;
	 if(ibFuncRemote(rhandle,handle,DVRSP,gpib_address,&tmp,1) & ERR){
	    ret=-1;
	    Close();
	    handle=0;
	 }
      }
     
      usleep(delay);
      break;

    case DEVICE_SERIAL:
      
      
      switch(serial_speed){
       case 0:
	 speed=B150;
	 break;
       case 1:
	 speed=B200;
	 break;
       case 2:
	 speed=B300;
	 break;
       case 3:
	 speed=B600;
	 break;
       case 4:
	 speed=B1200;
	 break;
       case 5:
	 speed=B1800;
	 break;
       case 6:
	 speed=B2400;
	 break;
       case 7:
	 speed=B4800;
	 break;
       case 8:
	 speed=B9600;
	 break;	 
       case 9:
	 speed=B19200;
	 break;	 
       case 10:
	 speed=B38400;
	 break;	 
       case 11:
	 speed=B57600;
	 break;	 
       case 12:
	 speed=B115200;
	 break;	 	 
       case 13:
	 speed=B230400;
	 break;	 	 	 
      }
      
      switch(serial_data){
       case 0:
	 dbits=CS8;
	 break;
       case 1:
	 dbits=CS7;
	 break;
       case 2:
	 dbits=CS6;
	 break;
       case 3:
	 dbits=CS5;
	 break;
      }
	
      switch(serial_stop){
       case 0:
	 break;
       case 1:
	 flgs |=CSTOPB;
	 break;
      }

      switch(serial_parity){
       case 0:
	 break;
       case 1:
	 flgs |=PARENB;
	 break;
       case 2:
	 flgs |=PARENB|PARODD;
	 break;
      }      
      
      bzero(&options,sizeof(options));
      cfsetispeed(&options, speed);
      cfsetospeed(&options, speed);
      options.c_cflag |= (CLOCAL | CREAD);
      options.c_cflag |= dbits | flgs;    
      options.c_cc[VMIN]=1;
	 
      if(host.isEmpty()){

	 handle=open(path.data(),flags | O_NOCTTY );
	 if(handle==-1){
	    handle=0;
	    return(0);
	 }

	 fcntl(handle, F_SETFL, FASYNC|FNDELAY);
	 tcsetattr(handle, TCSANOW, &options);
      
      }
      else{
	 gethostname(hostname,sizeof(hostname));
	 
	 rhandle=clnt_create(host.data(),RSERIALPROG,RSERIALVERS,"tcp");
	 if(rhandle==NULL){
	    errlog=clnt_spcreateerror(host.data());
	    return(-3);
	 }
	 rhandle->cl_auth = authunix_create(hostname, getuid(), getgid(),0,NULL);
	 
	 memset(&req,0,sizeof(req));
	 
	 req.client=hostname;
	 req.device=path.data();
	 req.arg=flags | O_NOCTTY;
	 req.request=REQ_OPEN;
	 result=rserial_gethandle_1(&req,rhandle,ntimeout);
	 if(result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    return(-3);
	 }
	 handle=*result;
	 if(handle==-1){
	    handle=0;
	    return(0);
	 }
	 
	 req.handle=handle;
	 req.arg=FASYNC | FNDELAY;
	 req.request=REQ_SETFL;
	 req_result=rserial_dorequest_1(&req,rhandle,ntimeout);
	 if(req_result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	 }

	 mybuff=(char *)calloc(sizeof(options)+4,sizeof(char));
	 memcpy(mybuff,&options,sizeof(options));

	 req.handle=handle;;
	 req.arg=TCSANOW;
	 req.request=REQ_TCSET;
	 req.buffer.buffer_len=sizeof(options);
	 req.buffer.buffer_val=mybuff;
	 req_result=rserial_dorequest_1(&req,rhandle,ntimeout);
	 free(mybuff);
	 if(req_result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	 }	 
      
      }
      ret = handle;
      break;

    case DEVICE_VOID:

      if(host.isEmpty()){

	 handle=open(path.data(),flags);
	 if(handle==-1){
	    handle=0;
	    return(0);
	 }

      }
      else{
	 gethostname(hostname,sizeof(hostname));
	 
	 rhandle=clnt_create(host.data(),RSERIALPROG,RSERIALVERS,"tcp");
	 if(rhandle==NULL){
	    errlog=clnt_spcreateerror(host.data());
	    return(-3);
	 }
	 rhandle->cl_auth = authunix_create(hostname, getuid(), getgid(),0,NULL);
	 
	 memset(&req,0,sizeof(req));
	 
	 req.client=hostname;
	 req.device=path.data();
	 req.arg=flags;
	 req.request=REQ_OPEN;
	 result=rserial_gethandle_1(&req,rhandle,ntimeout);
	 if(result==NULL){
	    errlog=clnt_sperror(rhandle,FAIL_MSG);
	    return(-3);
	 }
	 handle=*result;
	 if(handle==-1){
	    handle=0;
	    return(0);
	 }
      }
      ret = handle;
      break;      
      
   }
return(ret);   
}

int SamplinDevice::GpibFunc( int fnc )
{
   int ret=-1;
   char tmp;
   
   
   if(handle==0)return(-1);
   if(type!=DEVICE_GPIB)return(-2);

   switch(fnc){
    case GPIB_DVRSP:
      if(host.isEmpty()){
	 ibFunc(handle,DVRSP,gpib_address,&tmp);
      }
      else{
	 ibFuncRemote(rhandle,handle,DVRSP,gpib_address,&tmp,1);
      }
      ret=tmp;
      break;
   }
   return(ret);
}


QDataStream &operator<<( QDataStream &s, SamplinDevice &dev )
{

   s << dev.getTitle();
   s << dev.getName();
   s << dev.getPath();
   s << dev.getHost();
   s << dev.getType();
   s << dev.getEndian();
   s << dev.getFlags();
   s << dev.getDelay();
   s << dev.getSpeed();
   s << dev.getDatabits();
   s << dev.getStopbits();
   s << (int)255;
   s << dev.getParity();
   s << dev.getTimeout();   
   s << dev.getNTimeout();
   s << (int)254;
   s << dev.getAddress();
   s << dev.getEos();
   s << dev.getMaster();
   s << (int)0;
   return(s);
}

QDataStream &operator>>( QDataStream &s, SamplinDevice &dev )
{
QString str;
int i;
char ch;
   
   s >> str;
   dev.setTitle(str);
   s >> str;
   dev.setName(str);
   s >> str;
   dev.setPath(str);
   s >> str;
   dev.setHost(str);   
   s >> i;
   dev.setType(i);
   s >> i;
   dev.setEndian(i);
   s >> i;
   dev.setFlags(i);
   s >> i;
   dev.setDelay(i);
   s >> i;
   dev.setSpeed(i);
   s >> i;
   dev.setDatabits(i);
   s >> i;
   dev.setStopbits(i);
   s >> i;
   if(i!=255){
      dev.setParity(i);
//      dev.setTimeout(10);
      dev.setNTimeout(10);
   }
   else{
      s >> i;
      dev.setParity(i);
      s >> i;
      dev.setTimeout(i);
      s >> i;
      dev.setNTimeout(i);
      s >> i;
      if(i==255){
	 s >> i;
	 dev.setAddress(i);
	 s >> ch;
	 dev.setEos(ch);
	 s >> ch;
	 dev.setMaster(ch);
	 s >> i;
      }
      else if(i==254){
	 s >> i;
	 dev.setAddress(i);
	 s >> i;
	 dev.setEos(i);
	 s >> ch;
	 dev.setMaster(ch);
	 s >> i;
      }

   
   }

   return(s);
}

QDataStream &operator>>( QDataStream &s, QList<SamplinDevice> &devs)
{
   uint count,i;
   SamplinDevice *p;
   
   devs.clear();
   s >> count;
   for(i=0;i<count;++i){
      p = new SamplinDevice();
      s >> *p;
      devs.append(p);
   }
   
   return(s);
}

QDataStream &operator<<( QDataStream &s, QList<SamplinDevice> &devs)
{
   SamplinDevice *p;
   
   s << devs.count();
   for(p=devs.first();p!=0;p=devs.next())
     s << *p;
   
   return(s);
}

