/**************************************************************************
 * $Id: PropBox.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "PropBox.h"
#include <qfontmetrics.h>
#include <qpainter.h>
#include <qstrlist.h>
#include <qkeycode.h>
#include <qscrollbar.h>
#include <qpixmap.h>
#include <qapplication.h>

#include "PropBox.moc"

Q_DECLARE(QListM, PropBoxItem);

class QLBItemList : public QListM(PropBoxItem) // internal class
{
    int compareItems( GCI i1, GCI i2);
public:
    int timerId;				//### bincomp
};

int QLBItemList::compareItems( GCI i1, GCI i2)
{
    PropBoxItem *lbi1 = (PropBoxItem *)i1;
    PropBoxItem *lbi2 = (PropBoxItem *)i2;
    return strcmp( lbi1->text(), lbi2->text() );
}


static inline bool checkInsertIndex( const char *method, const char * name,
				     int count, int *index)
{
    bool range_err = (*index > count);
#if defined(CHECK_RANGE)
    if ( *index > count )
	warning( "x PropBox::%s: (%s) Index %i out of range",
		 method, name ? name : "<no name>", *index );
#endif
    if ( *index < 0 )				// append
	*index = count;
    return !range_err;
}

static inline bool checkIndex( const char *method, const char * name,
			       int count, int index )
{
    bool range_err = ((uint)index >= (uint)count);
#if defined(CHECK_RANGE)
    if ( range_err )
	warning( "y PropBox::%s: (%s) Index %d out of range",
		 method, name ? name : "<no name>", index );
#endif
    return !range_err;
}

PropBoxItem::PropBoxItem()
{
    selected = FALSE;
}

PropBoxItem::~PropBoxItem()
{
}

PropBoxText::PropBoxText( const char *text )
    :PropBoxItem()
{
    setText( text );
}

PropBoxText::~PropBoxText()
{
}

void PropBoxText::paint( QPainter *p )
{
    QFontMetrics fm = p->fontMetrics();
    p->drawText( 3,  fm.ascent() + fm.leading()/2, text() );
}

int PropBoxText::height( const PropBox *lb ) const
{
    if ( lb )
	return lb->fontMetrics().lineSpacing() + 2;
    return -1;
}

int PropBoxText::width( const PropBox *lb ) const
{
    if ( lb )
	return lb->fontMetrics().width( text() ) + 6;
    return -1;
}

PropBoxPixmap::PropBoxPixmap( const QPixmap &pixmap )
    : PropBoxItem()
{
    pm = pixmap;
}

PropBoxPixmap::~PropBoxPixmap()
{
}

void PropBoxPixmap::paint( QPainter *p )
{
    p->drawPixmap( 3, 0, pm );
}

int PropBoxPixmap::height( const PropBox * ) const
{
    return pm.height();
}

int PropBoxPixmap::width( const PropBox * ) const
{
    return pm.width() + 6;
}

//### How to provide new member variables while keeping binary compatibility:

//#if QT_VERSION == 200
//#error "Remove QListBox dict."
//#endif

#include "qintdict.h"

static QIntDict<int> *qlb_maxLenDict = 0;

static void cleanupListbox()
{
    delete qlb_maxLenDict;
    qlb_maxLenDict = 0;
}

PropBox::PropBox( QWidget *parent, const char *name, WFlags f )
    : QTableView( parent, name, f )
{
    doDrag	  = TRUE;
    doAutoScroll  = TRUE;
    current	  = -1;
    isTiming	  = FALSE;
    stringsOnly	  = TRUE;
    multiSelect   = FALSE;
    goingDown	  = FALSE;
    itemList	  = new QLBItemList;
    CHECK_PTR( itemList );
    itemList->timerId = 0;
    setCellWidth( 0 );
    QFontMetrics fm = fontMetrics();
    setCellHeight( fm.lineSpacing() + 1 );
    setNumCols( 1 );
    setTableFlags( Tbl_autoVScrollBar|Tbl_autoHScrollBar | //Tbl_snapToVGrid |
		   Tbl_smoothVScrolling | Tbl_clipCellPainting  );
    switch ( style() ) {
	case WindowsStyle:
	case MotifStyle:
	    setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
	    setBackgroundMode( PaletteBase );
	    break;
	default:
	    setFrameStyle( QFrame::Panel | QFrame::Plain );
	    setLineWidth( 1 );
    }
    setFocusPolicy( StrongFocus );
    if ( !qlb_maxLenDict ) {
	qlb_maxLenDict = new QIntDict<int>;
	CHECK_PTR( qlb_maxLenDict );
	qAddPostRoutine( cleanupListbox );
    }
}

//#if QT_VERSION == 200
//#error "Remove QListBox pointer."
//#endif

static PropBox * changedListBox = 0;

PropBox::~PropBox()
{
    if ( changedListBox == this )
	changedListBox = 0;
    goingDown = TRUE;
    clearList();
    if ( qlb_maxLenDict )
	qlb_maxLenDict->remove( (long)this );
    delete itemList;
}

void PropBox::setFont( const QFont &font )
{
    QWidget::setFont( font );
    if ( stringsOnly )
	setCellHeight( fontMetrics().lineSpacing() + 1 );
    // else ...?

    updateCellWidth();
}

uint PropBox::count() const
{
    return itemList->count();
}

void PropBox::insertStrList( const QStrList *list, int index )
{
    if ( !checkInsertIndex( "insertStrList", name(), count(), &index ) )
	return;
    if ( !list ) {
#if defined(CHECK_NULL)
	ASSERT( list != 0 );
#endif
	return;
    }
    QStrListIterator it( *list );
    const char *txt;
    if ( index < 0 )
	index = itemList->count();
    while ( (txt=it.current()) ) {
	++it;
	insert( new PropBoxText(txt), index++, FALSE );
    }
    if ( currentItem() < 0 && numRows() > 0 && hasFocus() )
	setCurrentItem( 0 );
    updateNumRows( TRUE );
    if ( autoUpdate() && isVisible() )
	repaint();
}

void PropBox::insertStrList( const char **strings, int numStrings, int index )
{
    if ( !checkInsertIndex( "insertStrList", name(), count(), &index ) )
	return;
    if ( !strings ) {
#if defined(CHECK_NULL)
	ASSERT( strings != 0 );
#endif
	return;
    }
    if ( index < 0 )
	index = itemList->count();
    int i = 0;
    while ( (numStrings<0 && strings[i]!=0) || i<numStrings ) {
	insert( new PropBoxText(strings[i]), index + i, FALSE );
	i++;
    }
    updateNumRows( TRUE );
    if ( currentItem() < 0 && numRows() > 0 && hasFocus() )
	setCurrentItem( 0 );
    if ( autoUpdate() && isVisible() )
	repaint();
}

void PropBox::insertItem( const PropBoxItem *lbi, int index )
{
   if ( !checkInsertIndex( "insertItem", name(), count(), &index ) )
	return;
    if ( !lbi ) {
#if defined ( CHECK_NULL )
	ASSERT( lbi != 0 );
#endif
	return;
    }
    if ( stringsOnly ) {
	stringsOnly = FALSE;
	setCellHeight( 0 );
    }
   insert( lbi, index, TRUE );
   updateNumRows( FALSE );
   if ( autoUpdate() ){
      if ( currentItem() < 0 && numRows() > 0 && hasFocus() )
	setCurrentItem( 0 );

      repaint();
   }
}

void PropBox::insertItem( const char *text, int index )
{
    if ( !checkInsertIndex( "insertItem", name(), count(), &index ) )
	return;
    if ( !text ) {
#if defined ( CHECK_NULL )
	ASSERT( text != 0 );
#endif
	return;
    }
    insert( new PropBoxText(text), index, TRUE );
    updateNumRows( FALSE );
    if ( currentItem() < 0 && numRows() > 0 && hasFocus() )
	setCurrentItem( 0 );
    if ( autoUpdate() && itemVisible(index) ) {
	int x, y;
	colXPos( 0, &x );
	rowYPos( index, &y );
	repaint( x, y, -1, -1 );
    }
}

void PropBox::insertItem( const QPixmap &pixmap, int index )
{
    if ( !checkInsertIndex( "insertItem", name(), count(), &index ) )
	return;
    if ( stringsOnly ) {
	stringsOnly = FALSE;
	setCellHeight( 0 );
    }
    insert( new PropBoxPixmap(pixmap), index, TRUE );
    updateNumRows( FALSE );
    if ( currentItem() < 0 && numRows() > 0 && hasFocus() )
	setCurrentItem( 0 );
    if ( autoUpdate() && itemVisible(index) ) {
	int x, y;
	colXPos( index, &x );
	rowYPos( index, &y );
	repaint( x, y, -1, -1 );
    }
}

void PropBox::inSort( const PropBoxItem *lbi )
{
    if ( !lbi->text() ) {
#if defined (CHECK_NULL)
	ASSERT( lbi->text() != 0 );
#endif
	return;
    }

    itemList->inSort( lbi );
    int index = itemList->at();
    itemList->remove();
    insertItem( lbi, index );
}

void PropBox::inSort( const char *text )
{
    if ( !text ) {
#if defined ( CHECK_NULL )
	ASSERT( text != 0 );
#endif
	return;
    }
    PropBoxText lbi( text );
    itemList->inSort(&lbi);
    int index = itemList->at();
    itemList->remove();
    insertItem( text, index );
}

void PropBox::removeItem( int index )
{
    if ( !checkIndex( "removeItem", name(), count(), index ) )
	return;
    bool currentChanged = ( current == index );

    if ( current >= index && current > 0 )
	current--;
    bool    updt = autoUpdate() && itemVisible( index );
    PropBoxItem *lbi = itemList->take( index );
    int w             = lbi->width( this );
    updateNumRows( w == cellWidth() );
    delete lbi;
    if ( count() == 0 ) {
	current = -1;
    } else if ( currentChanged ) {
	QString tmp = 0;
	if ( item( currentItem() ) )
	    tmp = item( currentItem() )->text();
	emit highlighted( current );
	if ( !tmp.isNull() )
	    emit highlighted( tmp );
    }
    if ( updt )
	repaint();
}

void PropBox::clear()
{
    clearList();
    updateNumRows( TRUE );
    if ( autoUpdate() )
	erase();
}

const char *PropBox::text( int index ) const
{
    if ( (uint)index >= count() )
	return 0;
    return itemList->at(index)->text();
}

const QPixmap *PropBox::pixmap( int index ) const
{
    if ( (uint)index >= count() )
	return 0;
    return itemList->at(index)->pixmap();
}

void PropBox::changeItem( const char *text, int index )
{
    if ( !checkIndex( "changeItem", name(), count(), index ) )
	return;
    change( new PropBoxText(text), index );
}

void PropBox::changeItem( const QPixmap &pixmap, int index )
{
    if ( !checkIndex( "changeItem", name(), count(), index ) )
	return;
    change( new PropBoxPixmap(pixmap), index );
}

void PropBox::changeItem( const PropBoxItem *lbi, int index )
{
    if ( !checkIndex( "changeItem", name(), count(), index ) )
	return;
    change( lbi, index );
}

bool PropBox::autoUpdate() const
{
    return QTableView::autoUpdate();
}

void PropBox::setAutoUpdate( bool enable )
{
    QTableView::setAutoUpdate( enable );
}

int PropBox::numItemsVisible() const
{
    return (lastRowVisible() - topCell() + 1);
}

int PropBox::currentItem() const
{
    return current;
}

void PropBox::setCurrentItem( int index )
{

   if ( index == current )
	return;
    if ( !checkIndex( "setCurrentItem", name(), count(), index ) )
	return;
    int oldCurrent = current;
    current	   = index;
    updateItem( oldCurrent );
    updateItem( current, FALSE ); // Do not clear, current marker covers item
//    QString tmp = 0;
//    if(currentItem()!=-1)
//     if ( item( currentItem() ) )
//	tmp = item( currentItem() )->text();
    emit highlighted( current );
//    if ( !tmp.isNull() )
//	emit highlighted( tmp );
}

void PropBox::centerCurrentItem()
{
    int top;
    if ( stringsOnly )
	top = current - numItemsVisible() / 2; // ###
    else
	top = current - numItemsVisible() / 2;
    if ( top < 0 )
	top = 0;
    int max = maxRowOffset();
    if ( top > max )
	top = max;
    setTopItem( top );
}

int PropBox::topItem() const
{
    return topCell();
}

void PropBox::setTopItem( int index )
{
    setTopCell( index );
}

bool PropBox::dragSelect() const
{
    return doDrag;
}

void PropBox::setDragSelect( bool enable )
{
    doDrag = enable;
}

bool PropBox::autoScroll() const
{
    return doAutoScroll;
}

void PropBox::setAutoScroll( bool enable )
{
    doAutoScroll = enable;
}

bool PropBox::autoScrollBar() const
{
    return testTableFlags( Tbl_autoVScrollBar );
}

void PropBox::setAutoScrollBar( bool enable )
{
    if ( enable )
	setTableFlags( Tbl_autoVScrollBar );
    else
	clearTableFlags( Tbl_autoVScrollBar );
}

bool PropBox::scrollBar() const
{
    return testTableFlags( Tbl_vScrollBar );
}

void PropBox::setScrollBar( bool enable )
{
    if ( enable )
	setTableFlags( Tbl_vScrollBar );
    else
	clearTableFlags( Tbl_vScrollBar );
}

bool PropBox::autoBottomScrollBar() const
{
    return testTableFlags( Tbl_autoHScrollBar );
}

void PropBox::setAutoBottomScrollBar( bool enable )
{
    if ( enable )
	setTableFlags( Tbl_autoHScrollBar );
    else
	clearTableFlags( Tbl_autoHScrollBar );
}

bool PropBox::bottomScrollBar() const
{
    return testTableFlags( Tbl_hScrollBar );
}

void PropBox::setBottomScrollBar( bool enable )
{
    if ( enable )
	setTableFlags( Tbl_hScrollBar );
    else
	clearTableFlags( Tbl_hScrollBar );
}

bool PropBox::smoothScrolling() const
{
    return testTableFlags( Tbl_smoothVScrolling );
}

void PropBox::setSmoothScrolling( bool enable )
{
    if ( enable )
	setTableFlags( Tbl_smoothVScrolling );
    else
	clearTableFlags( Tbl_smoothVScrolling );
}

PropBoxItem *PropBox::item( int index ) const
{
    if (!checkIndex( "item", name(), count(), index ) )
	return 0;
    return itemList->at( index );
}

int PropBox::cellHeight( int index )
{
    if ( stringsOnly )
	return QTableView::cellHeight();
   PropBoxItem *lbi = item( index );
    return lbi ? lbi->height(this) : 0;
}

int PropBox::itemHeight() const
{
   return stringsOnly ? ((PropBox*)this)->cellHeight( 0 ) : -1;
}

int PropBox::itemHeight( int index ) const
{

   int ret= ((PropBox*)this)->cellHeight( index );

   return ret;
}


bool PropBox::itemVisible( int index )
{
    return rowIsVisible( index );
}

void PropBox::paintCell( QPainter *p, int row, int col )
{

   PropBoxItem *lbi = itemList->at( row );
    if ( !lbi )
	return;
    QColorGroup g = colorGroup();
    if ( isSelected( row ) ) {
	QColor	 fc;				// fill color
	if ( style() == WindowsStyle )
	    fc = QApplication::winStyleHighlightColor();
	else
	    fc = g.text();
	p->fillRect( 0, 0, cellWidth(col), cellHeight(row), fc );
	p->setPen( style() == WindowsStyle ? white : g.base() );
	p->setBackgroundColor( fc );
    } else {
	p->setBackgroundColor( g.base() );
	p->setPen( g.text() );
    }
    lbi->paint( p );
    if ( current == row && hasFocus() ) {
	if ( style() == WindowsStyle ) {
	    p->drawWinFocusRect( 1, 1, cellWidth(col)-2 , cellHeight(row)-2,
				 QApplication::winStyleHighlightColor() );
	} else {
	    if ( isSelected( row ) )
		p->setPen( g.base() );
	    else
		p->setPen( g.text() );
	    p->setBrush( NoBrush );
	    p->drawRect( 1, 1, cellWidth(col)-2 , cellHeight(row)-2 );
	}
    }
    p->setBackgroundColor( g.base() );
    p->setPen( g.text() );
}

void PropBox::mousePressEvent( QMouseEvent *e )
{
   int itemClicked = findItem( e->pos().y() );
   
      if ( itemClicked != -1 ) {
	 setCurrentItem( itemClicked );
	 toggleCurrentItem();
	 emit mouseClick(e,itemClicked);
      }
   else if ( contentsRect().contains( e->pos() ) &&
	    lastRowVisible() >= (int) count() ) {
      setCurrentItem( count()-1 );
      toggleCurrentItem();
      emit mouseClick(e,count()-1);
   }
}

void PropBox::mouseReleaseEvent( QMouseEvent *e )
{
    if ( doDrag )
	mouseMoveEvent( e );
    if ( isTiming ) {
	killTimer( itemList->timerId );
	isTiming = FALSE;
    }
    emitChangedSignal( FALSE );
}

void PropBox::mouseDoubleClickEvent( QMouseEvent *e )
{
    mouseReleaseEvent( e );
    if ( currentItem() >= 0 ) {
	ASSERT(item( currentItem() ));
       QString tmp = item( currentItem() )->text();
	emit selected( currentItem());
	if ( !tmp.isNull() )
	    emit selected( tmp );
    }
}

void PropBox::mouseMoveEvent( QMouseEvent *e )
{
    if ( doDrag && (e->state() & (RightButton|LeftButton|MidButton)) != 0 ) {
	int itemClicked = findItem( e->pos().y() );
	if ( itemClicked >= 0 ) {
	    if ( isTiming ) {
		killTimer( itemList->timerId );
		isTiming = FALSE;
	    }
	    if ( multiSelect ) {
		bool s = currentItem() >= 0
			 ? isSelected( currentItem() ) : TRUE;
		int i = QMIN( itemClicked, currentItem() );
		if ( i < 0 )
		    i = 0;
		while( i <= itemClicked || i <= currentItem() ) {
		    setSelected( i, s );
		    i++;
		}
	    }
	    setCurrentItem( itemClicked );	// already current -> return
	    return;
	} else {
	    if ( !doAutoScroll )
		return;
	    if ( e->pos().y() < frameWidth() )
		scrollDown = FALSE;
	    else
		scrollDown = TRUE;
	    if ( !isTiming ) {
		isTiming = TRUE;
		itemList->timerId = startTimer( 100 );
	    }
	}
    }
}

void PropBox::keyPressEvent( QKeyEvent *e )
{
    if ( numRows() == 0 )
	return;
    if ( currentItem() < 0 )
	setCurrentItem( topItem() );

    int pageSize, delta;

    switch ( e->key() ) {
    case Key_Up:
	if ( currentItem() > 0 ) {
	    ensureCurrentVisible( currentItem() - 1 );
	    if ( e->state() & ShiftButton )
		toggleCurrentItem();
	}
	e->accept();
	break;
    case Key_Down:
	if ( currentItem() < (int)count() - 1 ) {
	    ensureCurrentVisible( currentItem()+1 );
	    if ( e->state() & ShiftButton )
		toggleCurrentItem();
	}
	e->accept();
	break;
    case Key_Next:
	if ( style() == MotifStyle ) {
	    delta = currentItem() - topItem();
	    pageSize = QMAX( 1, lastRowVisible() - 1 - topItem() );
	    setTopItem( QMIN( topItem() + pageSize, (int)count() - 1 ) );
	    setCurrentItem( QMIN( topItem() + delta, (int)count() - 1 ) );
	} else {
	    pageSize = QMAX( 1, lastRowVisible() - 1 - topItem() );
	    setTopItem( QMIN(currentItem(),(int)count()-pageSize+1) );
	    setCurrentItem( QMIN(lastRowVisible(), (int)count()-1) );
	}
	e->accept();
	break;
    case Key_Prior:
	if ( style() == MotifStyle ) {
	    delta = currentItem() - topItem();
	    pageSize = QMAX( 1, lastRowVisible() - 1 - topItem() );
	    setTopItem( QMAX( topItem() - pageSize, 0 ) );
	    setCurrentItem( QMAX( topItem() + delta, 0 ) );
	} else {
	    pageSize = QMAX( 1, lastRowVisible() - topItem() );
	    setTopItem( QMAX(0,currentItem()-pageSize+1) );
	    setCurrentItem( topItem() );
	}
	e->accept();
	break;

    case Key_Space:
	 {
	    int yc,cur;
	    cur=currentItem();
	    toggleCurrentItem();
	    e->accept();
	    rowYPos(cur,&yc);
	    emit mouseClick(&QMouseEvent(Event_MouseButtonPress,QPoint(width()/2,yc+itemHeight(cur)),RightButton,0),cur);
	    break;
	 }
       
    case Key_Return:
    case Key_Enter:
	if ( currentItem() >= 0 ) {
	    QString tmp = item( currentItem() )->text();
	    emit selected( currentItem());
	    if ( !tmp.isEmpty() )
		emit selected( tmp );
	}
	// do NOT accept here.  qdialog.
	break;
    default:
	break;
    }
    emitChangedSignal( FALSE );
}

void PropBox::focusInEvent( QFocusEvent * )
{
    emitChangedSignal( FALSE );
    if ( currentItem() < 0 && numRows() > 0 )
	setCurrentItem( topItem() );
    updateCell( currentItem(), 0); //show focus
}

void PropBox::focusOutEvent( QFocusEvent * )
{
    emitChangedSignal( FALSE );
    if ( currentItem() >= 0 )
	updateCell( currentItem(), 0); //show lack of focus
}

void PropBox::resizeEvent( QResizeEvent *e )
{
    QTableView::resizeEvent( e );
    setCellWidth( QMAX((int)maxItemWidth(), viewWidth()) );
}

void PropBox::timerEvent( QTimerEvent * )
{
    if ( scrollDown ) {
	if ( currentItem() + 1 < (int)count() ) {
	    int y = QMAX(currentItem(),lastRowVisible())+1;
	    if ( y >= (int)count() )
		y = count() - 1;
	    if ( currentItem() >= 0 && multiSelect ) {
		bool s = isSelected( currentItem() );
		int i = currentItem();
		while( i <= y ) {
		    setSelected( i, s );
		    i++;
		}
	    }
	    ensureCurrentVisible( y );
	}
    } else {
	if ( topItem() > 0 ) {
	    setTopItem( topItem() - 1 );
	    if ( currentItem() > 0 && multiSelect ) {
		bool s = isSelected( currentItem() );
		int i = currentItem();
		while( i >= topItem() ) {
		    setSelected( i, s );
		    i--;
		}
	    }
	    setCurrentItem( topItem() );
	}
    }
}

bool PropBox::itemYPos( int index, int *yPos ) const
{

    return rowYPos( index, yPos );
}

int PropBox::findItem( int yPos ) const
{
    return findRow( yPos );
}

void PropBox::updateItem( int index, bool erase )
{
    updateCell( index, 0,  erase );
}

void PropBox::clearList()
{
    stringsOnly = TRUE;
    PropBoxItem *lbi;
    while ( itemList->count() ) {
	lbi = itemList->take( 0 );
	delete lbi;
    }
    if ( goingDown || QApplication::closingDown() )
	return;
    bool a = autoUpdate();
    setAutoUpdate( FALSE );
    updateNumRows( TRUE );
    current = -1;
    setTopCell( 0 );
    setAutoUpdate( a );
}

void PropBox::updateCellWidth()
{
    PropBoxItem *lbi = itemList->first();
    int maxW = 0;
    int w;
    while ( lbi ) {
	w = lbi->width( this );
	if ( w > maxW )
	    maxW = w;
	lbi = itemList->next();
    }
    setMaxItemWidth( maxW );
    setCellWidth( QMAX( maxW, viewWidth() ) );
}

void PropBox::insert( const PropBoxItem *lbi, int index,
		       bool updateCellWidth )
{
#if defined(CHECK_RANGE)
    ASSERT( lbi );
    ASSERT( (uint)index <= itemList->count() );
#endif
    itemList->insert( index, lbi );
    if ( current == index )
	current++;
    if ( updateCellWidth ) {
	int w = lbi->width( this );
	if ( w > maxItemWidth() )
	    setMaxItemWidth( w );
	if ( w > cellWidth() )
	    setCellWidth( w );
    }
}

void PropBox::change( const PropBoxItem *lbi, int index )
{
#if defined(CHECK_RANGE)
    ASSERT( lbi );
    ASSERT( (uint)index < itemList->count() );
#endif

    PropBoxItem *old = itemList->take( index );
    int w = old->width( this );
    int h = old->height( this );
    delete old;
    itemList->insert( index, lbi );
    if ( w == cellWidth() ) {		     // I.e. index was the widest item
	updateCellWidth();
    }
    else {
	int ww = lbi->width( this );
	if ( ww > maxItemWidth() )
	    setMaxItemWidth( ww );
	if ( ww > cellWidth() )
	    setCellWidth( ww );
    }
   int nh = cellHeight( index );
   int y;
    if ( autoUpdate() && rowYPos( index, &y ) ) {
	if ( nh == h )
	    repaint( frameWidth(), y, viewWidth(), h );
	else
	    repaint( frameWidth(), y, viewWidth(), viewHeight() - y );
    }
}

void PropBox::updateNumRows( bool updateWidth )
{
    bool autoU = autoUpdate();
    if ( autoU )
	setAutoUpdate( FALSE );
    bool sbBefore = testTableFlags( Tbl_vScrollBar );
    setNumRows( itemList->count() );
    if ( updateWidth || sbBefore != testTableFlags(Tbl_vScrollBar) )
	updateCellWidth();
    if ( autoU )
	setAutoUpdate( TRUE );
}

long PropBox::maxItemWidth() const
{
    if ( !qlb_maxLenDict )
	return 0;
    return (long) qlb_maxLenDict->find( (long)this );
}

long PropBox::maxItemWidth()
{
    // This is only here for binary compatibility
    if ( !qlb_maxLenDict )
	return 0;
    return (long) qlb_maxLenDict->find( (long)this );
    // This is only here for binary compatibility
}

void PropBox::setMaxItemWidth( int len )
{
    ASSERT( qlb_maxLenDict );
    qlb_maxLenDict->remove( (long)this );
    if ( len )
	qlb_maxLenDict->insert( (long)this, (int*)len );
}

void PropBox::setMultiSelection( bool enable )
{
    if ( enable != (bool)multiSelect ) {
	multiSelect = enable;
	update();
    }
}

void PropBox::toggleCurrentItem()
{
    if ( !multiSelect || currentItem() < 0 )
	return;

   PropBoxItem * i = item( currentItem() );
   if ( !i )
	return;

    i->selected = !i->selected;
    updateItem( currentItem() );
    emitChangedSignal( TRUE );
}

void PropBox::setSelected( int index, bool select )
{
    if ( !multiSelect ) {
	if ( select ) {
	    setCurrentItem( index );
	} else {
	    if ( index == current )
		clearSelection();
	}
	return;
    }

    if ( currentItem() < 0 )
	return;

   PropBoxItem *lbi = item( index );
    if ( !lbi || lbi->selected == select )
	return;

    lbi->selected = select;
    updateItem( index );
    emitChangedSignal( TRUE );
}

bool PropBox::isSelected( int i ) const
{
    if ( !multiSelect )
	return i == current;

   PropBoxItem * lbi = item( i );
    return lbi ? lbi->selected : FALSE;
}

void PropBox::clearSelection()
{
    if ( multiSelect ) {
	for ( int i = 0; i < (int)count(); i++ )
	    setSelected( i, FALSE );
    } else {
	int i = current;
	if ( hasFocus() ) {
	    current = 0;
	    updateItem( current );
	} else {
	    current = -1;
	}
	updateItem( i );
    }
}

void PropBox::emitChangedSignal( bool lazy ) {
    if ( !multiSelect )
	return;

    if ( changedListBox && (!lazy || changedListBox != this) )
	emit changedListBox->selectionChanged();

    changedListBox = lazy ? this : 0;
}

void PropBox::ensureCurrentVisible( int newCurrent )
{
    if ( newCurrent < 0 )
	newCurrent = currentItem();
    if ( newCurrent < topItem() ) {
	setTopItem( newCurrent );
    } else if ( newCurrent >= lastRowVisible() ) {
	int h = viewHeight();
	int i = newCurrent;
	while( i > 0 && h > 0 ) {
	   h -= cellHeight( i );
	    if ( h >= 0 )
		i--;
	}
	if ( h <= 0 && i < newCurrent ) // make sure ALL of y is visible
	    i++;
	setTopItem( i );
    }
    if ( newCurrent != currentItem() )
	setCurrentItem( newCurrent );
}

void PropBox::setFixedVisibleLines( int lines )
{
    int ls = fontMetrics().lineSpacing() + 1; // #### explain +1
    // #### What about auto-scrollbars?
    int sb = testTableFlags(Tbl_hScrollBar)
		? horizontalScrollBar()->height() : 0;
    setFixedHeight( frameWidth()*2 + ls*lines + sb );
    return;
}

QSize PropBox::sizeHint() const
{
    QSize sz = QTableView::sizeHint();

    int w = (int)maxItemWidth() + 2*frameWidth();
    if ( testTableFlags(Tbl_vScrollBar) )
	w += verticalScrollBar()->width();
    sz.setWidth(w);

    // For when setFixedVisibleLines is used
    int h = maximumSize().height();
    if ( h<1000 ) sz.setHeight(h);

    return sz;
}
