#include <qwt_math.h>
#include "qwt_autoscl.h"

//------------------------------------------------------------
//.H QwtAutoScale|3|27/04/97|Qwt Widget Library|Qwt Programmer's manual
//.S NAME
//  QwtAutoScale - The Qwt Autoscaler
//
//.S SYNOPSIS
//  #include <qwt_autoscl.h>
//
//.S DESCRIPTION
//	This class can be  used to generate a scale which may span 
//	multiple ranges of values. A scale consists of a lower boundary,
//	an upper boundary, and a step size which divides it into subintervals.
//	The tick marks of the scale are the boundaries of these subintervals.
//	A quick look at the \l{#EXAMPLE}{example} below will give you an idea
//	of how the autoscaler works.
//.P
//	The autoscaler produces "reasonable"  step sizes. For linear scales,
//	the step size will fit into the scheme {1,2,5}*10^n,
//	where n is an integer. In \l{#QwtScale::setOptions}{logarithmic}
//	mode, the step size is measured in *decades*
//	and will be adjusted to fit the scheme {1,2,3,5}*10^n,
//	where n is a natural number including zero.
//.P
//	The step size can be manipulated indirectly using \r{QwtScale::setMaxIntv}.
//.P
//	Using the \r{QwtAutoScale::setOptions} member function,
//	QwtAutoScale can be configured to
//	create different layouts of scales.
//.P
//	The autoscaling algorithm can be partly or completely disabled
//	(even temporarily) if a user-defined scale is desired. This can be done with
//	the \r{QwtAutoscale::setFixedScale} function. It can be switched off
//	again with \r{QwtScale::setAutoScale}.
//.P
//	The two \l{QwtAutoScale::adjust (1)}{QwtAutoScale::adjust}
//	members are used to extend
//	the scale if necessary in order to include another range or array of
//	values. The \r{QwtAutoScale::reset} resets the scale to zero.
//.P
//	In order to determine the values of scale tick marks, the
//	\r{QwtAutoScale::firstMark}, \r{QwtAutoScale::lastMark},
//	\r{QwtAutoScale::markValue}, and \r{QwtAutoScale::markCnt}
//	members are available.
//
//.S PUBLIC MEMBERS 
//.R
//	QwtAutoScale::QwtAutoScale	-- constructor
//	QwtAutoScale::~QwtAutoScale	-- destructor
//    
//	QwtAutoScale::setMaxIntv	-- set maximal number of subdivisions
//	QwtAutoScale::setFixedScale	-- set a fixed scale range and (optionally)
//					   step size
//	QwtAutoScale::setAutoScale	-- re-enable autoscaling
//	QwtAutoScale::adjust (1)	-- adjust the scale in order to include
//					   all values in a specified array
//	QwtAutoScale::adjust (2)	-- adjust the scale in order to include
//					   a specified range
//	QwtAutoScale::reset		-- reset the scale to zero
//	QwtAutoScale::setOptions	-- specify options
//	QwtAutoScale::setMargins (1)	-- specify boundary margins
//	QwtAutoScale::setMargins (2)	-- specify boundary margins
//	QwtAutoScale::setReference 	-- specify reference point
//	QwtAutoScale::scaleMin		-- return lower boundary
//	QwtAutoScale::scaleMax		-- return upper boundary
//	QwtAutoScale::step		-- return step size
//	QwtAutoScale::firstMark		-- return first scale tick
//	QwtAutoScale::lastMark		-- return last scale tick
//	QwtAutoScale::markValue		-- return value of a scale mark with
//					   a specified index
//	QwtAutoScale::markCnt		-- return number of scale ticks
//
//.S    EXAMPLE
//.c
//      #include<qwt_autoscl.h>
//      #include<iostream.h>
//
//      double x1[100];
//      double x2[200];
//      double range_min, range_max;
//
//      QwtAutoScale as;
//
//      ... determine x1 and x1, range_min and range_max here ...
//
//      as.reset();                             // clear it
//      as.adjust(range_min, range_max);      // include a range
//      as.adjust(x1,100);                    // include an array
//      as.adjust(x2,200);                    // include another array
//
//      for (i=0;i<as.scaleDiv().majCnt(); i++)
//      {
//              cout << "Scale tick " << i << " at " << as.scaleDiv().majMark(i) << "\n";
//      }       
//
//.S	NOTE
//	For logarithmic scales, the step size as well as
//	the margins are measured in *decades*.
//------------------------------------------------------------

//------------------------------------------------------------
//.C	MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------------

const double MinEps=1.0e-10;

//------------------------------------------------------------
//
//.S	QwtAutoScale::QwtAutoScale
//	Construct a QwtAutoScale instance
//
//.s	Description
//
//.s	Syntax
//.f	 QwtAutoScale::QwtAutoScale()
//
//------------------------------------------------------------
QwtAutoScale::QwtAutoScale ()
{
    d_autoScale = true;
    d_scaleOpt = None;
    
    d_minValue = 0.0;
    d_maxValue = 0.0;
    d_scaleMin = 0.0;
    d_scaleMax = 0.0;
    d_loMargin = 0.0;
    d_hiMargin = 0.0;
    d_step = 0.0;
    d_maxMajor = 8;
    d_maxMinor = 5;
    d_reset = 1;
    d_autoRebuild = true;
   d_ref=0;
   d_scaleInv=false;
   
}

//------------------------------------------------------------
//
//.S	QwtAutoScale::~QwtAutoScale
//	Destroy a QwtAutoScale instance
//
//.s	Syntax
//.f	 QwtAutoScale::~QwtAutoScale()
//
//------------------------------------------------------------
QwtAutoScale::~QwtAutoScale ()
{
}




//------------------------------------------------------------
//
//.S	QwtAutoScale::adjust (1)
//	Adjust the scale to a given array of input values.
//
//.s	Parameters
//.p	double *x	-- array of input values
//	int num		-- size of x
//	int reset = 0	-- if != 0 reset the scale's contents.
//
//.s	Return Value
//
//.s	Description
//	This member function extends the boundaries of the scale
//	and re-calculates the step size if necessary in order
//	to include all values in the array. If the reset
//	parameter has nonzero value, the previous state will
//	be cleared.
//
//.s	Syntax
//.f	void QwtAutoScale::adjust(double *., int num, int reset)
//
//------------------------------------------------------------
void QwtAutoScale::adjust(double *x, int num, int reset)
{
    int i;
    if (d_reset || reset)
       d_minValue = d_maxValue = x[0];
    for (i = 0; i < num; i++)
    {
	if (x[i] > d_maxValue)
	   d_maxValue = x[i];
	if (x[i] < d_minValue)
	   d_minValue = x[i];
    }
    d_reset = 0;
    if (d_autoRebuild) build();
    
}



//------------------------------------------------------------
//
//.S	QwtAutoScale::adjust (2)
//	Adjust the scale to a specified interval
//
//.s	Parameters
//.p	double vmin	--	lower range of the specified interval
//	double vmax	--	upper range of the specified interval
//	int reset	--	if nonzero, reset the scale. Defaults to 0.
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//	This member function extends the boundaries of the scale
//	and re-calculates the step size if necessary in order
//	to include a specified range of values. If the reset
//	parameter has nonzero value, the previous state will
//	be cleared.
//
//.s	Syntax
//.f	void QwtAutoScale::adjust(double vmin, double vmax, int reset)
//
//------------------------------------------------------------
void QwtAutoScale::adjust(double vmin, double vmax, int reset)
{ 
    double mxv,mnv;
    
    mxv = qwtMax(vmin,vmax);
    mnv = qwtMin(vmin,vmax);
    
    if (d_reset || reset)
    {
	d_minValue = mnv;
	d_maxValue = mxv;
    }
    else
    {
	if (d_minValue > mnv) 
	   d_minValue = mnv;
	if (d_maxValue < mxv)
	   d_maxValue = mxv;
    }
    d_reset = 0;
    if (d_autoRebuild) build();
    
}

//------------------------------------------------------------
//.-
//.S	QwtAutoScale::build
//	Re-build the scale
//
//.s	Description
//
//.s	Syntax
//.f	 void QwtAutoScale::build()
//
//------------------------------------------------------------
void QwtAutoScale::build() 
{

   if (d_reset) return;
    
    if (d_autoScale)
    {
	if (d_scaleOpt & Logarithmic) 
	   buildLogScale(); 
	else 
	   buildLinScale();
    }
    else{
       if(d_scaleInv)
	 d_scldiv.rebuild(d_scaleMax, d_scaleMin, d_maxMajor, d_maxMinor,
			  bool(d_scaleOpt & Logarithmic), d_step,d_ref); 
       else d_scldiv.rebuild(d_scaleMin, d_scaleMax, d_maxMajor, d_maxMinor,
			     bool(d_scaleOpt & Logarithmic), d_step,d_ref); 
    }
   update();
   
}


//------------------------------------------------------------
//.-
//.S	QwtAutoScale::buildLinScale
//	Build a linear scale
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void QwtAutoScale::buildLinScale()
//
//------------------------------------------------------------
void QwtAutoScale::buildLinScale ()
{

    double delta, dec, ticks, minval, maxval, step;


    ticks = double (d_maxMajor);


    //
    // If in Autoscale Mode, adjust minval and maxval according to
    // the active scale options, and add the margins
    //
    if (!d_autoScale) return;
    
    minval = d_minValue;		// scale boundaries are based on the
    maxval = d_maxValue;		// data.


   //
    // add / subtract margins
    //
    if (d_loMargin > 0.0)
       minval -= d_loMargin;
    if (d_hiMargin > 0.0)
       maxval += d_hiMargin;

    //
    //	Correct minval / maxval according to the scale options
    //
   if (d_scaleOpt & Symmetric)
     {
	delta = qwtMax(fabs(d_ref - maxval),fabs(d_ref - minval));
	maxval = d_ref + delta;
	minval = d_ref - delta;	
    }
   else if (d_scaleOpt & IncludeRef)
    {
	if (maxval < d_ref) 
	   maxval = d_ref;
	else if (minval > d_ref) 
	   minval = d_ref;
    }
    //
    // first approximation of d_scaleMin and d_scaleMax
    //
    setRange(minval, maxval);
    delta = d_scaleMax - d_scaleMin;


    // dec := maximal power of ten which fits into the interval
    //   [d_scaleMin,d_scaleMax]
    dec = pow (10.0, floor (log10 (delta)));

    //
    //	The following magic line calculates the step size such that
    //      - The number of subintervals will not exceed the maximum
    //        as specified by the user
    //	    - The step size fits {1,2,5}*10^n with a natural number n  
    // 
    step = qwtCeil125(delta * 0.999999 / dec / ticks) * dec;

    //
    //	determine he final values of scaleMin and scaleMax
    //
   
   if (! (d_scaleOpt & Floating) )
    {
       // adjust of d_scaleMin and d_scaleMax such that both are integer
       // multiples of the step size.
       d_scaleMin = step * floor ((d_scaleMin + MinEps * step) / step);
       d_scaleMax = step * ceil ((d_scaleMax - MinEps * step) / step);
    }

    if (d_scaleOpt & Inverted)
    {
//	step = -step;
	d_scldiv.rebuild(d_scaleMax, d_scaleMin, d_maxMajor, d_maxMinor,
			 false, step);
    }
    else
    {
	d_scldiv.rebuild(d_scaleMin, d_scaleMax, d_maxMajor, d_maxMinor,
			 false, step);
    }
    
}


//------------------------------------------------------------
//.-
//.S	QwtAutoScale::buildLogScale
//	build a logarithmic scale
//
//.s	Syntax
//.f	void QwtAutoScale::buildLogScale()
//
//------------------------------------------------------------
void QwtAutoScale::buildLogScale ()
{

    double ticks, decades;
    double delta, minval, maxval, fpart, ipart, step;

    if (!d_autoScale) return;

    
    minval = d_minValue;	// the calculation of scale divisions is based on the
    maxval = d_maxValue;	// input data.

    if (d_loMargin > 0.0)
       minval /= pow(10.0, d_loMargin);
    if (d_hiMargin > 0.0)
       maxval *= pow(10.0, d_hiMargin);

    if (d_scaleOpt & Symmetric)
    {
	delta = qwtMax(maxval / d_lref,  d_lref / minval); 
	maxval = d_lref * delta;
	minval = d_lref / delta;	
    }
    else if (d_scaleOpt & IncludeRef)
    {
	if (maxval < d_lref) 
	   maxval = d_lref;
	else if (minval > d_lref) 
	   minval = d_lref;
    }

    if (d_maxMajor > 0)
       ticks = double (d_maxMajor);
    else
       ticks = 1;

    setRange(minval, maxval);

    // decades included in the interval
    decades = fabs(log10 (d_scaleMax / d_scaleMin));

    // calculate step size in decades
    if ((decades > 1.0) && (decades > ticks))
    {
	// One interval contains more than one decade.
	// The number of decades in an interval is adjusted
	// to be a multiple of 2,3,5, or 10.
	fpart = modf (log10 (ceil (decades * 0.999999 / ticks)), &ipart);
	if (fpart < MinEps)
	   fpart = 1.0;
	else if ((fpart - LOG10_2) < MinEps)
	   fpart = 2.0;
	else if ((fpart - LOG10_3) < MinEps)
	   fpart = 3.0;
	else if ((fpart - LOG10_5) < MinEps)
	   fpart = 5.0;
	else
	   fpart = 10.0;

	step = pow (10, ipart) * fpart;

    }
    else				// The minimal step size is one decade.
    {
	step = 1.0;
    };
    
    if (!(d_scaleOpt & Floating))
    {
	d_scaleMin = pow (10, step * floor ((log10(d_scaleMin) + MinEps * step) / step));
	d_scaleMax = pow (10, step * ceil ((log10(d_scaleMax) - MinEps * step) / step));
    }

    if (d_scaleOpt & Inverted)
    {
	step = -step;
	d_scldiv.rebuild(d_scaleMax, d_scaleMin, d_maxMajor, d_maxMinor, true,
			 step);
    }
    else
    {
	d_scldiv.rebuild(d_scaleMin, d_scaleMax, d_maxMajor, d_maxMinor,
			 true, step);
    }

}

//------------------------------------------------------------
//
//.S	QwtAutoScale::changeOptions
//	Set or reset specified scale options
//
//.s	Parameters
//.p	int opt		--	or'ed scale options
//	bool tf 	--	If true, set the specified options.
//				If false, reset these options.
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void QwtAutoScale::changeOptions(int opt, bool addit)
//.s	See also
//      \r{QwtAutoScale::setOptions}
//------------------------------------------------------------
void QwtAutoScale::changeOptions(int opt, bool tf)
{
    if (tf)
       d_scaleOpt |= opt;
    else
       d_scaleOpt &= (!opt);
    build();
}

//------------------------------------------------------------
//
//.S	QwtAutoScale::option
//	Returns 'true' if the specified option is set.
//.s Parameters
//	int opt
//
//.s Return Type bool
//
//.s See also
//   \r{QwtAutoscale::setOptions}
//------------------------------------------------------------
bool QwtAutoScale::option(int opt) const
{
  return bool(d_scaleOpt & opt);
}

//------------------------------------------------------------
//
//.S	QwtAutoScale::reset
//	Set boundaries and step size to zero.
//
//.s	Description
//	This member function resets an AutoScale instance 
//	to its initial state. It is needed to clean up
//	the scale before or
//	after subsequent \r{QwtAutoScale::adjust} calls.
//	The boundaries of the scale and the step size
//	are set to zero.
//
//.s	Note
//	A reset doesn't affect the margins.
//
//.s	Syntax
//.f	void QwtAutoScale::reset()
//
//------------------------------------------------------------
void QwtAutoScale::reset()
{
    d_reset = true;
    d_scldiv.reset();
    d_minValue = 0;
    d_maxValue = 0;
   d_scaleMin=0;
   d_scaleMax=0;
   d_step = 0;
}


//------------------------------------------------------------
//
//.S	QwtAutoScale::setAutoScale
//	Enable autoscaling
//
//.s	Description
//	This function is used to switch back to autoscaling mode
//	if the scale has temporarily been frozen (see \rQwtAutoScale::setScale).
//.P	If autoscaling
//	is reactivated, the scale will be rebuild, which
//	means that
//.i	-- if adjust or setMaxIntv have been called in between, the scale
//	   will be adjusted to the new conditions.
//	-- if none of these functions and no reset has been called, the old state will
//		be restored.
//	-- if only reset has been called in between, nothing will happen.
//
//.s    See also
//	\r{QwtAutoScale::setScale}
//.s	Syntax
//.f	void QwtAutoScale::setAutoScale(double xmin, double xmax, double step)
//
//------------------------------------------------------------
void QwtAutoScale::setAutoScale()
{
    d_autoScale = true;
    build();
}

void QwtAutoScale::setAutoScale(bool val)
{
    d_autoScale = val;
}

//------------------------------------------------------------
//
//.S	QwtAutoScale::setMargins (1)
//	Specify margins at the scale's endpoints
//
//.s	Parameters
//.p	double mlo     --	minimum distance between
//				the scale's lower boundary
//				and the smallest enclosed value
//	double mhi	--	minimum distance between 
//				the scale's upper boundary
//				and the greatest enclosed value
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//	This function can be used to specify a minimum
//	distance between the scale's boundaries
//      and the values or range boundaries enclosed by the scale.
//	
//.s	Syntax
//.f	void QwtAutoScale::setMargins(double mlo, double mhi)
//
//.s	Note
//.i	-- With logaritmic scales, the margins are measured in
//	   decades.
//	-- The margins will not be changed by any other member function.
//	   You should remember this when you call \r{QwtAutoScale::reset}
//	   or change from a linear to a logarithmic scale. 
//	   
//------------------------------------------------------------
void QwtAutoScale::setMargins(double mlo, double mhi)
{
    d_loMargin = qwtMax(mlo,0.0);
    d_hiMargin = qwtMax(mhi,0.0);
    build();
}

//------------------------------------------------------------
//
//.S	QwtAutoScale::setMargins (2)
//	Specify margins at the scale's endpoints
//
//.s	Parameters
//.p	double m	-- minimum distance between
//			   the scale boundaries and the enclosed values
//
//.s	Description
//	In contrast to \r{QwtAutoScale::setmargins (1)}, this
//	function sets both margins to the same value.
//
//.s	Syntax
//.f	void QwtAutoScale::setMargins(double m)
//
//.s	Note
//.i	-- With logaritmic scales, the margins are measured in
//	   decades.
//	-- The margins will not be changed by any other member function.
//	   You should remember this when you call \r{QwtAutoScale::reset}
//	   or change from a linear to a logarithmic scale. 
//
//------------------------------------------------------------
void QwtAutoScale::setMargins(double m)
{
    setMargins(m,m);
}

//------------------------------------------------------------
//
//.S	QwtAutoScale::setMaxMajor
//	Specify the maximum number of major intervals
//
//.s	Parameters
//.p	int mx		--	maximum number of subintervals
//
//.s	Description
//	The maximum number of subintervals is used by the autoscaler
//	in order to determine the step width.
//
//.s	Syntax
//.f	void QwtAutoScale::setMaxMajor(int mx)
//
//------------------------------------------------------------
void QwtAutoScale::setMaxMajor(int mx)
{
    d_maxMajor = qwtMax(mx,1);
    d_maxMajor = qwtMin(mx, 10000);
    build();
}

//------------------------------------------------------------
//
//.S	QwtAutoScale::setMaxMinor
//	Specify the maximum number of minor scale intervals
//
//.s	Parameters
//.p	int mx
//
//.s	Description
//
//.s	Syntax
//.f	void QwtAutoScale::setMaxMinor(int mx)
//
//------------------------------------------------------------
void QwtAutoScale::setMaxMinor(int mx)
{
    d_maxMinor = qwtMin(qwtMax(mx,0), 100);
    build();
}


//------------------------------------------------------------
//.-
//.S	QwtAutoScale::setRange 
//
//
//.s	Parameters
//.p	double x1, double x2
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void QwtAutoScale::setRange(double x1, double x2)
//
//------------------------------------------------------------
void QwtAutoScale::setRange(double x1, double x2)
{
    double delta;
    double minval, maxval;
    

     
   minval = qwtMin(x1, x2);
   maxval = qwtMax(x1, x2);

    if (d_scaleOpt & Logarithmic)
    {
	minval = qwtMin(qwtMax(minval, LOG_MIN), LOG_MAX);
	maxval = qwtMin(qwtMax(maxval,LOG_MIN), LOG_MAX);
    }
    
    delta = maxval - minval;
	
    if (delta <= 0.0)		// all values are equal
    {				
	if (minval > 0)
	{
	    d_scaleMin = minval * 0.5;
	    d_scaleMax = maxval * 1.5;
	}
	else if (minval < 0)
	{
	    d_scaleMin = minval * 1.5;
	    d_scaleMax = maxval * 0.5;
	}
	else		      // all values are zero
	{			
	    d_scaleMin = -0.5;
	    d_scaleMax = 0.5;
	}

	delta = d_scaleMax - d_scaleMin;

    }
    else			// the normal case
    {				
	d_scaleMin = minval;
	d_scaleMax = maxval;
    }

}



//------------------------------------------------------------
//
//.S	QwtAutoScale::setScale
//	Specify a user-defined scale and switch off autoscaling
//	
//
//.s	Parameters
//.p	double xmin	--	user-defined lower boundary
//	double xmax	--	user-defined upper boundary
//	double step = 0	--	user-defined step size
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//	A fixed scale may be used for different purposes, e.g.
//	zooming. If the step argument is left out or less or equal
//	to zero, the autoscaler will kindly calculate the step
//	size for you. The fixed-scale mode can switched off using
//	\r{QwtAutoScale::setAutoScale}, which restores the
//	previous values.
//
//.s	Note
//.i	-- if xmin > xmax, xmax will be the lower boundary.
//	-- if xmin == xmax, the autoscaler will not take you for
//		       serious and make a more intelligent suggestion.
//	-- Options and margins have no effect while autoscaling is switched off.
//
//.s	Syntax
//.f	void QwtAutoScale::setScale(double xmin, double xmax, double step)
//
//------------------------------------------------------------
void QwtAutoScale::setScale(double xmin, double xmax, double step)
{

    // turn autoscaling off and set the
    // scale limits to the desired values
   if(finite(xmin)&&finite(xmax)&&finite(step)){
      setRange(xmin,xmax);
      d_autoScale = false;
      d_step = qwtMin(fabs(step),fabs(xmax-xmin));
   }
   else d_autoScale=true;
   
   d_scaleInv=bool(xmin>xmax);
   // rebuild the scale
   build ();

}




//------------------------------------------------------------
//
//.S	QwtAutoScale::setOptions
//	Reset scale options and set specified options
//
//.s	Parameters
//.p	int opt		--	Combined set of options
//
//.s	Description
//	The behaviour of the autoscaling algorithm can be changed
//	with the following options:
//.t
//	QwtAutoScale::None	--	reset all options.
//	QwtAutoscale::IncludeZero --	Build a scale which includes
//					the origin (if linear) or
//					the point 10^0 (if logarithmic).
//	QwtAutoScale::ZeroSymmetric	--	Build a scale which is symmetric
//					to the origin (if linear) or
//					to the point 10^0 (if logarithmic).
//	QwtAutoScale::Logarithmic	--	Build a logarithmic scale.
//	QwtAutoScale::Floating	--	The endpoints of the scale
//					are supposed to be equal the outmost included
//					values plus the specified
//					\l{#QwtAutoScale::setMargins (1)}{margins}.
//					If this option is *not* set, the endpoints
//					of the scale will be integer multiples
//					of the step size.
//	QwtAutoScale::Inverted --	Turn the upside down.
//
//.s	Note
//.i	-- If the type of scale division is changed from logarithmic to linear
//	   or vice versa, the margins will not be transformed. Note that
//	    the margins count in decades if the scale is logarithmic.
//      -- If a linear scale contains negative values, switching to a
//	   logarithmic scale will cut them off and set the lower scale
//	   boundary to its lowest possible value of 1.0E-100. This effect
//	   is reversible if you
//	   switch back to a linear scale.
//	-- The options have no effect while autoscaling is turned off
//		(see \r{QwtAutoScale::setScale})
//
//.s	Syntax
//.f	void QwtAutoScale::setOptions(int opt)
//
//.s	Example
//.c
//	#include <qwt_autoscl.h>
//
//	void main() 
//	{
//		QwtAutoScale as;
//
//		// build linear scale with default settings
//		as.adjust(17.45, 87344.0);
//
//				...
//
//		// change to logarithmic scale with floating ends
//		as.setOptions(QwtAutoScale::Floating | QwtAutoscale::Logarithmic);
//
//				...
//
//		// change to linear, zero-symmetric scale
//		as.setOptions(QwtAutoScale::ZeroSymmetric);
//
//				...
//	}
//
//.s	See also
//	\r{QwtAutoScale::changeOptions} for a description of the
//	possible options
//------------------------------------------------------------
void QwtAutoScale::setOptions(int opt)
{
    d_scaleOpt = opt;
    build();
}


//------------------------------------------------------------
//
//.S	QwtAutoScale::setReference
//	Specify a reference point
//
//.s	Parameters
//.p	double r	--	new reference value
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//	The reference point is needed if the autoscaler options IncludeRef or
//	Symmetric are active. Its default value is 0 for linear scales and 1 for
//	logarithmic scales.
//
//.s	Note
//	The reference value for logarithmic scales is limited to
//	( LOG_MIN / 2 <= reference <= LOG_MAX /2 ). If the specified
//	value is less than LOG_MIN, it will
//	be set to *1.0* for logarithmic scales.
//
//.s	Syntax
//.f	void QwtAutoScale::setReference(double r)
//
//------------------------------------------------------------
void QwtAutoScale::setReference(double r)
{
    d_ref = r;
    
    if (r > LOG_MIN / 2) 
    {
	d_lref = qwtMin(r, LOG_MAX / 2);
    }
    else
       d_lref = 1.0;
    build();
}

void QwtAutoScale::update(void)
{
}






