#include "qwt_symbol.h"
#include <qrect.h>
#include <qpainter.h>

//------------------------------------------------------------
//.H QwtSymbol | 3 | 27/04/96 | Qwt Widget Library | Qwt Programmer's Manual
//
//.S NAME
//	QwtSymbol - A class to draw symbols
//
//.S SYNOPSIS
//	#include <qwt_symbol.h>
//
//.S PUBLIC MEMBERS 
//
//.S PROTECTED MEMBERS 
//
//.S DESCRIPTION
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.S	QwtSymbol::QwtSymbol
//	Constructor
//
//.s	Description
//	The symbol is constructed with gray interior,
//	black outline with zero width, no size and style 'None'.
//
//.s	Syntax
//.f	 QwtSymbol::QwtSymbol()
//
//------------------------------------------------------------
QwtSymbol::QwtSymbol()
: d_brush(gray), d_pen(black), d_size(0,0)
{
    d_style = QwtSymbol::None;
}



//------------------------------------------------------------
//
//.S	QwtSymbol::QwtSymbol
//
//.s	Parameters
//.p	int st -- Symbol Style
//	QBrush &bd -- brush to fill the interior
//	QPen &pn -- outline pen 
//	QSize &size --size 
//
//.s	Syntax
//.f	 QwtSymbol::QwtSymbol(int st, const QBrush &bd, const QPen &pn, const QSize &s)
//
//------------------------------------------------------------
QwtSymbol::QwtSymbol(int st, const QBrush &br, const QPen &pn, const QSize &s)
: d_brush(br), d_pen(pn), d_size(s)
{
    d_style = st;
}

//------------------------------------------------------------
//
//.S	QwtSymbol::~QwtSymbol
//	Destructor
//
//.s	Syntax
//.f	 QwtSymbol::~QwtSymbol()
//
//------------------------------------------------------------
QwtSymbol::~QwtSymbol()
{
    
}


//------------------------------------------------------------
//
//.S	QwtSymbol::setSize
//	Specify the symbol's size
//
//.s	Parameters
//.p	int w -- width
//	int h = -1 -- height
//
//.s	Description
//	If the 'h' parameter is left out or less than 0,
//	and the 'w' parameter is greater than or equal to 0,
//	the symbol size will be set to (w,w).
//
//.s	Syntax
//.f	void QwtSymbol::setSize(int size)
//
//------------------------------------------------------------
void QwtSymbol::setSize(int w, int h)
{
    if ((w >= 0) && (h < 0)) h = w;
    d_size = QSize(h,w);
    
}

//------------------------------------------------------------
//
//.S	QwtSymbol::setBrush
//	Assign a brush
//
//.s	Parameters
//.p	const QBrush &br	--	brush
//
//.s	Description
//	The brush is used to draw the interior of the symbol.
//
//.s	Syntax
//.f	void QwtSymbol::setBrush(const QBrush &br)
//
//------------------------------------------------------------
void QwtSymbol::setBrush(const QBrush &br)
{
    d_brush = br;
}

//------------------------------------------------------------
//
//.S	QwtSymbol::setPen
//	Assign a pen
//
//.s	Parameters
//.p	const QPen &pn -- pen
//
//.s	Description
//	The pen is used to draw the symbol's outline.
//
//.s	Syntax
//.f	void QwtSymbol::setPen(const QPen &pn)
//
//------------------------------------------------------------
void QwtSymbol::setPen(const QPen &pn)
{
    d_pen = pn;
}

//------------------------------------------------------------
//
//.S	QwtSymbol::draw
//	Draw the symbol at a point (x,y).
//
//.s	Parameters
//.p	QPainter *p, int x, int y
//
//.s	Description
//	The point (x,y) is the upper left corner of a rectangle with
//	the symbol's size.
//
//.s	Syntax
//.f	void QwtSymbol::draw(QPainter *p, int x, int y)
//
//------------------------------------------------------------
void QwtSymbol::draw(QPainter *p, int x, int y)
{
    QRect r (x,y,d_size.width(), d_size.height());
    draw(p, r);
}


//------------------------------------------------------------
//
//.S	QwtSymbol::draw
//	Draw the symbol into a bounding rectangle.
//
//.s	Parameters
//.p	QPainter *p, const QRect& r
//
//.s	Description
//	This function overrides the symbol's size settings,
//	but it doesn't modify them.
//
//.s	Syntax
//.f	void QwtSymbol::draw(QPainter *p, const QRect& r)
//
//------------------------------------------------------------
void QwtSymbol::draw(QPainter *p, const QRect& r)
{

    int w2 = r.width() / 2;
    int h2 = r.height() / 2;

    p->setBrush(d_brush);
    p->setPen(d_pen);
    
    switch(d_style)
    {
    case QwtSymbol::Ellipse:
	p->drawEllipse(r);
	break;
    case QwtSymbol::Rect:
  	p->drawRect(r);
    break;
    case QwtSymbol::Diamond:
       d_pa.resize(4);
       d_pa.setPoint(0, r.x() + w2, r.y());
       d_pa.setPoint(1, r.right(), r.y() + h2);
       d_pa.setPoint(2, r.x() + w2, r.bottom());
       d_pa.setPoint(3, r.x(), r.y() + h2);
       p->drawPolygon(d_pa);
	break;
    case QwtSymbol::Cross:
	p->drawLine(r.x() + w2, r.y(), r.x() + w2, r.bottom());
	p->drawLine(r.x(), r.y() + h2, r.right(), r.y() + h2);
	break;
    case QwtSymbol::XCross:
	p->drawLine(r.x(), r.y(), r.right(), r.bottom());
	p->drawLine(r.x(), r.bottom(), r.right(), r.top());
	break;
    case QwtSymbol::Triangle:
    case QwtSymbol::UTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.x() + w2, r.y());
	d_pa.setPoint(1, r.right(), r.bottom());
	d_pa.setPoint(2, r.x(), r.bottom());
	p->drawPolygon(d_pa);
	break;
    case QwtSymbol::DTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.x(), r.y());
	d_pa.setPoint(1, r.right(), r.y());
	d_pa.setPoint(2, r.x() +  w2, r.bottom());
	p->drawPolygon(d_pa);
	break;
    case QwtSymbol::LTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.x(), r.y());
	d_pa.setPoint(1, r.right(), r.y() + h2);
	d_pa.setPoint(2, r.x(), r.bottom());
	p->drawPolygon(d_pa);
	break;
    case QwtSymbol::RTriangle:
	d_pa.resize(3);
	d_pa.setPoint(0, r.right(), r.y());
	d_pa.setPoint(1, r.x(), r.y() + h2);
	d_pa.setPoint(2, r.right(), r.bottom());
	p->drawPolygon(d_pa);
	break;
    default:
	break;
    }

}

//------------------------------------------------------------
//
//.S	QwtSymbol::draw
//	Draw the symbol at a specified point
//
//.s	Parameters
//.p	QPainter *p	-- painter
//	const QPoint &pt -- point
//
//.s	Description
//	The point sp/ecifies the upper left corner of a
//	rectangle with the symbol's size.
//
//.s	Syntax
//.f	void QwtSymbol::draw(QPainter *p, const QPoint &p)
//
//------------------------------------------------------------
void QwtSymbol::draw(QPainter *p, const QPoint &pt)
{
    draw(p, pt.x(), pt.y());
}

//------------------------------------------------------------
//
//.S	QwtSymbol::setStyle
//	Specify the symbol style
//
//.s	Parameters
//.p	int s -- style
//
//.s	Description
//	The following styles are defined:
//.t	QwtSymbol::None	--	No Style. The symbol cannot be drawn.
//      QwtSymbol::Ellipse --	Ellipse or circle
//      QwtSymbol::Rect --	Rectangle
//      QwtSymbol::Diamond --	Diamond
//      QwtSymbol::Triangle --	Triangle pointing upwards
//      QwtSymbol::DTriangle -- Triangle pointing downwards
//      QwtSymbol::UTriangle -- Trinagle pointing upwards
//      QwtSymbol::LTriangle -- Triangle pointing left
//      QwtSymbol::RTriangle -- Triangle pointing right
//      QwtSymbol::Cross --	Cross
//      QwtSymbol::XCross --	Diagonal cross
//
//.s	Syntax
//.f	void QwtSymbol::setStyle(int s)
//
//------------------------------------------------------------
void QwtSymbol::setStyle(int s)
{
    d_style = s;
}

void QwtSymbol::receiveData(QDataStream &s){

   s >> d_size;
   s >> d_pen;
   s >> d_style;
   s >> d_brush;
}

void QwtSymbol::sendData(QDataStream &s) const {

   s << d_size;
   s << d_pen;
   s << d_style;
   s << d_brush;
}
  

QDataStream &operator<<( QDataStream &s, QwtSymbol const &sym ){

   sym.sendData(s);
   return(s);
}

QDataStream &operator>>( QDataStream &s, QwtSymbol &sym ){

   sym.receiveData(s);
   return(s);
}
