c     -*- mode: FORTRAN -*-
c
c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1998 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c
c     If you use this program for your scientific work, please cite it according to
c     the file CITATION included with this package.
c
c     krot-arnirot
c     a program to calculate rotational resolved vibrational/vibronic bands


#include "arni.h"



      subroutine asyrot( J, rotcon, npar, mat, size )
c     Add the standard rigid rotor hamiltonian contributions to mat

c     Input:   J      - main quantum number
c     Input:   rotcon - array with inertial parameter:
c                       ( A, B, C, DX, DY, DZ, Dj, Djk, Dk, sdj, sdk )
c     Input:   npar   - dimension of rotcon
c     Input:   size   - dimension of the matrix 
c     InOut:   mat    - rotational constants matrix

      A   = rotcon(1)
      B   = rotcon(2)
      C   = rotcon(3)


      return
      end


c     --------------------------------------------------------------------------
      subroutine watson( J, rotcon, npar, mat, size )
c     standard Watson hamiltonian (A reduction) up to quartic terms - no axis reorientation 
c     (J.K.G. Watson, Vibrational Spectra and Structure 6, 1 (1977), ed J.R. Durig)

      implicit none

      integer        i, K, J, npar, size
      real*8         rotcon( npar ), mat( size, size )
      real*8         A, B, C
c     real*8         DX, DY, DZ
      real*8         Dj, Djk, Dk, sdj, sdk

      ARNIROT_LAUNCH ("Launching watson - calculating hamiltonian without axis reorientation.")

c     define constants of the hamiltonian
      Dj  = rotcon(7)
      Djk = rotcon(8)
      Dk  = rotcon(9)
      sdj = rotcon(10)
      sdk = rotcon(11)
c     calculation of reduced rotational constants (Watson, Vib. Spectra Struct. 6, 1 (1977), p. 40):
c      A   = rotcon(1) - 2*Dj
c      B   = rotcon(2) - 2*Dj - Djk + 2*sdj + 2*sdk
c      C   = rotcon(3) - 2*Dj - Djk - 2*sdj - 2*sdk
      A   = rotcon(1)
      B   = rotcon(2)
      C   = rotcon(3)


#ifdef DEBUG
      if ( J .eq. 1 ) then
         write(*,11) A, B, C, Dj, Djk, Dk, sdj, sdk
   11    format(/,'A  =',f14.6,', B  =',f14.6,', C  =',f14.6,
     *          /,'Dj =',f14.6,', Djk=',f14.6,', Dk =',f14.6,
     *          /,'dj =',f14.6,', dk =',f14.6,/)
      end if
#endif

c     calculation of matrix elements (Watson, Vib. Spectra Struct. 6, 1 (1977), p. 42):
      do K = -J, J - 2
         i = K + J + 1
c        diagonal elements
         mat( i, i )     = 0.5 * ( B + C ) * ( J * ( J + 1 ) - K*K ) + A * K*K
     *                    - Dj * ( J*J ) * ( ( J + 1 ) * ( J + 1 ) )
     *                    - Djk * J * ( J + 1 ) * ( K*K )
     *                    - Dk * ( ( K*K ) * ( K*K ) )
c        two-off-diagonal elements
         mat( i + 2, i ) = ( 0.25 * ( B - C ) - sdj * J * ( J + 1 )
     *                    - 0.5 * sdk * ( ( ( K + 2 ) * ( K + 2 ) ) + ( K*K ) ) )
     *                    * dsqrt( dfloat( ( J * ( J + 1 ) - K * ( K + 1 ) )
     *                                   * ( J * ( J + 1 ) - ( K + 1 ) * ( K + 2 ) ) ) )
         mat( i, i + 2 ) = mat( i + 2, i )
      end do
c     plus some more diagonal elements
      do K = J - 1, J
         i = K + J + 1
         mat( i, i )     = 0.5 * ( B + C ) * ( J * ( J + 1 ) - K*K ) + A * K*K
     *                    - Dj * ( J*J ) * ( ( J + 1 ) * ( J + 1 ) )
     *                    - Djk * J * ( J + 1 ) * ( K*K )
     *                    - Dk * ( ( K*K ) * ( K*K ) )
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine rrhfax( J, A, B, C, ham, dmham )
c     rigid rotor A'Jz**2, B'Jx**2 and C'Jy**2 terms for axis switched calculations

      implicit none

      integer        dmham, J
      real*8         A, B, C, ham(dmham,dmham)

      ARNIROT_LAUNCH ( "Launching rrhfax." )

c     if ( shorti.eq.0 ) write(*,*) 'This is still old code - quartic terms are not used.'

c     evaluate the <i|H|i> rigid rotor matrix elements
      call rrii( A, B, C, J, ham, dmham )
c     evaluate the <i|H|i+-2> rigid rotor matrix elements
      call rri2( A, B, C, J, ham, dmham )

      return
      end


c------------------------------------------------------------------------------
      subroutine rrii( A, B, C, J, ham, dmham )
c     evaluate the rigid rotor diagonal matrix elements <i|H|i>

      implicit none

      integer        dmham, i, J, K
      real*8         A, B, C, ham(dmham,dmham)

      ARNIROT_LAUNCH ("Launching rrii.")

      K = -J - 1
      do i = 1, 2*J+1, 1
         K = K + 1
c        fill the diagonal matrix elements for the asymmetric rotor
         ham(i,i) = 0.5*(B + C)*(J*(J + 1) - K*K) + A*K*K
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine rri2( A, B, C, J, ham, dmham )
c     evaluate the two-off-diagonal rigid rotor matrix elements <i|H|i+-2>

      implicit none

      integer        dmham, i, ip2, J, K
      real*8         A, B, C, ham(dmham,dmham), x1, x2

      ARNIROT_LAUNCH ("Launching rri2.")

      K = -(J + 1)
      do i = 1, 2*J-1, 1
         K = K + 1
         ip2 = i + 2
c        off-diagonal <K| |K+2> terms
         x1 = J*(J + 1) - K*(K + 1)
         x2 = J*(J + 1) - (K + 1)*(K + 2)
c        fill in the hamiltonian terms which are off-diagonal in 2 qn
         ham(i,ip2) = 0.25*(B - C)*dsqrt(x1*x2)
c        fill the equivalent i+2,i elements without recalculating
         ham(ip2,i) = ham(i,ip2)
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jyjx( mat, dmham, J, con )
c     fill JyJx+JxJy matrix (used for derivatives in ax. sw. calculations)
c     and multiply elements with a constant (for additional hamiltonian terms)

      implicit none

      integer        dmham, i, J, K
      real*8         con, mat(dmham,dmham), y

      ARNIROT_LAUNCH ( "Launching jyjx." )

c     this routine is changed by only preliminary understanding (Jochen),
c     look at original arnirot-code for differences.

c     only two-off-diagonal elements
      do K = -J, J - 2
         i = K + J + 1
         y = 0.5 * con * dsqrt( dfloat( ( J * ( J + 1 ) - K * ( K + 1 ) )
     *                                * ( J * ( J + 1 ) - ( K + 1 ) * ( K + 2 ) ) ) )
         mat( i, i + 2 ) = mat( i, i + 2 ) - y
         mat( i + 2, i ) = mat( i + 2, i ) - y
c        (instead of: mat(i+2,i) = mat(i+2,i) - mat(i,i+2))
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jzjx( mat, dmham, J, con )
c     fill JzJx+JxJz matrix (used for derivatives in ax. sw. calculations)
c     and multiply elements with a constant (for additional hamiltonian terms)

      implicit none

      integer        dmham, i, J, K
      real*8         con, mat(dmham,dmham), y

      ARNIROT_LAUNCH ( "Launching jzjx." )

c     this routine is changed by only preliminary understanding (Jochen)
c     - look at original arnirot-code for differences.

c     only one-off-diagonal elements
      do K = -J, J - 1
         i = K + J + 1
         y = 0.5 * con * dsqrt( dfloat( J * ( J + 1 ) - K * ( K + 1 ) ) ) * ( 2 * K + 1 )
         mat( i, i + 1 ) = mat( i, i + 1 ) + y
         mat( i + 1, i ) = mat( i + 1, i ) + y
c        (instead of: mat(i+1,i) = mat(i+1,i) + mat(i,i+1))
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jzjy( mat, dmham, J, con )
c     fill JzJy+JyJz matrix (used for derivatives in ax. sw. calculations)
c     and multiply elements with a constant (for additional hamiltonian terms)

      implicit none

      integer        dmham, i, J, K
      real*8         con, mat(dmham,dmham), y

      ARNIROT_LAUNCH ( "Launching jzjy." )

c     this routine is changed by only preliminary understanding (Jochen)
c     - look at original arnirot-code for differences.

c     only one-off-diagonal elements
      do K = -J, J - 1
         i = K + J + 1
         y = 0.5 * con * dsqrt( dfloat( J * ( J + 1 ) - K * ( K + 1 ) ) ) * ( 2 * K + 1 )
         mat( i, i + 1 ) = mat( i, i + 1 ) - y
         mat( i + 1, i ) = mat( i + 1, i ) + y
c        (instead of: mat(i+1,i) = mat(i+1,i) - mat(i,i+1))
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jx( mat, dmham, J, con )
c     fill Jx matrix (used for derivatives in calculations including linear Dx terms)
c     and multiply elements with a constant (for additional hamiltonian terms)

      implicit none

      integer        dmham, i, J, K
      real*8         con, mat(dmham,dmham), y

      ARNIROT_LAUNCH ( "Launching jx." )

c     this routine is changed by only preliminary understanding (Jochen)
c     - look at original arnirot-code for differences.

c     only one-off-diagonal elements
      do K = -J, J - 1
         i = K + J + 1
         y = 0.5 * con * dsqrt( dfloat( J * ( J + 1 ) - K * ( K + 1 ) ) )
         mat( i, i + 1 ) = mat( i, i + 1 ) + y
         mat( i + 1, i ) = mat( i + 1, i ) + y
c        (instead of: mat(i+1,i) = mat(i+1,i) + mat(i,i+1))
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jy( mat, dmham, J, con )
c     fill Jy matrix (used for derivatives in calculations including linear Dy terms)
c     and multiply elements with a constant (for additional hamiltonian terms)

      implicit none

      integer        dmham, i, J, K
      real*8         con, mat(dmham,dmham), y

      ARNIROT_LAUNCH ( "Launching jy." )

c     this routine is changed by only preliminary understanding (Arnim)
c     - look at original arnirot-code for differences.

c     only one-off-diagonal elements
      do K = -J, J - 1
         i = K + J + 1
         y = 0.5 * con * dsqrt( dfloat( J * ( J + 1 ) - K * ( K + 1 ) ) )
c        upper diagonal is negative
         mat( i, i + 1 ) = mat( i, i + 1 ) - y
c        lower diagonal is positive
         mat( i + 1, i ) = mat( i + 1, i ) + y
c        (instead of: mat(i+1,i) = mat(i+1,i) - mat(i,i+1))
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine jz( mat, dmham, J, con )
c     fill Jz matrix (used for derivatives in calculations including linear Dz terms)
c     and multiply elements with a constant (for additional hamiltonian terms)

      implicit none

      integer        dmham, i, J, K
      real*8         con, mat(dmham,dmham)

      ARNIROT_LAUNCH ( "Launching jz." )

c     only diagonal elements
      do K = -J, J
         i = K + J + 1
         mat( i, i ) = mat( i, i ) + con * K
      end do

      return
      end
