// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_DIALOG_CALCULATION_H
#define KROT_DIALOG_CALCULATION_H


#include "calculationParameter.h"
#include "constants.h"
#include "rawData.h"
#include "lined.h"

#include <qgroupbox.h>
#include <qradiobutton.h>
#include <qtabdialog.h>



class KConfig;
class QBoxLayout;
class QCheckBox;
class QLayout;
class QWidget;



// Abstract base classes -------------------------------------------------------


/**
 * Parameter input box.
 *
 * Abstract base class of all GroupBoxes for parameter input.
 *
 * This is a QGroupBox plus one additional abstract function to get back the
 * actual parameter from any ParameterBox.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class ParameterBox : public QGroupBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor.
     *
     * @param title  User visible title of this ParameterBox. This is passed to the
     *               @ref QGroupBox constructor.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    ParameterBox( const char *title, QWidget *parent, const char *name=0 );
    
    /**
     * Get the current values of the input fields.
     */
    virtual void getValues( CalculationParameter& par ) const = 0;

    /**
     * Enable/disable all input to this widget.
     * This is used by control instances that know wether these inputs shall
     * be used or not.
     *
     * @param enable True to enable input, false to disable it.
     */
    virtual void setEnabled( bool on );


protected:

    bool enabled;
};



/**
 * Base class for QTabDialog pages.
 *
 * This is the base of all TabDialog pages. It provides an abstract function to
 * get back the actual parameters from this page.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class TabPage : public QFrame
{
    Q_OBJECT

public:
    
    TabPage( QWidget *parent, const char *name=0 );

    /**
     * Update the parameter to the current values of this page.
     *
     * @param Calculationparameter structure to copy data into.
     */
    virtual void getValues( CalculationParameter& par ) const = 0;

    /**
     * Enable/disable all input to this widget.
     * This is used by control instances that know wether these inputs shall
     * be used or not.
     *
     * @param enable True to enable input, false to disable it.
     */
    virtual void setEnabled( bool ) = 0;

    
protected:
    
    QBoxLayout *layout;
};



/**
 * Base class of all calculation dialogs.
 *
 * Generic dialog to enter calculation start parameter and to call calculation
 * program.
 *
 * @author Jochen Kpper
 * @version 1999/01/21
 */
class DialogCalculation : public QTabDialog
{
    Q_OBJECT

public:

    DialogCalculation( const CalculationParameter& par, QWidget *parent=0, const char *name=0 );


protected:
    
    /**
     * Update the calculation parameter to the values in the user inputs.
     */
    virtual void getValues() = 0;

    /**
     * Set the dialogs values to the given ones.
     */
    void setValues( const CalculationParameter& par );
    

protected slots:

    void accept();

    /**
     * Overloaded so pressing a button does not close the dialog when it's
     * unwanted (like the help button).
     *
     * The dialog is closed if reallyDone is true, nothing is done otherwise.
     *
     * @param status The result of this dialog to return to the calling exec.
     */
    void done( int status );

    /**
     * Help facility. This needs to be overloaded to jump to the correct help
     * entry point.
     */
    virtual void help();

    void reject();
    

signals:

    /**
     * Start calculation.
     *
     * @param par Parameter to do calculation on.
     */
    void calculate( const CalculationParameter& par );

    /**
     * Send a message to the user.
     *
     * @param msg Message to provide to the user.
     */
    void message( const QString& msg );
    
protected:

    CalculationParameter par;

    bool reallyDone;
};



// Parameter boxes -------------------------------------------------------------



/**
 * Axis tilting angles for the excited state. These are represented by three
 * Eulerian angles.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class AxisTiltBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    AxisTiltBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;

    /**
     * Enable/disable all input to this widget.
     * This is used by control instances that know wether these inputs shall
     * be used or not.
     *
     * @param enable True to enable input, false to disable it.
     */
    virtual void setEnabled( bool on );

    
protected slots:
 
    /**
     * Enable / disable the inputs for axis tilting parameter.
     *
     * @param on True to enable, false to disable the page.
     */
    void enableAxisTilt( bool on );

protected:
    
    LblFloated *angle[ 3 ];

    QCheckBox *flag;
};



/**
 * General Calcualtion Parameter.
 *
 * Input for general calculation parameter as Jmax and DeltaKmax.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class ControlParameterBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    ControlParameterBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;

    
protected:
    
    LblInted *dkmax, *jmax;
};



/**
 * Quartic centrifugal distortion terms of the reduced Watson hamiltonian.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class CentrifugalBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    CentrifugalBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;

    
protected:
    
    ChkLblFloated *terms[ 2 ][ 5 ];
};



/**
 * Hybrid character
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class HybridBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * Create a QGroupBox containing three ChkLblFloat for input of relative
     * intensities of a, b, and c type transitions.
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    HybridBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;


private:

    /**
     * These inputs are needed everytime, so it doesnt make sense to disable
     * them.
     */
    virtual void setEnabled( bool ) {};

    
protected:
    
    LblFloated *hybrid[ 3 ];
};



/**
 * Linear distortion terms.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class LinearBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    LinearBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;

    
protected:
    
    ChkLblFloated *terms[ 2 ][ 3 ];
};



/**
 * Origin input.
 *
 * This is a GroupBox containing an input and a checkmark for the vibronic origin.
 *
 * (Here we might implement a unit-selection for the origin as well. Right now
 * everything in here is MHz. The interface has to keep that convention at any
 * time !)
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class OriginBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * Create a QGroupBox containing a ChkLblFloat for input.
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    OriginBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;


protected slots:

    /**
     * save new unit to KConfig object.
     *
     * @param btn The pressed buttons id.
     */
    void unitChanged( int btn );


private:

    /**
     * These inputs are needed everytime, so it doesnt make sense to disable
     * them.
     */
    virtual void setEnabled( bool ) {};

    
protected:

    enum { ID_CM, ID_MHZ };

    static const char config_OriginUnit[] = "OriginUnit";
    
    KConfig *config;
    
    ChkLblFloated *origin;
    
    QRadioButton *cmRB, *mhzRB;
};



/**
 * Rotational constants input.
 *
 * This is a GroupBox containing inputs and checkmarks for two sets of three
 * rotational constants each.
 * The constants are called A, B, C and primed for gound and excited state.
 * They are layouted two in a row, three rows.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class RotationalConstantsBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * Create a QGroupBox containing six ChkLblFloats for input.
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    RotationalConstantsBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;


private:

    /**
     * These inputs are needed everytime, so it doesnt make sense to disable
     * them.
     */
    virtual void setEnabled( bool ) {};

    
protected:
    
    ChkLblFloated *Ag, *Bg, *Cg, *Ae, *Be, *Ce;
};



/**
 * Nuclear spin statistical weights.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class SpinStatisticBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    SpinStatisticBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;

    
protected:
    
    LblInted *nssw[ 4 ];
};



/**
 * Temperature input.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class TemperatureBox : public ParameterBox
{
    Q_OBJECT
    
public:

    /**
     * Constructor
     *
     * Create a QGroupBox containing a ChkLblFloat for input.
     *
     * @param par    Values to use on creation.
     * @param parent Parent widget. This is passed to the @ref QGroupBox constructor.
     * @param name   Internal name. This is passed to the @ref QGroupBox constructor.
     */
    TemperatureBox( const CalculationParameter& par, QWidget *parent, const char *name=0 );

    /**
     * Get the current values of the rotational constants
     */
    virtual void getValues( CalculationParameter& par ) const;


private:

    /**
     * These inputs are needed everytime, so it doesnt make sense to disable
     * them.
     */
    virtual void setEnabled( bool ) {};

    
protected:
    
    LblFloated *temp;
};





// QTabDialog pages ------------------------------------------------------------



/**
 * First page for calculation dialog when using arnirot.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class PageOneArnirot : public TabPage
{
    Q_OBJECT

public:
    
    PageOneArnirot( const CalculationParameter& par, QWidget *parent=0, const char *name=0 );

    /**
     * Update the parameter to the current values of this page.
     *
     * This is done by calling getValues for all ParameterBox children and then
     * storing the local parameter values into par.
     *
     * @param par Calculationparameter structure to copy data in.
     */
    virtual void getValues( CalculationParameter& par ) const;


protected slots:

    /**
     * Enable / disable the input page for centrifugal distortion parameter.
     * This only emits enableCentrifugalDistortion( bool )
     *
     * @param on True to enable, false to disable the page.
     */
    void slotCentDist( bool on );

    /**
     * Enable / disable the input page for linear distortion parameter.
     *
     * @param on True to enable, false to disable the page.
     */
    void slotLinDist( bool on );

    
signals:

    /**
     * Enable / disable the input page for centrifugal distortion parameter.
     *
     * @param on True to enable, false to disable the page.
     */
    void centDistToggled( bool on );

    /**
     * Enable / disable the input page for linear distortion parameter.
     *
     * @param on True to enable, false to disable the page.
     */
    void linDistToggled( bool on );


private:

    /**
     * These inputs are needed everytime, so it doesnt make sense to disable
     * them.
     */
    virtual void setEnabled( bool ) {};

    
protected:

    ParameterBox *origin, *rotConstants, *control;
    
    QCheckBox *linDist, *centDist; 
};



/**
 * Widget for input of centrifugal distortion parameter.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class PageCentrifugal : public TabPage
{
    Q_OBJECT
    
public:
    
    /**
     * Constructor
     *
     * Build up the page layout.
     *
     * @param parent Passed to QFrame constructor.
     * @param name   Passed to QFrame constructor.
     */
    PageCentrifugal( const CalculationParameter& par, QWidget *parent=0, const char *name=0 );

    /**
     * Update the parameter to the current values of this page.
     *
     * This is done by calling getValues for all ParameterBox children and then
     * storing the local parameter values into par.
     *
     * @param par Calculationparameter structure to copy data in.
     */
    virtual void getValues( CalculationParameter& par ) const;

    /**
     * Enable/disable all input to this widget.
     * This is used by control instances that know wether these inputs shall
     * be used or not.
     *
     * @param enable True to enable input, false to disable it.
     */
    virtual void setEnabled( bool on );


protected:
    
    ParameterBox *cent;
};



/**
 * Widget for input of centrifugal distortion parameter.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class PageLinear : public TabPage
{
    Q_OBJECT
    
public:
    
    /**
     * Constructor
     *
     * Build up the page layout.
     *
     * @param parent Passed to QFrame constructor.
     * @param name   Passed to QFrame constructor.
     */
    PageLinear( const CalculationParameter& par, QWidget *parent=0, const char *name=0 );

    /**
     * Update the parameter to the current values of this page.
     *
     * This is done by calling getValues for all ParameterBox children and then
     * storing the local parameter values into par.
     *
     * @param par Calculationparameter structure to copy data in.
     */
    virtual void getValues( CalculationParameter& par ) const;

    /**
     * Enable/disable all input to this widget.
     * This is used by control instances that know wether these inputs shall
     * be used or not.
     *
     * @param enable True to enable input, false to disable it.
     */
    virtual void setEnabled( bool on );


protected:
    
    ParameterBox *lin;
};



/**
 * Widget for input of standard intensity parameter.
 *
 * The parameter are temperature, transition hybrid character
 * This should be usable for any calculation program/input dialog that uses
 * these parameters.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class PageIntensity : public TabPage
{
    Q_OBJECT
    
public:
    
    /**
     * Constructor
     *
     * Build up the page layout.
     *
     * @param parent Passed to QFrame constructor.
     * @param name   Passed to QFrame constructor.
     */
    PageIntensity( const CalculationParameter& par, QWidget *parent=0, const char *name=0 );

    /**
     * Update the parameter to the current values of this page.
     *
     * This is done by calling getValues for all ParameterBox children and then
     * storing the local parameter values into par.
     *
     * @param par Calculationparameter structure to copy data in.
     */
    virtual void getValues( CalculationParameter& par ) const;


private:

    virtual void setEnabled( bool on );

    
protected:
    
    ParameterBox *axisTilt, *hybrid, *nssw, *temp;
};



// calculation dialogs ---------------------------------------------------------



/**
 * @short Dialog to start arnirot calculation.
 *
 * Dialog to enter arnirot calculation start parameter and to call arnirot.
 *
 * @author Jochen Kpper
 * @version 1998/12/22
 */
class DialogCalculationArnirot : public DialogCalculation
{
    Q_OBJECT

public:

    /**
     * Constructor
     */
    DialogCalculationArnirot( const CalculationParameter& param, QWidget *parent=0, const char *name=0 );

    /**
     * Display the dialog and return result.
     *
     * This is overloaded from \see QDialog::exec to display the front page
     * of the dialog whenever called.
     *
     * @return The dialogs result.
     */
    int exec();

	
protected:
    
    /**
     * Update the parameter to the current values of this page.
     *
     * This is done by calling getValues for each page child.
     */
    virtual void getValues();

    
protected slots:

    /**
     * Help facility. This needs to be overloaded to jump to the correct help
     * entry point.
     */
    virtual void help();

    /**
     * Enable / disable the input page for centrifugal distortion parameter.
     * This only emits enableCentrifugalDistortion( bool )
     *
     * @param on True to enable, false to disable the page.
     */
    void enableCentDist( bool on );

    /**
     * Enable / disable the input page for linear distortion parameter.
     * This only emits enableCentrifugalDistortion( bool )
     *
     * @param on True to enable, false to disable the page.
     */
    void enableLinDist( bool on );

    
protected:

    static const char nameCentrifugalDistortion[] = "ArnirotPageCentrifugalDistortion",
	nameIntensity[] = "ArnirotPageIntensity",
	nameLinearDistortion[] = "ArnirotPageLinearDistortion",
	nameOne[] = "ArnirotPageOne";
    
    PageOneArnirot *one;

    PageIntensity *intensity;
    
    PageCentrifugal *cent;

    PageLinear *lin;
};



#include "calculation_inline.h"



#endif
