// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_SIMULATION_INLINE_H
#define KROT_SIMULATION_INLINE_H



inline Simulation::Simulation( KRotData *krotdata )
    : DataSet( krotdata ), epsilon( 1e-3 ), wG( 0 ), wL( 0 ), lns(),
      profile( LINESHAPE_STICK ), profiledata( 0 ), profilevalid( false ),
      scaling( 100.0 )
{
    clr = ::red;
}



inline Simulation::Simulation( const Simulation& sim )
    : DataSet( sim ), epsilon( sim.epsilon ), wG( sim.wG ), wL( sim.wL ), lns( sim.lns ),
      profile(sim.profile), profiledata( sim.profiledata ), profilevalid( sim.profilevalid ),
      scaling( sim.scaling )
{
    clr = ::red;
}


inline Simulation::Simulation( const vector< Transition >& lines, KRotData *krotdata )
    : DataSet( krotdata ), epsilon( 1e-3 ), wG( 0 ), wL( 0 ), lns( lines ),
      profile( LINESHAPE_STICK ), profiledata( 0 ), profilevalid( false ),
      scaling( 100.0 )
{
    clr = ::red;
    KDEBUG( KDEBUG_INFO, KDEBUG_KROT_LAUNCH, "Constructing Simulation from given vector< transition >" );
    // sort Assignments by frequency (using Transition::operator<).
    // we use stable_sort since it has a better worst case complexity then sort itself !
    // Moreover it doesn't reorder equal assignments
    stable_sort( lns.begin(), lns.end() );
    KROT_DEBUG3( "%d lns in the frequency range %Ld  - %Ld read.",
		 lns.size(), lns.front().frequency(), lns.back().frequency() );
    setName( "New Simulation" );
    fold();
}



inline Simulation& Simulation::operator=( const Simulation& sim )
{
    // check for object identity
    if( this == &sim )
	return *this;
    DataSet::operator=( sim );
    epsilon = sim.epsilon;
    wG = sim.wG;
    lns = sim.lns;
    wL = sim.wL;
    profile = sim.profile;
    return *this;
}



inline double Simulation::gauss() const
{
    return wG;
}



inline Simulation::Lineshape Simulation::lineshape() const
{
    return profile;
}



inline double Simulation::lorentz() const
{
    return wL;
}



inline const vector<Transition>& Simulation::lines() const
{
    return lns;
}



inline void Simulation::setGauss( const double val )
{
    if( wG != val ) {
	profilevalid = false;
	wG =  val > 0.0 ? val : -val;
    }
}



inline void Simulation::setLineshape( const Lineshape val )
{
    if( profile != val ) {
	profilevalid = false;
	profile = val;
    }
}



inline void Simulation::setLorentz( const double val )
{
    if( wL != val ) {
	profilevalid = false;
	wL = val > 0.0 ? val : -val;
    }
}



inline double Simulation::gauss( double x )
    // wG is the FWHM
{
    static const double c1 =  2.0 * Constants::sqrt__ln2_div_pi();
    static const double c2 = -4.0 * Constants::ln2();
    return c1 / wG * exp( c2 * sqr( x / wG ) );
}



inline double Simulation::gaussLimit()
{
    return 3.0 * wG;
}



inline double Simulation::lorentz( double x )
{
    static const double k = 2.0 / Constants::pi();
    return wL * k / ( sqr( wL ) + 4.0 * sqr( x ) );
}



inline double Simulation::lorentzLimit()
{
    return 10.0 * wL;
}



inline double Simulation::voigt( double x )
{
    static const double c1 = 2 * Constants::sqrt_ln2() / Constants::sqrt_pi();
    return ( c1 / wG * humlicek_v12( 2 * Constants::sqrt_ln2() * x / wG,
					 Constants::sqrt_ln2() * wL / wG ) );
}



inline double Simulation::voigtLimit()
{
    return 10.0 * wL + 3.0 * wG;
}



#endif



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
