// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "dataset.h"
#include "krotData.h"
#include "rawData.h"
#include "spectrum.h"

#include <algorithm>
#include <climits>
#include <fstream.h>



Spectrum::Spectrum( const DataSet& set )
    : DataSet( set )
{
}



Spectrum::Spectrum( KRotData *krotdata )
    : DataSet( krotdata )
{
}



Spectrum::Spectrum( const RawData& import, KRotData *krotData )
    : DataSet( krotData )
{
    int64_t start;
    u_int64_t size;
    double *newdata = import.linearize( delta(), start, size );
    setData( start, start + size - 1, newdata );
    setName( import.name() );
}



void Spectrum::loadXY( const QString& fileName )
{
    KROT_LAUNCH1( "Launching Spectrum::loadXY with filename %s", fileName.data() );
    ifstream stream( fileName.data(), ios::in|ios::nocreate );
    if( ! stream )
	  throw FileNotFound( fileName.data() );
    vector< double > x, y;
    // read data
    while( ! stream.eof() ) {
		double  freq;
		double  inten;
		stream >> freq >> inten;
		x.push_back( freq );
		y.push_back( inten );
    }
    int64_t start = static_cast< int64_t >( x.front() );
    int64_t stop = static_cast< int64_t >( x.back() );
    if( start > stop )
	    swap( start, stop );
    u_int64_t size = stop - start + 1;
    double *newdata = new double [ size ];
    double  x1 = x[ 0 ], x2 = x[ 1 ], y1 = y[ 0 ], y2 = y[ 1 ];
    newdata[ 0 ] = y[ 0 ];
    int64_t c = 1;
    for( u_int64_t i=1; i<size; i++ ) {
		int64_t xc = start + static_cast< int64_t >( i * delta() );
		while( x2 < xc ) {
			c++;
			x1 = x2;    y1 = y2;
			x2 = x[ c ];  y2 = y[ c ];
		}
		newdata[ i ] = y1 + ( xc - x1 ) * ( y2 -y1 ) / ( x2 - x1 );
	}
    setData( start, stop, newdata );
    setName( fileName );
    
    KROT_DEBUG1( "%d data points read.", size );
#if SIZEOF_LONG >= 8
    KROT_DEBUG3( "Startfrequenz: %ld, delta: %d,  stop: %ld", start, delta(), stop );
#else    
    KROT_DEBUG3( "Startfrequenz: %lld, delta: %d,  stop: %lld", start, delta(), stop );
#endif
}



//* Local Variables:
//* c-file-style: "Stroustrup"
//* mode: C++
//* End:
