/****************************************************************
**
** Qvumeter: a VU-meter widget for Qt
** By Andrew D. Balsa <andrebalsa@altern.org>, June 1998
** Version 0.0.3
** Licensing: my code under GNU/GPL, Qt code under Qt special license
**
** Implementation file.
** Other files included are the header file and a main.cpp demo program.
**
** Basically this is a Qt Frame paintable widget, of geometry determined by
** the type of meter, the number of steps and the pixel dimensions of the
** GIF image used for each step. We Bitblt the LED pixmaps inside the frame
** in order to avoid flicker as much as possible.
**
** Only led, single or multiple color, basic damping, linear scale (vertical or
** horizontal) in version 0.0.3, 10 June 1998.
**
** Web site: http://www.altern.org/andrebalsa/widgets
**
** At this stage there is a nice feature implemented: the widget will accept any
** GIF file (including transparency information) as the basic LED. A few sample
** small GIF files are included here.
**
** Planned features include:
** 	- More damping options (see algorithm below). (version 0.0.4)
** 	- Linear/log/dB scales. (version 0.0.5)
** 	- Fluorescent look, LCD look. (version 0.0.6)
**	- Overload the constructor, cleanup the code. (version 0.0.7)
**	- Analog VU meter look. (version 0.1)
**	- Resizable. (version 0.2)
**	- Audio alerts using KDE audio server. (version 0.3)
**	- Resettable "Hold" marks. (version 0.4)
**	- Suggestions welcome for versions 0.5+ :-)
**

KSpectrum
Carl van Schaik (carl@leg.uct.ac.za)

This widget has been HIGHLY modified from the original for use in KSpectrum
by Carl van Schaik to provide greatly enhanced speed, various effects and
integration into KSpectrum.

****************************************************************/

#include "qvumeter.moc"

#include <qpixmap.h>
#include <qpainter.h>
#include <qimage.h>
#include "leds.h"

unsigned int maxLed, oldMax;

Qvumeter::Qvumeter( int ledNumber, int minRange, int maxRange, int vuDamping,
      int valStop1, int valStop2, int valStop3,
      unsigned char ledOrient,
      QWidget *parent, const char *name, WFlags f )
      : QFrame( parent, name, f )

{
    min = minRange;
    max = maxRange;
    minInput = 0;	// input value memories, min and max
    maxInput = 0;
    previousLed = 0;
    activeLed = 0;
    maxLed = 0;
    sticky = 0;
    mode = 1;
    stepsNo = ledNumber;
	if (stepsNo == 0) stepsNo = 10;
    dampingFactor = vuDamping;
    if ((ledOrient == 'H') || (ledOrient == 'h')) isVert = 0;
    else isVert = 1;	// default to vertical bar graph
    unsigned char i;

   // clamp damping factor value

    if (vuDamping < 0)
	dampingFactor = 0;
    else
        if (vuDamping > 100)
	    dampingFactor = 100;

    // also be careful with valStops

    valStop1 = valStop1 % stepsNo;
    valStop2 = valStop2 % stepsNo;
    valStop3 = valStop3 % stepsNo;
    if (valStop3) valStop2 = valStop2 % valStop3;
    if (valStop2) valStop1 = valStop1 % valStop2;
    
    //whitePix = new QPixmap((const char**)whiteled); // get blank pixmap
    whitePix = new QPixmap((const char**)blueled); // get bg pixmap

    // load led base pixmap
    if (ledOrient <= 'Z') {	// standard LED GIFs
        ledPixmap[0] = new QPixmap((const char**)greenled); // get led pixmap
        ledPixmap[1] = new QPixmap((const char**)yellowled); // get led pixmap
        ledPixmap[2] = new QPixmap((const char**)orangeled); // get led pixmap
        ledPixmap[3] = new QPixmap((const char**)redled); // get led pixmap
    }

    // get LED width and height; we assume they are all same size
    ledWidth = ledPixmap[0]->width();
    ledHeight = ledPixmap[0]->height();

    // initialize the led pixmap pointers
    for (i=0; i < valStop1; i++) pled[i] = ledPixmap[0];
    for (i=valStop1; i < valStop2; i++) pled[i] = ledPixmap[1];
    for (i=valStop2; i < valStop3; i++) pled[i] = ledPixmap[2];
    for (i=valStop3; i < MAXLEDS; i++) pled[i] = ledPixmap[3];

    // create the frame
    setFrameStyle( QFrame::Panel | QFrame::Sunken );   
    setLineWidth( 2 );

    if (isVert) setFixedSize( ledWidth + 4, (stepsNo *
				ledHeight) + 4 );
    setBackgroundMode ( NoBackground );

    // create the pixmap used to buffer the VU-meter pixmap
    // same size as our widget itself, background color

    fullHeight = stepsNo * (ledHeight);

    // setup the timer event if damping != 0

    if (dampingFactor != 0) 
	timer = startTimer( 4*dampingFactor );
}


void Qvumeter::setValue( int value )
{
  // this slot redraws the widget according to value

  unsigned int i;

  // clamp value
    if (value < min)
	value = min;
    else
        if (value > max)
	    value = max;

  // save minimum and maximum values
  if (value < minInput) minInput = value;
  if (value > maxInput) maxInput = value;

  activeLed = (value + ((max-min)/(2*stepsNo))) / ((max-min) / stepsNo);

  if (activeLed != previousLed)
    {
      if (activeLed > previousLed)
      {
	if (activeLed >= maxLed)
	  {
	    if (mode == 3)
	      {
		sticky = 50;
		if (maxLed > 0)
		  bitBlt( this, 2, fullHeight - (maxLed *
       		    ledHeight)+2, (QPixmap *)pled[maxLed]);
	      }
	    maxLed = activeLed;
	  }
	for (i=previousLed; i < activeLed; i++)
	    bitBlt( this, 2, fullHeight - ((i + 1) *
	     ledHeight)+2, (QPixmap *)pled[i]);

	previousLed = activeLed;
      }
    else
      {
	int t = previousLed;
	  previousLed = activeLed;

        if (mode != 2)
	  for (i=activeLed; i < t; i++)
	      bitBlt( this, 2, fullHeight - ((i + 1) *
	        ledHeight)+2, (QPixmap *)whitePix);

      }
  }
}

void Qvumeter::fullCalc()
{
  // this slot redraws the widget according to value
  unsigned int i;

    for (i=0; i < previousLed; i++)
       bitBlt( this, 2, 2+fullHeight - ((i + 1) *
              ledHeight), (QPixmap *)pled[i]);

    for (i=previousLed; i < max; i++)
       bitBlt( this, 2, 2+fullHeight - ((i + 1) *
              ledHeight), whitePix);
}

void Qvumeter::paintEvent( QPaintEvent * )
{
    // we simply copy our buffer to the widget
    fullCalc();

    // and paint the frame
    QPainter paint;
    paint.begin( this );
    drawFrame( &paint );   
    paint.end();
}


void Qvumeter::timerEvent( QTimerEvent * )
{
  unsigned int i,j;

  if (mode == 1)
  {
    if (activeLed < maxLed)
      { // damp fall
	j = maxLed--; // one led at a time
	
        int k = fullHeight - ((j+1) * ledHeight)+2;
        if (k >= 2)
	  bitBlt( this, 2, k, whitePix);

	bitBlt( this, 2, fullHeight - (j *
	  ledHeight)+2, ledPixmap[3]);
    }
    else if (maxLed == 0)
      {
	  bitBlt( this, 2, fullHeight - ((1) *
	    ledHeight)+2, whitePix);
      }
  }
  else if (mode == 2)
  {
    if (activeLed < maxLed)
      { // damp fall
        j = maxLed--; // one led at a time
	
	bitBlt( this, 2, fullHeight - ((j + 1) *
	  ledHeight)+2, whitePix);

        for (i=activeLed; i < j; i++)
	  bitBlt( this, 2, fullHeight - ((i + 1) *
	    ledHeight)+2, (QPixmap *)pled[i]);
    }
    else if (activeLed == 0)
      {
	bitBlt( this, 2, fullHeight - ((1) *
          ledHeight)+2, whitePix);
      }
  }
  else if (mode == 3)
  {
    if (activeLed >= maxLed)
      {
	sticky = 50;
	if (maxLed > 0)
	  bitBlt( this, 2, fullHeight - (maxLed *
	    ledHeight)+2, (QPixmap *)pled[maxLed]);
	maxLed = activeLed;
      }

    if (sticky > 0)
      {
        j = maxLed;
	sticky--;
	if ((j > 0) && (sticky > 47))
	  bitBlt( this, 2, fullHeight - (j *
	    ledHeight)+2, ledPixmap[3]);
      }
    else
      {
	j = maxLed;
	maxLed = 0;
	if (j > 0)
	  bitBlt( this, 2, fullHeight - (j *
	   ledHeight)+2, whitePix);
      }
  }
}

void Qvumeter::setMode( int n )
{
  mode = n;
  fullCalc();

  killTimer(timer);
  if (n == 1)
    {
      timer = startTimer( dampingFactor*4 );
    }
  else if (n == 2)
    {
      timer = startTimer( dampingFactor );
    }
  else if (n == 3)
    {
      timer = startTimer( dampingFactor ); 
    }
}

