// This file is part of kseti,
// a utility to manage SETI@home calculations on Un*x boxes...
//
// Copyright (C) 1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.



#include <qdir.h>
#include <qstrlist.h>
#include <qprinter.h>
#include <qpainter.h>

#include <kiconloader.h>
#include <kmsgbox.h>
#include <kfiledialog.h>

#include <kseti.h>
#include "ksetiview.h"
#include "ksetidoc.h"
#include "resource.h"



KsetiApp::KsetiApp()
{
    config = kapp->getConfig();
    initMenuBar();
    initToolBar();
    initStatusBar();
    initKeyAccel();
    initDocument();
    initView();
    readOptions();
    disableCommand( ID_FILE_PRINT );
}



KsetiApp::~KsetiApp()
{
}



void KsetiApp::initKeyAccel()
{
    key_accel = new KAccel( this );
    // file_menu accelerators
    key_accel->connectItem( KAccel::Open, this, SLOT( slotFileOpen() ) );
    key_accel->connectItem( KAccel::Close, this, SLOT( slotFileClose() ) );
    key_accel->connectItem( KAccel::Print, this, SLOT( slotFilePrint() ) );
    key_accel->connectItem( KAccel::Quit, this, SLOT( slotFileQuit() ) );
    key_accel->changeMenuAccel(	file_menu, ID_FILE_OPEN, KAccel::Open );
    key_accel->changeMenuAccel( file_menu, ID_FILE_CLOSE, KAccel::Close );
    key_accel->changeMenuAccel(	file_menu, ID_FILE_PRINT, KAccel::Print );
    key_accel->changeMenuAccel(	file_menu, ID_FILE_QUIT, KAccel::Quit );
    // help_menu accelerators
    key_accel->connectItem( KAccel::Help, kapp, SLOT( appHelpActivated() ) );
    key_accel->readSettings();	
}



void KsetiApp::initMenuBar()
{
    recent_files_menu = new QPopupMenu();
    connect( recent_files_menu, SIGNAL(activated(int)), SLOT(slotFileOpenRecent(int)) );

    file_menu = new QPopupMenu();
    file_menu->insertItem(kapp->getMiniIcon(), i18n("New &Window"), ID_FILE_NEW_WINDOW );
    file_menu->insertSeparator();
    file_menu->insertItem(Icon("fileopen.xpm"), i18n("&Open..."), ID_FILE_OPEN );
    file_menu->insertItem(i18n("Open &recent"), recent_files_menu, ID_FILE_OPEN_RECENT );
    file_menu->insertItem(i18n("&Close"), ID_FILE_CLOSE );
    file_menu->insertSeparator();
    file_menu->insertItem(Icon("fileprint.xpm"), i18n("&Print..."), ID_FILE_PRINT );
    file_menu->insertSeparator();
    file_menu->insertItem(i18n("E&xit"), ID_FILE_QUIT );
	
    view_menu = new QPopupMenu();
    view_menu->setCheckable(true);
    view_menu->insertItem(i18n("&Toolbar"), ID_VIEW_TOOLBAR);
    view_menu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR );

    QString aboutstring = kapp->appName() + " " + VERSION + "\n\n";
    aboutstring += "Copyright (C) 1999 Jochen Kpper\n\n";
    aboutstring += i18n( "Display your Un*x's SETI@home statistics." );
    help_menu = new QPopupMenu();
    help_menu = kapp->getHelpMenu(true, aboutstring );

    menuBar()->insertItem(i18n("&File"), file_menu);
    menuBar()->insertItem(i18n("&View"), view_menu);
    menuBar()->insertSeparator();
    menuBar()->insertItem(i18n("&Help"), help_menu);

    connect(file_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
    connect(file_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));
    connect(view_menu,SIGNAL(activated(int)),SLOT(commandCallback(int)));
    connect(view_menu,SIGNAL(highlighted(int)),SLOT(statusCallback(int)));
}



void KsetiApp::initToolBar()
{
    toolBar()->insertButton( Icon( "fileopen.xpm" ), ID_FILE_OPEN, true, i18n( "Open Calculation Folder" ) );
    toolBar()->insertButton( Icon( "fileprint.xpm" ), ID_FILE_PRINT, true, i18n( "Print" ) );
    toolBar()->insertSeparator();
    toolBar()->insertButton( Icon( "help.xpm" ), ID_HELP_CONTENTS, SIGNAL( clicked() ),
			     kapp, SLOT( appHelpActivated() ), true, i18n( "Help" ) );

    // INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE WITH toolBar(n)
    // CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars by their according number
  	
    // connect for invoking the slot actions
    connect( toolBar(), SIGNAL( clicked( int ) ), SLOT( commandCallback( int ) ) );
    // connect for the status help on holding icons pressed with the mouse button
    connect( toolBar(), SIGNAL( pressed( int ) ), SLOT( statusCallback( int ) ) );
}



void KsetiApp::initStatusBar()
{
    // STATUSBAR
    // TODO: add your own items you need for displaying current application status.
    statusBar()->insertItem( i18n( "Ready." ), ID_STATUS_MSG );
}



void KsetiApp::initDocument()
{
    doc = new KsetiDoc(this);
    doc->newDocument();
}



void KsetiApp::initView()
{ 
    // create the main widget here that is managed by KTMainWindow's view-region and
    // connect the widget to your document to display document contents.
    view = new KsetiView( this );
    doc->addView( view );
    setView( view );
    QString caption = kapp->getCaption();	
    setCaption( caption + ": " + doc->getTitle() );
    view->show();
}



void KsetiApp::enableCommand( int id_ )
    // enable menu and toolbar functions by their ID's
{
    menuBar()->setItemEnabled( id_, true );
    toolBar()->setItemEnabled( id_, true );
}



void KsetiApp::disableCommand( int id_ )
    // disable menu and toolbar functions by their ID's
{
    menuBar()->setItemEnabled( id_, false );
    toolBar()->setItemEnabled( id_, false );
}



void KsetiApp::addRecentFile( const char* file )
{
    if( recent_files.find(file) == -1 ){
	if( recent_files.count() < 5 )
	    recent_files.insert( 0, file );
	else {
	    recent_files.remove( 4 );
	    recent_files.insert( 0, file );
	}
	recent_files_menu->clear();
	for ( int i = 0 ; i < static_cast< int >( recent_files.count() ); i++ ) {
	    recent_files_menu->insertItem( recent_files.at( i ) );
	}
    }
}



void KsetiApp::openDocumentFile( const char* _cmdl )
{
    slotStatusMsg( i18n( "Opening calculation folder..." ) );
    doc->openDocument( _cmdl );
    slotStatusMsg( i18n( "Ready." ) );
}



KsetiDoc* KsetiApp::getDocument() const
{
    return doc;
}



void KsetiApp::saveOptions()
{	
    config->setGroup( "General Options" );
    config->writeEntry( "Geometry", size() );
    config->writeEntry( "Show Toolbar", toolBar()->isVisible() );
    config->writeEntry( "Show Statusbar",statusBar()->isVisible() );
    config->writeEntry( "MenuBarPos", (int)menuBar()->menuBarPos() );
    config->writeEntry( "ToolBarPos",  (int)toolBar()->barPos() );
    config->writeEntry( "Recent Files", recent_files );
}



void KsetiApp::readOptions()
{
    config->setGroup("General Options");
    // bar status settings
    bool bViewToolbar = config->readBoolEntry( "Show Toolbar", true );
    view_menu->setItemChecked( ID_VIEW_TOOLBAR, bViewToolbar );
    if( ! bViewToolbar )
	enableToolBar( KToolBar::Hide );
    bool bViewStatusbar = config->readBoolEntry( "Show Statusbar", true );
    view_menu->setItemChecked( ID_VIEW_STATUSBAR, bViewStatusbar );
    if( ! bViewStatusbar )
	enableStatusBar( KStatusBar::Hide );
    // bar position settings
    KMenuBar::menuPosition menu_bar_pos;
    menu_bar_pos=(KMenuBar::menuPosition)config->readNumEntry( "MenuBarPos", KMenuBar::Top );
    KToolBar::BarPosition tool_bar_pos;
    tool_bar_pos=(KToolBar::BarPosition)config->readNumEntry( "ToolBarPos", KToolBar::Top );
    menuBar()->setMenuBarPos(menu_bar_pos);
    toolBar()->setBarPos(tool_bar_pos);
	
    // initialize the recent file list
    recent_files.setAutoDelete( true );
    config->readListEntry( "Recent Files", recent_files );
    uint i;
    for ( i = 0 ; i < recent_files.count(); i++ ) {
	recent_files_menu->insertItem( recent_files.at( i ) );
    }
    QSize size=config->readSizeEntry( "Geometry" );
    if( ! size.isEmpty() )
	resize( size );
}



void KsetiApp::saveProperties( KConfig* )
{
    if( doc->getTitle() != i18n( "Untitled" ) && ! doc->isModified() ) {
	return;
    } else {
	QString filename = doc->getPathName() + doc->getTitle();	
	config->writeEntry( "filename", filename );
	config->writeEntry( "modified", doc->isModified() );
	// const char *tempname = kapp->tempSaveName( filename );
	// doc->saveDocument( tempname );
    }
}



void KsetiApp::readProperties(KConfig*)
{
    QString filename = config->readEntry("filename","");
    bool modified = config->readBoolEntry("modified",false);
    if( modified ) {
	bool b_canRecover;
	QString tempname = kapp->checkRecoverFile(filename,b_canRecover);
	if( b_canRecover ) {
	    doc->openDocument(tempname);
	    doc->setModified();
	    QFileInfo info(filename);
	    doc->pathName(info.absFilePath());
	    doc->title(info.fileName());
	    QFile::remove(tempname);
	}
    } else if( ! filename.isEmpty() ) {
	doc->openDocument( filename );
    }
    QString caption = kapp->getCaption();	
    setCaption( caption + ": " + doc->getTitle() );
}



bool KsetiApp::queryClose()
{
    // return doc->saveModified();
    return true;
}



bool KsetiApp::queryExit()
{
    saveOptions();
    return true;
}



void KsetiApp::slotFileNewWindow()
{
    slotStatusMsg( i18n( "Opening a new Application window..." ) );
    KsetiApp* new_window = new KsetiApp();
    new_window->show();
    slotStatusMsg( i18n( "Ready." ) );
}



void KsetiApp::slotFileNew()
{
    slotStatusMsg( i18n( "Creating new document...") );
    // if( ! doc->saveModified() )
    //     return;
    doc->newDocument();		
    QString caption = kapp->getCaption();	
    setCaption( caption + ": " + doc->getTitle() );
    slotStatusMsg( i18n( "Ready." ) );
}



void KsetiApp::slotFileOpen()
{
    slotStatusMsg( i18n("Opening calculation folder...") );
    // if( ! doc->saveModified() )
    //     return;
    QString dirToOpen = KFileDialog::getDirectory( QDir::homeDirPath(),
						   this, i18n( "Open Folder..." ) );
    if( ! dirToOpen.isEmpty() ) {
	doc->openDocument( dirToOpen );
	QString caption = kapp->getCaption();	
	setCaption( caption + ": " + doc->getTitle() );
	addRecentFile( dirToOpen );
    }
    slotStatusMsg( i18n( "Ready." ) );
}



void KsetiApp::slotFileOpenRecent(int id_)
{
    slotStatusMsg( i18n( "Opening calculation folder..." ) );
    // if( ! doc->saveModified() )
    //     return;
    doc->openDocument( recent_files.at( id_ ) );
    QString caption = kapp->getCaption();	
    setCaption( caption + ": " + doc->getTitle() );
    slotStatusMsg( i18n( "Ready." ) );
}



void KsetiApp::slotFileClose()
{
    slotStatusMsg( i18n( "Closing file..." ) );
    close();
}



void KsetiApp::slotFilePrint()
{
    slotStatusMsg( i18n( "Printing..." ) );
    QPrinter printer;
    if ( printer.setup( this ) ) {
	view->print( &printer );
    }
    slotStatusMsg( i18n( "Ready." ) );
}



void KsetiApp::slotFileQuit()
{
    saveOptions();
    // close the first window, the list makes the next one the first again.
    // This ensures that queryClose() is called on each window to ask for closing
    KTMainWindow* w;
    if(memberList){
	for(w=memberList->first(); w; w=memberList->first()){
	    // only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
	    // the window and the application stay open.
	    if(!w->close())
		break;
	}
    }	
}



void KsetiApp::slotViewToolBar()
{
    // turn Toolbar on or off
    if( view_menu->isItemChecked(ID_VIEW_TOOLBAR))
	view_menu->setItemChecked(ID_VIEW_TOOLBAR, false);
    else
	view_menu->setItemChecked(ID_VIEW_TOOLBAR, true);
    enableToolBar();
    slotStatusMsg( i18n( "Ready." ) );
}



void KsetiApp::slotViewStatusBar()
{
    //turn Statusbar on or off
    if( view_menu->isItemChecked(ID_VIEW_STATUSBAR))
	view_menu->setItemChecked(ID_VIEW_STATUSBAR, false);
    else
	view_menu->setItemChecked(ID_VIEW_STATUSBAR, true);
    enableStatusBar();
    slotStatusMsg( i18n( "Ready." ) );
}



void KsetiApp::slotStatusMsg(const char *text)
    // change status message permanently
{
    statusBar()->clear();
    statusBar()->changeItem( text, ID_STATUS_MSG );
}



void KsetiApp::slotStatusHelpMsg( const char *text )
    // change status message of whole statusbar temporary ( text, msec )
{
    statusBar()->message( text, 2000 );
}



void KsetiApp::commandCallback( int id_ )
{
    switch( id_ ) {
    case ID_FILE_NEW_WINDOW:
	slotFileNewWindow();
	break;
    case ID_FILE_NEW:
	slotFileNew();
	break;
    case ID_FILE_OPEN:
	slotFileOpen();
	break;
    case ID_FILE_CLOSE:
	slotFileClose();
	break;
    case ID_FILE_PRINT:
	slotFilePrint();
	break;
    case ID_FILE_QUIT:
	slotFileQuit();
	break;
    case ID_VIEW_TOOLBAR:
	slotViewToolBar();
	break;
    case ID_VIEW_STATUSBAR:
	slotViewStatusBar();
	break;
    default:
	break;
    }
}



void KsetiApp::statusCallback( int id_ ){
    switch (id_){
    case ID_FILE_NEW_WINDOW:
	slotStatusHelpMsg( i18n( "Opens a new application window" ) );
	break;
    case ID_FILE_OPEN:
	slotStatusHelpMsg( i18n( "Opens an SETI@home calculation folder" ) );
	break;
    case ID_FILE_OPEN_RECENT:
	slotStatusHelpMsg( i18n( "Opens a recently visited SETI@home calculation folder" ) );
	break;
    case ID_FILE_CLOSE:
	slotStatusHelpMsg( i18n( "Closes the actual SETI@home calculation folder" ) );
	break;
    case ID_FILE_PRINT:
	slotStatusHelpMsg( i18n( "Prints the current statistics" ) );
	break;
    case ID_FILE_QUIT:
    {
	QString caption = kapp->getCaption();
	slotStatusHelpMsg( i18n( "Exits " ) + caption );
    }
    break;
    case ID_VIEW_TOOLBAR:
	slotStatusHelpMsg( i18n( "Enables / disables the actual Toolbar" ) );
	break;
    case ID_VIEW_STATUSBAR:
	slotStatusHelpMsg( i18n( "Enables / disables the actual Statusbar" ) );
	break;
    default:
	break;
    }
}




//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
