// This file is part of kseti,
// a utility to manage SETI@home calculations on Un*x boxes..
//
// Copyright (C) 1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.



#include <iostream>

#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qtextstream.h>
#include <qwidget.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "ksetidoc.h"
#include "kseti.h"
#include "ksetiview.h"



QList<KsetiView>* KsetiDoc::viewList = 0L;



KsetiDoc::KsetiDoc(QWidget *parent, const char* name)
    : QObject(parent, name),
      stateFile( 0 ), userFile( 0 ), workFile( 0 ),
      timer( 0 )
{
    if( !viewList )
	viewList = new QList<KsetiView>;
    viewList->setAutoDelete(true);
}


KsetiDoc::~KsetiDoc()
{
    if( 0 != stateFile )
        delete stateFile;
    if( 0 != workFile )
        delete workFile;
    if( 0 != timer )
        delete timer;
}


void KsetiDoc::addView( KsetiView *m_pView )
{
    viewList->append(m_pView);
}


void KsetiDoc::removeView(KsetiView *m_pView)
{
    viewList->remove( m_pView );
}


const QString& KsetiDoc::getPathName() const
{
    return m_path;
}


void KsetiDoc::slotUpdateAllViews( KsetiView *pSender )
{
    if( viewList ) {
	KsetiView *w;
	for( w = viewList->first(); w; w = viewList->next() ) {
	    if( w != pSender)
		w->repaint();
	}
    }
}


void KsetiDoc::pathName( const char *path_name)
{
    m_path = path_name;
}


void KsetiDoc::title( const char *title)
{
    m_title = title;
}


const QString& KsetiDoc::getTitle() const
{
    return m_title;
}


void KsetiDoc::closeDocument()
{
    timer->stop();
    deleteContents();
    delete stateFile;
    delete userFile;
    delete workFile;
}


bool KsetiDoc::newDocument()
{
    // TODO: Add your document initialization code here
    b_modified = false;
    m_path=QDir::homeDirPath();
    m_title = i18n( "Untitled" );
    return true;
}


bool KsetiDoc::openDocument( const char* filename, const char* format )
{
    m_title = filename;
    m_path = filename;

    if( 0 != stateFile )
	delete stateFile;
    if( 0 != userFile )
	delete userFile;
    if( 0 != workFile )
	delete workFile;
    stateFile = new QFile( QString( filename ) + "/state.txt" );
    stateFile->open( IO_Raw | IO_ReadOnly );
    userFile = new QFile( QString( filename ) + "/user_info.txt" );
    userFile->open( IO_Raw | IO_ReadOnly );
    workFile = new QFile( QString( filename ) + "/work_unit.txt" );
    workFile->open( IO_Raw | IO_ReadOnly );
    readData();
    if( 0 == timer ) {
        timer = new QTimer( this );
        connect( timer, SIGNAL( timeout() ), this, SLOT( readData() ) );
    }
    timer->start( 5000, false );
    b_modified = false;
    return true;
}


// bool KsetiDoc::saveDocument(const char* filename, const char* format)
// {
//    // TODO: Add your document saving code here
//	  b_modified = false;
//    return true;
//}


void KsetiDoc::deleteContents()
{
    // TODO: Add implementation to delete the document contents
    // nothing to do here - Jochen !
}


void KsetiDoc::readData()
{
    // status file
    QTextStream state( stateFile );
    cpustring = readEntry( state, "cpu" );
    cpuvalue = cpustring.toDouble();
    int d = static_cast< int >( cpuvalue ) / 86400;
    int h = static_cast< int >( cpuvalue ) % 86400 / 3600;
    int m = static_cast< int >( cpuvalue ) % 3600 / 60;
    int s = static_cast< int >( cpuvalue ) % 60;
    if( d > 0 )
	cpustring.sprintf( "%dd %d:%d:%d", d, h, m, s );
    else
	cpustring.sprintf( "%d:%d:%d", h, m, s );
    progressstring = readEntry( state, "prog" );
    progressvalue = progressstring.toDouble();
    if( progressvalue != 0.0 ) {
	int d = static_cast< int >( cpuvalue / progressvalue ) / 86400;
	int h = static_cast< int >( cpuvalue / progressvalue ) % 86400 / 3600;
	int m = static_cast< int >( cpuvalue / progressvalue ) % 3600 / 60;
	int s = static_cast< int >( cpuvalue / progressvalue ) % 60;
	if( d > 0 )
	    overallstring.sprintf( "%dd %d:%d:%d", d, h, m, s );
	else
	    overallstring.sprintf( "%d:%d:%d", h, m, s );
    } else {
        overallstring = i18n( "infinity" );
    }
    // user file
    QTextStream user( userFile );
    emailstring = readEntry( user, "email_addr" );
    userstring = readEntry( user, "name" );
    resultvalue = readEntry( user, "results" ).toInt();
    // work file
    QTextStream work( workFile );
    datestring = readEntry( work, "time_recorded" );
    datestring = datestring.mid( datestring.find( "(" ) + 1,
				 datestring.find( ")" ) - datestring.find( "(" ) - 1 );
    // show new values
    slotUpdateAllViews( 0 );
    cout << "Update" << endl;
}



QString KsetiDoc::readEntry( QTextStream& stream, const QString& key )
{
    QString value;
    QRegExp regexp( QString( "^" ) + key + "=", false );
    stream.device()->at( 0 );
    while( ! stream.eof() ) {
        QString line = stream.readLine();
        if( 0 < line.contains( regexp ) ) {
	    value = line.mid( line.find( "=" ) + 1, 256 );
	    break;
        }
    }
    return value;
}




//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
