/**********************************************************************

	File: mainwindow.C

	Author:  Sven Schmidt
	Email:   sven.schmidt@cern.ch
	Version: 0.3
	Date:    July 5, 1998

	Description: K Answering Machine (kam) is a frontend for the 
	             vbox package (ISDN answering machine). It displays
		     your voice mail inbox and allows you to play and 
		     delete messages.

 *********************************************************************/

#include "mainwindow.H"
#include "kwm.h"


const int appwidth  = 500;
const int appheight = 300;
const int bwidth  = 50;
const int bheight = 24;


MainWindow::MainWindow() {

#ifdef TEST
  setCaption( "K Answering Machine - TEST VERSION" );
#else
  setCaption( "K Answering Machine" );
#endif
  resize( appwidth, appheight );
  setMinimumSize( appwidth, appheight );
  setMaximumSize( appwidth, appheight );
  

  // initialize global objects and connections

  Timer = new QTimer( this );
  connect( Timer, SIGNAL( timeout() ), SLOT( updateCallList() ) );
  connect( Timer, SIGNAL( timeout() ), SLOT( updateAnswerB() ) );

  PlayProcess = new TPlay();

  // get a pointer to KConfig
  Conf = kapp->getConfig();

  connect( this, SIGNAL( configUpdated() ), SLOT( updateCallList() ) );
  connect( this, SIGNAL( configUpdated() ), SLOT( updateAnswerB() ) );
  connect( this, SIGNAL( configUpdated() ), SLOT( updateAnswerB() ) );
  
  Accel = new QAccel( this );
  Accel->connectItem( Accel->insertItem( Key_Right ), 
		      this,
		      SLOT( highlightNext() ) );          
  Accel->connectItem( Accel->insertItem( Key_Down ), 
		      this,
		      SLOT( highlightNext() ) );          
  Accel->connectItem( Accel->insertItem( Key_Left ), 
		      this,
		      SLOT( highlightPrev() ) );          
  Accel->connectItem( Accel->insertItem( Key_Up ), 
		      this,
		      SLOT( highlightPrev() ) );          


  // Before doing anything application specific, get the configuration.
  // Some things rely on the vbox version number for example.
  ConfData = getConfig();


  // set up GUI objects

  PlayB = new QPushButton( this, "Play" );
  //PlayB->setToggleButton( TRUE );
  PlayB->setText( "Play" );
  //PlayB->setPixmap( QPixmap( "play.xpm" ) );
  PlayB->setAccel( Key_P );
  connect( PlayB, SIGNAL( clicked() ), this, SLOT( playPressed() ) );
  
  StopB = new QPushButton( this, "Stop" );
  StopB->setText( "Stop" );
  //StopB->setPixmap( QPixmap( "stop.xpm" ) );
  StopB->setAccel( Key_S );
  connect( StopB, SIGNAL(clicked()), this, SLOT(stopPressed()) );
  
  DeleteB = new QPushButton( this, "Delete" );
  DeleteB->setText( "Delete" );
  DeleteB->setAccel( Key_D );
  connect( DeleteB, SIGNAL(clicked()), this, SLOT(deletePressed()) );
  
  CheckB = new QPushButton( this, "Check" );
  CheckB->setText( "Check" );
  CheckB->setAccel( Key_C );
  connect( CheckB, SIGNAL(clicked()), this, SLOT(checkPressed()) );
  
  AnswerB = new QCheckBox( "Do Not Answer", this );
  AnswerB->setAccel( Key_N );
  connect( AnswerB, SIGNAL( clicked() ), this, SLOT( answerPressed() ) );

  ConfigB = new QPushButton( this, "Config" );
  ConfigB->setText( "Config" );
  ConfigB->setAccel( Key_K );
  connect( ConfigB, SIGNAL(clicked()), SLOT(configPressed()) );
  
  HideB = new QPushButton( this, "Hide" );
  HideB->setText( "Hide" );
  HideB->setAccel( Key_H );
  connect( HideB, SIGNAL(clicked()), SLOT(hidePressed()) );
  
  QuitB = new QPushButton( this, "Quit" );
  QuitB->setText( "Quit" );
  QuitB->setAccel( Key_Q );
  connect( QuitB, SIGNAL(clicked()), this, SLOT(quitPressed()) );

  // volume slider
  Volume = 3;
  VolumeS = new QSlider( 0,         // min
			 10,        // max
			 3,         // page step
			 Volume,    // initial value
			 QSlider::Horizontal, 
			 this, "Volume" );
  connect( VolumeS, SIGNAL( valueChanged( int ) ),
	   this, SLOT( setVolume( int ) ) );

  VolumeL = new QLabel( "Volume", this );
  VolumeL->setAlignment( AlignHCenter );

  CallList = new QStrList();
  Call.clear();

  ListBox = new KTabListBox( this, "ListBox", 4 );
  ListBox->setSeparator( '\t' );
  ListBox->setColumn( 0, "CallerID", 90 );
  ListBox->setColumn( 1, "Date", 110 );
  ListBox->setColumn( 2, "Time", 70 );
  ListBox->setColumn( 3, "Length", 40 );
  QString w = ConfData.ColumnWidth;
  for ( int i = 0; i < ListBox->numCols(); i++ ) {
    ListBox->setColumnWidth( i, w.left( w.find( ' ' ) ).toInt() );
    w = w.remove( 0, w.find( ' ' )+1 );
  }

  
  connect( ListBox, SIGNAL( selected(int,int) ),
	   SLOT( listDoubleClicked(int,int) ) );

  NewCountLCD = new QLCDNumber( this, "NewCountLCD" );
  {
    QColorGroup normal( QColor( QRgb(43008) ), 
			QColor( QRgb(8421504) ), 
			QColor( QRgb(16777215) ), 
			QColor( QRgb(6316128) ), 
			QColor( QRgb(10789024) ), 
			QColor( QRgb(0) ), 
			QColor( QRgb(32768) ) );
    QColorGroup disabled( QColor( QRgb(8421504) ), 
			  QColor( QRgb(12632256) ), 
			  QColor( QRgb(16777215) ), 
			  QColor( QRgb(6316128) ), 
			  QColor( QRgb(10789024) ), 
			  QColor( QRgb(8421504) ), 
			  QColor( QRgb(12632256) ) );
    QColorGroup active( QColor( QRgb(0) ), 
			QColor( QRgb(12632256) ), 
			QColor( QRgb(16777215) ), 
			QColor( QRgb(6316128) ), 
			QColor( QRgb(10789024) ), 
			QColor( QRgb(0) ), 
			QColor( QRgb(16777215) ) );
    QPalette palette( normal, disabled, active );
    NewCountLCD->setPalette( palette );
  }
  NewCountLCD->setNumDigits( 2 );
  NewCountLCD->setSegmentStyle( QLCDNumber::Filled );
  connect( this, SIGNAL( newCountChanged(int) ),
	   NewCountLCD, SLOT( display(int) ) );
  

  // Now that everything is set up verify configuration and write it
  // (to make sure there's a working version on disk).
  updateConfig( &ConfData );

}



MainWindow::~MainWindow() {

  writeConfig();

}




void MainWindow::updateCallList() {

  if ( ! InDir->exists() ) {
    QMessageBox::warning( this, "Error: ",
			  "Incoming directory\n"
			  "does not exist!\n", "OK", 0);
    return;
  }
  
  // get file list
  QFileInfoList *call = 
    (QFileInfoList *) InDir->entryInfoList( "*", QDir::Files, QDir::Name );
  
  // clear call list
  ListBox->clear();
  ListBox->setAutoUpdate( FALSE );

  Call.clear();
  NewCount = 0;
  
  // loop over files in incoming dir and add their names to the list
  for ( QFileInfo *file = call->first(); file != 0; file = call->next() ) {

    Call.insert( 0, new TCall( file ) );
    ListBox->insertItem( Call.current()->listBoxItem().data(), 0 );
    if ( Call.at( 0 )->isNew() ) {
      NewCount++;
      ListBox->changeItemColor( red, 0 );
    }

  }
  
  ListBox->setAutoUpdate( TRUE );
  ListBox->repaint();

  emit newCountChanged( NewCount );

}



void MainWindow::updateAnswerB() {

  QFileInfo f( stopPath().data() );
  AnswerB->setChecked( f.exists() );

}


void MainWindow::playPressed() {

  if ( ListBox->count() == 0 ) return;

  if ( ListBox->currentItem() == - 1 ) {
    // select first message
    ListBox->setCurrentItem( 0 );
  }
  
  // play selected message
  for ( int i=0; i < (int)ListBox->count(); i++ ) {
    if ( ListBox->isMarked( i ) ) {
      playCall( i );
      return;
    }
  }

  // no selected message - play the first one
  playCall( 0 );

}


void MainWindow::playCall( int id ) {

  // this should not happen
  if ( Call.count() <= (uint)id ) return;

  Call.at( id );
  
  // play selected messages
  if ( InDir->exists( Call.current()->fileName() ) ) {


    // create new play process
    PlayProcess->start( InDir->absFilePath(Call.current()->fileName()).data(),
			Volume );


    if ( Call.current()->isNew() ) {
      Call.current()->setNew( FALSE );
      // mark file as executable to indicate its been played
      if ( chmod( InDir->absFilePath( Call.current()->fileName() ),
	     S_IRUSR | S_IWUSR | S_IXUSR ) != 0 )
	return; // chmod error occured
      // mark item black
      ListBox->changeItemColor( black, Call.at() );
      // decrease LCD new count
      emit newCountChanged( --NewCount );
    }

  }
  
}




void MainWindow::stopPressed() {

  if ( PlayProcess ) {
    PlayProcess->stop();
  }

}


void MainWindow::quitPressed() {

  stopPressed();
  kapp->quit();

}


void MainWindow::deletePressed() {

  if ( (ListBox->count() == 0) || (ListBox->currentItem() == - 1) 
       || (Call.count() == 0) ) return;

  for ( int i=0; i < (int)ListBox->count(); i++ ) {
    if ( ListBox->isMarked( i ) ) deleteCall( i );
  }
  
}


void MainWindow::listDoubleClicked( int index, int ) {

  playCall( index );

}


void MainWindow::deleteCall( int id ) {
  // this should not happen
  if ( Call.count() <= (uint)id ) return;

  Call.at( id );

  if ( InDir->exists( Call.current()->fileName() ) ) {

    if ( Call.current()->isNew() ) {

      switch ( QMessageBox::warning( this, "Kam: Delete new message?",
				     "You havent listened to this call!\n"
				     "It could be that million dollar deal,\n"
				     "you know...\n"
				     "Sure to proceed?\n",
				     QMessageBox::Ok | QMessageBox::Default,
				     QMessageBox::Cancel | 
				     QMessageBox::Escape ) ) {
      case QMessageBox::Ok:
	emit newCountChanged( --NewCount );
	break;
      case QMessageBox::Cancel:
	return;
      }

    }

#ifdef TEST
    debug( "Removing %s\n", Call.current()->fileName().data() );
#else
    InDir->remove( Call.current()->fileName() );
#endif

    ListBox->removeItem( Call.at() );

    if ( ListBox->count() > 0 ) {
      if ( Call.at() >= (int)ListBox->count() )
	ListBox->markItem( (int)ListBox->count() - 1 );
      else
	ListBox->markItem( Call.at() );
    }

    Call.remove();
    if ( Call.count() == 0 ) DeleteB->setEnabled( FALSE );
  }


}


void MainWindow::checkPressed() {

  checkInbox();

}


void MainWindow::answerPressed() {

  if ( AnswerB->isChecked() ) {
    FILE *f = fopen( stopPath(), "w" );
    if( f ) fclose( f );
  } else {
    unlink( stopPath() );
  }

}


void MainWindow::hidePressed() {

  hideMainWindow();

}

void MainWindow::checkInbox() {

  updateCallList();

}


void MainWindow::configPressed() {

  TConfig conf( this, "Configuration", &ConfData );

  connect( &conf, SIGNAL( configChanged( TConfigData* ) ),
	   SLOT( updateConfig( TConfigData* ) ) );

  conf.exec();

}


TConfigData MainWindow::getConfig() {

  TConfigData  confdata;

  // set spool directory
  QString  spooldir = "/var/spool/vbox/";
  spooldir += cuserid( NULL ); // system call ( Linux man(3) )
  spooldir += "/";
  confdata.SpoolDir = Conf->readEntry( "SpoolDir", spooldir );

  // construct default incoming path
  QString  defaultdir = spooldir.data();
  defaultdir += "incoming/";

  // get incoming dir from config file or use default
  confdata.IndirPath = Conf->readEntry( "IncomingDir", defaultdir );


  // update interval default: 5 minutes
  confdata.UpdateInterval = Conf->readNumEntry( "UpdateInterval", 5*60 );

  confdata.VboxVersion = Conf->readNumEntry( "VboxVersion", 1 );

  confdata.Hidden = Conf->readBoolEntry( "Hidden", FALSE );

  // get column widths
  confdata.ColumnWidth = Conf->readEntry( "ColumnWidth", "90 110 70 40" );

  return confdata;

}



void MainWindow::updateConfig( TConfigData *conf ) {

  ConfData = *conf;

  Timer->stop();

  InDir = new QDir( ConfData.IndirPath.data() );

  // set Timer only if directory exists (to prevent the warning from
  // popping up at intervals ;-)
  if ( InDir->exists() ) {

    Timer->start( ConfData.UpdateInterval*1000 ); // time is saved in seconds
 
  }

  writeConfig();

  emit configUpdated();

}


void MainWindow::writeConfig() {

  debug( "Writing configuration!" );
  Conf->writeEntry( "SpoolDir", ConfData.SpoolDir );
  Conf->writeEntry( "IncomingDir", ConfData.IndirPath );
  Conf->writeEntry( "UpdateInterval", ConfData.UpdateInterval );
  Conf->writeEntry( "VboxVersion", ConfData.VboxVersion );
  Conf->writeEntry( "Hidden", !isVisible() );

  QString w;
  w.sprintf( "%i", ListBox->columnWidth( 0 ) );
  for ( int i=1; i < ListBox->numCols(); i++ )
    w.sprintf( "%s %i", w.data(), ListBox->columnWidth( i ) );
  Conf->writeEntry( "ColumnWidth", w );


}



void MainWindow::setVolume( int volume ) {

  Volume = volume;

}


void MainWindow::resizeEvent( QResizeEvent *ev ){
    KTopLevelWidget::resizeEvent( ev );

    PlayB->setGeometry( 10, 60, bwidth, bheight );
    StopB->setGeometry( PlayB->x(), PlayB->y() + PlayB->height() + 10, 
		       bwidth, bheight );
    DeleteB->setGeometry( StopB->x(), StopB->y() + StopB->height() + 10, 
			 bwidth, bheight );
    CheckB->setGeometry( DeleteB->x(), DeleteB->y() + DeleteB->height() + 10, 
			 bwidth, bheight );

    ConfigB->setGeometry( CheckB->x(), CheckB->y() + CheckB->height() + 10, 
			 bwidth, bheight );

    VolumeS->setGeometry( ConfigB->x(), ConfigB->y() + ConfigB->height() + 10,
			  bwidth, 16 );
    VolumeL->setGeometry( VolumeS->x(), 
			  VolumeS->y() + VolumeS->height(),
			  bwidth, 16 );

    AnswerB->setGeometry( VolumeL->x(), height() - bheight - 10, 
			  3*bwidth, bheight );

    QuitB->setGeometry( width() - bwidth -10, 
			height() - bheight - 10, 
			bwidth, bheight );

    HideB->setGeometry( QuitB->x() - QuitB->width() -10, 
			QuitB->y(), 
			bwidth, bheight );

    ListBox->setGeometry( PlayB->x() + bwidth + 10, 
			  10, 
			  width() - ( PlayB->x() + PlayB->width() +10 ) -10, 
			  height() - bheight -30 );
    NewCountLCD->setGeometry( 10, 10, bwidth, bwidth-10 );
}


bool MainWindow::isHidden() {

  return ConfData.Hidden;

}


void MainWindow::setHidden( bool status ) {

  ConfData.Hidden = status;

}


void MainWindow::show() {

  if ( ! isHidden() ) KTopLevelWidget::show();

}

void MainWindow::showMainWindow() {

  setHidden( FALSE );
  show();

}

void MainWindow::hideMainWindow() {

  setHidden( TRUE );
  hide();

}


void MainWindow::highlightPrev() {

  for ( int i=1; i < (int)ListBox->count(); i++ ) {
    if ( ListBox->isMarked( i ) ) {
      ListBox->unmarkAll();
      ListBox->markItem( i-1 );
      return;
    }
  }

}

void MainWindow::highlightNext() {

  for ( int i=0; i < (int)ListBox->count()-1; i++ ) {
    if ( ListBox->isMarked( i ) ) {
      ListBox->unmarkAll();
      ListBox->markItem( i+1 );
      return;
    }
  }

}



QString MainWindow::stopPath() {

  QString sp;

  switch( ConfData.VboxVersion ) {
  case 1:
    sp.sprintf( "%s/.vboxstop", QDir::home().path() );
    break;
  case 2:
    sp = ConfData.SpoolDir.data();
    sp += "vboxctrl-stop";
    break;
  }

  return sp;
}














//////////////////////////////////
// Docked Window implementation //
//////////////////////////////////

DockedWindow::DockedWindow( const char *name )
  : QPushButton( name ) {

    // save the original palette to be able to restore it
    StdPalette = palette();

    Menu = new QPopupMenu();
    Menu->insertItem( "Show KAM Window", this, SLOT(selectShow()) );
    Menu->insertItem( "Quit KAM", this, SLOT(selectQuit()) );

    connect( this, SIGNAL( rightClicked() ), this, SLOT( showMenu() ) );
    connect( this, SIGNAL( leftClicked() ), this, SLOT( selectShow() ) );

}


void DockedWindow::updateMessageCount( int count ) {

  QString s;
  s.sprintf( "%i", count );
  
  if ( count ) setPalette( QPalette( green ) );
  else setPalette( StdPalette );

  setText( s.data() );

}


void DockedWindow::showMenu() {

  // menu placement code "borrowed"
  // from kISDN ( C. Pfeiffer, T. Westheider )
  Menu->move(-1000,-1000);
  Menu->show();
  Menu->hide();
  QRect g = KWM::geometry( winId() );
  if ( g.x() > QApplication::desktop()->width()/2 
       && g.y() + Menu->height() > QApplication::desktop()->height() )
    Menu->popup( QPoint( g.x(), g.y() - Menu->height() ) );
  else 
    Menu->popup( QPoint( g.x() + g.width(), g.y() + g.height() ) );

}


void DockedWindow::selectShow() {

  emit showSelected();

}


void DockedWindow::selectQuit() {

  emit quitSelected();

}

void DockedWindow::mousePressEvent( QMouseEvent *event ) {

  if ( event->button() == RightButton )
    emit rightClicked();
  if ( event->button() == LeftButton )
    emit leftClicked();

}




#include "mainwindow.moc"


