/****************************************************************************
 * KPrintcap - encapsulates processing of /etc/printcap. Allows for adding,
 *  removing and editing printer records.
 *
 *  Copyright (C) 1998 Neil Hart (nhart@magma.ca)
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Id: kprintcap.cpp,v 1.13 1998/05/06 12:32:14 nhart Exp $
 *                                                                   
 ****************************************************************************/

#include "kprintcap.h"
#include "kfiledir.h"
#include "kprinterrs.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <qstring.h>
#include <qfile.h>
#include <qdir.h>
#include <qregexp.h>

#define LINELENGTH 1024

static char* RCSinfo = "$Id: kprintcap.cpp,v 1.13 1998/05/06 12:32:14 nhart Exp $";

// Constructor -- initialise all pointers to zero
KPrintcap::KPrintcap()
{
	firstPrintcapLine = NULL;
	lastPrintcapLine = NULL;

	int i;
	for (i = 0; i < PRINTCAPTABLESZ; i++)
	{
		printcapTable[i].firstLine          = NULL;
		printcapTable[i].lastLine           = NULL;
		printcapTable[i].printerDescription = NULL;
	}

	printcapTableIdx = 0;
}

// Destructor -- delete the linked list of printcap lines and the printer
//  description entries in the printcapTable. Close /etc/printcap.
KPrintcap::~KPrintcap()
{
	struct KPrintcapLine* currentPrintcapLine = firstPrintcapLine;
	struct KPrintcapLine* nextPrintcapLine = NULL;
	while (currentPrintcapLine != NULL)
	{
		nextPrintcapLine = currentPrintcapLine->next;
		if (currentPrintcapLine->line != NULL)
			delete currentPrintcapLine->line;
		delete currentPrintcapLine;
		currentPrintcapLine = nextPrintcapLine;
	}

	int i;
	for (i = 0; i < PRINTCAPTABLESZ; i++)
		if (printcapTable[i].printerDescription != NULL)
			delete printcapTable[i].printerDescription;
}

// Initialisation -- read /etc/printcap creating a KPrintcapLine for each
//  line of the file. When we find a printer record, add a KPrintcapTableEntry
//  for it.
bool KPrintcap::init()
{
	// read printcap into linked list
	if ( !readPrintcapIntoList() )
		return false;

	// step through linked list, finding printer records. An entry begins
	//  either with the printer name as the first characters of the line,
	//  for example, lp:, or with a KCMPRINTER-1 comment line. An entry
	//  ends when we find a line containing only whitespace (generally
	//  just a newline), or a comment line. If the latter, then we've found
	//  the start of a new entry (ie. there's no blank line between entries)
	bool withinPrinterEntry = false;
	KPrintcapLine* kplp = firstPrintcapLine;
	while ( kplp != NULL )
	{
		char first_char;
		memcpy(&first_char, kplp->line->data(), 1);
		if ( !withinPrinterEntry )
		{
			if ( kplp->line->contains("##KCMPRINTER-1##") != 0)
			{
				// first line of a kcmprinter entry
				withinPrinterEntry = true;
				printcapTable[printcapTableIdx].firstLine = kplp;
				continue;
			}
			if ( isalpha(first_char) )
			{
				// first line of a non-kcmprinter entry
				withinPrinterEntry = true;
				printcapTable[printcapTableIdx].firstLine = kplp;
				continue;
			}
		}
		else
		{
			if ( first_char == '\n' )
			{
				withinPrinterEntry = false;
				printcapTable[printcapTableIdx].lastLine = kplp->prev;
				QString* qsp = new QString(256);
				createPrinterDescription( printcapTableIdx, qsp );
				printcapTable[printcapTableIdx].printerDescription = qsp;
				printcapTableIdx++;
			}
		}
		
		kplp = kplp->next;
	}

	if ( withinPrinterEntry )
	{
		// the last entry goes right to the end of the file
		printcapTable[printcapTableIdx].lastLine = lastPrintcapLine;
		QString* qsp = new QString(256);
		createPrinterDescription( printcapTableIdx, qsp );
		printcapTable[printcapTableIdx].printerDescription = qsp;
		printcapTableIdx++;
	}

	return true;
}
		

int KPrintcap::numberOfPrinters()
{
	return printcapTableIdx;
}


int KPrintcap::addNewLocalPrinter(const QString& name, const QString& model, 
		const QString& spooldir, const QString& filter, const QString& device,
		const QString& note)
{
	QString* qsp;		// used as a temporary item for writing lines out
	KPrintcapLine* kplp;	// ditto

	// create new record in linked list
	// first, create a blank line, followed by the two comment records
	kplp = new KPrintcapLine;
	qsp = new QString("\n");
	kplp->line = qsp;
	addToLinkedList(kplp);
	printcapTable[printcapTableIdx].firstLine = kplp;
	printcapTable[printcapTableIdx].isLocal = true;

	kplp = new KPrintcapLine;
	qsp = new QString("##KCMPRINTER-1## ");
	*qsp += name;
	*qsp += " local {";
	*qsp += model;
	*qsp += "}\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString("##KCMPRINTER-2## {");
	*qsp += note;
	*qsp += "}\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	// now, add the lines for the name, spooldir, device and filter
	kplp = new KPrintcapLine;
	qsp = new QString( name.data() );
	*qsp += ":\\\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString( "\t:sd=" );
	*qsp += spooldir;
	*qsp += ":\\\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	kplp->line = new QString("\t:mx#0:\\\n");
	addToLinkedList(kplp);
	kplp = new KPrintcapLine;
	kplp->line = new QString("\t:sh:\\\n");
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString( "\t:lp=" );
	*qsp += device;
	*qsp += ":\\\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString( "\t:if=" );
	*qsp += filter;
	*qsp += ":\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	printcapTable[printcapTableIdx].lastLine = kplp;
	QString* printerDesc = new QString(512);
	printerDesc->sprintf("%-16.16s %1.64s on %1.16s", name.data(), model.data(),
		device.data());
	printcapTable[printcapTableIdx].printerDescription = printerDesc;
	printcapTableIdx++;
	
	// create spool directory
	int retval = createSpooldir( spooldir );
	if ( retval != RETURN_OK )
		return retval;

	// commit linked list to /etc/printcap
	if ( !writeListToPrintcap() )
		return CANT_WRITE_PRINTCAP;
	else
		return RETURN_OK;
}
	
int KPrintcap::addNewRemotePrinter(const QString& name,
	const QString& spooldir, const QString& host, const QString& remqueue,
	const QString& note)
{
	QString* qsp;		// used as a temporary item for writing lines out
	KPrintcapLine* kplp;	// ditto

	// create new record in linked list
	// first, create a blank line, followed by the two comment records
	kplp = new KPrintcapLine;
	qsp = new QString("\n");
	kplp->line = qsp;
	addToLinkedList(kplp);
	printcapTable[printcapTableIdx].firstLine = kplp;
	printcapTable[printcapTableIdx].isLocal = false;

	kplp = new KPrintcapLine;
	qsp = new QString("##KCMPRINTER-1## ");
	*qsp += name;
	*qsp += " remote\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString("##KCMPRINTER-2## {");
	*qsp += note;
	*qsp += "}\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	// now, add the lines for the name, spooldir, host and remote queue
	kplp = new KPrintcapLine;
	qsp = new QString( name.data() );
	*qsp += ":\\\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString( "\t:sd=" );
	*qsp += spooldir;
	*qsp += ":\\\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	kplp->line = new QString("\t:mx#0:\\\n");
	addToLinkedList(kplp);
	kplp = new KPrintcapLine;
	kplp->line = new QString("\t:sh:\\\n");
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString( "\t:rm=" );
	*qsp += host;
	*qsp += ":\\\n";
	kplp->line = qsp;
	addToLinkedList(kplp);

	kplp = new KPrintcapLine;
	qsp = new QString( "\t:rp=" );
	*qsp += remqueue;
	*qsp += ":\n";
	kplp->line = qsp;
	addToLinkedList(kplp);
	printcapTable[printcapTableIdx].lastLine = kplp;
	QString* printerDesc = new QString(512);
	printerDesc->sprintf("%-16.16s queue %1.64s on remote host %1.16s",
		name.data(), remqueue.data(), host.data());
	printcapTable[printcapTableIdx].printerDescription = printerDesc;
	printcapTableIdx++;
	
	// create spool directory
	int retval = createSpooldir( spooldir );
	if ( retval != RETURN_OK )
		return retval;

	// commit linked list to /etc/printcap
	if ( !writeListToPrintcap() )
		return CANT_WRITE_PRINTCAP;
	else
		return RETURN_OK;
}

// Remove a printer record. If the printerNumber is out of range, return
//  false. Also, optionally remove spool directory. If we run into problems,
//  return false.
bool KPrintcap::removePrinter(int printerNumber, bool removeDirs)
{
	// is number within range?
	if ( printerNumber > printcapTableIdx )
		return false;

	// save the spooldir so we can delete it later
	QString spooldir;
	if ( removeDirs )
		if ( !getPrinterSpooldir(printerNumber, &spooldir) )
			return false;

	// remove lines from printcap
	if ( !removeRangeOfLines( printcapTable[printerNumber].firstLine,
			printcapTable[printerNumber].lastLine) )
		return false;

	// now remove the spooling directory
	if ( removeDirs )
		removeSpooldir( spooldir );

	// remove the relevant entry from the printcapTable
	int i;
	for (i = printerNumber; i < printcapTableIdx; i++)
	{
		printcapTable[i].firstLine          = printcapTable[i + 1].firstLine;
		printcapTable[i].lastLine           = printcapTable[i + 1].lastLine;
		printcapTable[i].printerDescription =
			printcapTable[i + 1].printerDescription;
		printcapTable[i].isLocal            = printcapTable[i + 1].isLocal;
	}
	printcapTableIdx -= 1;

	// and commit the changes to printcap
	return writeListToPrintcap();
}


// find out whether printer is local. If we can't find out, then the record
//  is one which was edited directly by the user (or another tool), and we
//  don't know what to do with it. If this is the case, return false.
bool KPrintcap::getPrinterLocation(int printerNumber, bool* isLocal)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains("KCMPRINTER-1") != 0)
		{
			if ( currentLine->line->contains("local") != 0)
			{
				*isLocal = true;
				return true;
			}
			else
			{
				*isLocal = false;
				return true;
			}
		}
		currentLine = currentLine->next;
	}
	return false;
}


bool KPrintcap::getPrinterName(int printerNumber, QString* name)
{
	QString* nameLine;
	if ( !findPrinterName( printerNumber, &nameLine ) )
		return false;
	
	int index = nameLine->find(':');
	if ( index == -1 )
		return false;
	*name = QString( nameLine->left(index) );
	return true;
}

bool KPrintcap::getPrinterModel(int printerNumber, QString* model)
{
	QString* modelLine;
	if ( !findPrinterModel( printerNumber, &modelLine ) )
		return false;

	int p1 = modelLine->find('{');
	int p2 = modelLine->findRev('}');
	if ( (p1 == -1) || (p2 == -1) )
		return false;
	*model = QString( modelLine->mid( p1+1, p2-p1-1) );
	return true;
}

bool KPrintcap::getPrinterFilter(int printerNumber, QString* filter)
{
	QString* filterLine;
	if ( !findPrinterFilter( printerNumber, &filterLine ) )
		return false;

	int p1 = filterLine->find('=');
	int p2 = filterLine->findRev(':');
	if ( (p1 == -1) || (p2 == -1) )
		return false;
	*filter = QString( filterLine->mid( p1+1, p2-p1-1) );
	return true;
}

bool KPrintcap::getPrinterSpooldir(int printerNumber, QString* spooldir)
{
	QString* spoolLine;
	if ( !findPrinterSpooldir( printerNumber, &spoolLine ) ) 
		return false;

	int p1 = spoolLine->find('=');
	int p2 = spoolLine->findRev(':');
	if ( (p1 == -1) || (p2 == -1) )
		return false;
	*spooldir = QString( spoolLine->mid( p1+1, p2-p1-1) );
	return true;
}

bool KPrintcap::getPrinterDevice(int printerNumber, QString* device)
{
	QString* deviceLine;
	if ( !findPrinterDevice( printerNumber, &deviceLine ) )
		return false;

	int p1 = deviceLine->find('=');
	int p2 = deviceLine->findRev(':');
	if ( (p1 == -1) || (p2 == -1) )
		return false;
	*device = QString( deviceLine->mid( p1+1, p2-p1-1) );
	return true;
}

bool KPrintcap::getPrinterHost(int printerNumber, QString* host)
{
	QString* hostLine;
	if ( !findPrinterHost( printerNumber, &hostLine ) )
		return false;

	int p1 = hostLine->find('=');
	int p2 = hostLine->findRev(':');
	if ( (p1 == -1) || (p2 == -1) )
		return false;
	*host = QString( hostLine->mid( p1+1, p2-p1-1) );
	return true;
}

bool KPrintcap::getPrinterQueue(int printerNumber, QString* remqueue)
{
	QString* queueLine;
	if ( !findPrinterQueue( printerNumber, &queueLine ) )
		return false;

	int p1 = queueLine->find('=');
	int p2 = queueLine->findRev(':');
	if ( (p1 == -1) || (p2 == -1) )
		return false;
	*remqueue = QString( queueLine->mid( p1+1, p2-p1-1) );
	return true;
}

bool KPrintcap::getPrinterNote(int printerNumber, QString* note)
{
	QString* noteLine;
	if ( !findPrinterNote( printerNumber, &noteLine ) )
		return false;

	int p1 = noteLine->find('{');
	int p2 = noteLine->findRev('}');
	if ( (p1 == -1) || (p2 == -1) )
		return false;
	*note = QString( noteLine->mid( p1+1, p2-p1-1) );
	return true;
}

bool KPrintcap::getPrinterDescription(int printerNumber, QString* desc)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	*desc = QString( printcapTable[printerNumber].printerDescription->data() );
	return true;
}

bool KPrintcap::changePrinterName(int printerNumber, const char* name)
{
	QString* nameLine;
	if ( !findPrinterName( printerNumber, &nameLine ) )
		return false;

	// nameLine now points to a string of the form:
	// <name>:\n
	int pos = nameLine->find(':');
	nameLine->replace( 0, pos, name );

	// Now we check whether the entry was a kcmprinter entry. 
	//  If so, we need to modify the first comment line
	QString* nameLine2 = printcapTable[printerNumber].firstLine->line;
	if ( nameLine2->contains("KCMPRINTER-1") )
	{
		// the line will be of the form:
		// ##KCMPRINTER-1## <name> <location> <model>
		// so we search for the second space
		int pos2 = nameLine2->find( ' ', 17 );
		if ( pos2 != -1 )
			nameLine2->replace( 17, pos2 - 17, name );
	}

	createPrinterDescription( printerNumber,
		printcapTable[printerNumber].printerDescription );

	return ( writeListToPrintcap() );
}

bool KPrintcap::changePrinterModel(int printerNumber, const char* model)
{
	QString* modelLine;
	if ( !findPrinterModel( printerNumber, &modelLine ) )
		return false;

	// modelLine now points to a string of the form:
	// ##KCMPRINTER-1## <printername> <local> {<model>}
	int pos = modelLine->find('{');
	int pos2 = modelLine->findRev('}');
	if ( pos == -1 || pos2 == -1)
		return false;
	modelLine->replace( pos + 1, (pos2 - pos - 1), model );

	createPrinterDescription( printerNumber,
		printcapTable[printerNumber].printerDescription );

	return ( writeListToPrintcap() );
}

bool KPrintcap::changePrinterFilter(int printerNumber, const char* filter)
{
	QString* filterLine;
	if ( !findPrinterFilter( printerNumber, &filterLine ) )
		return false;

	// filterLine now points to a string of the form:
	// /t:if=<filter>:\n
	int pos = filterLine->find('=');
	int pos2 = filterLine->findRev(':');
	if ( pos == -1 || pos2 == -1 )
		return false;
	filterLine->replace( pos + 1, (pos2 - pos - 1), filter );
	return ( writeListToPrintcap() );
}

bool KPrintcap::changePrinterSpooldir(int printerNumber, const char* spooldir,
	bool removeDirs)
{
	QString* spoolLine;
	if ( !findPrinterSpooldir( printerNumber, &spoolLine ) ) 
		return false;

	// check whether the new spool directory already exists
	KDir kd( spooldir );
	if ( kd.exists() )			// does spool directory already exist?
		return false;

	// spoolLine now points to a string of the form:
	// \t:sd=<spooldir>:\n
	int pos = spoolLine->find('=');
	int pos2 = spoolLine->findRev(':');
	if ( pos == -1 || pos2 == -1 )
		return false;
	QString oldSpoolDir( (spoolLine->mid( pos + 1, (pos2 - pos - 1) ) ).data() );
	spoolLine->replace( pos + 1, (pos2 - pos - 1), spooldir );

	// now, create the new spooldir and optionally delete the old one
	if ( createSpooldir( QString( spooldir ) ) != RETURN_OK )
		return false;
	if ( removeDirs )
		removeSpooldir( oldSpoolDir );

	return ( writeListToPrintcap() );
}

bool KPrintcap::changePrinterDevice(int printerNumber, const char* device)
{
	QString* deviceLine;
	if ( !findPrinterDevice( printerNumber, &deviceLine ) )
		return false;

	// deviceLine now points to a line of the form:
	// \t:lp=<device>:\n
	int pos = deviceLine->find( '=' );
	int pos2 = deviceLine->findRev( ':' );
	if ( pos == -1 || pos2 == -1 )
		return false;
	deviceLine->replace( pos + 1, (pos2 - pos - 1), device );

	createPrinterDescription( printerNumber,
		printcapTable[printerNumber].printerDescription );

	return ( writeListToPrintcap() );
}

bool KPrintcap::changePrinterHost(int printerNumber, const char* host)
{
	QString* hostLine;
	if ( !findPrinterHost( printerNumber, &hostLine ) )
		return false;

	// hostLine now points to a line of the form:
	// \t:lp=<host>:\n
	int pos = hostLine->find( '=' );
	int pos2 = hostLine->findRev( ':' );
	if ( pos == -1 || pos2 == -1 )
		return false;
	hostLine->replace( pos + 1, (pos2 - pos - 1), host );

	createPrinterDescription( printerNumber,
		printcapTable[printerNumber].printerDescription );

	return ( writeListToPrintcap() );
}

bool KPrintcap::changePrinterQueue(int printerNumber, const char* remqueue)
{
	QString* queueLine;
	if ( !findPrinterQueue( printerNumber, &queueLine ) )
		return false;

	// queueLine now points to a line of the form:
	// \t:rp=<queue>:\n
	int pos = queueLine->find( '=' );
	int pos2 = queueLine->findRev( ':' );
	if ( pos == -1 || pos2 == -1 )
		return false;
	queueLine->replace( pos + 1, (pos2 - pos - 1), remqueue );

	createPrinterDescription( printerNumber,
		printcapTable[printerNumber].printerDescription );

	return ( writeListToPrintcap() );
}

bool KPrintcap::changePrinterNote(int printerNumber, const char* note)
{
	QString* noteLine;
	if ( !findPrinterNote( printerNumber, &noteLine ) )
		return false;

	// noteLine now points to a line of the form:
	// ##KCMPRINTER-2## {<note>}
	int pos = noteLine->find( '{' );
	int pos2 = noteLine->find( '}' );
	if ( pos == -1 || pos2 == -1 )
		return false;
	if ( pos == (pos2 - 1) )
		// there wasn't a note before, so we must insert the new one
		noteLine->insert( pos2, note );
	else
		noteLine->replace( pos + 1, ( pos2 - pos - 1 ), note );
	return ( writeListToPrintcap() );
}

bool KPrintcap::defaultPrinterName(QString* defaultName)
{
	if ( numberOfPrinters() == 0 )
	{
		// we're creating the first printer, so make it lp0|lp
		*defaultName = "lp0|lp";
		return true;
	}
		
	// table of booleans indicating that the corresponding name exists
	bool nameExists[100];
	int i;
	for (i = 0; i < 100; i++)
		nameExists[i] = false;

	QString currentName(256);
	QRegExp qre("lp[0-9][0-9]?");		// matches lp followed by one or
										//  two digits
	for (i = 0; i < printcapTableIdx; i++)
	{
		// get each printer name in turn
		if ( !getPrinterName(i, &currentName) )
			break;

		// search for names of the required form, marking those
		//  we find in the 'nameExists' table
		int pos = 0;
		while ( (pos = currentName.find(qre, pos)) != -1)
		{
			int prNum = 0;
			for ( pos += 2; isdigit(currentName[pos]); pos++)
				prNum = prNum * 10 + (currentName[pos] - '0');
			if (prNum < 100)
				nameExists[prNum] = true;
		}
	}

	// now find the first unused name
	for (i = 0; i < 100; i++)
		if ( !nameExists[i] )
			break;

	if ( i < 100 )
	{
		defaultName->sprintf("lp%d", i);
		return true;
	}
	else
		// we can't find a default name
		return false;
}

/*************** PRIVATE FUNCTIONS ****************************/

// Add a KPrintcapLine to the end of the linked list
void KPrintcap::addToLinkedList(struct KPrintcapLine* kpl)
{
	if ( firstPrintcapLine == NULL )
		firstPrintcapLine = kpl;
	if ( lastPrintcapLine != NULL )
		lastPrintcapLine->next = kpl;
	kpl->prev = lastPrintcapLine;
	kpl->next = NULL;
	lastPrintcapLine = kpl;
}

// Read printcap into the linked list and rewind file ready for updating
bool KPrintcap::readPrintcapIntoList()
{
	// open printcap
	if ( (printcap = fopen("/etc/printcap","r")) == NULL )
		return false;

	// while not at eof, read line from printcap and add to linked list
	char lineFromPrintcap[LINELENGTH];
	while ( (fgets(lineFromPrintcap, LINELENGTH, printcap)) != NULL )
	{
		struct KPrintcapLine* newLine = new KPrintcapLine;
		newLine->line = new QString(lineFromPrintcap);
		addToLinkedList( newLine );
	}
	if ( fclose(printcap) != 0 )
		return false;

	return true;
}

// Write linked list into /etc/printcap and rewind
bool KPrintcap::writeListToPrintcap()
{
	// open printcap
	if ( (printcap = fopen("/etc/printcap","w")) == NULL )
		return false;

	struct KPrintcapLine* currentLine = firstPrintcapLine;
	while ( currentLine != NULL)
	{
		if ( currentLine->line != NULL )
			if ( fputs(currentLine->line->data(), printcap) == EOF )
				return false;
		currentLine = currentLine->next;
	}
	
	// ensure buffers are flushed
	if ( fflush(printcap) != 0 )
		return false;

	if ( fclose(printcap) != 0 )
		return false;

	return true;
}
	
bool KPrintcap::createPrinterDescription( int printNum, QString* desc )
{
	QString name(256);
	if ( !getPrinterName( printNum, &name ) )
		return false;

	bool isLocal;
	if ( !getPrinterLocation( printNum, &isLocal ) )
		return false;
	if ( isLocal )
	{
		QString model(256);
		if ( !getPrinterModel( printNum, &model ) )
			model = "unknown printer type";
		QString device(256);
		if ( !getPrinterDevice( printNum, &device ) )
			return false;
		desc->sprintf("%-16.16s %1.64s on %1.16s", name.data(), model.data(),
			device.data());
	}
	else
	{
		QString host(256);
		if ( !getPrinterHost( printNum, &host ) )
			host = "unknown host";
		QString queue(256);
		if ( !getPrinterQueue( printNum, &queue ) )
			queue = "unknown remote queue";
		desc->sprintf("%-16.16s queue %1.64s on remote host %1.16s", name.data(),
			queue.data(), host.data());
	}
	return true;
}

bool KPrintcap::createSpoolDirFile(QString& sd, const char* name, int mode)
{
	KFile sdf(sd + name);
	if (!sdf.touch())
		return FALSE;
	if (!sdf.chown("root", "root"))
		return FALSE;
	if (!sdf.chmod(mode))
		return FALSE;
	return TRUE;
}

// remove the range of lines from start to end, inclusive
bool KPrintcap::removeRangeOfLines(KPrintcapLine* start, KPrintcapLine* end)
{
	if (start == NULL || end == NULL)
		return false;

	// first, rearrange the links so that the nodes in question are 
	// cut out of the list
	if ( start->prev != NULL )
		start->prev->next = end->next;
	if ( end->next != NULL )
		end->next->prev = start->prev;

	// ensure that if we're deleting the first or last entry, that
	// the relevant pointers don't point to invalid lines
	if ( firstPrintcapLine == start )
		firstPrintcapLine = end->next;
	if ( lastPrintcapLine == end )
		lastPrintcapLine = start->prev;

	// go through the list deleting each in turn. If start == end 
	//  (ie. there's only one node, we'll skip the loop and just delete
	//  one node.
	KPrintcapLine* current_node = start;
	KPrintcapLine* next_node;
	while (current_node != end)
	{
		next_node = current_node->next;
		if (current_node->line != NULL)
			delete current_node->line;
		delete current_node;
		current_node = next_node;
	}

	return true;
}
		
bool KPrintcap::findPrinterName(int printerNumber, QString** nameLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		char first_char;
		memcpy(&first_char, currentLine->line->data(), 1);
		if ( first_char == '\n' || first_char == '#' || first_char == '\t' )
		{
			currentLine = currentLine->next;
			continue;
		}
		// if line isn't empty, and doesn't start with a comment character
		//  or a tab, it must (should?) be the printer name
		*nameLine = currentLine->line;
		return true;
	}
	return false;
}

bool KPrintcap::findPrinterModel(int printerNumber, QString** modelLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains("KCMPRINTER-1") != 0)
		{
			if ( currentLine->line->contains("local") != 0)
			{
				*modelLine = currentLine->line;
				return true;
			}
			else
				return false;	// remote printers don't have model names
		}
		currentLine = currentLine->next;
	}
	return false;
}

bool KPrintcap::findPrinterFilter(int printerNumber, QString** filterLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains(":if=") != 0)
		{
			*filterLine = currentLine->line;
			return true;
		}
		currentLine = currentLine->next;
	}
	// FIXME -- there should be a more elegant solution
	if ( currentLine != NULL && currentLine->line != NULL )
	{
		if ( currentLine->line->contains(":if=") != 0)
		{
			*filterLine = currentLine->line;
			return true;
		}
	}
	return false;
}

bool KPrintcap::findPrinterSpooldir(int printerNumber, QString** spooldirLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains(":sd=") != 0)
		{
			*spooldirLine = currentLine->line;
			return true;
		}
		currentLine = currentLine->next;
	}
	return false;
}

bool KPrintcap::findPrinterDevice(int printerNumber, QString** deviceLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains(":lp=") != 0)
		{
			*deviceLine = currentLine->line;
			return true;
		}
		currentLine = currentLine->next;
	}
	return false;
}

bool KPrintcap::findPrinterHost(int printerNumber, QString** hostLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains(":rm=") != 0)
		{
			*hostLine = currentLine->line;
			return true;
		}
		currentLine = currentLine->next;
	}
	return false;
}

bool KPrintcap::findPrinterQueue(int printerNumber, QString** remqueueLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains(":rp=") != 0)
		{
			*remqueueLine = currentLine->line;
			return true;
		}
		currentLine = currentLine->next;
	}
	// FIXME -- there should be a more elegant solution
	if ( currentLine != NULL && currentLine->line != NULL )
	{
		if ( currentLine->line->contains(":rp=") != 0)
		{
			*remqueueLine = currentLine->line;
			return true;
		}
	}
	return false;
}

bool KPrintcap::findPrinterNote(int printerNumber, QString** noteLine)
{
	if ( printerNumber > printcapTableIdx )
		return false;

	KPrintcapLine* currentLine = printcapTable[printerNumber].firstLine;
	while ( currentLine != NULL && 
			currentLine != printcapTable[printerNumber].lastLine )
	{
		if ( currentLine->line->contains("KCMPRINTER-2") != 0)
		{
			*noteLine = currentLine->line;
			return true;
		}
		currentLine = currentLine->next;
	}
	return false;
}

void KPrintcap::removeSpooldir( const QString& spooldir )
{
	QDir d( spooldir );
	if ( d.exists() )
	{
		d.setFilter( QDir::Files | QDir::Hidden );
		for ( unsigned int i = 0; i < d.count(); i++ )
			d.remove( d[i], TRUE );
		::rmdir( spooldir.data() );
	}
}

int KPrintcap::createSpooldir( const QString& spooldir )
{
	QString spoolDirectory( spooldir.data() );
	KDir kd(spoolDirectory);
	if ( kd.exists() )			// does spool directory already exist?
		return SPOOLDIR_ALREADY_EXISTS;
	if ( !kd.mkdir(spoolDirectory) ) 	// try to make spool directory
		return CANT_CREATE_SPOOLDIR;
	if ( !kd.chown(spoolDirectory, "root", "lp") )
		return CANT_CHOWN_SPOOLDIR;
	if ( !kd.chmod(spoolDirectory, 0755) )
		return CANT_CHOWN_SPOOLDIR;
	if ( !createSpoolDirFile(spoolDirectory, "/.seq", 0664) )
		return CANT_CREATE_SEQ;
	if ( !createSpoolDirFile(spoolDirectory, "/errs", 0644) )
		return CANT_CREATE_ERRS;
	if ( !createSpoolDirFile(spoolDirectory, "/status", 0664) )
		return CANT_CREATE_STATUS;
	if ( !createSpoolDirFile(spoolDirectory, "/lock", 0664) )
		return CANT_CREATE_LOCK;

	return RETURN_OK;
}

