/***************************************************************************
                          tablet.cpp  -  description
                             -------------------
    begin                : Fri Jul 9 1999
    copyright            : (C) 1999 by Heiner Lamprecht
    email                : heiner@kijumfo.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kflog.h>

#include <map.h>
#include <mapcalc.h>
#include <resource.h>
#include <tablet.h>

//#include <iostream>
#include <ctype.h>
#include <math.h>
#include <stdlib.h>

bool Tablet::isPresent()
{
  XExtensionVersion* version;
  bool present;

  version = XGetExtensionVersion(display, INAME);

  if(version && (version != (XExtensionVersion*) NoSuchExtension)) {
    present = version->present;
    XFree(version);
    return present;
  } else {
    return false;
  }
}

XDeviceInfo* Tablet::findInfo(char* name, bool only_ext)
{
  XDeviceInfo* devices;
  int loop;
  int num_dev;
  int len = strlen(name);
  bool is_id = true;
  XID id = 0;

  for(loop = 0; loop < len; loop++) {
    if(!isdigit(name[loop])) {
      is_id = false;
      break;
    }
  }

  if(is_id) {
    id = atoi(name);
  }

  devices = XListInputDevices(display, &num_dev);

  for(loop = 0; loop < num_dev; loop++) {
    if((!only_ext || (devices[loop].use == IsXExtensionDevice)) &&
       ((!is_id && strcmp(devices[loop].name, name) == 0) ||
        (is_id && devices[loop].id == id))) {
      return &devices[loop];
    }
  }

  return NULL;
}

int Tablet::registerEvents(XDeviceInfo* info, bool handleProximity)
{
  int number = 0;        /* number of events registered */
  XEventClass eventList[7];
  int i;

  XDevice* device;
  Window root_win;
  unsigned long screen;
  XInputClassInfo* ip;

  screen = DefaultScreen(display);
  root_win = RootWindow(display, screen);

  device = XOpenDevice(display, info->id);

  if(!device) {
    KMsgBox* errorBox = new KMsgBox(parent,i18n("Not Found"),
          i18n("Warning\nThe inputdevice was not found!"),4,i18n("Damn!"));
    errorBox->exec();
    return 0;
  }

  if(device->num_classes > 0) {
    for(ip = device->classes, i = 0; i < info->num_classes; ip++, i++) {
      switch (ip->input_class) {
      case KeyClass:
        DeviceKeyPress(device, keyPressType, eventList[number]);
        number++;
        DeviceKeyRelease(device, keyReleaseType, eventList[number]);
        number++;
        break;
      case ButtonClass:
        DeviceButtonPress(device, buttonPressType, eventList[number]);
        number++;
        DeviceButtonRelease(device, buttonReleaseType, eventList[number]);
        number++;
        break;
      case ValuatorClass:
        DeviceMotionNotify(device, motionType, eventList[number]);
        number++;
        if(handleProximity) {
          ProximityIn(device, proximityInType, eventList[number]);
          number++;
          ProximityOut(device, proximityOutType, eventList[number]);
          number++;
        }
        break;
      default:
        break;
      }
    }

    if(XSelectExtensionEvent(display, root_win, eventList, number))
      return 0;
  }
  return number;
}

void Tablet::readEvents(bool onlyOne, QList<Location> *pointList)
{
  XEvent Event;

  pointList->~QList();

  int run = 1;
  bool isPressed = false;

  while(run) {
    XNextEvent(display, &Event);

    if(Event.type == motionType) {
      if(isPressed) {
        XDeviceMotionEvent* motion = (XDeviceMotionEvent *) &Event;
        Location* newPoint = new Location();
        newPoint->setLon(motion->axis_data[0]);
        newPoint->setLat(motion->axis_data[1]);

        pointList->append(newPoint);
      }
    } else if(Event.type == buttonPressType) {
      isPressed = true;
      XDeviceButtonEvent* button = (XDeviceButtonEvent *) &Event;

      Location* newPoint = new Location();
      newPoint->setLon(button->axis_data[0]);
      newPoint->setLat(button->axis_data[1]);
      pointList->append(newPoint);

      // Button 4 will exit input:
      if((button->button == 3) || onlyOne) {
        run = 0;
        break;
      }
    } else if(Event.type == buttonReleaseType) {
      isPressed = false;
    }
  }
}

void Tablet::getPoints(bool onlyOne, QList<Location> *pointList, char* name)
{
  //
  // Fehlerbehandlung noch nicht korrekt ...
  XDeviceInfo* info = findInfo(name, true);

  if(info) {
    if(registerEvents(info, true)) {
      readEvents(onlyOne, pointList);
    }
  }
}

Tablet::Tablet(QWidget* parent)
{

  motionType = INVALID_EVENT_TYPE;
  buttonPressType = INVALID_EVENT_TYPE;
  buttonReleaseType = INVALID_EVENT_TYPE;
  keyPressType = INVALID_EVENT_TYPE;
  keyReleaseType = INVALID_EVENT_TYPE;
  proximityInType = INVALID_EVENT_TYPE;
  proximityOutType = INVALID_EVENT_TYPE;

  display = XOpenDisplay(NULL);
}

Tablet::~Tablet()
{

}

/***************************************************************************
 *
 * The TabletDialog-Class
 *
 ***************************************************************************/
void TabletDialog::startDigitizing()
{
  if(scale == 0) {
    KMsgBox* box = new KMsgBox(this,i18n("Warning"),
                               i18n("The inputdevice has not \
been calibrated yet!"),4,i18n("OK"));
    box->exec();
    return;
  }
  tab->getPoints(onlyOne,&pointList, tabName);

  listLength = pointList.count();
  latList = new long[listLength];
  lonList = new long[listLength];

  Location* point;

  for(unsigned int loop = 0; loop < pointList.count(); loop++) {
    point = pointList.at(loop);

    double y = ((point->getLon() - point1D->getLon()) * cos(angle) +
                (point->getLat() - point1D->getLat()) * sin(angle))
      * scale + vectorMapY;
    double x = ((point->getLon() - point1D->getLon()) * sin(angle) -
                (point->getLat() - point1D->getLat()) * cos(angle))
      * scale + vectorMapX;

    latList[loop] = map2LambertLat(x, y, v1, v2);
    lonList[loop] = map2LambertLon(x, y, v1, v2);
  }
  pointList.~QList();
  delete(dW);
}

void TabletDialog::stopDigitizing()
{
  delete(dW);
}

void TabletDialog::getPointList(long* latitude, long* longitude)
{
  // is there a better way???
  for(int loop = 0; loop < listLength; loop++) {
    latitude[loop] = latList[loop];
    longitude[loop] = lonList[loop];
  }
  delete latList;
  delete lonList;
}

int TabletDialog::getListLength()
{
  return listLength;
}

void TabletDialog::calibrateTablet()
{
  if(!done) {
    done = true;
    dWL1->hide();
    cal->setText(i18n("Done"));
    calL->show();
    point1L->show();
    point1E1->show();
    point1E2->show();
    point1B->show();
    point2L->show();
    point2E1->show();
    point2E2->show();
    point2B->show();
    point3L->show();
    point3E1->show();
    point3E2->show();
    point3B->show();
  } else {
    if((strlen(point1E1->text()) == 0) ||
       (strlen(point1E2->text()) == 0) ||
       (strlen(point2E1->text()) == 0) ||
       (strlen(point2E2->text()) == 0) ||
       (strlen(point3E1->text()) == 0) ||
       (strlen(point3E2->text()) == 0)) return;

    if((point1D == NULL) || (point2D == NULL)|| (point3D == NULL)) return;

    done = false;
    cal->setText(i18n("Calibrate"));
    calL->hide();
    point1L->hide();
    point1E1->hide();
    point1E2->hide();
    point1B->hide();
    point2L->hide();
    point2E1->hide();
    point2E2->hide();
    point2B->hide();
    point3L->hide();
    point3E1->hide();
    point3E2->hide();
    point3B->hide();
    dWL1->show();

    double p1X, p1Y, p2X, p2Y, p3X, p3Y;
    double dX1, dY1, dX2, dY2, dX3, dY3;
    double dist1, dist2, dist3;

    long p1Xd, p1Yd, p2Xd, p2Yd, p3Xd, p3Yd;
    double dXd1, dYd1, dXd2, dYd2, dXd3, dYd3;
    double distd1, distd2, distd3;

    // Eingegebene Punkte:
    p1Y = numToRad(degreeToNum(point1E2->text()));
    p1X = numToRad(degreeToNum(point1E1->text()));
    p2Y = numToRad(degreeToNum(point2E2->text()));
    p2X = numToRad(degreeToNum(point2E1->text()));
    p3Y = numToRad(degreeToNum(point3E2->text()));
    p3X = numToRad(degreeToNum(point3E1->text()));

    dX1 = (calc_X_Lambert(p1X, p1Y, v1, v2) -
           calc_X_Lambert(p2X, p2Y, v1, v2));
    dY1 = (calc_Y_Lambert(p1X, p1Y, v1, v2) -
           calc_Y_Lambert(p2X, p2Y, v1, v2));
    dX2 = (calc_X_Lambert(p2X, p2Y, v1, v2) -
           calc_X_Lambert(p3X, p3Y, v1, v2));
    dY2 = (calc_Y_Lambert(p2X, p2Y, v1, v2) -
           calc_Y_Lambert(p3X, p3Y, v1, v2));
    dX3 = (calc_X_Lambert(p1X, p1Y, v1, v2) -
           calc_X_Lambert(p3X, p3Y, v1, v2));
    dY3 = (calc_Y_Lambert(p1X, p1Y, v1, v2) -
           calc_Y_Lambert(p3X, p3Y, v1, v2));

    dist1 = sqrt(dX1 * dX1 + dY1 * dY1);
    dist2 = sqrt(dX2 * dX2 + dY2 * dY2);
    dist3 = sqrt(dX3 * dX3 + dY3 * dY3);

    vectorMapX = calc_X_Lambert(p1X, p1Y, v1, v2);
    vectorMapY = calc_Y_Lambert(p1X, p1Y, v1, v2);

    // Digitalisierte Punkte:
    p1Xd = point1D->getLon();
    p1Yd = point1D->getLat();
    p2Xd = point2D->getLon();
    p2Yd = point2D->getLat();
    p3Xd = point3D->getLon();
    p3Yd = point3D->getLat();

    dXd1 = p1Xd - p2Xd;
    dYd1 = p1Yd - p2Yd;
    dXd2 = p2Xd - p3Xd;
    dYd2 = p2Yd - p3Yd;
    dXd3 = p1Xd - p3Xd;
    dYd3 = p1Yd - p3Yd;

    distd1 = sqrt(dXd1 * dXd1 + dYd1 * dYd1);
    distd2 = sqrt(dXd2 * dXd2 + dYd2 * dYd2);
    distd3 = sqrt(dXd3 * dXd3 + dYd3 * dYd3);

    // Scalierungsfaktor:
    scale = ((dist1 / distd1) + (dist2 / distd2) + (dist3 / distd3)) / 3;

    // Winkel:
    // Der Drehungswinkel wird anhand der Strecke a->b, bzw a'->b' bestimmt
    angle = atan(dX1 / dY1) - atan(dXd1 / dYd1);

    // Zuletzt: Abspeichern der Konfiguration:
    KConfig* config = kapp->getConfig();

    config->setGroup("Digitizer");
    config->writeEntry("Scale", scale);
    config->writeEntry("Angle", angle);
    config->writeEntry("TabletName", tabName);
    config->writeEntry("TabletVectorX", point1D->getLon()); // hier kein
    config->writeEntry("TabletVectorY", point1D->getLat()); // Objekt !!
    config->writeEntry("MapVectorX", vectorMapX);
    config->writeEntry("MapVectorY", vectorMapY);
  }
}

void TabletDialog::inputPoint1()
{
  tab->getPoints(true, &pointList, tabName);
  point1D = pointList.take(0);
}

void TabletDialog::inputPoint2()
{
  tab->getPoints(true, &pointList, tabName);
  point2D = pointList.take(0);
}

void TabletDialog::inputPoint3()
{
  tab->getPoints(true, &pointList, tabName);
  point3D = pointList.take(0);
}

TabletDialog::TabletDialog(QWidget* parent, bool onlyone)
  : QWidget(parent)
{

  onlyOne = onlyone;
  point1D = new Location();
  listLength = 0;

  ///////////////////////////////////////////////////////////////////
  // read the config file entries
  KConfig *config = kapp->getConfig();

  config->setGroup("General");
  v1 = numToRad(config->readNumEntry("Parallel1",
      degreeToNum(DEFAULT_V1)));
  v2 = numToRad(config->readNumEntry("Parallel2",
      degreeToNum(DEFAULT_V2)));

  config->setGroup("Digitizer");
  scale = config->readDoubleNumEntry("Scale", 0);
  angle = config->readDoubleNumEntry("Angle", 0);
  if(config->hasKey("TabletName")) {
    tabName = config->readEntry("TabletName", "tablet");
  } else {
    tabName = "tablet";
  }

  point1D->setLon(config->readLongNumEntry("TabletVectorX", 0));
  point1D->setLat(config->readLongNumEntry("TabletVectorY", 0));
  vectorMapX = config->readDoubleNumEntry("MapVectorX", 0);
  vectorMapY = config->readDoubleNumEntry("MapVectorY", 0);

  tab = new Tablet(parent);
  done = false;

  if(!tab->isPresent()) {
    KMsgBox* errorBox = new KMsgBox(parent,i18n("Not Found"),
          i18n("Warning\nThe inputdevice was not found!"),4,i18n("Damn!"));
    errorBox->exec();
    return;
  }

  QFont ttbf("Courier");
  QFont bold;
  bold.setBold(true);

  const char* caption;
  const char* text;

  caption = i18n("Digitizing the Map");
  text = i18n("\nNow you can start digitizing the map.\n\n\
The first button of the pad is used for giving an\n\
ongoing list of coordinates. With the second button\n\
You can give a single point.\n\n\
To stop digitizing, press the third button\n");

  dW = new QDialog(parent, "dialog", true);

  dWLayout = new QGridLayout(dW,11,12,15,1);

  QLabel* dWIcon = new QLabel(dW);
  dWIcon->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  dWLayout->addMultiCellWidget(dWIcon,0,7,0,0);

  dWL1 = new QLabel(dW);
  dWL1->setText(text);
  dWLayout->addMultiCellWidget(dWL1,0,3,2,10);

  cal = new QPushButton(i18n("Calibrate"), dW);
  cal->setMinimumHeight(25);
  cal->setMaximumHeight(35);
  cal->setMinimumWidth(75);
  dWLayout->addMultiCellWidget(cal,7,7,2,4);

  // Widgets for calibrating the tablet
  calL = new QLabel(dW);
  calL->setText(i18n("Please enter two coordinates\n\
and give the correspondending points.\n\n\
The coordinates must have the following form:\n\
4913,1254N / [0]0905,4621E"));
  dWLayout->addMultiCellWidget(calL,0,0,2,10);

  point1L = new QLabel(dW);
  point1L->setText(i18n("Point 1:"));
  point1L->setMinimumWidth(80);
  point1E1 = new KRestrictedLine(dW);
  point1E1->setValidChars("1234567890.'\",NSWE");
  point1E1->setFont(ttbf);
  point1E1->setMinimumWidth(130);
  point1E1->setMaximumHeight(30);
  point1E1->setMinimumHeight(25);
  point1E2 = new KRestrictedLine(dW);
  point1E2->setValidChars("1234567890.'\",NSWE");
  point1E2->setFont(ttbf);
  point1E2->setMinimumWidth(130);
  point1E2->setMinimumHeight(25);
  point1B = new QPushButton(dW);
  point1B->setPixmap(Icon("mini/tablet.xpm"));
  point1B->setMinimumWidth(30);
  dWLayout->addWidget(point1L,1,2);
  dWLayout->addMultiCellWidget(point1E1,1,1,4,5);
  dWLayout->addMultiCellWidget(point1E2,1,1,7,8);
  dWLayout->addWidget(point1B,1,10);

  point2L = new QLabel(dW);
  point2L->setText(i18n("Point 2:"));
  point2L->setMinimumWidth(80);
  point2E1 = new KRestrictedLine(dW);
  point2E1->setValidChars("1234567890.'\",NSWE");
  point2E1->setFont(ttbf);
  point2E1->setMinimumWidth(130);
  point2E1->setMaximumHeight(30);
  point2E1->setMinimumHeight(25);
  point2E2 = new KRestrictedLine(dW);
  point2E2->setValidChars("1234567890.'\",NSWE");
  point2E2->setFont(ttbf);
  point2E2->setMinimumWidth(130);
  point2E2->setMinimumHeight(25);
  point2B = new QPushButton(dW);
  point2B->setPixmap(Icon("mini/tablet.xpm"));
  point2B->setMinimumWidth(30);
  dWLayout->addWidget(point2L,3,2);
  dWLayout->addMultiCellWidget(point2E1,3,3,4,5);
  dWLayout->addMultiCellWidget(point2E2,3,3,7,8);
  dWLayout->addWidget(point2B,3,10);

  point3L = new QLabel(dW);
  point3L->setText(i18n("Point 3:"));
  point3L->setMinimumWidth(80);
  point3E1 = new KRestrictedLine(dW);
  point3E1->setValidChars("1234567890.'\",NSWE");
  point3E1->setFont(ttbf);
  point3E1->setMinimumWidth(130);
  point3E1->setMaximumHeight(30);
  point3E1->setMinimumHeight(25);
  point3E2 = new KRestrictedLine(dW);
  point3E2->setValidChars("1234567890.'\",NSWE");
  point3E2->setFont(ttbf);
  point3E2->setMinimumWidth(130);
  point3E2->setMinimumHeight(25);
  point3B = new QPushButton(dW);
  point3B->setPixmap(Icon("mini/tablet.xpm"));
  point3B->setMinimumWidth(30);
  dWLayout->addWidget(point3L,5,2);
  dWLayout->addMultiCellWidget(point3E1,5,5,4,5);
  dWLayout->addMultiCellWidget(point3E2,5,5,7,8);
  dWLayout->addWidget(point3B,5,10);

  // Hiding the widgets:
  calL->hide();
  point1L->hide();
  point1E1->hide();
  point1E2->hide();
  point1B->hide();
  point2L->hide();
  point2E1->hide();
  point2E2->hide();
  point2B->hide();
  point3L->hide();
  point3E1->hide();
  point3E2->hide();
  point3B->hide();

  // Control-Buttons
  QPushButton* ok = new QPushButton(i18n("Start"),dW);
  ok->setMinimumHeight(25);
  ok->setMaximumHeight(35);
  ok->setMinimumWidth(75);
  ok->setMaximumWidth(75);
  QPushButton* cancel = new QPushButton(i18n("Cancel"), dW);
  cancel->setMinimumWidth(75);
  cancel->setMaximumWidth(75);
  dWLayout->addWidget(ok,9,5);
  dWLayout->addWidget(cancel,9,7);

  dWLayout->addColSpacing(0,150);
  dWLayout->addColSpacing(1,10);
  dWLayout->addColSpacing(3,10);
  dWLayout->addColSpacing(4,80);
  dWLayout->addColSpacing(6,15);
  dWLayout->addColSpacing(8,80);
  dWLayout->addColSpacing(9,10);

  dWLayout->setColStretch(0,0);
  dWLayout->setColStretch(1,0);
  dWLayout->setColStretch(2,0);
  dWLayout->setColStretch(3,0);
  dWLayout->setColStretch(4,0);
  dWLayout->setColStretch(5,0);
  dWLayout->setColStretch(6,0);
  dWLayout->setColStretch(7,0);
  dWLayout->setColStretch(8,0);
  dWLayout->setColStretch(9,0);
  dWLayout->setColStretch(10,0);
  dWLayout->setColStretch(11,1);

  dWLayout->addRowSpacing(2,10);
  dWLayout->addRowSpacing(4,10);
  dWLayout->addRowSpacing(6,10);
  dWLayout->addRowSpacing(8,10);
  dWLayout->addRowSpacing(10,10);

  dWLayout->setRowStretch(0,1);
  dWLayout->setRowStretch(7,1);

  dWLayout->activate();

  this->connect(ok, SIGNAL(clicked()), SLOT(startDigitizing()));
  this->connect(cancel, SIGNAL(clicked()), SLOT(stopDigitizing()));
  this->connect(cal, SIGNAL(clicked()), SLOT(calibrateTablet()));
  this->connect(point1B, SIGNAL(clicked()), SLOT(inputPoint1()));
  this->connect(point2B, SIGNAL(clicked()), SLOT(inputPoint2()));
  this->connect(point3B, SIGNAL(clicked()), SLOT(inputPoint3()));

  dW->exec();
}

TabletDialog::~TabletDialog()
{
  tab->~Tablet();
}

