/***********************************************************************
**
**   igc.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include <stdlib.h>

#include <igc.h>

#include <flight.h>
#include <mapcalc.h>
#include <wp.h>

#include <qtextstream.h>

void loadIGC(QFile* flightDataFile, Flight* currentFlight)
{
  QString pilotName, gliderType, gliderID, date;
  char latChar, lonChar;
  bool launched = false, append = true, isFirst = true;
  int dtime, lat, latmin, latTemp, lon, lonmin, lonTemp;
  int hh = 0, mm = 0, ss = 0, curTime = 0, preTime = 0;
  struct flightPoint* flightRoute;
  unsigned int routeLength = 0;

  flightPoint oldPoint, current;

  if(flightDataFile->open(IO_ReadOnly)) {
    QTextStream t(flightDataFile);
    QString s;
    while(!t.eof()) {
      s = t.readLine();
      if(s.mid(0,1) == "H") {
        // We have a headerline
        if(s.mid(5,5) == "PILOT") {
          pilotName = s.mid(11,100);
        } else if(s.mid(5,10) == "GLIDERTYPE") {
          gliderType = s.mid(16,100);
        } else if(s.mid(5,8) == "GLIDERID") {
          gliderID = s.mid(14,100);
        } else if(s.mid(1,4) == "FDTE") {
          // Hier bislang nur "deutsches" Format.
          date = s.mid(5,2) + "." + s.mid(7,2) + "." + s.mid(9,2);
        }
      } else if(s.mid(0,1) == "B") {
        // We have a point
        // sscanf has some problems (in glibc-2.1.{1,2})
        // we must find a replacement sometime
        sscanf(s.mid(1,23), "%2d%2d%2d%2d%5d%1c%3d%5d%1c",
            &hh, &mm, &ss, &lat, &latmin, &latChar, &lon, &lonmin, &lonChar);
        latTemp = lat * 600000 + latmin * 10;
        lonTemp = lon * 600000 + lonmin * 10;

        if(latChar == 'S') {
          latTemp = -latTemp;
        }

        if(lonChar == 'W') {
          lonTemp = -lonTemp;
        }

        curTime = 3600 * hh + 60 * mm + ss;
        current.time = curTime;
        current.latitude = latTemp;
        current.longitude = lonTemp;
        current.nearWP = false;

        sscanf(s.mid(25,10), "%5d%5d", &current.height, &current.gpsHeight);

        if(isFirst) {
          oldPoint = current;
          preTime = curTime;
          isFirst = false;
          current.speed = 0;
          current.vario = 0;
          continue;
        }
        //
        // dtime may change, even if the intervall, in wich the
        // logger gets the position, is allways the same. If the
        // intervall is f.e. 10 sec, dtime may change to 11 or 9 sec.
        //
        dtime = curTime - preTime;

        double ddist = dist(current.latitude, current.longitude,
              oldPoint.latitude, oldPoint.longitude);
        float dheight = current.height - oldPoint.height;

        current.speed = 3600 * ddist / dtime;  // [km/h]
        current.vario = dheight / dtime * 1.0; // [m/s]

        if(launched) {
          routeLength++;
          flightRoute = (struct flightPoint*) realloc(flightRoute,
                                     routeLength * sizeof(flightPoint));
          flightRoute[routeLength - 1] = current;
          if(!append) {
            break;
          }
          if((current.speed < 20) && (current.vario < 1.5)) {
            // Now we can stop reading the file!
            append = false;
          }
        } else {
          if((current.speed > 20) && (current.vario > 1.5)) {
            launched = true;
            flightRoute = new flightPoint[2];
            flightRoute[0] = oldPoint;
            flightRoute[1] = current;
            routeLength = 2;
          }
        }
        oldPoint = current;
        preTime = curTime;
      } else if(s.mid(0,1) == "C") {
        if((((s.mid(8,1) == "N") || (s.mid(8,1) == "S")) ||
            ((s.mid(17,1) == "W") || (s.mid(17,1) == "E"))) &&
           (s.mid(18,20) != 0))  {
          // We have a waypoint
          sscanf(s.mid(1,17), "%2d%5d%1c%3d%5d%1c",
              &lat, &latmin, &latChar,
              &lon, &lonmin, &lonChar);
          latTemp = lat * 600000 + latmin * 10;
          lonTemp = lon * 600000 + lonmin * 10;

          if(latChar == 'S') {
            latTemp = -latTemp;
          }

          if(lonChar == 'W') {
            lonTemp = -lonTemp;
          }

          currentFlight->appendWaypoint(s.mid(18,20), latTemp, lonTemp);
        }
      } else if(s.mid(0,1) == "L") {
        // We have a comment, let's ignore it ...
      }
    }
    currentFlight->setFlightHeader(pilotName, gliderType, gliderID, date);
    currentFlight->setRoute(flightRoute, routeLength);
    flightDataFile->close();
  }
}
