/***********************************************************************
**
**   map.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  1999 by Heiner Lamprecht ( heiner@kijumfo.de )
**   begin        :  Mon Jun  7 1999
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include <iostream>

#include <unistd.h>
#include <cmath>

// mapconfig must be loaded at this point, because
// otherwise we will run into deep trouble...
#include <mapconfig.h>

#include <kapp.h>
#include <kconfig.h>
#include <klocale.h>
#include <kpopmenu.h>
#include <qbitmap.h>
#include <qfont.h>
#include <qpainter.h>
#include <qregexp.h>

#include <basemapelement.h>
#include <kflog.h>
#include <map.h>
#include <mapcalc.h>
#include <resource.h>

void Map::mouseMoveEvent(QMouseEvent* event)
{
  extern double _currentScale;
/*
  // Wird zur Zeit noch nicht bentigt, daher erst mal
  // auskommentieren.
  // War nur ein erster Test ...

  QPainter window;
  window.begin(this);
  window.setRasterOp(XorROP);
  window.setPen(QPen(QColor(0,255,0), 2));
  window.drawLine(_start, _current);
  window.drawLine(_start, event->pos());
  window.end();
*/
  // remember the current mouse position
  _current = event->pos();

  double mapX, mapY;

  mapX = calc_X_Lambert(numToRad(mapCenterLat), 0);
  mapY = calc_Y_Lambert(numToRad(mapCenterLat), 0);

  double dX = ( this->width() / 2 ) - ( mapX / _currentScale * RADIUS);
  double dY = ( this->height() / 2 ) - ( mapY / _currentScale * RADIUS);

  QPoint current = event->pos();
  double x = current.x();
  double y = current.y();

  x = ( x - dX ) * _currentScale / RADIUS;
  y = ( y - dY ) * _currentScale / RADIUS;

  struct point loc;
  map2Lambert(y, x, &loc);

  app->showCoords(loc.latitude, (loc.longitude + mapCenterLon));
}

void Map::mousePressEvent(QMouseEvent* event)
{
  extern double _currentScale;

  QPoint current = event->pos();
  if(event->button() == 4) {
    // middle-button pressed
    double mapX, mapY;

    mapX = calc_X_Lambert(numToRad(mapCenterLat), 0);
    mapY = calc_Y_Lambert(numToRad(mapCenterLat), 0);

    double dX = ( this->width() / 2 ) - ( mapX / _currentScale * RADIUS);
    double dY = ( this->height() / 2 ) - ( mapY / _currentScale * RADIUS);

    double x = current.x();
    double y = current.y();

    x = ( x - dX ) * _currentScale / RADIUS;
    y = ( y - dY ) * _currentScale / RADIUS;

    struct point loc;
    map2Lambert(y, x, &loc);

    mapCenterLat = loc.latitude;
    mapCenterLon += loc.longitude;

    redrawMap();

  } else if(event->button() == 1) {
    // left-button pressed
//    _start = event->pos();
  } else if(event->button() == 2) {
    // right-button pressed
    int dxTemp, dyTemp;
    for(unsigned int loop = 0; loop < posNum; loop++) {
      dxTemp = current.x() - xPos[loop];
      dyTemp = current.y() - yPos[loop];
      if((dxTemp*dxTemp < 145) && (dyTemp*dyTemp < 145)) {
        app->slotShowObjectData(loop);
      }
    }

	  KPopupMenu* helpMenu = new KPopupMenu("Info");
	  bool show = false;

    QRegion* testReg;
    BaseMapElement* hitElement;
    QString text;

    MapContents* mContents = app->getMapContents();

    if(showCity) {
      for(unsigned int loop = 0; loop < cityRegList->count(); loop++) {
        testReg = cityRegList->at(loop);
        text = i18n("City:");
        if(testReg->contains(current)) {
          hitElement = mContents->getElement(MapContents::CityList,
            loop);
          text = text + ' ' + hitElement->getName();
          helpMenu->setTitle(text);
          show = true;
        }
      }
    }

    if(showAirspace) {
      for(unsigned int loop = 0; loop < airspaceRegList->count(); loop++) {
        testReg = airspaceRegList->at(loop);
        if(testReg->contains(current)) {
          hitElement = mContents->getElement(MapContents::AirspaceList,
            loop);
          // Icon knnte "Aufgabe" der Klasse "Airspace" werden ...
          switch(hitElement->getTypeID()) {
            case BaseMapElement::AirC:
            case BaseMapElement::AirCtemp:
              text = "C";
              break;
            case BaseMapElement::AirD:
            case BaseMapElement::AirDtemp:
              text = "D";
              break;
            case BaseMapElement::AirElow:
            case BaseMapElement::AirEhigh:
              text = "E";
              break;
            case BaseMapElement::AirF:
              text = "F";
              break;
            case BaseMapElement::Restricted:
            case BaseMapElement::Danger:
              text = "Danger";
              break;
            case BaseMapElement::ControlD:
              text = "Control";
              break;
            default:
              text = "?";
              break;
          }
          text = text + ' ' + hitElement->getName();
          helpMenu->insertItem(Icon(KApplication::kde_datadir() +
              "/kflog/pics/mini-k.xpm"), text);
          show = true;
        }
      }
    }
    if(show) {
     	helpMenu->exec(event->globalPos());
    }
  }
}
/*
void Map::openToolTip(QMouseEvent* event)
{
  delete toolTip;
  QPoint pos = event->globalPos();
  toolTip = new QLabel( 0, "myToolTip",
                  WStyle_Customize | WStyle_NoBorder |
                  WStyle_Tool );

  toolTip->setText("Platzinfo\nSegelflugplatz");
  toolTip->setGeometry(pos.x(), pos.y(), 150, 50);
  toolTip->setFrameStyle(QFrame::Panel | QFrame::Raised);
  toolTip->show();

  KPopupMenu* helpMenu = new KPopupMenu("Platzinfo");
  QString text;
  text.sprintf("Segelflugplatz Nr. %d", loop);
  helpMenu->insertItem(text);
  helpMenu->insertItem("Lnge der Landebahn: 450m");
  helpMenu->exec(event->globalPos());
}
*/

void Map::paintEvent(QPaintEvent* event = 0)
{
  bitBlt(this, 0, 0, &pixBuffer);
}

void Map::drawGrid()
{
  extern double _scale[];
  extern double _currentScale;

  double drawX, drawY, mapX, mapY, projX, projY;

  mapX = calc_X_Lambert(numToRad(mapCenterLat), 0);
  mapY = calc_Y_Lambert(numToRad(mapCenterLat), 0);

  QPainter gridP;
  gridP.begin(&pixGrid);

  /////////////////////////////////////////
  //
  // Displaying the grid
  if(mapCenterLon > 0) {
    double mapX, mapY;

    mapX = calc_X_Lambert(numToRad(mapCenterLat), 0);
    mapY = calc_Y_Lambert(numToRad(mapCenterLat), 0);

    double dX = ( this->width() / 2 ) - ( mapX / _currentScale * RADIUS);
    double dY = ( this->height() / 2 ) - ( mapY / _currentScale * RADIUS);

    double x1 = 0;
    double y1 = 0;

    x1 = ( x1 - dX ) * _currentScale / RADIUS;
    y1 = ( y1 - dY ) * _currentScale / RADIUS;

    struct point loc1;
    map2Lambert(y1, x1, &loc1);

    double x2 = this->width();
    double y2 = 0;

    x2 = ( x2 - dX ) * _currentScale / RADIUS;
    y2 = ( y2 - dY ) * _currentScale / RADIUS;

    struct point loc2;
    map2Lambert(y2, x2, &loc2);

    gridP.setBrush(NoBrush);
    gridP.setPen(QPen(QColor(0,0,0), 2));

    double x3 = this->width();
    double y3 = this->height();

    x3 = ( x3 - dX ) * _currentScale / RADIUS;
    y3 = ( y3 - dY ) * _currentScale / RADIUS;

    struct point loc3;
    map2Lambert(y3, x3, &loc3);

    long lon1 = (loc1.longitude + mapCenterLon) / 600000;
    long lon2 = (loc2.longitude + mapCenterLon) / 600000 + 1;

    // Obere Kante des Bereichs:
    long lat1 = loc1.latitude / 600000 + 1;
    // Untere Kante des Bereichs:
    long lat2 = loc3.latitude / 600000 - 1;

    double lat1int = numToRad(lat1 * 600000);
    double lat2int = numToRad(lat2 * 600000);

    // Step between two longitude-lines (in 1/60 degree)
    int lineWidth = 2;
    int step = 60;
    if(_currentScale < _scale[5]) {
      step = 30;
    }
    if(_currentScale < _scale[3]) {
      step = 10;
      lineWidth = 1;
    }

    double latTemp = 0;
    double lonTemp = 0;
    int size = (int) ((lat1 - lat2 + 1) * (60.0 / step));
    for(int loop = 0; loop < size ; loop++) {
      latTemp = lat2 + loop;
      latTemp = numToRad((long)(latTemp * 600000));
      lonTemp = 0;

      int size = (lon2 - lon1 + 1) * 10;

      QPointArray pointArray(size);

      for(int lonloop = 0; lonloop < size; lonloop++) {
        lonTemp = lon1 + (lonloop * 0.1);
        lonTemp = numToRad((long)((lonTemp * 600000) - mapCenterLon));

        projX = calc_X_Lambert(latTemp,lonTemp);
        projY = calc_Y_Lambert(latTemp,lonTemp);

        drawX = projX * RADIUS / _currentScale + dX;
        drawY = projY * RADIUS / _currentScale + dY;
        pointArray.setPoint(lonloop, drawX, drawY);
      }

      // Draw the small lines between:
      int number = (int) (60.0 / step);

      for(int loop2 = 1; loop2 < number; loop2++) {
        latTemp = lat2 + (loop * (step / 60.0));
        latTemp = numToRad((long)(latTemp * 600000));
        lonTemp = 0;

        size = (lon2 - lon1 + 1) * 10;
        QPointArray pointArray2(size);

        for(int lonloop = 0; lonloop < size; lonloop++) {
          lonTemp = lon1 + (lonloop * 0.1);
          lonTemp = numToRad((long)((lonTemp * 600000) - mapCenterLon));

          projX = calc_X_Lambert(latTemp,lonTemp);
          projY = calc_Y_Lambert(latTemp,lonTemp);

          drawX = projX * RADIUS / _currentScale + dX;
          drawY = projY * RADIUS / _currentScale + dY;
          pointArray2.setPoint(lonloop, drawX, drawY);
        }
        if(loop2 == (number / 2.0)) {
          gridP.setPen(QPen(QColor(0,0,0), 2, DashLine));
          gridP.drawPolyline(pointArray2);
//        } else {
//          gridP.setPen(QPen(QColor(255,0,0), 2, DotLine));
//          gridP.drawPolyline(pointArray2);
        }
      }
      // Draw the main lines
      gridP.setPen(QPen(QColor(0,0,0), 2));
      gridP.drawPolyline(pointArray);
    }

    // Now the latitudes:
    gridP.setBrush(NoBrush);
    for(long loop = lon1; loop <= lon2; loop++) {
      double temp = numToRad((loop * 600000) - mapCenterLon);

      double projX1 = calc_X_Lambert(lat1int, temp);
      double projY1 = calc_Y_Lambert(lat1int, temp);

      double drawX1 = projX1 * RADIUS / _currentScale + dX;
      double drawY1 = projY1 * RADIUS / _currentScale + dY;

      double projX2 = calc_X_Lambert(lat2int, temp);
      double projY2 = calc_Y_Lambert(lat2int, temp);

      double drawX2 = projX2 * RADIUS / _currentScale + dX;
      double drawY2 = projY2 * RADIUS / _currentScale + dY;

      // Draw the main latitudes:
      gridP.setPen(QPen(QColor(0,0,0), 2));
      gridP.drawLine(drawX1, drawY1, drawX2, drawY2);

      // Draw the small lines between:
      int number = (int) (60.0 / step);

      for(int loop2 = 1; loop2 < number; loop2++) {
        temp = numToRad((long)((loop + (loop2 * step / 60.0)) * 600000)
                - mapCenterLon);

        lat1int = numToRad(lat1 * 600000);
        lat2int = numToRad(lat2 * 600000);

        projX1 = calc_X_Lambert(lat1int, temp);
        projY1 = calc_Y_Lambert(lat1int, temp);

        drawX1 = projX1 * RADIUS / _currentScale + dX;
        drawY1 = projY1 * RADIUS / _currentScale + dY;

        projX2 = calc_X_Lambert(lat2int, temp);
        projY2 = calc_Y_Lambert(lat2int, temp);

        drawX2 = projX2 * RADIUS / _currentScale + dX;
        drawY2 = projY2 * RADIUS / _currentScale + dY;

        if(loop2 == (number / 2.0)) {
          gridP.setPen(QPen(QColor(0,0,0), 2, DashLine));
        } else {
          gridP.setPen(QPen(QColor(0,0,0), lineWidth, DotLine));
        }
        gridP.drawLine(drawX1, drawY1, drawX2, drawY2);
      }
    }
  }
  gridP.end();
}

void Map::drawMap()
{
  if(!isEnabled) {
    return;
  }

  QPainter addSitesP;
  QPainter airP;
  QPainter airSpaceP;
  QPainter cityP;
  QPainter gliderP;
  QPainter hydroP;
  QPainter landP;
  QPainter navP;
  QPainter outP;
  QPainter railP;
  QPainter roadP;
  QPainter topoP;
  QPainter waypointP;

  addSitesP.begin(&pixAddSites);
  airP.begin(&pixAirport);
  airSpaceP.begin(&pixAirspace);
  cityP.begin(&pixCity);
  gliderP.begin(&pixGlider);
  hydroP.begin(&pixHydro);
  landP.begin(&pixLand);
  navP.begin(&pixNav);
  outP.begin(&pixOut);
  railP.begin(&pixRail);
  roadP.begin(&pixRoad);
  topoP.begin(&pixTopo);
  waypointP.begin(&pixWaypoints);

  delete airspaceRegList;
  delete cityRegList;
  airspaceRegList = new QList<QRegion>;
  cityRegList = new QList<QRegion>;
  airspaceRegList->setAutoDelete(true);
  cityRegList->setAutoDelete(true);

  extern double _scale[];
  extern int _scaleBorder[];
  extern double _currentScale;

  MapContents* mContents = app->getMapContents();

  double mapX, mapY;

  mapX = calc_X_Lambert(numToRad(mapCenterLat), 0);
  mapY = calc_Y_Lambert(numToRad(mapCenterLat), 0);

  double dX = ( this->width() / 2 ) - ( mapX / _currentScale * RADIUS);
  double dY = ( this->height() / 2 ) - ( mapY / _currentScale * RADIUS);

  BaseMapElement* _current;

  delete[] xPos;
  delete[] yPos;

  posNum = mContents->getListLength(MapContents::GliderList);
  xPos = new int[posNum];
  yPos = new int[posNum];

  // Statusbar noch nicht "genial" eingestellt ...
  app->slotSetProgress(0);

  // First we display the glaciers:
  if(_currentScale <= _scale[_scaleBorder[11]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::TopoList); loop++) {
      _current = mContents->getElement(MapContents::TopoList, loop);
      _current->drawMapElement(&topoP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(5);

  // Now we take the hydrography:
  for(unsigned int loop = 0;
          loop < mContents->getListLength(MapContents::HydroList); loop++) {
    _current = mContents->getElement(MapContents::HydroList, loop);
    _current->drawMapElement(&hydroP, dX, dY, mapCenterLon);
  }
  app->slotSetProgress(10);

  // Now the small roads:
  for(unsigned int loop = 0;
          loop < mContents->getListLength(MapContents::RoadList); loop++) {
    _current = mContents->getElement(MapContents::RoadList, loop);
    _current->drawMapElement(&roadP, dX, dY, mapCenterLon);
  }

  app->slotSetProgress(15);

  // Now the highways
  if(_currentScale <= _scale[_scaleBorder[12]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::HighwayList); loop++) {
      _current = mContents->getElement(MapContents::HighwayList, loop);
      _current->drawMapElement(&roadP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(20);

  // If needed, here are the highwayentries:
  if(_currentScale <= _scale[_scaleBorder[13]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::HighwayEntryList); loop++) {
      _current = mContents->getElement(MapContents::HighwayEntryList, loop);
      _current->drawMapElement(&roadP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(25);

  // Here are the railways and aerial-railways
  for(unsigned int loop = 0;
          loop < mContents->getListLength(MapContents::RailList); loop++) {
    _current = mContents->getElement(MapContents::RailList, loop);
    _current->drawMapElement(&railP, dX, dY, mapCenterLon);
  }

  app->slotSetProgress(30);
/*
  // If needed, here are the stations
  // Hier stimmt die Liste noch nicht!
  if(isRail) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::RailList); loop++) {
      _current = mContents->getElement(MapContents::RailList, loop);
      _current->drawMapElement(&railP, dX, dY, mapCenterLon);
    }
  }
*/
  app->slotSetProgress(35);

  // Now we come to the cities:
  for(unsigned int loop = 0;
          loop < mContents->getListLength(MapContents::CityList); loop++) {
    _current = mContents->getElement(MapContents::CityList, loop);
    cityRegList->append(_current->drawRegion(&cityP, dX, dY, mapCenterLon));
  }

  app->slotSetProgress(40);

  // Here are the villages:
  if(_currentScale <= _scale[_scaleBorder[21]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::VillageList); loop++) {
      _current = mContents->getElement(MapContents::VillageList, loop);
      _current->drawMapElement(&cityP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(45);

  // No we display the additional landmarks:
  if(_currentScale <= _scale[_scaleBorder[16]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::LandmarkList); loop++) {
      _current = mContents->getElement(MapContents::LandmarkList, loop);
      _current->drawMapElement(&landP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(50);

  // Here we have the obstacles:
  if(_currentScale <= _scale[_scaleBorder[8]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::ObstacleList); loop++) {
      _current = mContents->getElement(MapContents::ObstacleList, loop);
      _current->drawMapElement(&topoP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(55);

  // Here are the reporting-points:
  if(_currentScale <= _scale[_scaleBorder[23]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::ReportList); loop++) {
      _current = mContents->getElement(MapContents::ReportList, loop);
      _current->drawMapElement(&navP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(60);

  // And here the radio-navigation-facilities:
  if(_currentScale <= _scale[_scaleBorder[23]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::NavList); loop++) {
      _current = mContents->getElement(MapContents::NavList, loop);
      _current->drawMapElement(&navP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(65);
  // Now we display the airspace-structure:
  if(_currentScale <= _scale[_scaleBorder[22]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::AirspaceList); loop++) {
      _current = mContents->getElement(MapContents::AirspaceList, loop);
      airspaceRegList->append(_current->drawRegion(&airSpaceP, dX, dY, mapCenterLon));
    }
  }

  app->slotSetProgress(70);

  // Here are the international airports:
  if(_currentScale <= _scale[_scaleBorder[24]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::IntAirportList); loop++) {
      _current = mContents->getElement(MapContents::IntAirportList, loop);
      _current->drawRegion(&airP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(75);
  // Here are the bigger airports:
  if(_currentScale <= _scale[_scaleBorder[24]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::AirportList); loop++) {
      _current = mContents->getElement(MapContents::AirportList, loop);
      _current->drawMapElement(&airP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(80);

  // Here are the additional airports and sites:
  if(_currentScale <= _scale[_scaleBorder[28]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::AddSitesList); loop++) {
      _current = mContents->getElement(MapContents::AddSitesList, loop);
      _current->drawMapElement(&addSitesP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(85);

  // Here are the outlanding-fields:
  if(_currentScale <= _scale[_scaleBorder[26]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::OutList); loop++) {
      _current = mContents->getElement(MapContents::OutList, loop);
      _current->drawMapElement(&outP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(90);

  // Okay, here are the glider-sites:
  if(_currentScale <= _scale[_scaleBorder[25]]) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::GliderList); loop++) {
      _current = mContents->getElement(MapContents::GliderList, loop);
      _current->drawMapElement(&gliderP, dX, dY, mapCenterLon);
    }
  }

  app->slotSetProgress(95);

/*
  // Last but not least, we display the privat waypoints:
  if(isWaypoint && (_currentScale <= _scale[1])) {
    for(unsigned int loop = 0;
            loop < mContents->getListLength(MapContents::WaypointList); loop++) {
      _current = mContents->getElement(MapContents::WaypointList, loop);
      _current->drawMapElement(mapP, dX, dY, mapCenterLon);
    }
  }
*/
/*
  // Now the new digitized objects:
  for(unsigned int loop = 0; loop < mContents->getListLength(20); loop++) {
    _current = mContents->getElement(20, loop);
    _current->drawMapElement(digitP, dX, dY, mapCenterLon);
  }
*/
  if(displayFlights) {
    drawFlight();
  }
  app->slotSetProgress(100);

  // Closing the painter ...
  addSitesP.end();
  airP.end();
  airSpaceP.end();
  cityP.end();
  gliderP.end();
  hydroP.end();
  landP.end();
  navP.end();
  outP.end();
  railP.end();
  roadP.end();
  topoP.end();
  waypointP.end();
}

void Map::resizeEvent(QResizeEvent* event)
{

  pixBuffer.fill(white);
  pixBuffer.resize(event->size());

  pixAddSites.resize(event->size());
  pixAirport.resize(event->size());
  pixAirspace.resize(event->size());
  pixCity.resize(event->size());
  pixFlight.resize(event->size());
  pixGlider.resize(event->size());
  pixGrid.resize(event->size());
  pixHydro.resize(event->size());
  pixLand.resize(event->size());
  pixNav.resize(event->size());
  pixOut.resize(event->size());
  pixRail.resize(event->size());
  pixRoad.resize(event->size());
  pixTopo.resize(event->size());
  pixWaypoints.resize(event->size());

  if(isEnabled) {
    redrawMap();
  }
}

void Map::redrawMap()
{
  pixAddSites.fill(white);
  pixAirport.fill(white);
  pixAirspace.fill(white);
  pixCity.fill(white);
  pixFlight.fill(white);
  pixGlider.fill(white);
  pixGrid.fill(white);
  pixHydro.fill(white);
  pixLand.fill(white);
  pixNav.fill(white);
  pixOut.fill(white);
  pixRail.fill(white);
  pixRoad.fill(white);
  pixTopo.fill(white);
  pixWaypoints.fill(white);

  pixBuffer.fill(white);

  drawGrid();
  drawMap();

  slotShowLayer();
}

void Map::slotRedrawMap()
{
  redrawMap();
}

void Map::slotShowLayer()
{
  showAddSites = main->toolBar()->isButtonOn(ID_LAYER_ADDSITES);
  showAirport  = main->toolBar()->isButtonOn(ID_LAYER_AIRPORT);
  showAirspace = main->toolBar()->isButtonOn(ID_LAYER_AIRSPACE);
  showCity     = main->toolBar()->isButtonOn(ID_LAYER_CITY);
  showFlight   = main->toolBar()->isButtonOn(ID_LAYER_FLIGHT);
  showGlider   = main->toolBar()->isButtonOn(ID_LAYER_GLIDER);
  showHydro    = main->toolBar()->isButtonOn(ID_LAYER_HYDRO);
  showLand     = main->toolBar()->isButtonOn(ID_LAYER_LAND);
  showNav      = main->toolBar()->isButtonOn(ID_LAYER_NAV);
  showOut      = main->toolBar()->isButtonOn(ID_LAYER_OUT);
  showRail     = main->toolBar()->isButtonOn(ID_LAYER_RAIL);
  showRoad     = main->toolBar()->isButtonOn(ID_LAYER_ROAD);
  showTopo     = main->toolBar()->isButtonOn(ID_LAYER_TOPO);
  showWaypoint = main->toolBar()->isButtonOn(ID_LAYER_WAYPOINT);

  main->menuBar()->setItemChecked(ID_LAYER_ADDSITES, showAddSites);
  main->menuBar()->setItemChecked(ID_LAYER_AIRPORT, showAirport);
  main->menuBar()->setItemChecked(ID_LAYER_AIRSPACE, showAirspace);
  main->menuBar()->setItemChecked(ID_LAYER_CITY, showCity);
  main->menuBar()->setItemChecked(ID_LAYER_FLIGHT, showFlight);
  main->menuBar()->setItemChecked(ID_LAYER_GLIDER, showGlider);
  main->menuBar()->setItemChecked(ID_LAYER_HYDRO, showHydro);
  main->menuBar()->setItemChecked(ID_LAYER_LAND, showLand);
  main->menuBar()->setItemChecked(ID_LAYER_NAV, showNav);
  main->menuBar()->setItemChecked(ID_LAYER_OUT, showOut);
  main->menuBar()->setItemChecked(ID_LAYER_RAIL, showRail);
  main->menuBar()->setItemChecked(ID_LAYER_ROAD, showRoad);
  main->menuBar()->setItemChecked(ID_LAYER_TOPO, showTopo);
  main->menuBar()->setItemChecked(ID_LAYER_WAYPOINT, showWaypoint);

  // deleting the current pixmap and displaying the grid:
  bitBlt(&pixBuffer, 0, 0, &pixGrid, 0, 0, -1, -1);

  if(showHydro) {
    bitBlt(&pixBuffer, 0, 0, &pixHydro, 0, 0, -1, -1, NotEraseROP);
  }
  if(showRoad) {
    bitBlt(&pixBuffer, 0, 0, &pixRoad, 0, 0, -1, -1, NotEraseROP);
  }
  if(showRail) {
    bitBlt(&pixBuffer, 0, 0, &pixRail, 0, 0, -1, -1, NotEraseROP);
  }
  if(showCity) {
    bitBlt(&pixBuffer, 0, 0, &pixCity, 0, 0, -1, -1, NotEraseROP);
  }
  if(showLand) {
    bitBlt(&pixBuffer, 0, 0, &pixLand, 0, 0, -1, -1, NotEraseROP);
  }
  if(showTopo) {
    bitBlt(&pixBuffer, 0, 0, &pixTopo, 0, 0, -1, -1, NotEraseROP);
  }
  if(showAirport) {
    bitBlt(&pixBuffer, 0, 0, &pixAirport, 0, 0, -1, -1, NotEraseROP);
  }
  if(showNav) {
    bitBlt(&pixBuffer, 0, 0, &pixNav, 0, 0, -1, -1, NotEraseROP);
  }
  if(showAddSites) {
    bitBlt(&pixBuffer, 0, 0, &pixAddSites, 0, 0, -1, -1, NotEraseROP);
  }
  if(showGlider) {
    bitBlt(&pixBuffer, 0, 0, &pixGlider, 0, 0, -1, -1, NotEraseROP);
  }
  if(showOut) {
    bitBlt(&pixBuffer, 0, 0, &pixOut, 0, 0, -1, -1, NotEraseROP);
  }
  if(showWaypoint) {
    bitBlt(&pixBuffer, 0, 0, &pixWaypoints, 0, 0, -1, -1, NotEraseROP);
  }
  if(showAirspace) {
    bitBlt(&pixBuffer, 0, 0, &pixAirspace, 0, 0, -1, -1, NotEraseROP);
  }
  if(showFlight) {
    bitBlt(&pixBuffer, 0, 0, &pixFlight, 0, 0, -1, -1, NotEraseROP);
  }

  paintEvent();
}

void Map::slotScaleAdd()
{
  extern double _scale[];
  extern double _currentScale;
cerr << _currentScale << " " << _scale[1] << endl;
  if((_currentScale / 1.25) >= _scale[1]) {
    _currentScale = (_currentScale / 1.25);
    redrawMap();
  }
}

void Map::slotScaleSub()
{
  extern double _scale[];
  extern double _currentScale;
cerr << _currentScale  << " " << _scale[6] << endl;
  if((_currentScale * 1.25) <= _scale[6]) {
    _currentScale = (_currentScale * 1.25);
    redrawMap();
  }
}

void Map::slotEnableMap(bool paint)
{
  isEnabled = paint;
}

void Map::slotDrawFlight()
{
  if(flightList.count() == 0) {
    return;
  }

  displayFlights = true;

  drawFlight();
  slotShowLayer();
}

void Map::slotConfigureMap()
{
  MapConfig* confD = new MapConfig(main);
}

void Map::slotDeleteFlight()
{
  unsigned int index = 0;
  if(flightList.count() == 1) {
    index = 0;
  } else {
/*
    dW = new QDialog(parentWidget, "dialog", true);

    QLabel* titleLabel = new QLabel(
        i18n("Which flight do You want to delete?"), dW);
    QListView* flView = new QListView(dW);
    flView->setMinimumWidth(220);
    flView->addColumn(i18n("Nr."),45);
    flView->addColumn(i18n("Date"),205);
    flView->addColumn(i18n("Start-time"),200);
    flView->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    flView->setAllColumnsShowFocus(true);
    flView->setRootIsDecorated(true);

    QPushButton* cancel = new QPushButton(i18n("Cancel"), dW);

    QGridLayout* dWLayout = new QGridLayout(dW,5,5,15,1);
    dWLayout->addMultiCellWidget(titleLabel,0,0,0,2);
    dWLayout->addMultiCellWidget(flView,2,2,0,4);
    dWLayout->addWidget(cancel,4,4);

    dWLayout->activate();

    dW->exec();
    index = 1; // nur temporr ...
*/
  }
  flightList.remove(index);

  view->delWPListView();
  pixFlight.fill(white);

  drawFlight();
  slotShowLayer();
}

void Map::drawFlight()
{
  QPainter flightP;
  flightP.begin(&pixFlight);

  extern double _currentScale;

  double mapX, mapY;

  mapX = calc_X_Lambert(numToRad(mapCenterLat), 0);
  mapY = calc_Y_Lambert(numToRad(mapCenterLat), 0);

  double dX = ( this->width() / 2 ) - ( mapX / _currentScale * RADIUS );
  double dY = ( this->height() / 2 ) - ( mapY / _currentScale * RADIUS );

  for(unsigned int loop = 0; loop < flightList.count(); loop++) {
    flightList.at(loop)->drawMapElement(&flightP, dX, dY, mapCenterLon);
  }
  flightP.end();
}

void Map::loadFlight(MapContents* mContents, QFile* flightDataFile)
{
  flightList.append(new Flight(mContents, flightDataFile));
  slotDrawFlight();
  view->showFlightData(flightList.current());
}

void Map::loadFlight(MapContents* mContents, QString flightDataFileName)
{
  QFile* flightDataFile = new QFile(flightDataFileName);
  flightList.append(new Flight(mContents, flightDataFile));
  slotDrawFlight();
  view->showFlightData(flightList.current());
}

void Map::loadWaypoint(MapContents* mContents, QString waypointFileName)
{
  QFile* waypointFile = new QFile(waypointFileName);

   if(waypointFile->open(IO_ReadOnly)) {
    QTextStream t( waypointFile );
    // Typen Namen
    QString s, height, typ, name, frequency, direction,
            length, surface, alias;
    QList<QString> runwayList;
    bool isObject = false;
    runwayList.setAutoDelete(true);
    long latitude = 0, longitude = 0;
    int list = 1, runway = 0;
    while(!t.eof()) {
      s = t.readLine().simplifyWhiteSpace();
      if(s == "") {
        if(isObject && (typ != "") && (name != "")) {
          //  write into file;
          QFile mapFile("/tmp/kflog_wend.out");
          if(!mapFile.open(IO_ReadWrite)) {
            KMsgBox* errorBox = new KMsgBox(parentWidget,i18n("Error"),
            i18n("Cannot access the mapfile!"),4,i18n("OK"));
            errorBox->exec();
            return;
          }
          mapFile.at(mapFile.size());
          QTextStream mapOut(&mapFile);

          name = name.replace(QRegExp("\""), "");
          mapOut << "[NEW]\n"
                 << "TYPE=" << typ << endl
                 << "NAME=" << name << endl
                 << "LIST=" << list << endl
                 << "ELEV=" << height << endl;
          if(frequency != "") {
            mapOut << "FREQUENCY=" << frequency << endl;
          } else if(alias != "") {
            mapOut << "ALIAS=" << alias << endl;
          }

          unsigned int length = runwayList.count();
          for(unsigned int loop = 0; loop < length; loop++) {
            QString temp = *runwayList.take(0);
            mapOut << "RUNWAY=" << temp << endl;
          }
          mapOut << "AT=" << latitude << " " << longitude << endl
                 << "[END]" << endl;
                     // lschen der Variablen
          isObject = false;
          typ = "";
          list = 1;
          name = "";
          height = "";
          frequency = "";
          alias = "";
          latitude = 0;
          longitude = 0;
         }
       } else if(s.mid(0,2) == "at") {
        isObject = true;
        runway = 0;
        runwayList.~QList();
        // New Object
        unsigned int loop = 0;
        // Koordinaten
        if((s.mid(11,1) == "N") || (s.mid(11,1) == "S")) {
          latitude = degreeToNum(s.mid(3,9));
          loop = 13;
        } else if((s.mid(12,1) == "N") || (s.mid(12,1) == "S")) {
          latitude = degreeToNum(s.mid(3,10));
          loop = 14;
        }

        if((s.mid(loop + 9,1) == "E") || (s.mid(loop + 9,1) == "W")) {
          longitude = degreeToNum(s.mid(loop,9));
          loop += 9;
        } else if((s.mid(loop + 10,1) == "E") ||
                  (s.mid(loop + 10,1) == "W")) {
          longitude = degreeToNum(s.mid(loop,10));
          loop += 10;
        } else if((s.mid(loop + 11,1) == "E") ||
                  (s.mid(loop + 11,1) == "W")) {
          longitude = degreeToNum(s.mid(loop,11));
          loop += 11;
        } else {
        }
        //Hoehe
        height = s.mid(loop + 6,10);
      // Typen
      } else if(s.mid(0,11) == "GLIDER_SITE") {
        typ = "glider";
        list = 3;
        name = s.mid(12,100);
      } else if(s.mid(0,15) == "AIRPORT_CIV_MIL") {
        typ = "civ-mil-air";
        name = s.mid(16,100);
      } else if(s.mid(0,8) == "AIRFIELD") {
        typ = "airfield";
        name = s.mid(9,100);
      } else if(s.mid(0,16) == "SPECIAL_AIRFIELD") {
        // ???????????
      } else if(s.mid(0,11) == "AIRPORT_MIL") {
        typ = "mil-air";
        name = s.mid(12,25);
      } else if(s.mid(0,11) == "INTLAIRPORT") {
        typ = "int-airport";
        name = s.mid(12,25);
      } else if(s.mid(0,7) == "AIRPORT") {
        typ = "airport";
        name = s.mid(8,25);
      } else if(s.mid(0,9) == "frequency") {
        frequency = s.mid(10,7);
      } else if(s.mid(0,6) == "runway") {
        runway++;
        QString* temp = 0;
        // search for space (hopefully, there will only be one!)
        unsigned int loop = 0;
        for(loop = 11; loop < strlen(s); loop++) {
          if(s.mid(loop, 1) == " ") {
            break;
          }
        }
        direction = s.mid(7,3);
        if(direction.mid(0,2) > "36") {
          direction = "0" + direction.mid(0,1);
        } else if(direction.mid(0,2) == "00") {
          direction = "36";
        } else {
          direction = direction.mid(0,2);
        }
        length = s.mid(11,loop - 11);

        if(s.mid(loop + 1,8) == "CONCRETE") {
          temp = new QString("C " + direction + " " + length);
        } else if(s.mid(loop + 1,4) == "GRAS") {
          temp = new QString("G " + direction + " " + length);
        } else if(s.mid(loop + 1,7) == "ASPHALT") {
          temp = new QString("A " + direction + " " + length);
        }
        runwayList.append(temp);
      } else if(s.mid(0,5) == "alias") {
        alias = s.mid(5,4);
      } else {
        warning("KFLog: unknown entry found in ascii-map-file");
      }
    }
  }
}

Map::Map(QWidget *parent = 0, KFLogApp *m = 0, KFLogView* nview = 0)
  : QWidget(parent)
{
  extern double _v1;
  extern double _v2;
  extern double _var1;
  extern double _var2;
  extern double _scale[];
  extern int _scaleBorder[];
  extern bool _showElements[];
  extern double _currentScale;

  main = m;

  posNum = 1;
  xPos = new int[1];
  yPos = new int[1];

  view = nview;

  airspaceRegList = new QList<QRegion>;
  cityRegList = new QList<QRegion>;
  airspaceRegList->setAutoDelete(true);
  cityRegList->setAutoDelete(true);
  displayFlights = false;

  parentWidget = parent;
  KConfig *config = kapp->getConfig();

  config->setGroup("General");
  _currentScale = config->readDoubleNumEntry("MapScale", 100.0);

  mapCenterLon = config->readNumEntry("MapCenterX",
      degreeToNum("0900,0000\"E"));
  mapCenterLat = config->readNumEntry("MapCenterY",
      degreeToNum("4800,0000\"N"));
  _v1 = numToRad(config->readNumEntry("Parallel1",
      degreeToNum(DEFAULT_V1)));
  _v2 = numToRad(config->readNumEntry("Parallel2",
      degreeToNum(DEFAULT_V2)));

  _var1 = cos(_v1)*cos(_v1);
  _var2 = sin(_v1)+sin(_v2);

  config->setGroup("MapLayer");
  showAddSites = config->readBoolEntry("AddSites", true);
  showAirport = config->readBoolEntry("Airport", true);
  showAirspace = config->readBoolEntry("Airspace", true);
  showCity = config->readBoolEntry("City ", true);
  showFlight = config->readBoolEntry("Flight", true);
  showGlider = config->readBoolEntry("Glider", true);
  showHydro = config->readBoolEntry("Hydro", true);
  showLand = config->readBoolEntry("Land", true);
  showNav = config->readBoolEntry("Nav", true);
  showOut = config->readBoolEntry("Out", true);
  showRail = config->readBoolEntry("Rail", true);
  showRoad = config->readBoolEntry("Road", true);
  showTopo = config->readBoolEntry("Topo", true);
  showWaypoint = config->readBoolEntry("Waypoint", true);

  config->setGroup("MapScale");
  _scale[0] = config->readNumEntry("Scale0", 10);
  _scale[1] = config->readNumEntry("Scale1", 70);
  _scale[2] = config->readNumEntry("Scale2", 100);
  _scale[3] = config->readNumEntry("Scale3", 150);
  _scale[4] = config->readNumEntry("Scale4", 280);
  _scale[5] = config->readNumEntry("Scale5", 640);
  _scale[6] = config->readNumEntry("Scale6", 1000);

  config->setGroup("MapBorder");
  _scaleBorder[0] = config->readNumEntry("ShoreLine",6);
  _scaleBorder[1] = config->readNumEntry("BigHydro",6);
  _scaleBorder[2] = config->readNumEntry("MidHydro",5);
  _scaleBorder[3] = config->readNumEntry("SmallHydro",3);
  _scaleBorder[4] = config->readNumEntry("Dam",2);
  _scaleBorder[5] = config->readNumEntry("RailTrack",5);
  _scaleBorder[6] = config->readNumEntry("AerialTrack",3);
  _scaleBorder[7] = config->readNumEntry("Station",3);
  _scaleBorder[8] = config->readNumEntry("Obstacle",4);
  _scaleBorder[9] = config->readNumEntry("Spot",2);
  _scaleBorder[10] = config->readNumEntry("Pass",2);
  _scaleBorder[11] = config->readNumEntry("Glacier",4);
  _scaleBorder[12] = config->readNumEntry("Highway",6);
  _scaleBorder[13] = config->readNumEntry("HighwayEntry",2);
  _scaleBorder[14] = config->readNumEntry("MidRoad",4);
  _scaleBorder[15] = config->readNumEntry("SmallRoad",3);
  _scaleBorder[16] = config->readNumEntry("Landmark",2);
  _scaleBorder[17] = config->readNumEntry("HugeCity",6);
  _scaleBorder[18] = config->readNumEntry("BigCity",5);
  _scaleBorder[19] = config->readNumEntry("MidCity",4);
  _scaleBorder[20] = config->readNumEntry("SmallCity",3);
  _scaleBorder[21] = config->readNumEntry("Village",3);
  _scaleBorder[22] = config->readNumEntry("Airspace",6);
  _scaleBorder[23] = config->readNumEntry("Radio",4);
  _scaleBorder[24] = config->readNumEntry("Airport",4);
  _scaleBorder[25] = config->readNumEntry("GliderSites",4);
  _scaleBorder[26] = config->readNumEntry("Outlanding",3);
  _scaleBorder[27] = config->readNumEntry("Waypoints",3);
  _scaleBorder[28] = config->readNumEntry("AddSites",3);

  config->setGroup("ShowElements");
  _showElements[0] = config->readBoolEntry("VOR", true);
  _showElements[1] = config->readBoolEntry("VORDME", true);
  _showElements[2] = config->readBoolEntry("VORTAC", true);
  _showElements[3] = config->readBoolEntry("NDB", true);
  _showElements[4] = config->readBoolEntry("CompPoint", true);
  _showElements[5] = config->readBoolEntry("IntAirport", true);
  _showElements[6] = config->readBoolEntry("Airport", true);
  _showElements[7] = config->readBoolEntry("MilAirport", true);
  _showElements[8] = config->readBoolEntry("CivMilAirport", true);
  _showElements[9] = config->readBoolEntry("Airfield", true);
  _showElements[10] = config->readBoolEntry("ClosedAirfield", true);
  _showElements[11] = config->readBoolEntry("CivHeliport", true);
  _showElements[12] = config->readBoolEntry("MilHeliport", true);
  _showElements[13] = config->readBoolEntry("AmbHeliport", true);
  _showElements[14] = config->readBoolEntry("UltraLight", true);
  _showElements[15] = config->readBoolEntry("HangGlider", true);
  _showElements[16] = config->readBoolEntry("Parachute", true);
  _showElements[17] = config->readBoolEntry("Ballon", true);

  if(showAddSites) {
    main->toolBar()->getButton(ID_LAYER_ADDSITES)->toggle();
  }
  if(showAirport) {
    main->toolBar()->getButton(ID_LAYER_AIRPORT)->toggle();
  }
  if(showAirspace) {
    main->toolBar()->getButton(ID_LAYER_AIRSPACE)->toggle();
  }
  if(showCity) {
    main->toolBar()->getButton(ID_LAYER_CITY)->toggle();
  }
  if(showFlight) {
    main->toolBar()->getButton(ID_LAYER_FLIGHT)->toggle();
  }
  if(showGlider) {
    main->toolBar()->getButton(ID_LAYER_GLIDER)->toggle();
  }
  if(showHydro) {
    main->toolBar()->getButton(ID_LAYER_HYDRO)->toggle();
  }
  if(showLand) {
    main->toolBar()->getButton(ID_LAYER_LAND)->toggle();
  }
  if(showNav) {
    main->toolBar()->getButton(ID_LAYER_NAV)->toggle();
  }
  if(showOut) {
    main->toolBar()->getButton(ID_LAYER_OUT)->toggle();
  }
  if(showRail) {
    main->toolBar()->getButton(ID_LAYER_RAIL)->toggle();
  }
  if(showRoad) {
    main->toolBar()->getButton(ID_LAYER_ROAD)->toggle();
  }
  if(showTopo) {
    main->toolBar()->getButton(ID_LAYER_TOPO)->toggle();
  }
  if(showWaypoint) {
    main->toolBar()->getButton(ID_LAYER_WAYPOINT)->toggle();
  }

  app = main;

  isEnabled = false;

  // defining the cursor for the map:
  static unsigned char cross_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00,
    0x00, 0x80, 0x01, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00, 0x98, 0x19, 0x00,
    0x00, 0x8c, 0x31, 0x00, 0x00, 0x86, 0x61, 0x00, 0x00, 0x83, 0xc1, 0x00,
    0x00, 0xc1, 0x83, 0x00, 0x80, 0x60, 0x06, 0x01, 0x80, 0x30, 0x0c, 0x01,
    0xe0, 0x9f, 0xf9, 0x07, 0xe0, 0x9f, 0xf9, 0x07, 0x80, 0x30, 0x0c, 0x01,
    0x80, 0x60, 0x06, 0x01, 0x00, 0xc1, 0x83, 0x00, 0x00, 0x83, 0xc1, 0x00,
    0x00, 0x86, 0x61, 0x00, 0x00, 0x8c, 0x31, 0x00, 0x00, 0x98, 0x19, 0x00,
    0x00, 0xe0, 0x07, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0x80, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  };
  QBitmap cross(32, 32, cross_bits, true);
  QCursor crossCursor(cross, cross);

  this->setMouseTracking(true);
  this->setBackgroundColor(QColor(255,255,255));
  this->setCursor(crossCursor);
}

Map::~Map()
{
  extern double _v1;
  extern double _v2;
  extern double _currentScale;
  extern double _scale[];
  extern int _scaleBorder[];
  extern bool _showElements[];

  KConfig *config = kapp->getConfig();

  config->setGroup("General");
  config->writeEntry("MapScale", _currentScale);
  config->writeEntry("MapCenterX", mapCenterLon);
  config->writeEntry("MapCenterY", mapCenterLat);
  if((_v1 != numToRad(degreeToNum("5400'0000\"N"))) ||
     (_v2 != numToRad(degreeToNum("5000'0000\"N")))) {
    config->writeEntry("Parallel1", radToNum(_v1));
    config->writeEntry("Parallel2", radToNum(_v2));
  }

  config->setGroup("MapLayer");
  config->writeEntry("AddSites", showAddSites);
  config->writeEntry("Airport", showAirport);
  config->writeEntry("Airspace", showAirspace);
  config->writeEntry("City", showCity);
  config->writeEntry("Flight", showFlight);
  config->writeEntry("Glider", showGlider);
  config->writeEntry("Hydro", showHydro);
  config->writeEntry("Land", showLand);
  config->writeEntry("Nav", showNav);
  config->writeEntry("Out", showOut);
  config->writeEntry("Rail", showRail);
  config->writeEntry("Road", showRoad);
  config->writeEntry("Topo", showTopo);
  config->writeEntry("Waypoint", showWaypoint);

  config->setGroup("MapScale");
  config->writeEntry("Scale0", _scale[0]);
  config->writeEntry("Scale1", _scale[1]);
  config->writeEntry("Scale2", _scale[2]);
  config->writeEntry("Scale3", _scale[3]);
  config->writeEntry("Scale4", _scale[4]);
  config->writeEntry("Scale5", _scale[5]);
  config->writeEntry("Scale6", _scale[6]);

  config->setGroup("MapBorder");
  config->writeEntry("ShoreLine",_scaleBorder[0]);
  config->writeEntry("BigHydro",_scaleBorder[1]);
  config->writeEntry("MidHydro",_scaleBorder[2]);
  config->writeEntry("SmallHydro",_scaleBorder[3]);
  config->writeEntry("Dam",_scaleBorder[4]);
  config->writeEntry("RailTrack",_scaleBorder[5]);
  config->writeEntry("AerialTrack",_scaleBorder[6]);
  config->writeEntry("Station",_scaleBorder[7]);
  config->writeEntry("Obstacle",_scaleBorder[8]);
  config->writeEntry("Spot",_scaleBorder[9]);
  config->writeEntry("Pass",_scaleBorder[10]);
  config->writeEntry("Glacier",_scaleBorder[11]);
  config->writeEntry("Highway",_scaleBorder[12]);
  config->writeEntry("HighwayEntry",_scaleBorder[13]);
  config->writeEntry("MidRoad",_scaleBorder[14]);
  config->writeEntry("SmallRoad",_scaleBorder[15]);
  config->writeEntry("Landmark",_scaleBorder[16]);
  config->writeEntry("HugeCity",_scaleBorder[17]);
  config->writeEntry("BigCity",_scaleBorder[18]);
  config->writeEntry("MidCity",_scaleBorder[19]);
  config->writeEntry("SmallCity",_scaleBorder[20]);
  config->writeEntry("Village",_scaleBorder[21]);
  config->writeEntry("Airspace",_scaleBorder[22]);
  config->writeEntry("Radio",_scaleBorder[23]);
  config->writeEntry("Airport",_scaleBorder[24]);
  config->writeEntry("GliderSites",_scaleBorder[25]);
  config->writeEntry("Outlanding",_scaleBorder[26]);
  config->writeEntry("Waypoints",_scaleBorder[27]);
  config->writeEntry("AddSites",_scaleBorder[28]);

  config->setGroup("ShowElements");
  config->writeEntry("VOR", _showElements[0]);
  config->writeEntry("VORDME", _showElements[1]);
  config->writeEntry("VORTAC", _showElements[2]);
  config->writeEntry("NDB", _showElements[3]);
  config->writeEntry("CompPoint", _showElements[4]);
  config->writeEntry("IntAirport", _showElements[5]);
  config->writeEntry("Airport", _showElements[6]);
  config->writeEntry("MilAirport", _showElements[7]);
  config->writeEntry("CivMilAirport", _showElements[8]);
  config->writeEntry("Airfield", _showElements[9]);
  config->writeEntry("ClosedAirfield", _showElements[10]);
  config->writeEntry("CivHeliport", _showElements[11]);
  config->writeEntry("MilHeliport", _showElements[12]);
  config->writeEntry("AmbHeliport", _showElements[13]);
  config->writeEntry("UltraLight", _showElements[14]);
  config->writeEntry("HangGlider", _showElements[15]);
  config->writeEntry("Parachute", _showElements[16]);
  config->writeEntry("Ballon", _showElements[17]);

  config->setGroup(0);
}
