/***********************************************************************
**
**   mapconfig.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include <iostream>

#include <basemapelement.h>
#include <mapconfig.h>

#include <kseparator.h>
#include <qgroupbox.h>
#include <qslider.h>
#include <qspinbox.h>

int MapConfig::setScaleValue(int value)
{
  if(value <= 40) {
    return (value * 5);
  } else if(value <= 70) {
    return (200 + (value - 40) * 10);
  } else if(value <= 95) {
    return (500 + (value - 70) * 20);
  } else if(value <= 105){
    return (1000 + (value - 95) * 50);
  } else {
    return (2000 + (value - 105) * 100);
  }
}

int MapConfig::getScaleValue(int scale)
{
  if(scale <= 200) {
    return (scale / 5);
  } else if (scale <= 500) {
    return ((scale - 200) / 10 + 40);
  } else if (scale <= 1000) {
    return ((scale - 500) / 20 + 70);
  } else if(scale <= 2000) {
    return ((scale - 1000) / 50 + 95);
  } else {
    return ((scale - 2000) / 100 + 125);
  }
}

void MapConfig::slotSetuLimit(int limit)
{
  uLimitN->display(setScaleValue(limit));
}

void MapConfig::slotSetBorder1(int limit)
{
  scale1N->display(setScaleValue(limit));
}

void MapConfig::slotSetBorder2(int limit)
{
  scale2N->display(setScaleValue(limit));
}

void MapConfig::slotSetBorder3(int limit)
{
  scale3N->display(setScaleValue(limit));
}

void MapConfig::slotSetBorder4(int limit)
{
  scale4N->display(setScaleValue(limit));
}

void MapConfig::slotSetBorder5(int limit)
{
  scale5N->display(setScaleValue(limit));
}

void MapConfig::slotSetDefaultValues()
{
  lLimit->setValue(getScaleValue(10));
  lLimitN->display(10);
  uLimit->setValue(getScaleValue(1000));
  uLimitN->display(1000);
  scale1->setValue(getScaleValue(70));
  scale1N->display(70);
  scale2->setValue(getScaleValue(100));
  scale2N->display(100);
  scale3->setValue(getScaleValue(150));
  scale3N->display(150);
  scale4->setValue(getScaleValue(280));
  scale4N->display(280);
  scale5->setValue(getScaleValue(640));
  scale5N->display(640);

  coastB->setValue(6);
  lakeBigB->setValue(6);
  lakeMidB->setValue(5);
  lakeSmallB->setValue(3);
  damB->setValue(2);
  railB->setValue(5);
  aerialB->setValue(3);
  stationB->setValue(3);
  obstacleB->setValue(4);
  spotB->setValue(2);
  passB->setValue(2);
  iceB->setValue(4);
  highB->setValue(6);
  highEB->setValue(2);
  roadMidB->setValue(4);
  roadSmallB->setValue(3);
  landB->setValue(2);
  hugeCityB->setValue(6);
  bigCityB->setValue(5);
  midCityB->setValue(4);
  smallCityB->setValue(3);
  villageB->setValue(3);
  airspaceB->setValue(6);
  radioB->setValue(4);
  airportB->setValue(4);
  gliderB->setValue(4);
  outB->setValue(3);
  waypointB->setValue(3);
  addAirportB->setValue(3);

  vorB->setChecked(true);
  vordmeB->setChecked(true);
  vortacB->setChecked(true);
  ndbB->setChecked(true);
  compB->setChecked(true);
  intairB->setChecked(true);
  airB->setChecked(true);
  milAirB->setChecked(true);
  civmilAirB->setChecked(true);
  airfieldB->setChecked(true);
  closedAirB->setChecked(true);
  civHeliB->setChecked(true);
  milHeliB->setChecked(true);
  ambHeliB->setChecked(true);
  ulfB->setChecked(true);
  hangB->setChecked(true);
  parachuteB->setChecked(true);
  ballonB->setChecked(true);
}

void MapConfig::slotSetValues()
{
  extern double _scale[];
  extern int _scaleBorder[];
  extern bool _showElements[];

  _scale[0] = lLimitN->intValue();
  _scale[1] = scale1N->intValue();
  _scale[2] = scale2N->intValue();
  _scale[3] = scale3N->intValue();
  _scale[4] = scale4N->intValue();
  _scale[5] = scale5N->intValue();
  _scale[6] = uLimitN->intValue();

  _scaleBorder[0] = ((QString)coastB->text()).toInt();
  _scaleBorder[1] = ((QString)lakeBigB->text()).toInt();
  _scaleBorder[2] = ((QString)lakeMidB->text()).toInt();
  _scaleBorder[3] = ((QString)lakeSmallB->text()).toInt();
  _scaleBorder[4] = ((QString)damB->text()).toInt();
  _scaleBorder[5] = ((QString)railB->text()).toInt();
  _scaleBorder[6] = ((QString)aerialB->text()).toInt();
  _scaleBorder[7] = ((QString)stationB->text()).toInt();
  _scaleBorder[8] = ((QString)obstacleB->text()).toInt();
  _scaleBorder[9] = ((QString)spotB->text()).toInt();
  _scaleBorder[10] = ((QString)passB->text()).toInt();
  _scaleBorder[11] = ((QString)iceB->text()).toInt();
  _scaleBorder[12] = ((QString)highB->text()).toInt();
  _scaleBorder[13] = ((QString)highEB->text()).toInt();
  _scaleBorder[14] = ((QString)roadMidB->text()).toInt();
  _scaleBorder[15] = ((QString)roadSmallB->text()).toInt();
  _scaleBorder[16] = ((QString)landB->text()).toInt();
  _scaleBorder[17] = ((QString)hugeCityB->text()).toInt();
  _scaleBorder[18] = ((QString)bigCityB->text()).toInt();
  _scaleBorder[19] = ((QString)midCityB->text()).toInt();
  _scaleBorder[20] = ((QString)smallCityB->text()).toInt();
  _scaleBorder[21] = ((QString)villageB->text()).toInt();
  _scaleBorder[22] = ((QString)airspaceB->text()).toInt();
  _scaleBorder[23] = ((QString)radioB->text()).toInt();
  _scaleBorder[24] = ((QString)airportB->text()).toInt();
  _scaleBorder[25] = ((QString)gliderB->text()).toInt();
  _scaleBorder[26] = ((QString)outB->text()).toInt();
  _scaleBorder[27] = ((QString)waypointB->text()).toInt();
  _scaleBorder[28] = ((QString)addAirportB->text()).toInt();

  _showElements[0] = vorB->isChecked();
  _showElements[1] = vordmeB->isChecked();
  _showElements[2] = vortacB->isChecked();
  _showElements[3] = ndbB->isChecked();
  _showElements[4] = compB->isChecked();
  _showElements[5] = intairB->isChecked();
  _showElements[6] = airB->isChecked();
  _showElements[7] = milAirB->isChecked();
  _showElements[8] = civmilAirB->isChecked();
  _showElements[9] = airfieldB->isChecked();
  _showElements[10] = closedAirB->isChecked();
  _showElements[11] = civHeliB->isChecked();
  _showElements[12] = milHeliB->isChecked();
  _showElements[13] = ambHeliB->isChecked();
  _showElements[14] = ulfB->isChecked();
  _showElements[15] = hangB->isChecked();
  _showElements[16] = parachuteB->isChecked();
  _showElements[17] = ballonB->isChecked();

  this->done(0);
}

void MapConfig::loadValues()
{  extern double _scale[];
  extern int _scaleBorder[];
  extern bool _showElements[];

  lLimit->setValue(getScaleValue(_scale[0]));
  lLimitN->display(_scale[0]);
  uLimit->setValue(getScaleValue(_scale[6]));
  uLimitN->display(_scale[6]);
  scale1->setValue(getScaleValue(_scale[1]));
  scale1N->display(_scale[1]);
  scale2->setValue(getScaleValue(_scale[2]));
  scale2N->display(_scale[2]);
  scale3->setValue(getScaleValue(_scale[3]));
  scale3N->display(_scale[3]);
  scale4->setValue(getScaleValue(_scale[4]));
  scale4N->display(_scale[4]);
  scale5->setValue(getScaleValue(_scale[5]));
  scale5N->display(_scale[5]);

  coastB->setValue(_scaleBorder[0]);
  lakeBigB->setValue(_scaleBorder[1]);
  lakeMidB->setValue(_scaleBorder[2]);
  lakeSmallB->setValue(_scaleBorder[3]);
  damB->setValue(_scaleBorder[4]);
  railB->setValue(_scaleBorder[5]);
  aerialB->setValue(_scaleBorder[6]);
  stationB->setValue(_scaleBorder[7]);
  obstacleB->setValue(_scaleBorder[8]);
  spotB->setValue(_scaleBorder[9]);
  passB->setValue(_scaleBorder[10]);
  iceB->setValue(_scaleBorder[11]);
  highB->setValue(_scaleBorder[12]);
  highEB->setValue(_scaleBorder[13]);
  roadMidB->setValue(_scaleBorder[14]);
  roadSmallB->setValue(_scaleBorder[15]);
  landB->setValue(_scaleBorder[16]);
  hugeCityB->setValue(_scaleBorder[17]);
  bigCityB->setValue(_scaleBorder[18]);
  midCityB->setValue(_scaleBorder[19]);
  smallCityB->setValue(_scaleBorder[20]);
  villageB->setValue(_scaleBorder[21]);
  airspaceB->setValue(_scaleBorder[22]);
  radioB->setValue(_scaleBorder[23]);
  airportB->setValue(_scaleBorder[24]);
  gliderB->setValue(_scaleBorder[25]);
  outB->setValue(_scaleBorder[26]);
  waypointB->setValue(_scaleBorder[27]);
  addAirportB->setValue(_scaleBorder[28]);

  vorB->setChecked(_showElements[0]);
  vordmeB->setChecked(_showElements[1]);
  vortacB->setChecked(_showElements[2]);
  ndbB->setChecked(_showElements[3]);
  compB->setChecked(_showElements[4]);
  intairB->setChecked(_showElements[5]);
  airB->setChecked(_showElements[6]);
  milAirB->setChecked(_showElements[7]);
  civmilAirB->setChecked(_showElements[8]);
  airfieldB->setChecked(_showElements[9]);
  closedAirB->setChecked(_showElements[10]);
  civHeliB->setChecked(_showElements[11]);
  milHeliB->setChecked(_showElements[12]);
  ambHeliB->setChecked(_showElements[13]);
  ulfB->setChecked(_showElements[14]);
  hangB->setChecked(_showElements[15]);
  parachuteB->setChecked(_showElements[16]);
  ballonB->setChecked(_showElements[17]);
}

MapConfig::MapConfig(QWidget* parent)
  : KWizard(parent, 0, true)
{
  this->setCaption(i18n("Configure Map"));
  this->setEnableArrowButtons(true);
  this->setCancelButton();
  this->setHelpButton();
  this->setOkButton();
  this->setDefaultButton();
  this->setEnablePopupMenu(false);

  /////////////////////////////////////////////////////////////////////
  //
  // title page:
  KWizardPage* page0 = new KWizardPage();
  page0->title = i18n("Welcome to the map-configure-wizard");
  page0->w = new QWidget();
  page0->w->setMinimumWidth(550);
  page0->w->setMinimumHeight(300);

  QPixmap pm;
  pm.load(KApplication::kde_datadir() + "/kflog/pics/digit_title.xpm");

  QLabel* page0Icon = new QLabel(page0->w);
  page0Icon->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  page0Icon->setBackgroundPixmap(pm);

  QString text = i18n("Here You can configure, which map-elements should be\n"
                  "displayed in the map.\n\n"
                  "First You have to define the scale-range for the map\n"
                  "and five different borders. These borders are used to\n"
                  "reduce the map-elements displayed.\n\n"
                  "After that, You can determine, until which border each\n"
                  "map-element should be displayed. The border \"0\" means\n"
                  "that this element should not be displayed at all;\n"
                  "a \"6\" says, that this element will allways be displayed.");

  QLabel* page0Text = new QLabel(text, page0->w);
  page0Text->setMinimumHeight(page0Text->sizeHint().height());

  QGridLayout* page0Layout = new QGridLayout(page0->w,4,3,10,1);
  page0Layout->addMultiCellWidget(page0Icon,0,3,0,0);
  page0Layout->addWidget(page0Text,2,2);

  page0Layout->addColSpacing(0,150);
  page0Layout->setColStretch(0,0);
  page0Layout->addColSpacing(1,10);
  page0Layout->setColStretch(1,0);
  page0Layout->setColStretch(2,1);
  page0Layout->addRowSpacing(0,15);
  page0Layout->setRowStretch(1,1);
  page0Layout->addRowSpacing(3,15);

  /////////////////////////////////////////////////////////////////////
  //
  // first page:
  KWizardPage* page1 = new KWizardPage();
  page1->title = i18n("Scale-range and Borders");
  page1->w = new QWidget();
  page1->w->setMinimumWidth(550);
  page1->w->setMinimumHeight(350);

  QLabel* lLimitL = new QLabel(i18n("lower scale-limit"), page1->w);
  lLimit = new QSlider(10,100,1,0,QSlider::Horizontal, page1->w);
  lLimit->setMinimumHeight(lLimit->sizeHint().height() + 5);
  lLimit->setMaximumHeight(lLimit->sizeHint().height() + 15);
  lLimitN = new QLCDNumber(5,page1->w);

  QLabel* uLimitL = new QLabel(i18n("upper scale-limit"), page1->w);
  uLimit = new QSlider(10,120,1,0,
        QSlider::Horizontal, page1->w);
  uLimit->setMinimumHeight(uLimit->sizeHint().height() + 5);
  uLimit->setMaximumHeight(uLimit->sizeHint().height() + 15);
  uLimitN = new QLCDNumber(5,page1->w);

  KSeparator* sep = new KSeparator(page1->w);

  QLabel* comment = new QLabel(i18n("Height of one map-pixel, given in meters"),
        page1->w);
  comment->setAlignment(AlignVCenter|AlignRight);
  comment->setMinimumHeight(comment->sizeHint().height());

  QLabel* scale1L = new QLabel(i18n("border 1"), page1->w);
  scale1 = new QSlider(2,110,1,0,
        QSlider::Horizontal, page1->w);
  scale1->setMinimumHeight(scale1->sizeHint().height() + 5);
  scale1->setMaximumHeight(scale1->sizeHint().height() + 15);
  scale1N = new QLCDNumber(5,page1->w);

  QLabel* scale2L = new QLabel(i18n("border 2"), page1->w);
  scale2 = new QSlider(2,110,1,0,
        QSlider::Horizontal, page1->w);
  scale2->setMinimumHeight(scale2->sizeHint().height() + 5);
  scale2->setMaximumHeight(scale2->sizeHint().height() + 15);
  scale2N = new QLCDNumber(5,page1->w);

  QLabel* scale3L = new QLabel(i18n("border 3"), page1->w);
  scale3 = new QSlider(2,110,1,0,
        QSlider::Horizontal, page1->w);
  scale3->setMinimumHeight(scale3->sizeHint().height() + 5);
  scale3->setMaximumHeight(scale3->sizeHint().height() + 15);
  scale3N = new QLCDNumber(5,page1->w);

  QLabel* scale4L = new QLabel(i18n("border 4"), page1->w);
  scale4 = new QSlider(2,110,1,0,
        QSlider::Horizontal, page1->w);
  scale4->setMinimumHeight(scale4->sizeHint().height() + 5);
  scale4->setMaximumHeight(scale4->sizeHint().height() + 15);
  scale4N = new QLCDNumber(5,page1->w);

  QLabel* scale5L = new QLabel(i18n("border 5"), page1->w);
  scale5 = new QSlider(2,110,1,0,
        QSlider::Horizontal, page1->w);
  scale5->setMinimumHeight(scale5->sizeHint().height() + 5);
  scale5->setMaximumHeight(scale5->sizeHint().height() + 15);
  scale5N = new QLCDNumber(5,page1->w);

  QGridLayout* page1Layout = new QGridLayout(page1->w, 18, 5, 10, 1);
  page1Layout->addMultiCellWidget(comment,0,0,2,4);
  page1Layout->addWidget(lLimitL,2,0);
  page1Layout->addWidget(lLimit,2,2);
  page1Layout->addWidget(lLimitN,2,4);
  page1Layout->addWidget(uLimitL,4,0);
  page1Layout->addWidget(uLimit,4,2);
  page1Layout->addWidget(uLimitN,4,4);
  page1Layout->addMultiCellWidget(sep,6,6,0,4);
  page1Layout->addWidget(scale1L,8,0);
  page1Layout->addWidget(scale1,8,2);
  page1Layout->addWidget(scale1N,8,4);
  page1Layout->addWidget(scale2L,10,0);
  page1Layout->addWidget(scale2,10,2);
  page1Layout->addWidget(scale2N,10,4);
  page1Layout->addWidget(scale3L,12,0);
  page1Layout->addWidget(scale3,12,2);
  page1Layout->addWidget(scale3N,12,4);
  page1Layout->addWidget(scale4L,14,0);
  page1Layout->addWidget(scale4,14,2);
  page1Layout->addWidget(scale4N,14,4);
  page1Layout->addWidget(scale5L,16,0);
  page1Layout->addWidget(scale5,16,2);
  page1Layout->addWidget(scale5N,16,4);

  page1Layout->addColSpacing(0,lLimitL->sizeHint().width() + 15);
  page1Layout->setColStretch(0,0);
  page1Layout->setColStretch(2,3);
  page1Layout->addColSpacing(3,15);
  page1Layout->addColSpacing(4,lLimitN->sizeHint().width() + 15);

  connect(lLimit, SIGNAL(valueChanged(int)), lLimitN, SLOT(display(int)));
  connect(uLimit, SIGNAL(valueChanged(int)), SLOT(slotSetuLimit(int)));
  connect(scale1, SIGNAL(valueChanged(int)), SLOT(slotSetBorder1(int)));
  connect(scale2, SIGNAL(valueChanged(int)), SLOT(slotSetBorder2(int)));
  connect(scale3, SIGNAL(valueChanged(int)), SLOT(slotSetBorder3(int)));
  connect(scale4, SIGNAL(valueChanged(int)), SLOT(slotSetBorder4(int)));
  connect(scale5, SIGNAL(valueChanged(int)), SLOT(slotSetBorder5(int)));

  /////////////////////////////////////////////////////////////////////
  //
  // second page:
  KWizardPage* page2 = new KWizardPage();
  page2->title = i18n("Map-Elements");
  page2->w = new QWidget();
  page2->w->setMinimumWidth(550);
  page2->w->setMinimumHeight(350);

  QGridLayout* page2Layout = new QGridLayout(page2->w,22,11,10,1);

  QGroupBox* hydroGroup = new QGroupBox(i18n("Hydrography:"), page2->w);

  QLabel* coastL = new QLabel(i18n("Shore line"), page2->w);
  coastL->setMinimumHeight(coastL->sizeHint().height());
  coastB = new QSpinBox(0,6,1, page2->w);
  coastB->setMaximumHeight(coastB->sizeHint().height() + 5);
  QLabel* lakeBigL = new QLabel(i18n("Lake/River (big)"), page2->w);
  lakeBigL->setMinimumHeight(lakeBigL->sizeHint().height());
  lakeBigB = new QSpinBox(0,6,1, page2->w);
  lakeBigB->setMaximumHeight(lakeBigB->sizeHint().height() + 5);
  QLabel* lakeMidL = new QLabel(i18n("Lake/River (medium)"), page2->w);
  lakeMidL->setMinimumHeight(lakeMidL->sizeHint().height());
  lakeMidB = new QSpinBox(0,6,1, page2->w);
  lakeMidB->setMaximumHeight(lakeMidB->sizeHint().height() + 5);
  QLabel* lakeSmallL = new QLabel(i18n("Lake/River (small)"), page2->w);
  lakeSmallL->setMinimumHeight(lakeSmallL->sizeHint().height());
  lakeSmallB = new QSpinBox(0,6,1, page2->w);
  lakeSmallB->setMaximumHeight(lakeSmallB->sizeHint().height() + 5);
  QLabel* damL = new QLabel((QString)i18n("Dam") + ", " + i18n("Lock") + ", ...",
    page2->w);
  damL->setMinimumHeight(damL->sizeHint().height());
  damB = new QSpinBox(0,6,1, page2->w);
  damB->setMaximumHeight(damB->sizeHint().height() + 5);

  QGroupBox* topoGroup = new QGroupBox(i18n("Topography:"), page2->w);

  QLabel* obstacleL = new QLabel(i18n("Obstacle"), page2->w);
  obstacleL->setMinimumHeight(obstacleL->sizeHint().height());
  obstacleB = new QSpinBox(0,6,1, page2->w);
  obstacleB->setMaximumHeight(obstacleB->sizeHint().height() + 5);
  QLabel* spotL = new QLabel(i18n("Spot"), page2->w);
  spotL->setMinimumHeight(spotL->sizeHint().height());
  spotB = new QSpinBox(0,6,1, page2->w);
  spotB->setMaximumHeight(spotB->sizeHint().height() + 5);
  QLabel* passL = new QLabel(i18n("Pass"), page2->w);
  passL->setMinimumHeight(passL->sizeHint().height());
  passB = new QSpinBox(0,6,1, page2->w);
  passB->setMaximumHeight(passB->sizeHint().height() + 5);
  QLabel* iceL = new QLabel(i18n("Glacier"), page2->w);
  iceL->setMinimumHeight(iceL->sizeHint().height());
  iceB = new QSpinBox(0,6,1, page2->w);
  iceB->setMaximumHeight(iceB->sizeHint().height() + 5);

  QGroupBox* railGroup = new QGroupBox(i18n("Railway:"), page2->w);

  QLabel* railL = new QLabel(i18n("Railway"), page2->w);
  railL->setMinimumHeight(railL->sizeHint().height());
  railB = new QSpinBox(0,6,1, page2->w);
  railB->setMaximumHeight(railB->sizeHint().height() + 5);
  QLabel* aerialL = new QLabel(i18n("Aerial Railway"), page2->w);
  aerialL->setMinimumHeight(aerialL->sizeHint().height());
  aerialB = new QSpinBox(0,6,1, page2->w);
  aerialB->setMaximumHeight(aerialB->sizeHint().height() + 5);
  QLabel* stationL = new QLabel(i18n("Stations, ..."), page2->w);
  stationL->setMinimumHeight(stationL->sizeHint().height());
  stationB = new QSpinBox(0,6,1, page2->w);
  stationB->setMaximumHeight(stationB->sizeHint().height() + 5);

  QGroupBox* roadGroup = new QGroupBox(i18n("Roads:"), page2->w);
  QLabel* highL = new QLabel(i18n("Highway"), page2->w);
  highL->setMinimumHeight(highL->sizeHint().height());
  highB = new QSpinBox(0,6,1, page2->w);
  highB->setMaximumHeight(highB->sizeHint().height() + 5);
  QLabel* highEL = new QLabel(i18n("Highwayentry"), page2->w);
  highEL->setMinimumHeight(highEL->sizeHint().height());
  highEB = new QSpinBox(0,6,1, page2->w);
  highEB->setMaximumHeight(highEB->sizeHint().height() + 5);
  QLabel* roadMidL = new QLabel(i18n("Roads (medium)"), page2->w);
  roadMidL->setMinimumHeight(roadMidL->sizeHint().height());
  roadMidB = new QSpinBox(0,6,1, page2->w);
  roadMidB->setMaximumHeight(roadMidB->sizeHint().height() + 5);
  QLabel* roadSmallL = new QLabel(i18n("Roads (small)"), page2->w);
  roadSmallL->setMinimumHeight(roadSmallL->sizeHint().height());
  roadSmallB = new QSpinBox(0,6,1, page2->w);
  roadSmallB->setMaximumHeight(roadSmallB->sizeHint().height() + 5);

  page2Layout->addMultiCellWidget(hydroGroup,0,12,0,4);
  page2Layout->addWidget(coastL,2,1);
  page2Layout->addWidget(coastB,2,3);
  page2Layout->addWidget(lakeBigL,4,1);
  page2Layout->addWidget(lakeBigB,4,3);
  page2Layout->addWidget(lakeMidL,6,1);
  page2Layout->addWidget(lakeMidB,6,3);
  page2Layout->addWidget(lakeSmallL,8,1);
  page2Layout->addWidget(lakeSmallB,8,3);
  page2Layout->addMultiCellWidget(damL,10,11,1,1);
  page2Layout->addMultiCellWidget(damB,10,11,3,3);

  page2Layout->addMultiCellWidget(topoGroup,0,9,6,10);
  page2Layout->addWidget(obstacleL,2,7);
  page2Layout->addWidget(obstacleB,2,9);
  page2Layout->addWidget(spotL,4,7);
  page2Layout->addWidget(spotB,4,9);
  page2Layout->addWidget(passL,6,7);
  page2Layout->addWidget(passB,6,9);
  page2Layout->addWidget(iceL,8,7);
  page2Layout->addWidget(iceB,8,9);

  page2Layout->addMultiCellWidget(railGroup,14,21,0,4);
  page2Layout->addWidget(railL,16,1);
  page2Layout->addWidget(railB,16,3);
  page2Layout->addWidget(aerialL,18,1);
  page2Layout->addWidget(aerialB,18,3);
  page2Layout->addWidget(stationL,20,1);
  page2Layout->addWidget(stationB,20,3);

  page2Layout->addMultiCellWidget(roadGroup,11,21,6,10);
  page2Layout->addMultiCellWidget(highL,13,14,7,7);
  page2Layout->addMultiCellWidget(highB,13,14,9,9);
  page2Layout->addWidget(highEL,16,7);
  page2Layout->addWidget(highEB,16,9);
  page2Layout->addWidget(roadMidL,18,7);
  page2Layout->addWidget(roadMidB,18,9);
  page2Layout->addWidget(roadSmallL,20,7);
  page2Layout->addWidget(roadSmallB,20,9);

  page2Layout->setColStretch(0,0);
  page2Layout->addColSpacing(0,10);
  page2Layout->setColStretch(1,3);
  page2Layout->setColStretch(2,0);
  page2Layout->addColSpacing(2,10);
  page2Layout->setColStretch(3,1);
  page2Layout->setColStretch(4,0);
  page2Layout->addColSpacing(4,10);
  page2Layout->setColStretch(5,0);
  page2Layout->addColSpacing(5,10);
  page2Layout->setColStretch(6,0);
  page2Layout->addColSpacing(6,10);
  page2Layout->setColStretch(7,3);
  page2Layout->setColStretch(8,0);
  page2Layout->addColSpacing(8,10);
  page2Layout->setColStretch(9,1);
  page2Layout->setColStretch(10,0);
  page2Layout->addColSpacing(10,10);

  page2Layout->addRowSpacing(0,15);
  page2Layout->setRowStretch(0,0);
  page2Layout->addRowSpacing(1,5);
  page2Layout->setRowStretch(1,0);
  page2Layout->setRowStretch(2,1);
  page2Layout->addRowSpacing(3,5);
  page2Layout->setRowStretch(3,0);
  page2Layout->setRowStretch(4,1);
  page2Layout->addRowSpacing(5,5);
  page2Layout->setRowStretch(5,0);
  page2Layout->setRowStretch(6,1);
  page2Layout->addRowSpacing(7,5);
  page2Layout->setRowStretch(7,0);
  page2Layout->setRowStretch(8,1);
  page2Layout->addRowSpacing(9,5);
  page2Layout->setRowStretch(9,0);
  page2Layout->setRowStretch(10,1);
  page2Layout->addRowSpacing(11,10);
  page2Layout->setRowStretch(11,1);
  page2Layout->setRowStretch(12,1);
  page2Layout->setRowStretch(13,1);
  page2Layout->addRowSpacing(14,10);
  page2Layout->setRowStretch(14,1);
  page2Layout->addRowSpacing(15,5);
  page2Layout->setRowStretch(16,1);
  page2Layout->addRowSpacing(17,5);
  page2Layout->setRowStretch(17,0);
  page2Layout->setRowStretch(18,1);
  page2Layout->addRowSpacing(19,5);
  page2Layout->setRowStretch(19,0);
  page2Layout->setRowStretch(20,1);
  page2Layout->addRowSpacing(21,5);
  page2Layout->setRowStretch(21,0);

  /////////////////////////////////////////////////////////////////////
  //
  // third page:
  KWizardPage* page3 = new KWizardPage();
  page3->title = i18n("Map-Elements");
  page3->w = new QWidget();
  page3->w->setMinimumWidth(550);
  page3->w->setMinimumHeight(350);

  QGridLayout* page3Layout = new QGridLayout(page3->w,22,11,10,1);

  QLabel* landL = new QLabel(i18n("Landmarks"), page3->w);
  landL->setMinimumHeight(landL->sizeHint().height());
  landB = new QSpinBox(0,6,1, page3->w);
  landB->setMaximumHeight(landB->sizeHint().height() + 5);
  QLabel* airspaceL = new QLabel(i18n("Airspace Structure"), page3->w);
  airspaceB = new QSpinBox(0,6,1, page3->w);

  QGroupBox* cityGroup = new QGroupBox(i18n("Build-up areas:"), page3->w);
  QLabel* hugeCityL = new QLabel(i18n("City (> 500.000)"), page3->w);
  hugeCityL->setMinimumHeight(hugeCityL->sizeHint().height());
  hugeCityB = new QSpinBox(0,6,1, page3->w);
  hugeCityB->setMaximumHeight(hugeCityB->sizeHint().height() + 5);

  QLabel* bigCityL = new QLabel(i18n("City (100.000 - 500.000)"), page3->w);
  bigCityL->setMinimumHeight(bigCityL->sizeHint().height());
  bigCityB = new QSpinBox(0,6,1, page3->w);
  bigCityB->setMaximumHeight(bigCityB->sizeHint().height() + 5);
  QLabel* midCityL = new QLabel(i18n("City (50.000 - 100.000)"), page3->w);
  midCityL->setMinimumHeight(midCityL->sizeHint().height());
  midCityB = new QSpinBox(0,6,1, page3->w);
  midCityB->setMaximumHeight(midCityB->sizeHint().height() + 5);
  QLabel* smallCityL = new QLabel(i18n("City (20.000 - 50.000)"), page3->w);
  smallCityL->setMinimumHeight(smallCityL->sizeHint().height());
  smallCityB = new QSpinBox(0,6,1, page3->w);
  smallCityB->setMaximumHeight(smallCityB->sizeHint().height() + 5);
  QLabel* villageL = new QLabel(i18n("Village"), page3->w);
  villageL->setMinimumHeight(villageL->sizeHint().height());
  villageB = new QSpinBox(0,6,1, page3->w);
  villageB->setMaximumHeight(villageB->sizeHint().height() + 5);

  QGroupBox* radioGroup = new QGroupBox(i18n("Radio Navigation Facilities:"),
            page3->w);
  QLabel* showRadioL = new QLabel(i18n("show Facilities"), page3->w);
  radioB = new QSpinBox(0,6,1, page3->w);

  vorB = new QCheckBox(i18n("VOR"), page3->w);
  vordmeB = new QCheckBox(i18n("VOR / DME"), page3->w);
  vortacB = new QCheckBox(i18n("VOR-TACAN"), page3->w);
  ndbB = new QCheckBox(i18n("NDB"), page3->w);
  compB = new QCheckBox(i18n("Compulsory reporting point"), page3->w);

  page3Layout->addWidget(landL,0,1);
  page3Layout->addWidget(landB,0,3);
  page3Layout->addWidget(airspaceL,0,7);
  page3Layout->addWidget(airspaceB,0,9);

  page3Layout->addMultiCellWidget(cityGroup,2,13,0,4);
  page3Layout->addWidget(hugeCityL,4,1);
  page3Layout->addWidget(hugeCityB,4,3);
  page3Layout->addWidget(bigCityL,6,1);
  page3Layout->addWidget(bigCityB,6,3);
  page3Layout->addWidget(midCityL,8,1);
  page3Layout->addWidget(midCityB,8,3);
  page3Layout->addWidget(smallCityL,10,1);
  page3Layout->addWidget(smallCityB,10,3);
  page3Layout->addWidget(villageL,12,1);
  page3Layout->addWidget(villageB,12,3);

  page3Layout->addMultiCellWidget(radioGroup,2,15,6,10);
  page3Layout->addWidget(showRadioL,4,7);
  page3Layout->addWidget(radioB,4,9);
  page3Layout->addMultiCellWidget(vorB,6,6,7,9);
  page3Layout->addMultiCellWidget(vordmeB,8,8,7,9);
  page3Layout->addMultiCellWidget(vortacB,10,10,7,9);
  page3Layout->addMultiCellWidget(ndbB,12,12,7,9);
  page3Layout->addMultiCellWidget(compB,14,14,7,9);

  page3Layout->setColStretch(0,0);
  page3Layout->addColSpacing(0,10);
  page3Layout->setColStretch(1,3);
  page3Layout->setColStretch(2,0);
  page3Layout->addColSpacing(2,10);
  page3Layout->setColStretch(3,1);
  page3Layout->setColStretch(4,0);
  page3Layout->addColSpacing(4,10);
  page3Layout->setColStretch(5,0);
  page3Layout->addColSpacing(5,10);
  page3Layout->setColStretch(6,0);
  page3Layout->addColSpacing(6,10);
  page3Layout->setColStretch(7,3);
  page3Layout->setColStretch(8,0);
  page3Layout->addColSpacing(8,10);
  page3Layout->setColStretch(9,1);
  page3Layout->setColStretch(10,0);
  page3Layout->addColSpacing(10,10);

  page3Layout->setRowStretch(0,1);
  page3Layout->addRowSpacing(1,5);
  page3Layout->setRowStretch(1,0);
  page3Layout->addRowSpacing(2,15);
  page3Layout->setRowStretch(2,1);
  page3Layout->addRowSpacing(3,5);
  page3Layout->setRowStretch(3,0);
  page3Layout->setRowStretch(4,1);
  page3Layout->addRowSpacing(5,5);
  page3Layout->setRowStretch(5,0);
  page3Layout->setRowStretch(6,1);
  page3Layout->addRowSpacing(7,5);
  page3Layout->setRowStretch(7,0);
  page3Layout->setRowStretch(8,1);
  page3Layout->addRowSpacing(9,5);
  page3Layout->setRowStretch(9,0);
  page3Layout->setRowStretch(10,1);
  page3Layout->addRowSpacing(11,5);
  page3Layout->setRowStretch(11,0);
  page3Layout->setRowStretch(12,1);
  page3Layout->setRowStretch(13,1);
  page3Layout->addRowSpacing(14,10);
  page3Layout->setRowStretch(14,1);
  page3Layout->addRowSpacing(15,5);
  page3Layout->setRowStretch(16,1);
  page3Layout->addRowSpacing(17,5);
  page3Layout->setRowStretch(17,0);
  page3Layout->setRowStretch(18,1);
  page3Layout->addRowSpacing(19,5);
  page3Layout->setRowStretch(19,0);
  page3Layout->setRowStretch(20,1);
  page3Layout->addRowSpacing(21,5);
  page3Layout->setRowStretch(21,0);

  /////////////////////////////////////////////////////////////////////
  //
  // fourth page:
  KWizardPage* page4 = new KWizardPage();
  page4->title = i18n("Airports and Sites");
  page4->w = new QWidget();
  page4->w->setMinimumWidth(550);
  page4->w->setMinimumHeight(350);

  QGridLayout* page4Layout = new QGridLayout(page4->w,23,11,10,1);

  QGroupBox* airportGroup = new QGroupBox(i18n("Airports"), page4->w);
  QLabel* showAirL = new QLabel(i18n("show Airports"), page4->w);
  showAirL->setMinimumHeight(showAirL->sizeHint().height());
  airportB = new QSpinBox(0,6,1, page4->w);
  airportB->setMaximumHeight(airportB->sizeHint().height() + 5);

  intairB = new QCheckBox(i18n("International Airport"), page4->w);
  intairB->setMinimumHeight(intairB->sizeHint().height());
  intairB->setMaximumHeight(intairB->sizeHint().height() + 5);
  airB = new QCheckBox(i18n("Airport, Airfield IFR"), page4->w);
  airB->setMinimumHeight(airB->sizeHint().height());
  airB->setMaximumHeight(airB->sizeHint().height() + 5);
  milAirB = new QCheckBox(i18n("Military Aerodrome"), page4->w);
  milAirB->setMinimumHeight(milAirB->sizeHint().height());
  milAirB->setMaximumHeight(milAirB->sizeHint().height() + 5);
  civmilAirB = new QCheckBox(i18n("Airfield (civil/military)"), page4->w);
  airfieldB = new QCheckBox(i18n("Airfield"), page4->w);
  closedAirB = new QCheckBox(i18n("Airfield closed"), page4->w);
  civHeliB = new QCheckBox(i18n("civ. Heliport"), page4->w);
  milHeliB = new QCheckBox(i18n("mil. Heliport"), page4->w);
  ambHeliB = new QCheckBox(i18n("amb. Heliport"), page4->w);

  QLabel* gliderL = new QLabel(i18n("Glider site"), page4->w);
  gliderL->setMinimumHeight(gliderL->sizeHint().height());
  gliderB = new QSpinBox(0,6,1, page4->w);
  gliderB->setMaximumHeight(gliderB->sizeHint().height() + 5);
  QLabel* outL = new QLabel(i18n("Outlandingfields"), page4->w);
  outL->setMinimumHeight(outL->sizeHint().height());
  outB = new QSpinBox(0,6,1, page4->w);
  outB->setMaximumHeight(outB->sizeHint().height() + 5);
  QLabel* waypointL = new QLabel(i18n("Waypoints"), page4->w);
  waypointL->setMinimumHeight(waypointL->sizeHint().height());
  waypointB = new QSpinBox(0,6,1, page4->w);
  waypointB->setMaximumHeight(waypointB->sizeHint().height() + 5);

  QGroupBox* addSitesGroup = new QGroupBox(i18n("Additional Sites"),
      page4->w);
  QLabel* showAddSitesL = new QLabel(i18n("show additional sites"), page4->w);
  addAirportB = new QSpinBox(0,6,1, page4->w);
  ulfB = new QCheckBox(i18n("Ultra light flying"), page4->w);
  hangB = new QCheckBox(i18n("Hang glider site"), page4->w);
  parachuteB = new QCheckBox(i18n("Parachute jumping site"), page4->w);
  ballonB = new QCheckBox(i18n("Free ballon site"), page4->w);

  page4Layout->addMultiCellWidget(airportGroup,0,9,0,10);
  page4Layout->addWidget(showAirL,2,1);
  page4Layout->addWidget(airportB,2,3);
  page4Layout->addMultiCellWidget(intairB,4,4,1,4);
  page4Layout->addMultiCellWidget(airB,6,6,1,4);
  page4Layout->addMultiCellWidget(milAirB,8,8,1,4);
  page4Layout->addMultiCellWidget(civmilAirB,4,4,6,6);
  page4Layout->addMultiCellWidget(airfieldB,6,6,6,6);
  page4Layout->addMultiCellWidget(closedAirB,8,8,6,6);
  page4Layout->addMultiCellWidget(civHeliB,4,4,8,9);
  page4Layout->addMultiCellWidget(milHeliB,6,6,8,9);
  page4Layout->addMultiCellWidget(ambHeliB,8,8,8,9);

  page4Layout->addWidget(gliderL,13,1);
  page4Layout->addWidget(gliderB,13,3);
  page4Layout->addWidget(outL,15,1);
  page4Layout->addWidget(outB,15,3);
  page4Layout->addWidget(waypointL,17,1);
  page4Layout->addWidget(waypointB,17,3);

  page4Layout->addMultiCellWidget(addSitesGroup,11,18,5,10);
  page4Layout->addWidget(showAddSitesL,13,6);
  page4Layout->addWidget(addAirportB,13,8);
  page4Layout->addMultiCellWidget(ulfB,15,15,6,6);
  page4Layout->addMultiCellWidget(hangB,17,17,6,6);
  page4Layout->addMultiCellWidget(parachuteB,15,15,8,9);
  page4Layout->addMultiCellWidget(ballonB,17,17,8,9);

  page4Layout->setColStretch(0,0);
  page4Layout->addColSpacing(0,10);
  page4Layout->setColStretch(1,2);
  page4Layout->setColStretch(2,0);
  page4Layout->addColSpacing(2,10);
  page4Layout->setColStretch(3,1);
  page4Layout->setColStretch(4,0);
  page4Layout->addColSpacing(4,15);
  page4Layout->setColStretch(5,0);
  page4Layout->addColSpacing(5,10);
  page4Layout->setColStretch(6,3);
  page4Layout->addColSpacing(7,10);
  page4Layout->setColStretch(7,0);
  page4Layout->setColStretch(8,1);
  page4Layout->setColStretch(9,2);
  page4Layout->addColSpacing(10,10);

  page4Layout->setRowStretch(0,1);
  page4Layout->addRowSpacing(1,5);
  page4Layout->setRowStretch(1,0);
  page4Layout->setRowStretch(2,1);
  page4Layout->addRowSpacing(3,5);
  page4Layout->setRowStretch(3,0);
  page4Layout->setRowStretch(4,1);
  page4Layout->addRowSpacing(5,5);
  page4Layout->setRowStretch(5,0);
  page4Layout->setRowStretch(6,1);
  page4Layout->addRowSpacing(7,5);
  page4Layout->setRowStretch(7,0);
  page4Layout->setRowStretch(8,1);
  page4Layout->addRowSpacing(9,5);
  page4Layout->setRowStretch(9,0);
  page4Layout->setRowStretch(10,1);
  page4Layout->addRowSpacing(11,5);
  page4Layout->setRowStretch(11,0);
  page4Layout->setRowStretch(12,1);
  page4Layout->setRowStretch(13,1);
  page4Layout->addRowSpacing(14,5);
  page4Layout->setRowStretch(14,0);
  page4Layout->setRowStretch(15,1);
  page4Layout->addRowSpacing(16,5);
  page4Layout->setRowStretch(16,0);
  page4Layout->setRowStretch(17,1);
  page4Layout->addRowSpacing(18,5);
  page4Layout->setRowStretch(18,0);

  // adding the pages
  this->addPage(page0);
  this->addPage(page1);
  this->addPage(page2);
  this->addPage(page3);
  this->addPage(page4);

  this->setPageEnabled(0, true);
  this->setPageEnabled(1, true);
  this->setPageEnabled(2, true);
  this->setPageEnabled(3, true);
  this->setPageEnabled(4, true);

  connect(getOkButton(), SIGNAL(clicked()), SLOT(slotSetValues()));
  connect(getDefaultButton(), SIGNAL(clicked()),
        SLOT(slotSetDefaultValues()));
  connect(getCancelButton(), SIGNAL(clicked()), SLOT(reject()));

  loadValues();

  this->exec();
}

MapConfig::~MapConfig()
{

}
