/***********************************************************************
**
**   mapcontents.h
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#ifndef MAPCONTENTS_H
#define MAPCONTENTS_H

// include qt-header
#include <qlist.h>

#include <areaelement.h>
#include <airport.h>
#include <airspace.h>
#include <basemapelement.h>
#include <elevpoint.h>
#include <glidersite.h>
#include <intairport.h>
#include <lineelement.h>
#include <radiopoint.h>
#include <singlepoint.h>

/**
  * This class provides functions for accessing the contents of the map.
  *
  * @author Heiner Lamprecht, Florian Ehinger
  * @version 0.1
  */
class MapContents
{
 public:
  MapContents();
  ~MapContents();
  /**
    * Adds a new area-element to the given list.
    */
  void addElement(unsigned int listIndex, AreaElement* newElement);
  /**
    * Adds a new airport to the airportlist.
    */
  void addElement(Airport* newElement);
  /**
    * Adds a new airspace to the airspacelist.
    */
  void addElement(Airspace* newElement);
  /**
    * Adds a new glider to the gliderlist.
    */
  void addElement(GliderSite* newElement);
  /**
    * Adds a new international airport to the list.
    */
  void addElement(IntAirport* newElement);
  /**
    * Adds a new line-element to the given list.
    */
  void addElement(unsigned int listIndex, LineElement* newElement);
  /**
    * Adds a new radio-navigation-facility to the radiolist.
    */
  void addElement(RadioPoint* newElement);
  /**
    * Adds a new singlepoint-element to the given list.
    */
  void addElement(unsigned int listIndex, SinglePoint* newElement);
  /**
    * Adds a new elevation-element to the given list.
    */
  void addElement(unsigned int listIndex, ElevPoint* newElement);
  /**
    * Returns the current length of the given list.
    */
  unsigned int getListLength(int listIndex);
  /**
    * Returns a BaseMapElement-pointer to the given mapelement in the list.
    */
  BaseMapElement* getElement(int listIndex, unsigned int index);
  /**
    * Returns a pointer to the given waypoint.
    */
  SinglePoint* getWayPoint(unsigned int index);
  /**
    * Returns the number of waypoints.
    */
  unsigned int getWayPointNumber();
  /**
    * The listid.
    */
  enum listID {NotSet, IntAirportList, AirportList, GliderList, AddSitesList,
       OutList, NavList, AirspaceList, ObstacleList, ReportList, CityList,
       VillageList, LandmarkList, HighwayList, HighwayEntryList, RoadList,
       RailList, StationList, HydroList, TopoList, WaypointList, DigitList};

 private:
  /** */
  QList<IntAirport> intairportList;
  /**
    * airportList contains all international airports.
    */
  QList<Airport> airportList;
  /**
    * gliderList contains all glider-sites.
    */
  QList<GliderSite> gliderList;
  /**
    * addSitesList contains all, ultra-light,
    * hang-glider-sites, free-ballon-sites, parachute-jumping-sites.
    */
  QList<SinglePoint> addSitesList;
  /**
    * outList contains all outlanding-fields.
    */
  QList<ElevPoint> outList;
  /**
    * navList contains all radio navigation facilities.
    */
  QList<RadioPoint> navList;
  /**
    * airspaceList contails all airspaces.
    */
  QList<Airspace> airspaceList;
  /**
    * obstacleList contains all obstacles and -groups, as well
    * as the spots and passes.
    */
  QList<ElevPoint> obstacleList;
  /**
    * reportList contains all reporting points.
    */
  QList<SinglePoint> reportList;
  /**
    * cityList contails all cities.
    */
  QList<AreaElement> cityList;
  /**
    * villageList contains all villages.
    */
  QList<SinglePoint> villageList;
  /**
    * landmarkList contains all landmarks.
    */
  QList<SinglePoint> landmarkList;
  /**
    * highwayList contains all highways.
    */
  QList<LineElement> highwayList;
  /**
    * highEntryList contains all highwayentries.
    */
  QList<SinglePoint> highEntryList;
  /**
    * roadList contails all roads.
    */
  QList<LineElement> roadList;
  /**
    * railList contains all railways and aerial railways.
    */
  QList<LineElement> railList;
  /**
    * stationList contains all stations.
    */
  QList<SinglePoint> stationList;
  /**
    * hydroList contains all shorelines, rivers, lakes, ...
    */
  QList<LineElement> hydroList;
  /**
    * topoList contains all topographical objects.
    */
  QList<AreaElement> topoList;
  /**
    * waypointList contains the listID of the waypoint.
    */
  unsigned int* waypointList;
  /**
    * waypointIndex contains the index of the waypoin.
    */
  unsigned int* waypointIndex;
  /**
    * Contains the number of waypoints.
    */
  unsigned int waypointNumber;
};

#endif
