////////////////////////////////////////////////////////////////////////
//  Originated by Logikos, Inc.  http://www.logikos.com
//
//  Class       : KGrepUI
//
//  File Name   : kgrepui.cc
//
//  Originated  : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract    : See notes in kgrepui.h .
//
//  Compiler    : egcs-1.1.2, qt 1.44, KDE 1.1.1
//
//  Revisions   :
//
////////////////////////////////////////////////////////////////////////

// Compiler includes

// Project includes
#include "kgrepui.moc"
#include "kgrepedit.h"
#include "kgrepoptiondlg.h"
#include "kgrepsearchdlg.h"

// Constants
//  Toolbar IDs
const int TOOLBAR_SEARCH = 1;
const int TOOLBAR_PREV_MATCH = 2;
const int TOOLBAR_NEXT_MATCH = 3;
const int TOOLBAR_EXT_EDITOR = 4;

// Globals

// Static attributes



////////////////////////////////////////////////////////////////////////
//
//  Function      : KGrepUI
//
//  Originated    : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : Constructor.  Creates menus, toolbar, etc...
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
KGrepUI::KGrepUI() : KTMainWindow()
{
    // Local variables
    QPixmap editPixmap;
    QPixmap exitPixmap;
    QPixmap nextPixmap;
    QString pixdir;
    QPixmap prevPixmap;
    QPixmap searchPixmap;


    // BEGIN
    pixdir = kapp->kde_toolbardir() + "/";
    editPixmap.load(pixdir + "pencil.xpm");
    exitPixmap.load(pixdir + "exit.xpm");
    searchPixmap.load(pixdir + "search.xpm");
    prevPixmap.load(pixdir + "prev.xpm");
    nextPixmap.load(pixdir + "next.xpm");

    m_fileMenu = new QPopupMenu();
    m_fileMenu->insertItem(i18n("&Options"), this, SLOT(slotOptions()));
    m_fileMenu->insertSeparator();
    m_menuItemExtEditorID = m_fileMenu->insertItem(editPixmap,             // pixmap
                                                   i18n("E&dit File"),     // text
                                                   this,                   // receiver
                                                   SLOT(slotExtEditor()),  // slot
                                                   CTRL+Key_E              // accelerator
                                                   );
    m_fileMenu->insertSeparator();
    m_fileMenu->insertItem(exitPixmap,                 // pixmap
                           i18n("E&xit"),              // text
                           this,                       // receiver
                           SLOT(slotExit()),           // slot
                           CTRL+Key_X                  // accelerator
                           );

    m_searchMenu = new QPopupMenu();
    m_menuItemSearchID = m_searchMenu->insertItem(searchPixmap,           // pixmap
                                                  i18n("New &Search"),    // text
                                                  this,                   // receiver
                                                  SLOT(slotNewSearch()),  // slot
                                                  CTRL+Key_S              // accelerator
                                                  );
    m_searchMenu->insertSeparator();
    m_menuItemPrevID = m_searchMenu->insertItem(prevPixmap,               // pixmap
                                                i18n("&Previous Match"),  // text
                                                this,                     // receiver
                                                SLOT(slotPrevMatch()),    // slot
                                                CTRL+Key_P                // accelerator
                                                );
    m_menuItemNextID = m_searchMenu->insertItem(nextPixmap,               // pixmap
                                                i18n("&Next Match"),      // text
                                                this,                     // receiver
                                                SLOT(slotNextMatch()),    // slot
                                                CTRL+Key_N                // accelerator
                                                );

    m_helpMenu = kapp->getHelpMenu(TRUE, "      KGrep (version 0.1.6)\n\n                 by\n\nKeith Brown (kbrown@logikos.com)\n                 and\nJeff Cody (jcody@logikos.com)");
  
    m_menubar = new KMenuBar(this);
    m_menubar->insertItem(i18n("&File"), m_fileMenu);
    m_menubar->insertItem(i18n("&Search"), m_searchMenu);
    m_menubar->insertItem(i18n("&Help"), m_helpMenu);

    m_toolbar = new KToolBar(this);
    m_toolbar->insertButton(searchPixmap,            // pixmap
                            TOOLBAR_SEARCH,          // ID
                            SIGNAL(clicked()),       // signal
                            this,                    // receiver
                            SLOT(slotNewSearch()),   // slot
                            true,                    // enabled?
                            i18n("New Search")       // tooltip
                            );
    m_toolbar->insertButton(prevPixmap,              // pixmap
                            TOOLBAR_PREV_MATCH,      // ID
                            SIGNAL(clicked()),       // signal
                            this,                    // receiver
                            SLOT(slotPrevMatch()),   // slot
                            true,                    // enabled?
                            i18n("Previous Match")   // tooltip
                            );
    m_toolbar->insertButton(nextPixmap,              // pixmap
                            TOOLBAR_NEXT_MATCH,      // ID
                            SIGNAL(clicked()),       // signal
                            this,                    // receiver
                            SLOT(slotNextMatch()),   // slot
                            true,                    // enabled?
                            i18n("Next Match")       // tooltip
                            );
    m_toolbar->insertButton(editPixmap,              // pixmap
                            TOOLBAR_EXT_EDITOR,      // ID
                            SIGNAL(clicked()),       // signal
                            this,                    // receiver
                            SLOT(slotExtEditor()),   // slot
                            true,                    // enabled?
                            i18n("Edit File")        // tooltip
                            );

    m_statusbar = new KStatusBar(this);
    m_statusbar->setInsertOrder(KStatusBar::RightToLeft);
    m_statusbar->insertItem(i18n("Total Matching Files:   0   "), 1);
    m_statusbar->insertItem(i18n("Total Matching Lines:   0   "), 2);
    m_statusbar->setAlignment(1, AlignRight);
    m_statusbar->setAlignment(2, AlignRight);

    m_panner  = new KNewPanner(this,                   // parent
                               "panner",               // internal name
                               KNewPanner::Horizontal, // orientation
                               KNewPanner::Percent,    // percent division
                               35                      // initial percent division
	                           );
    m_panner->resize(size());

    m_listbox = new KTabListBox(m_panner,   // parent
                                "listbox",  // internal name
                                3           // columns
                                );
    m_listbox->setColumn(0,
                         i18n(" File"),
                         100,
                         KTabListBox::TextColumn
                         );
    m_listbox->setColumn(1,
                         i18n(" Line #"),
                         50,
                         KTabListBox::TextColumn
                         );
    m_listbox->setColumn(2,
                         i18n(" Matched Output"),
                         445,
                         KTabListBox::TextColumn
                         );

    m_edit = new KGrepEdit(kapp, m_panner);

    // Set up the "bars"
    setMenu(m_menubar);
    setStatusBar(m_statusbar);
    addToolBar(m_toolbar);

    // Set up the main client widget
    setView(m_panner);
    m_panner->activate(m_listbox, m_edit);

    // Create the option dialog object
    m_optionDlg = new KGrepOptionDlg(this,          // parent
                                     "optiondlg"    // internal name
                                     );

    // Create the search dialog object
    m_searchDlg = new KGrepSearchDlg(0,             // no parent since it's modeless
                                     "searchdlg"    // internal name
                                     );
    m_searchDlg->hide();
    m_listbox->setSeparator(203);

    // We'll watch for an OK from the search dialog to fire off our grep process
    connect(m_searchDlg, SIGNAL(okSelected(QString)), this, SLOT(slotDoGrep(QString)));

    // Hook the listbox single and double click in...
    connect(m_listbox,                              // container of our grep results
            SIGNAL(highlighted(int,int)),           // emitted when user single-clicks a row
            this,                                   // KGrepUI Instance
            SLOT(slotGrepResultSelected(int,int))   // slot to open file up & goto line number
            );
    connect(m_listbox,                              // container of our grep results
            SIGNAL(selected(int,int)),              // emitted when user double-clicks a row
            this,                                   // KGrepUI Instance
            SLOT(slotGrepResultSelected(int,int))   // slot to open file up & goto line number
            );

    // Start out with Previous and Next Match disabled
    enableDependentBtns(false);

    return;

}   //  KGrepUI::KGrepUI


////////////////////////////////////////////////////////////////////////
//
//  Function      : ~KGrepUI
//
//  Originated    : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : Destructor.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
KGrepUI::~KGrepUI()
{
    // Free allocated memory
    // NOTE: (From the memory-leak hunting HOWTO) "... when dealing with QWidgets
    //       and derived Qt and KDE subclasses: The destructor of a parent
    //       object destroys all child objects. This saves you from writing your
    //       own clean-up code for widgets that have been constructed with a
    //       pointer to the parent."

    return;

}   // end KGrepUI::~KGrepUI


////////////////////////////////////////////////////////////////////////
//
//  Function      : closeEvent
//
//  Originated    : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : Override of KTMainWindow function.  This is automatically
//                  called when the main window widget is close.  Cleans
//                  up the widgets and everything nicely.
//
//  Parameters    : pointer to the close event
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::closeEvent(QCloseEvent *)
{
    // BEGIN
    // Clean up KDE widgets nicely
    deleteAll();

    // Exit app.  Qt takes over here...
    kapp->quit();

    return;

}   // end KGrepUI::closeEvent


////////////////////////////////////////////////////////////////////////
//
//  Function      : enableDependentBtns
//
//  Originated    : 9-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : Enable/disable previous match, next match, and edit
//                  file toolbar buttons and menu items.  All these
//                  widgets are dependent on matches existing.
//
//  Parameters    : enable - boolean indicating if the widgets are
//                           enabled or disabled
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::enableDependentBtns(bool enable)
{
    // BEGIN
    m_menubar->setItemEnabled(m_menuItemExtEditorID, enable);
    m_menubar->setItemEnabled(m_menuItemNextID, enable);
    m_menubar->setItemEnabled(m_menuItemPrevID, enable);
    m_toolbar->setItemEnabled(TOOLBAR_EXT_EDITOR, enable);
    m_toolbar->setItemEnabled(TOOLBAR_NEXT_MATCH, enable);
    m_toolbar->setItemEnabled(TOOLBAR_PREV_MATCH, enable);

    return;

}   // end KGrepUI::enableDependentBtns()


////////////////////////////////////////////////////////////////////////
//
//  Function      : processOutput
//
//  Originated    : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//                               Jeff Cody   (jcody@logikos.com)
//
//  Abstract      : Formats the results of the grep process for display
//                  in the listbox and displays it.  Updates status bar.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::processOutput()
{
    // Local variables
    char*   buf = new char[10];             // Temp buffer
    QString currFile;                       // Current file name for file counting
    int     index;                          // Index into string
    int     numFiles = 0;                   // Number of files matched
    int     numLines = 0;                   // Number of lines matched
    QString prevFile;                       // Previous file name for file counting
    QString rightString;                    // Scratchpad QString object
    QString text;                           // Temporary text string


    // BEGIN
    index = 1;

    // Make sure there is something to display
    if (m_resultLength > 0)
    {
        // Parse the output and display it in the listbox
        while (index > 0)
        {
            index = m_resultString.find('\n', 0, TRUE);       // We'll separate on newlines for now

            // Once we have a line, we save it off and insert it.
            if (index++ > 0)
            {
                // Grab the entire string after the first line
                m_resultLength -= index;
                rightString = m_resultString.right(m_resultLength);

                // Whack off the newline
                m_resultString.truncate(index);

                // Here we change the two :'s surrounding the line number to ASCII
                // char 203 for field separator use, and check the current file name
                // to count the total number of files matched.
                int pos = m_resultString.find(':');

                if (pos < 0)
                {
                    // Add two separator characters to the end for binary file matches
                    char buf[8];
                    sprintf(buf, "%c %c ", 203, 203);
                    m_resultString += buf;

                    numFiles++;
                }   // end if
                else
                {
                    currFile = m_resultString.left(pos);

                    if (prevFile != currFile)
                    {
                        numFiles++;
                        prevFile = currFile;
                    }   // end if

                    m_resultString.insert(pos, 203);
                    m_resultString.remove(pos+1, 1);    // remove the :

                    pos = m_resultString.find(':');     // find the : after the line #
                    m_resultString.insert(pos, 203);
                    m_resultString.remove(pos+1, 1);    // remove the :
                }   // end else

                // Insert it into the listbox
                m_listbox->insertItem(m_resultString);

                // Use the remaining string the next go-around
                m_resultString = rightString;
            }   // end if
        }   // end while

        // Update the status bar information.
        numLines = m_listbox->numRows();
        if (numLines < 10000)
        {
            sprintf(buf, "%d   ", numLines);
        }   // end if
        else
        {
            sprintf(buf, "lots");
        }   // end else

        if (numLines < 10)
        {
            text =  i18n("Total Matching Lines:   ");
            text += i18n(buf);
        }   // end if
        else if (numLines < 100)
        {
            text =  i18n("Total Matching Lines:  ");
            text += i18n(buf);
        }   // end else if
        else if (numLines < 1000)
        {
            text =  i18n("Total Matching Lines: ");
            text += i18n(buf);
        }   // end else if
        else
        {
            text =  i18n("Total Matching Lines:");
            text += i18n(buf);
        }   // end else

        m_statusbar->changeItem(text, 2);

        if (numFiles < 10000)
        {
            sprintf(buf, "%d   ", numFiles);
        }   // end if
        else
        {
            sprintf(buf, "lots");
        }   // end else

        if (numFiles < 10)
        {
            text =  i18n("Total Matching Files:   ");
            text += i18n(buf);
        }   // end if
        else if (numFiles < 100)
        {
            text =  i18n("Total Matching Files:  ");
            text += i18n(buf);
        }   // end else if
        else if (numFiles < 1000)
        {
            text =  i18n("Total Matching Files: ");
            text += i18n(buf);
        }   // end else if
        else
        {
            text =  i18n("Total Matching Files:");
            text += i18n(buf);
        }   // end else

        m_statusbar->changeItem(text, 1);

        // Enable the navigation buttons and menu items
        enableDependentBtns(true);

        // Select the first match
        m_listbox->setCurrentItem(0, 0);
        slotGrepResultSelected(0, 0);
    }   // end if
    else
    {
        // Nothing to display, disable next and prev buttons, update status bar
        enableDependentBtns(false);

        m_statusbar->changeItem(i18n("Total Matching Files:   0"), 1);
        m_statusbar->changeItem(i18n("Total Matching Lines:   0"), 2);
    }   // end else

    delete [] buf;

    return;

}   // end KGrepUI::processOutput()


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotDoGrep
//
//  Originated    : 7-Sept-99  - Jeff Cody (jcody@logikos.com)
//
//  Abstract      : This sets up the string for the grep process
//                  commandline, and then spawns the grep process.  The
//                  spawned process will emit SIGNALS for Stdout and Stderr
//                  output.  The spawned process is blocking because it is
//                  a royal pain in the butt to get non-blocking output/signals
//                  to behave in a manner that is sane.
//
//  Parameters    : grepOptions - QString containing the pattern, location(s)
//                                and other options to pass to grep.
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotDoGrep(QString grepOptions)
{
    // Empty and refill the listbox and editor with the new info.
    m_listbox->clear();
    m_prevFilename = "";
    m_edit->newFile();

    m_grepProcess = new KShellProcess();

    // set up signal and slot for stdout...
    connect(m_grepProcess,                                  //our grep process
            SIGNAL(receivedStdout(KProcess*, char*, int)),  //when grep outputs
            this,                                           //KGrepUI
            SLOT(slotGrepResults(KProcess*, char*, int))    //SLOT
            );

    // TODO - handle stderr and look for errors

    *m_grepProcess << "grep" << grepOptions;                // set up the executable

    // Clear out our output container
    m_resultString.truncate(0);
    m_resultLength = 0;

    // Let the grep'ing begin!
    m_grepProcess->start(KProcess::Block, KProcess::AllOutput);

    // Clean up pointer
    delete m_grepProcess;

    // Once the grep is done, process its output
    processOutput();

    return;

} // end KGrepUI::slotDoGrep()


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotExit
//
//  Originated    : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : Event-handler for application exit.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotExit()
{
    // BEGIN
    close();

    return;

}   // end KGrepUI::slotExit


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotExtEditor
//
//  Originated    : 23-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : Opens the selected file in the user specified
//                  external editor.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotExtEditor()
{
    // Local variables
    int      currRow;                 // The currently selected row
    QString  filename;                // The currently selected filename
    QString  line;                    // The currently selected line
    KProcess proc;                    // External editor process


    // BEGIN
    currRow = m_listbox->currentItem();

    if (currRow < 0)
    {
        KMsgBox msg(0, 0, i18n("You must select an item first."));
        msg.show();
    }   // end if
    else
    {
        filename = m_listbox->text(currRow, 0);               // get filename
        line = m_listbox->text(currRow, 1);                   // get line number

        if (line.isEmpty())
        {
            KMsgBox msg(0, 0, i18n("Can't open the selected file."));
            msg.show();
        }   // end if
        else
        {
            const Editor* editor = m_optionDlg->getEditorInfo();

            if (editor->xapp() != true)
            {
                // This is a terminal-based app
                proc << "konsole" << "-e";
            }   // end if
            proc << editor->name();

            // Fire off the external editor
            if (strlen(editor->cla()) == 0)
            {
                // No line number command-line argument specified
                proc << filename;
            }   // end if
            else
            {
                QString lineCla = editor->cla() + line;
                proc << lineCla << filename;
            }   // end else

            proc.start(KProcess::DontCare);
        }   // end else
    }   // end else

    return;

}   // end KGrepUI::slotExtEditor


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotGrepResults
//
//  Originated    : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//                               Jeff Cody   (jcody@logikos.com)
//
//  Abstract      : Handles the signal we get when the grep process
//                  generates output to stdout.  We assign the output
//                  to a QString for later manipulation.  Output comes
//                  in in 1024 chunks, so this function may be called
//                  more than once per grep process run.
//
//  Parameters    : proc          - pointer to process that initiated the
//                                  corresponding signal
//                  results       - 1024 or less chunk of output data
//                  result_length - length of result string
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotGrepResults(KProcess *proc, char *results, int result_length)
{
    // Local variables
    char buf[1025];

    if (result_length > 0)
    {
        // We get this signal after every 1024 bytes.  Here we save the
        // results off into a safe buffer to avoid core dumps while
        // adding a null terminator.
        snprintf(buf, result_length+1, "%s", results);

        // Update the non-volatile results information
        m_resultString += buf;
        m_resultLength += result_length;
    }   // end if

    return;

}   // end KGrepUI::slotGrepResults()


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotGrepResultsSelected
//
//  Originated    : 7-Sept-99  - Jeff Cody (jcody@logikos.com)
//
//  Abstract      : This is the slot which receives the row which the
//                  user clicked in the list box containing the
//                  grep results. This opens a file up as READONLY,
//                  places the cursor at the line containing the
//                  specific match, and that line is selected.
//
//  Parameters    : row - row number in listbox
//                  col   - which column in the selected row
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotGrepResultSelected(int row, int col)
{
    // Local variables
    QString filename;                // The currently selected filename
    QString line;                    // The currently selected line


    // BEGIN
    filename = m_listbox->text(row, 0);               // get filename
    line = m_listbox->text(row, 1);                   // ascii string of line number

    // Check to see if we're in the same file
    if (m_prevFilename != filename)
    {
        // We're going into a new file.  See if we need to save
        // the old one first, then load it up at the proper line.
        m_prevFilename = filename;

        // Add check here to save modified file when edit is READWRITE, if and
        // when the embedded editor is ever made READWRITE.

        // TODO - the stupid warning about QGList::locate ..., QMulitLineEdit::paintCell
        //        is happening here when we go from a long file to a shorter one.  My
        //        attempts to fix with as setCursorPosition(0, 0), unHighlightResult
        //        didn't work.
        m_edit->loadFile(filename, KEdit::OPEN_READONLY);
    }   // end if

    m_edit->setCursorPosition(atoi(line)-1, 0, FALSE);  // position the cursor
    m_edit->highlightResult();                          // highlight the line

    return;

}   // end KGrepUI::slotGrepResultSelected()


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotNewSearch
//
//  Originated    : 7-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : When the user clicks the new search button or menu
//                  item, show the modeless dialog.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotNewSearch()
{
    // BEGIN
    m_searchDlg->show();

    return;

}   // end KGrepUI::slotNewSearch


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotNextMatch
//
//  Originated    : 9-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : When the user clicks the next match button or menu
//                  item, attempt to go to the next match.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotNextMatch()
{
    // Local variables
    int nextRow = 0;                // The row that will be next
    int numRows = 0;                // Number of rows (matches) in the listbox
    int selRow  = 0;                // Currently selected row


    // BEGIN
    numRows = m_listbox->numRows();
    selRow  = m_listbox->currentItem();

    nextRow = (selRow + 1) % numRows;

    m_listbox->setCurrentItem(nextRow, 0);

    return;

}   // end KGrepUI::slotNextMatch


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotOptions
//
//  Originated    : 27-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : Opens up the Options dialog.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotOptions()
{
    // BEGIN
    m_optionDlg->exec();

    return;

}   // end KGrepUI::slotOptions


////////////////////////////////////////////////////////////////////////
//
//  Function      : slotPrevMatch
//
//  Originated    : 9-Sept-99  - Keith Brown (kbrown@logikos.com)
//
//  Abstract      : When the user clicks the previous match button or
//                  menu item, attempt to go to the previous match.
//
//  Parameters    : none
//
//  Return Value  : none
//
//  Exceptions    : none
//
//  Revisions     : none
//
////////////////////////////////////////////////////////////////////////
void KGrepUI::slotPrevMatch()
{
    // Local variables
    int numRows = 0;                // Number of rows (matches) in the listbox
    int prevRow = 0;                // The previous row
    int selRow  = 0;                // Currently selected row


    // BEGIN
    numRows = m_listbox->numRows();
    selRow  = m_listbox->currentItem();

    if (selRow == 0)
    {
        prevRow = numRows - 1;
    }   // end if
    else
    {
        prevRow = selRow - 1;
    }   // end else

    m_listbox->setCurrentItem(prevRow, 0);

    return;

}   // end KGrepUI::slotPrevMatch



//  end file kgrepui.cc
