/*
 *            klm: A lm_sensors front end for the KDE project
 *
 * $Id: StdFan.cpp,v 1.1 1999/02/06 20:23:58 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>

#include "StdFan.h"

#define RANGE_INCREMENT 100
#define RANGE_LOWER_LIMIT 4000
#define RANGE_UPPER_LIMIT 10000

StdFan::StdFan( 
  int sysc1, 
  int sysc2, 
  int sysc3,
  int sysc4,
  double grMin, 
  double grMax,
  char *defaultName	) : Sensor( sysc1, sysc2, sysc3, sysc4, grMin, grMax )
{
  SensorValue val;
  QString sensorName;
  double scaleFactor;
  
  // attempt to read the sensor config from the application config
  // and write the config to the lm_sensors module. 
  // must be root for the write operation to succeed. If it fails
  // the klm will read existing data from the sensors.
  if(Sensor::getSettings( val, sensorName, scaleFactor ))
  {
    setScale( scaleFactor );
    write( val );
    setName( sensorName );		
  }
  else
  {
    setName( defaultName );
  }
}

StdFan::~StdFan()
{
}

bool StdFan::save()
{
  QString sensorName;
  SensorValue val;
  
  name( sensorName );
  if ( read( val ) )
  {
    return Sensor::saveSettings( val, sensorName, scale() );
  }
  
  return false;
}

void StdFan::name( QString &name )
{
  name = sensorName;
}

bool StdFan::setName( QString &name )
{
  sensorName = name;	
  return true;
}

double StdFan::scale()
{
  long scaleBuffer[20];
  int numNames;
  int *sysctlAddr;
  
  getScaleFactor( 
    LM78_SYSCTL_FAN_DIV, 
    numNames, 
    sizeof(scaleBuffer),
    scaleBuffer );
  
  sysctlAddr = Sensor::getAddr();
  
  return (double)(scaleBuffer[sysctlAddr[3]-LM78_SYSCTL_FAN1]);
}

bool StdFan::setScale( double scaleFactor )
{
  return Sensor::setScaleFactor( 
    LM78_SYSCTL_FAN_DIV, 
    LM78_SYSCTL_FAN1, 
    scaleFactor );
}

bool StdFan::read( SensorValue &val )
{
  if (!Sensor::getFan( val ))
  {
    val.value = val.min = val.max = 0;
    return false;		
  }
  
  return true;
}	

bool StdFan::write( SensorValue &val )
{
  return Sensor::writeFan( val );
}

bool StdFan::displayDecimals()
{
  return false;
}

bool StdFan::displayUnits()
{
  return false;
}

void StdFan::units( QString &unitStr )
{
  unitStr="RPM";
}

bool StdFan::highAlarmSettable()
{
  return false;
}

bool StdFan::increaseGraphRange()
{
  SensorValue val;
  double min;
  double max;
  
  Sensor::graphRange( min, max );
  
  max += RANGE_INCREMENT;
  
  if (max > RANGE_UPPER_LIMIT)
  {
    max = RANGE_UPPER_LIMIT;
  }
  
  if(!read( val ))
  {
    return false;
  }
  
  if ( fabs(val.value) > max )
  {
    max = fabs(val.value) + (RANGE_INCREMENT * 2);
  }
  
  return Sensor::setGraphRange( min ,max );
}

bool StdFan::decreaseGraphRange()
{
  SensorValue val;
  double min;
  double max;
  
  Sensor::graphRange( min, max );
  
  max -= RANGE_INCREMENT;
  
  if (max < RANGE_LOWER_LIMIT)
  {
    max = RANGE_LOWER_LIMIT;
  }
  
  if(!read( val ))
  {
    return false;
  }
  
  if ( fabs(val.value) > max )
  {
    max = fabs(val.value) + (RANGE_INCREMENT * 2);
  }
  
  return Sensor::setGraphRange( min ,max );	
}

bool StdFan::increaseScale()
{
  double sc;
  
  sc = scale();
  
  switch((int)sc)
  {
    case 1:
      sc = 2;
      break;
    case 2:
      sc = 4;
      break;
    case 4:
      sc = 8;
      break;
    case 8:
      sc = 1;
      break;
    default:
      sc = 2;  // lm_sensors default setting. 
      break;		
  }
  
  return setScale( sc );
}

bool StdFan::decreaseScale()
{
  double sc;
  
  sc = scale();
  
  switch((int)sc)
  {
    case 1:
      sc = 8;
      break;
    case 2:
      sc = 1;
      break;
    case 4:
      sc = 2;
      break;
    case 8:
      sc = 4;
      break;
    default:
      sc = 2;  // lm_sensors default setting. 
      break;		
  }
  
  return setScale( sc );
}

bool StdFan::isAlarm()
{
  SensorValue val;
  
  read( val );
  val.min = fabs(val.min);
  val.max = fabs(val.max);
  val.value = fabs(val.value);
  
  return ( (val.value < val.min) || (val.value > val.max) );
}
