/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "config.h"

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <strings.h>
#include <stdarg.h>

#include "inter.h"
#include "server.h"
#include "lpr_local.h"
#include "lpr_remote.h"
#include "lpr_common.h"
#include "netware.h"


#define i18n(s)	s


const struct server servers[] = {
	{
		i18n("BSD-lpr / LPRng (local)"),
		lpr_local_set_printer,
		lpr_local_q_main,
		lpr_local_p_main,
		lpr_local_r_main,
		lpr_q_parse,
		lpr_p_parse,
		lpr_r_parse,
		lpr_printer_list,
		1,
		0
	},
	{
		i18n("BSD-lpr / LPRng (remote)"),
		lpr_remote_set_printer,
		lpr_remote_q_main,
		lpr_remote_p_main,
		lpr_remote_r_main,
		lpr_q_parse,
		lpr_p_parse,
		lpr_r_parse,
		NULL,
		0,
		1
	},
	{
		i18n("NetWare"),
		netware_set_printer,
		netware_q_main,
		netware_p_main,
		netware_r_main,
		netware_q_parse,
		netware_p_parse,
		netware_r_parse,
		NULL,
		1,
		1
	}
};


const struct queue_item empty_queue_item = { 0, NULL, NULL, NULL, 0 };
const struct queue empty_queue = { 0, 0, 0, NULL };


int pipe_r = -1;
int pipe_w = -1;

static int exit_on_fail = 0;

static int suid_root = 0;
static int full_root = 0;


static char username_[100];
const char * const username = username_;


void free_queue_item(struct queue_item *item)
{
	if (!item)
		return;
	free(item->job_name);
	free(item->user);
	free(item->job_id);
	free(item);
}


void free_queue(struct queue *queue)
{
	int i;

	if (!queue)
		return;
	for (i = 0 ; i < queue->item_count ; ++i)
		free_queue_item(queue->items[i]);
	free(queue->items);
	free(queue);
}


int check_root(void)
{
	full_root = 0;
	suid_root = 0;
	if (!getuid()) {
		fprintf(stderr, "klp: Warning: running with full root privileges\n");
		full_root = 1;
		return 1;
	} else if (!geteuid()) {
		suid_root = 1;
		return 1;
	}
	return 0;
}	


void drop_root(void)
{
	if (suid_root && setuid(getuid()) == -1) {
		perror("klp: setuid(non-root)");
		fprintf(stderr, "klp: Warning: could not drop root privileges\n");
	}
	full_root = 0;
	suid_root = 0;
}


int tmp_as_root(void)
{
	if (!full_root && !suid_root)
		return 0;
#ifdef HAVE_SETEUID
	if (suid_root && seteuid(0) == -1) {
		perror("klp: seteuid(root)");
		return 0;
	}
#endif
	return 1;
}


void tmp_drop_root(void)
{
#ifdef HAVE_SETEUID
	if (suid_root && seteuid(getuid()) == -1) {
		perror("klp: seteuid(non-root)");
		suid_root = 0;
		full_root = 1;
		fprintf(stderr, "klp: Warning: could not temporarily drop root privileges\n");
	}
#endif
}


int start_server(void)
{
	int i;
	int nfd;
	pid_t server_pid;
	int pipe1[2];
	int pipe2[2];
	struct passwd *pwd;

	if (!(pwd = getpwuid(getuid()))) {
		perror("klp: getpwuid");
		return 0;
	}
	
	strncpy(username_, pwd->pw_name, 100);
	username_[99] = '\0';

	check_root();

	/*
	 *	Make sure file descriptors 0, 1, and 2 are valid
	 *	(otherwise, pipe() might use these, and we'll have
	 *	 interference between inter process communication and
	 *	 std{in,out,err}).
	 */
	if ((nfd = open("/dev/null", O_RDONLY)) == -1) {
		perror("klp: open");
		return 0;
	}
	if (nfd) {
		int dfd;
		if ((dfd = dup(nfd)) == -1) {
			perror("klp: dup");
			close(nfd);
			return 0;
		}
		if (dfd)
			close(dfd);
		close(nfd);
	}
	for (i = 1 ; i < 2 ; ++i) {
		if ((nfd = open("/dev/null", O_WRONLY)) == -1) {
			perror("klp: open");
			return 0;
		}
		if (nfd > 2) {
			int dfd;
			if ((dfd = dup(nfd)) == -1) {
				perror("klp: dup");
				close(nfd);
				return 0;
			}
			if (dfd > 2)
				close(dfd);
			close(nfd);
		}
	}

	if ((pipe(pipe1) == -1) || (pipe(pipe2) == -1)) {
		perror("klp: pipe");
		return 0;
	}
	if ((server_pid = fork()) == -1) {
		perror("klp: fork");
		return 0;
	}
	if (!server_pid) {
		close(pipe1[0]);
		pipe_w = pipe1[1];
		pipe_r = pipe2[0];
		close(pipe2[1]);
		exit_on_fail = 1;
		exit(server_main());
	}
	pipe_r = pipe1[0];
	close(pipe1[1]);
	close(pipe2[0]);
	pipe_w = pipe2[1];
	return 1;
}


int iwrite_char(char c)
{
	if ((pipe_w == -1) || (write(pipe_w, &c, 1) != 1)) {
		if (exit_on_fail)
			exit(EXIT_FAILURE);
		else
			return 0;
	}
	return 1;
}


int iwrite_int(int i)
{
	if ((pipe_w == -1) || (write(pipe_w, &i, sizeof(i)) != sizeof(i))) {
		if (exit_on_fail)
			exit(EXIT_FAILURE);
		else
			return 0;
	}
	return 1;
}


int iwrite_string(const char *fmt, ...)
{
	int len;
	char *s = NULL;
	int size;

	if (!fmt || (pipe_w == -1))
		goto fail;
#ifdef HAVE_VSNPRINTF
	size = strlen(fmt) + 10;
#else
	size = strlen(fmt) + 1024;
#endif
	if (!(s = malloc(size)))
		goto fail;
	while (1) {
		va_list ap;
		va_start(ap, fmt);
#ifdef HAVE_VSNPRINTF
		len = vsnprintf(s, size, fmt, ap);
#else
		len = vsprintf(s, fmt, ap);
#endif
		va_end(ap);
		if ((len >= 0) && (len < size))
			break;
#ifndef HAVE_VSNPRINTF
		fprintf(stderr, "klp: Fatal: vsprintf() wrote past end of buffer\n");
		abort();
#endif		
		size *= 2;
		if (!(s = realloc(s, size)))
			goto fail;
	}
	if (!iwrite_int(len))
		goto fail;
	if (write(pipe_w, s, len) != len)
		goto fail;
	return 1;

fail:
	xfree(s);
	if (exit_on_fail)
		exit(EXIT_FAILURE);
	return 0;
}



int iread_char(char *c)
{
	if ((pipe_r == -1) || (read(pipe_r, c, 1) != 1)) {
		if (exit_on_fail)
			exit(EXIT_FAILURE);
		return 0;
	}
	return 1;
}


int iread_int(int *i)
{
	if ((pipe_r == -1) || (read(pipe_r, i, sizeof(*i)) != sizeof(*i))) {
		if (exit_on_fail)
			exit(EXIT_FAILURE);
		return 0;
	}
	return 1;
}


int iread_string(char **s)
{
	int l;

	*s = NULL;
	if (!iread_int(&l))
		goto fail;
	if (!(*s = malloc(l + 1)))
		goto fail;
	(*s)[l] = '\0';
	if (read(pipe_r, *s, l) != l) {
		xfree(*s);
		goto fail;
	}
	return 1;

fail:
	if (exit_on_fail)
		exit(EXIT_FAILURE);
	return 0;
}
