/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <o.gantz@tu-bs.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "downloaddlg.h"
#include "downloaddlg.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qwidget.h>
#include <qprogressdialog.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "modem.h"
#include "global.h"
#include "preferences.h"



DownloadDlg::DownloadDlg( QWidget * parent, const char * name, bool modal, WFlags f ) : QProgressDialog( parent, name, modal, f )
{
	modem = new Modem( this, "modem" );
	CHECK_PTR( modem );
	init();

	setMinimumDuration( 0 );

	connect( modem, SIGNAL(gotLine(const char *)), SLOT(fetchModemLine(const char *)) );
	connect( modem, SIGNAL(gotXBlock(const uchar *, int)), SLOT(fetchXModemBlock(const uchar *, int)) );
	connect( modem, SIGNAL(xmodemDone(bool)), SLOT(fetchXModemDone(bool)) );
	connect( modem, SIGNAL(timeout()), SLOT(fetchTimeout()) );
	connect( this, SIGNAL(cancelled()), SLOT(cancel()) );

	setCaption( i18n("Download") );
}


DownloadDlg::~DownloadDlg()
{
	reset();
}


bool DownloadDlg::startDownload( const char * device, int speed )
{
	static int speeds[] = { 300, 2400, 9600, 19200, 38400, 57600, 115200, 230400 };

	if (status)
		return FALSE;

	modem->setDevice( device );
	modem->setSpeed( speeds[speed] );
	modem->setData( 8 );
	modem->setParity( 'N' );
	modem->setStop( 1 );

	if (!modem->open()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Cannot open modem device."), KMsgBox::EXCLAMATION);
		return FALSE;
	}
	if (!modem->dsrOn()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem is off."), KMsgBox::EXCLAMATION);
		modem->close();
		return FALSE;
	}
	if (!modem->ctsOn()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem is busy."), KMsgBox::EXCLAMATION);
		modem->close();
		return FALSE;
	}

	modem->writeLine("");
	usleep(250000);
	modem->flush();
	modem->writeLine("ATZ");

	status = 1;

	return TRUE;
}


void DownloadDlg::cancel()
{
	QProgressDialog::cancel();

	if (status == 3)
		modem->abortXModem();
	else
		reset();
}


void DownloadDlg::fetchModemLine( const char * line )
{
	char buff[81], *p;
	int year, month, day, hour, minute;
	fitem_t * fitem;

	switch (status) {
		case 1:					/* "ATZ" sent	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				if (prefs->genLoadNew())
					modem->writeLine("AT$JDIR=,FAX,N");
				else
					modem->writeLine("AT$JDIR=,FAX");
				status++;
			}
			break;
		case 2:					/* "AT$JDIR=,FAX[,N]" sent	*/
			if (strlen(line) > 59) {
				usleep(250000);
				strncpy(buff, line, 80);
				buff[80] = 0;
				fitem = (fitem_t *)malloc(sizeof(fitem_t));
				fitem->next = fitems;
				fitems = fitem;
				if ((p = strchr(buff, ',')))
					*p = 0;
				strncpy(fitems->name, buff, 18);
				fitems->name[18] = 0;
				sscanf(&buff[41], "%d.%d.%d, %d:%d, %d", &day, &month, &year, &hour, &minute, &fitems->size);
				sprintf(fitems->time, "%04d-%02d-%02d  %02d:%02d", year, month, day, hour, minute);
				fitems->saved = FALSE;
				nfiles++;
			} else if (!strcmp(line, "OK")) {
				usleep(250000);
				count = 0;
				status++;
				downloadFile();
			}
			break;
		case 3:					/* XModem confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);

				if (prefs->genDeleteFax()) {
					sprintf(buff, "AT$JDEL=\"%s\"", fitems->name);
					modem->writeLine( buff );
					status++;
				} else {
					setProgress( fitems->size + 1 );
					fitem = fitems;
					fitems = fitems->next;
					free(fitem);
					downloadFile();
				}
			}
			break;
		case 4:					/* Delete confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				setProgress( fitems->size + 1 );
				fitem = fitems;
				fitems = fitems->next;
				free(fitem);
				status = 3;
				downloadFile();
			}
	}
}


void DownloadDlg::fetchXModemBlock( const uchar * block, int size )
{
	int diff;

	if (fd && ((diff = fitems->size - progress()) > 0)) {
		if (size > diff)
			size = diff;
		write(fd, (const void *)block, size);
		setProgress( progress () + size );
	}
}


void DownloadDlg::fetchXModemDone( bool success )
{
	if (fd) {
		::close(fd);
		fd = 0;
		fitems->saved = success;
		if (success)
			emit gotFile( fitems->name, fitems->time, fitems->size );
		else {
			sprintf(msgbuff, "%s: %s", i18n("Error"), i18n("Download failed."));
			emit message( msgbuff );

			reset();
		}
		setLabelText( i18n("Waiting for confirmation...") );
	}
}


void DownloadDlg::fetchTimeout()
{
	KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem response timeout."), KMsgBox::EXCLAMATION);

	sprintf(msgbuff, "%s: %s", i18n("Error"), i18n("Modem response timeout."));
	emit message( msgbuff );

	reset();
}


void DownloadDlg::init()
{
	status = 0;

	fitems = 0;
	nfiles = 0;
	count = 0;

	fd = 0;
}


void DownloadDlg::reset()
{
	fitem_t * p;

	QProgressDialog::reset();

	modem->close();

	while (fitems) {
		p = fitems;
		fitems = fitems->next;
		free(p);
	}

	if (fd)
		::close(fd);

	init();

	emit done();
}


void DownloadDlg::downloadFile()
{
	char buff[80];

	if (fitems) {
		count++;
		if ((fd = creat(expand_path(fitems->name), 0644)) == -1) {
			KMsgBox::message( 0, i18n("File Error"), i18n("Cannot create file."), KMsgBox::EXCLAMATION);
			sprintf(msgbuff, "%s: %s!", i18n("Error"), i18n("Download failed"));
			emit message( msgbuff );
			reset();

			return;
		}
		sprintf(buff, "AT$JDNL=\"%s\"", fitems->name);
		modem->writeLine(buff);
		sprintf(buff, i18n("Receiving file %d of %d..."), count, nfiles);
		setLabelText( buff );
		setTotalSteps( fitems->size + 1 );
		setProgress( 0 );
		modem->receiveXModem( TRUE );
	} else {
		if (nfiles)
			sprintf(msgbuff, i18n("Received %d file(s)."), nfiles);
		else
			strcpy(msgbuff, i18n("No facsimile in modem."));
		emit message( msgbuff );

		reset();
	}
}
