/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdlib.h>
#include <string.h>

#include <qglobal.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>

#include <kapp.h>
#include <kiconloader.h>
#include <ksimpleconfig.h>

#include "faxview.h"
#include "global.h"
#include "mlofile.h"
#include "senderaliases.h"



FaxViewItem::FaxViewItem(QListView *parent = 0, const char *name = 0) : QListViewItem(parent, name)
{
	m_new = false;
	m_sender[0] = 0;
	m_pages = 0;
	m_size = 0;
	m_file = false;
}


FaxViewItem::~FaxViewItem()
{
}


void FaxViewItem::setName(const QString& name)
{
	setText(0, name);
	setText(2, mloDateStr(name));
}


void FaxViewItem::setNew(bool isnew)
{
	m_new = isnew;
	setPixmap(1, m_new ? Icon("yes.xpm") : Icon("no.xpm"));
}


void FaxViewItem::setSender(const char *sender_)
{
	QString s = QString(sender_);

	strncpy(m_sender, (const char *)s.stripWhiteSpace(), 20);
	m_sender[20] = 0;
	setAlias();
}


void FaxViewItem::setPages(int pages_)
{
	QString s;

	m_pages = pages_;
	s.sprintf("%3d", pages_);
	setText(4, s);
}


void FaxViewItem::setSize(uint size_)
{
	QString s;

	m_size = size_;
	s.sprintf("%7u", size_);
	setText(5, s);
}


void FaxViewItem::setFile(bool hasfile)
{
	m_file = hasfile;
	setPixmap(6, m_file ? Icon("yes.xpm") : Icon("no.xpm"));
}


void FaxViewItem::setAlias()
{
	setText(3, SENDER_ALIAS(m_sender));
}


void FaxViewItem::setParams(const QString& name_, bool isnew_, const char *sender_, int pages_, uint size_)
{
	setName(name_);
	setNew(isnew_);
	setSender(sender_);
	setPages(pages_);
	setSize(size_);
	setFile(QFile::exists(expandPath(name_)));
}


FaxView::FaxView(QWidget *parent, const char *name) : QListView(parent, name)
{
	config = kapp->getConfig();

	selectedItem = 0;

	addColumn(i18n("Name"));
	addColumn(i18n("New"));
	addColumn(i18n("Date (Time)"));
	addColumn(i18n("Sender"));
	addColumn(i18n("Pages"));
	addColumn(i18n("Size"));
	addColumn(i18n("File"));
	setColumnWidth(0, 140);
	setColumnWidth(1, 40);
	setColumnWidth(2, 140);
	setColumnWidth(3, 160);
	setColumnWidth(4, 50);
	setColumnWidth(5, 60);
	setColumnWidth(6, 40);
	setColumnAlignment(1, AlignVCenter | AlignRight);
	setColumnAlignment(4, AlignVCenter | AlignHCenter);
	setColumnAlignment(5, AlignVCenter | AlignRight);
	setMinimumSize(sizeHint());
	setSorting(0, false);
	setAllColumnsShowFocus(true);

	contextMenu = new QPopupMenu(0, "contextmenu");
	CHECK_PTR(contextMenu);
	contextMenu->insertItem(Icon("print_preview.xpm"), i18n("&View..."), ID_FILE_VIEW);
	contextMenu->insertItem(Icon("fileprint.xpm"), i18n("&Print..."), ID_FILE_PRINT);
	contextMenu->insertItem(Icon("filefloppy.xpm"), i18n("&Export..."), ID_FILE_EXPORT);
	contextMenu->insertItem(Icon("filemail.xpm"), i18n("&Mail..."), ID_FILE_MAIL);
	contextMenu->insertItem(Icon("filedel.xpm"), i18n("De&lete"), ID_FILE_DELETE);
	contextMenu->insertItem(i18n("&Sender Alias..."), ID_FILE_SENDER_ALIAS);

	connect(this, SIGNAL(selectionChanged(QListViewItem *)), SLOT(selectionChangedSlot(QListViewItem *)));
	connect(this, SIGNAL(doubleClicked(QListViewItem *)), SLOT(doubleClickedSlot(QListViewItem *)));
	connect(this, SIGNAL(returnPressed(QListViewItem *)), SLOT(returnPressedSlot(QListViewItem *)));
	connect(this, SIGNAL(rightButtonClicked(QListViewItem *, const QPoint&, int)), SLOT(rightButtonClickedSlot(QListViewItem *, const QPoint&, int)));
	connect(senderaliases, SIGNAL(applyButtonPressed()), SLOT(refreshAliases()));
	connect(contextMenu, SIGNAL(activated(int)), SLOT(contextMenuSlot(int)));
}


FaxView::~FaxView()
{
	delete contextMenu;
}


void FaxView::readOptions()
{
	config->setGroup("FaxWindow Options");

	setColumnWidth(0, config->readNumEntry("NameWidth", 140));
	setColumnWidth(1, config->readNumEntry("NewWidth", 40));
	setColumnWidth(2, config->readNumEntry("DateWidth", 140));
	setColumnWidth(3, config->readNumEntry("SenderWidth", 160));
	setColumnWidth(4, config->readNumEntry("PagesWidth", 50));
	setColumnWidth(5, config->readNumEntry("SizeWidth", 60));
	setColumnWidth(6, config->readNumEntry("FileWidth", 40));
}


void FaxView::saveOptions()
{
	config->setGroup("FaxWindow Options");
	
	config->writeEntry("NameWidth", columnWidth(0));
	config->writeEntry("NewWidth", columnWidth(1));
	config->writeEntry("DateWidth", columnWidth(2));
	config->writeEntry("SenderWidth", columnWidth(3));
	config->writeEntry("PagesWidth", columnWidth(4));
	config->writeEntry("SizeWidth", columnWidth(5));
	config->writeEntry("FileWidth", columnWidth(6));
}


int FaxView::loadDir()
{
	KSimpleConfig *sconfig;
	int i, number = 0;
	QString s;
	FaxViewItem *item;

	clear();
	
	sconfig = new KSimpleConfig(expandPath(KMLOFAX_FAXDIR), true);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		s.sprintf("Entry %d", i);
		sconfig->setGroup(s);
		item = new FaxViewItem(this);
		CHECK_PTR(item);
		item->setParams(sconfig->readEntry("FileName"), sconfig->readBoolEntry("New"), sconfig->readEntry("Sender"), sconfig->readNumEntry("Pages"), sconfig->readNumEntry("Size"));
	}
	delete sconfig;
	
	return number;
}


void FaxView::saveDir()
{
	KSimpleConfig *sconfig;
	QString s;
	FaxViewItem *item;
	int i;

	QFile::remove(expandPath(KMLOFAX_FAXDIR));
	sconfig = new KSimpleConfig(expandPath(KMLOFAX_FAXDIR), false);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", childCount());
	item = (FaxViewItem *)firstChild();
	for (i = 0; (item); i++) {
		s.sprintf("Entry %d", i);
		sconfig->setGroup(s);
		sconfig->writeEntry("FileName", item->name());
		sconfig->writeEntry("New", item->isNew());
		sconfig->writeEntry("Sender", item->sender());
		sconfig->writeEntry("Pages", item->pages());
		sconfig->writeEntry("Size", item->size());
		item = (FaxViewItem *)item->nextSibling();
	}
	sconfig->sync();
	delete sconfig;
}


void FaxView::purgeDir()
{
	FaxViewItem *item, *item_buff;

	item = (FaxViewItem *)firstChild();
	while (item) {
		if (!(item->hasFile())) {
			item_buff = item;
			item = (FaxViewItem *)item->nextSibling();
			delete item_buff;
		}
		else
			item = (FaxViewItem *)item->nextSibling();
	}
}


int FaxView::rebuildDir()
{
	MLOFile mlofile;
	FaxViewItem *item;
	int number = 0;

	QDir d(expandPath(""));
	const QFileInfoList *files;
	
	d.setNameFilter("[1-9][0-9][0-9][0-9][01][0-9][0-3][0-9][012][0-9][0-5][0-9][0-5][0-9].FAX");
	files = d.entryInfoList();

	if (files) {
		clear();
		kapp->setOverrideCursor(waitCursor);
		QFileInfoListIterator it(*files);
		QFileInfo *f;
		while ((f=it.current()) != 0) {
			++it;
			mlofile.setName(f->filePath());
			if (!(mlofile.open()))
				continue;
			item = new FaxViewItem(this);
			CHECK_PTR(item);
			item->setParams(f->fileName(), false, mlofile.sender(), mlofile.pages(), mlofile.size());
			mlofile.close();
			number++;
		}
		kapp->restoreOverrideCursor();
	}

	return number;
}


void FaxView::addFaxItem(const QString& name)
{
	FaxViewItem *item;
	MLOFile mlofile;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (item->name() == name)
			break;
	
	if (!item)
		item = new FaxViewItem(this);

	item->setName(name);
	
	mlofile.setName(name);
	if (mlofile.open()) {
		item->setNew(true);
		item->setSender(mlofile.sender());
		item->setPages(mlofile.pages());
		item->setSize(mlofile.size());
		item->setFile(true);
		mlofile.close();
	}
	else {
		item->setNew(false);
		item->setSender("");
		item->setPages(0);
		item->setSize(0);
		item->setFile(false);
	}
}


void FaxView::setSelectedFaxNew(bool isnew)
{
	if (selectedItem)
		selectedItem->setNew(isnew);
}


bool FaxView::selectedFaxNew()
{
	if (selectedItem)
		return selectedItem->isNew();

	return false;
}


QString FaxView::selectedFaxName()
{
	if (selectedItem)
		return selectedItem->name();

	return QString(0);
}


const char *FaxView::selectedFaxSender()
{
	if (selectedItem)
		return selectedItem->sender();

	return 0;
}


QString FaxView::selectedFaxAlias()
{
	if (selectedItem)
		return selectedItem->alias();

	return QString(0);
}


void FaxView::setSelectedFaxFile(bool fexists)
{
	if (selectedItem)
		selectedItem->setFile(fexists);
}


bool FaxView::selectedFaxFile()
{
	if (selectedItem)
		return selectedItem->hasFile();

	return false;
}


void FaxView::enableCommand(int id_, bool enable)
{
	contextMenu->setItemEnabled(id_, enable);
}


void FaxView::gotSelection(bool on)
{
	enableCommand(ID_FILE_SENDER_ALIAS, on);
	
	if (on)
		on = selectedFaxFile();

	enableCommand(ID_FILE_VIEW, on);
	enableCommand(ID_FILE_PRINT, on);
	enableCommand(ID_FILE_EXPORT, on);
	enableCommand(ID_FILE_MAIL, on);
	enableCommand(ID_FILE_DELETE, on);
}


void FaxView::selectionChangedSlot(QListViewItem *item)
{
	selectedItem = (FaxViewItem *)item;
	
	gotSelection((selectedItem));
	
	emit hasSelection((selectedItem));
}


void FaxView::doubleClickedSlot(QListViewItem *)
{
	if (selectedItem)
		emit command(ID_FILE_VIEW);
}


void FaxView::returnPressedSlot(QListViewItem *item)
{
	if (item) {
		setSelected(item, true);
		emit command(ID_FILE_VIEW);
	}
}


void FaxView::rightButtonClickedSlot(QListViewItem *item, const QPoint& point, int)
{
	if (item) {
		setSelected(item, true);
		contextMenu->popup(point);
	}
}


void FaxView::refreshAliases()
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		item->setAlias();
}


void FaxView::contextMenuSlot(int id_)
{
	emit command(id_);
}
