/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdlib.h>

#include <qglobal.h>
#include <qscrollbar.h>

#include <kapp.h>
#include <kiconloader.h>
#include <kmsgbox.h>

#include "preview.h"
#include "mlofile.h"
#include "printdlg.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "global.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>



FaxWidget *fw = 0;


char *shrinkPage(int w, int h, char *data)
{
	static const uchar shrink_tab[128] = {
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
		0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f
	};
	char *ndata;
	uchar *line, *nline;
	int nw, nh, i, j, k, bpl, nbpl;

	nw = w >> 1;
	nh = h >> 1;

	bpl = (((w-1) >> 4) + 1) << 1;
	nbpl = (((nw-1) >> 4) + 1) << 1;
	ndata = (char *)malloc(nbpl * nh);

	line = (uchar *)data;
	nline = (uchar *)ndata;

	for (i=0; i < nh; i++) {
		if (i & 1)
			line += bpl;
		for (j=0; j < nbpl; j++) {
			k = j << 1;
			if (j & 1)
				nline[j] = shrink_tab[line[k] >> 1] | (shrink_tab[line[k|1] >> 1] << 4);
			else
				nline[j] = shrink_tab[line[k] & 0x7f] | (shrink_tab[line[k|1] & 0x7f] << 4);
		}
		line += bpl;
		if (!(i & 1))
			line += bpl;
		nline += nbpl;
	}

	return ndata;
}


char *rotatePage(int w, int h, char *data, bool lsb_first)
{
	char *ndata;
	uchar *line, *nline, *nline2, buff[8];
	int nw, nh, i, j, k, l, max_lines, bpl, nbpl;

	nw = h;
	nh = w;
	
	bpl = (((w-1) >> 4) + 1) << 1;
	nbpl = (((nw-1) >> 4) + 1) << 1;
	ndata = (char *)malloc(nbpl * nh);
	
	for (i = 0; i < w; i += 8) { // bpl
		line = (uchar *)(data + bpl * h + (i >> 3));
		nline = (uchar *)(ndata + nbpl * i);
		max_lines = nh - i;
		if (max_lines > 8)
			max_lines = 8;

		for (j = 0; j < nw; j += 8) { // nbpl
			nline2 = nline;
			for (k = 0; k < max_lines; k++) {
				line -= bpl;
				buff[k] = *line;
			}
			for (k = 0; k < max_lines; k++) {
				if (lsb_first) {
					for (l = 0; l < 8; l++) {
						*nline2 = (uchar)((*nline2 << 1) | (buff[l] & 1));
						buff[l] >>= 1;
					}
				}
				else {
					for (l = 0; l < 8; l++) {
						*nline2 = (uchar)((*nline2 >> 1) | (buff[l] & 128));
						buff[l] <<= 1;
					}
				}
				*nline2 = xchg_endian[*nline2];
				nline2 += nbpl;
			}
			nline++;
		}
	}

	return ndata;
}


FaxWidget::FaxWidget(QWidget *parent, const char *name, WFlags f) : QWidget(parent, name, f)
{
	int screen;

	scrollx = 0;
	scrolly = 0;

	display = x11Display();
	screen = x11Screen();

	lsb_first = (BitmapBitOrder(display) == LSBFirst);
	
	window = XCreateSimpleWindow(display, winId(), 0, 0, 1, 1, 0,
		 BlackPixel(display, screen),
		 WhitePixel(display, screen));
	gc = XCreateGC(display, window, 0L, (XGCValues *)0);
	XSetForeground(display, gc, BlackPixel(display, screen));
	XSetBackground(display, gc, WhitePixel(display, screen));
	XSetFunction(display, gc, GXcopy);
	XSelectInput(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask);
	XFlush(display);

	move_cursor = XCreateFontCursor(display, XC_fleur);
	moving = false;

	image = 0;

	hscroll = new QScrollBar(QScrollBar::Horizontal, this, "prehscroll");
	CHECK_PTR(hscroll);
	hscroll->setFixedHeight(hscroll->sizeHint().height());
	vscroll = new QScrollBar(QScrollBar::Vertical, this, "prevscroll");
	CHECK_PTR(vscroll);
	vscroll->setFixedWidth(vscroll->sizeHint().width());

	connect(hscroll, SIGNAL(valueChanged(int)), this, SLOT(setXScroll(int)));
	connect(vscroll, SIGNAL(valueChanged(int)), this, SLOT(setYScroll(int)));
}


FaxWidget::~FaxWidget()
{
	deleteImage();
	XFreeCursor(display, move_cursor);
	XFreeGC(display, gc);
	XDestroyWindow(display, window);
}


void FaxWidget::setImage(int w, int h, char *data)
{
	deleteImage();

	im_width = w;
	im_height = h;

	if ((w) && (h)) {
		image = XCreateImage(display, DefaultVisual(display, x11Screen()),
			1, XYBitmap, 0, data, im_width, im_height, 16, 0);
		XMapRaised(display, window);
	} else {
		XUnmapWindow(display, window);
		XResizeWindow(display, window, 1, 1);
	}

	updateRects();

	repaint(false);
}


void FaxWidget::deleteImage()
{
	if (image) {
		image->data = 0;
		XDestroyImage(image);
		image = 0;
	}
}


bool FaxWidget::x11EventFilter(XEvent *ev)
{
	bool refresh = false;
	bool move_it = false;
	int mv_x = 0, mv_y = 0;
	int x1 = INT_MAX, y1 = INT_MAX, x2 = 0, y2 = 0, xb, yb;

	if (ev->xany.window != window)
		return false;
	
	do {
		switch(ev->type) {
			case ButtonPress:
				if (ev->xbutton.button == Button1) {
					XDefineCursor(display, window, move_cursor);
					moving = true;
					move_x = ev->xbutton.x;
					move_y = ev->xbutton.y;
				}
				break;
			case ButtonRelease:
				if (ev->xbutton.button == Button1) {
					XUndefineCursor(display, window);
					moving = false;
				}
				break;
			case MotionNotify:
				if (moving) {
					move_it = true;
					mv_x = ev->xmotion.x;
					mv_y = ev->xmotion.y;
				}
				break;
			case Expose:
				refresh = true;
				xb = ev->xexpose.x + ev->xexpose.width;
				yb = ev->xexpose.y + ev->xexpose.height;
				if (ev->xexpose.x < x1)
					x1 = ev->xexpose.x;
				if (ev->xexpose.y < y1)
					y1 = ev->xexpose.y;
				if (xb > x2)
					x2 = xb;
				if (yb > y2)
					y2 = yb;
			default:
				break;
		}
	} while (XCheckWindowEvent(display, window, ButtonPressMask | ButtonReleaseMask | Button1MotionMask | ExposureMask, ev));

	if (refresh && image)
		XPutImage(display, window, gc, image, scrollx + x1, scrolly + y1, x1, y1, x2-x1, y2-y1);

	if (move_it) {
		hscroll->setValue(hscroll->value() + move_x - mv_x);
		vscroll->setValue(vscroll->value() + move_y - mv_y);
		move_x = mv_x;
		move_y = mv_y;
	}

	return true;
}	


void FaxWidget::setXScroll(int x)
{
	scrollx = x;
	repaint(false);
}


void FaxWidget::setYScroll(int y)
{
	scrolly = y;
	repaint(false);
}


void FaxWidget::keyLeft()
{
	hscroll->setValue(hscroll->value() - hscroll->lineStep());
}


void FaxWidget::keyRight()
{
	hscroll->setValue(hscroll->value() + hscroll->lineStep());
}


void FaxWidget::keyUp()
{
	vscroll->setValue(vscroll->value() - vscroll->lineStep());
}


void FaxWidget::keyDown()
{
	vscroll->setValue(vscroll->value() + vscroll->lineStep());
}


void FaxWidget::keyPageUp()
{
	vscroll->setValue(vscroll->value() - vscroll->pageStep());
}


void FaxWidget::keyPageDown()
{
	vscroll->setValue(vscroll->value() + vscroll->pageStep());
}


void FaxWidget::keyHome()
{
	vscroll->setValue(vscroll->minValue());
}


void FaxWidget::keyEnd()
{
	vscroll->setValue(vscroll->maxValue());
}


void FaxWidget::paintEvent(QPaintEvent *)
{
	int w, h, x = 0, y = 0;

	w = width();
	if (vscroll->isVisible())
		w -= vscroll->width();

	h = height();
	if (hscroll->isVisible())
		h -= hscroll->height();
	
	if (w > im_width) {
		x = (w - im_width) >> 1;
		w = im_width;
	}
	if (h > im_height) {
		y = (h - im_height) >> 1;
		h = im_height;
	}
	if (image) {
		XMoveResizeWindow(display, window, x, y, w, h);
		XPutImage(display, window, gc, image, scrollx, scrolly, 0, 0, w, h);
	}
}


void FaxWidget::resizeEvent(QResizeEvent *)
{
	updateRects();
}


void FaxWidget::updateRects()
{
	int w, h;
	bool need_hscroll, need_vscroll;

	w = width();
	h = height();

	if ((need_hscroll = (im_width > w)))
		h -= hscroll->height();
	if ((need_vscroll = (im_height > h))) {
		w -= vscroll->width();
		if (!need_hscroll && (im_width > w)) {
			need_hscroll = true;
			h -= hscroll->height();
		}
	}
	
	if (need_hscroll) {
		hscroll->setRange(0, im_width - w);
		hscroll->setSteps(12, w);
		hscroll->setGeometry(0, h, w, hscroll->height());
		hscroll->show();
	} else {
		hscroll->hide();
		hscroll->setRange(0, 0);
		hscroll->setValue(0);
	}
	
	if (need_vscroll) {
		vscroll->setRange(0, im_height - h);
		vscroll->setSteps(12, h);
		vscroll->setGeometry(w, 0, vscroll->width(), h);
		vscroll->show();
	} else {
		vscroll->hide();
		vscroll->setRange(0, 0);
		vscroll->setValue(0);
	}
}



Preview::Preview(const char *name) : KTMainWindow(name)
{
	int i;

	config = kapp->getConfig();
	
	for (i=0; i < 3; i++)
		page_datas[i] = 0;
	page = pages = 0;
	page_width = page_height = 0;
	zoom_width = zoom_height = 0;
	zoom = 1;
	angle = 0;

	initMenuBar();
	initToolBar();
	initStatusBar();

	view = new FaxWidget(this, "faxwidget");
	CHECK_PTR(view);
	fw = view;
	setView(view);
	
	initKeyAccel();

	readOptions();

	setCaption(i18n("KMLOFax Preview"));
}


Preview::~Preview()
{
	deletePages();
}


void Preview::showFax(const QString& name, const QString& alias)
{
	slotStatusName("");
	slotStatusSender("");

	file.setName(name);
	if (!file.open()) {
		KMsgBox::message(0, i18n("File Error"), i18n("Cannot open facsimile file."), KMsgBox::EXCLAMATION);
		return;
	}
	pages = file.pages();
	file.close();

	fname = name;
	salias = alias;
	
	slotStatusName(fname);
	slotStatusSender(salias);

	loadPage(1);

	if (isVisible())
		raise();
	else
		show();
}


void Preview::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void Preview::readOptions()
{
	config->setGroup("Preview Options");

	bool toolBarOn = config->readBoolEntry("Show ToolBar", true);
	optionsMenu->setItemChecked(ID_PRE_OPTIONS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		enableToolBar(KToolBar::Hide);

	bool statusBarOn = config->readBoolEntry("Show StatusBar", true);
	optionsMenu->setItemChecked(ID_PRE_OPTIONS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		enableStatusBar(KStatusBar::Hide);

	menuBar()->setMenuBarPos((KMenuBar::menuPosition)config->readNumEntry("MenuBarPos", (int)KMenuBar::Top));
	
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
	
	QSize geoSize(650, 400);
	resize(config->readSizeEntry("Geometry", &geoSize));
}


void Preview::saveOptions()
{
	config->setGroup("Preview Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("MenuBarPos", (int)menuBar()->menuBarPos());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());
}


void Preview::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "prefilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(Icon("fileprint.xpm"), i18n("&Print..."), ID_PRE_FILE_PRINT);
	fileMenu->insertItem(Icon("filefloppy.xpm"), i18n("&Export..."), ID_PRE_FILE_EXPORT);
	fileMenu->insertItem(Icon("filemail.xpm"), i18n("&Mail..."), ID_PRE_FILE_MAIL);
	fileMenu->insertSeparator();
	fileMenu->insertItem(i18n("&Close"), ID_PRE_FILE_CLOSE);

	gotoMenu = new QPopupMenu(0, "pregotomenu");
	CHECK_PTR(gotoMenu);
	gotoMenu->insertItem(Icon("back.xpm"), i18n("&Previous Page"), ID_PRE_GOTO_PREVIOUS);
	gotoMenu->insertItem(Icon("forward.xpm"), i18n("&Next Page"), ID_PRE_GOTO_NEXT);
	gotoMenu->insertItem(Icon("start.xpm"), i18n("&First Page"), ID_PRE_GOTO_FIRST);
	gotoMenu->insertItem(Icon("finish.xpm"), i18n("&Last Page"), ID_PRE_GOTO_LAST);
	
	viewMenu = new QPopupMenu(0, "previewmenu");
	CHECK_PTR(viewMenu);
	viewMenu->insertItem(Icon("viewmag+.xpm"), i18n("Zoom &In"), ID_PRE_VIEW_ZOOM_IN);
	viewMenu->insertItem(Icon("viewmag-.xpm"), i18n("Zoom &Out"), ID_PRE_VIEW_ZOOM_OUT);
	viewMenu->insertItem(Icon("rotate.xpm"), i18n("&Rotate"), ID_PRE_VIEW_ROTATE);

	optionsMenu = new QPopupMenu(0, "preoptmenu");
	CHECK_PTR(optionsMenu);
	optionsMenu->setCheckable(true);
	optionsMenu->insertItem(i18n("Show &Toolbar"), ID_PRE_OPTIONS_TOOLBAR);
	optionsMenu->setItemChecked(ID_PRE_OPTIONS_TOOLBAR, true);
	optionsMenu->insertItem(i18n("Show Status&bar"), ID_PRE_OPTIONS_STATUSBAR);
	optionsMenu->setItemChecked(ID_PRE_OPTIONS_STATUSBAR, true);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Goto"), gotoMenu);
	menuBar()->insertItem(i18n("&View"), viewMenu);
	menuBar()->insertItem(i18n("&Options"), optionsMenu);
	
	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(gotoMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(viewMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(optionsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void Preview::initToolBar()
{
	toolBar()->insertButton(Icon("back.xpm"), ID_PRE_GOTO_PREVIOUS, true, i18n("Previous Page"));
	toolBar()->insertButton(Icon("forward.xpm"), ID_PRE_GOTO_NEXT, true, i18n("Next Page"));
	toolBar()->insertButton(Icon("start.xpm"), ID_PRE_GOTO_FIRST, true, i18n("First Page"));
	toolBar()->insertButton(Icon("finish.xpm"), ID_PRE_GOTO_LAST, true, i18n("Last Page"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(Icon("viewmag+.xpm"), ID_PRE_VIEW_ZOOM_IN, true, i18n("Zoom In"));
	toolBar()->insertButton(Icon("viewmag-.xpm"), ID_PRE_VIEW_ZOOM_OUT, true, i18n("Zoom Out"));
	toolBar()->insertButton(Icon("rotate.xpm"), ID_PRE_VIEW_ROTATE, true, i18n("Rotate"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(Icon("fileprint.xpm"), ID_PRE_FILE_PRINT, true, i18n("Print Facsimile"));
	toolBar()->insertButton(Icon("filefloppy.xpm"), ID_PRE_FILE_EXPORT, true, i18n("Export Facsimile"));
	toolBar()->insertButton(Icon("filemail.xpm"), ID_PRE_FILE_MAIL, true, i18n("Mail Facsimile"));
	toolBar()->insertButton(Icon("exit.xpm"), ID_PRE_FILE_CLOSE, true, i18n("Close"));
	toolBar()->alignItemRight(ID_PRE_FILE_CLOSE, true);
	
	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


void Preview::initStatusBar()
{
	statusBar()->insertItem("00000000000000.FAX", ID_PRE_STATUS_NAME);
	statusBar()->insertItem(i18n("000000000, Normal"), ID_PRE_STATUS_SIZE);
	statusBar()->setAlignment(ID_PRE_STATUS_SIZE, AlignHCenter);
	statusBar()->insertItem(i18n("Page 000 of 000"), ID_PRE_STATUS_PAGE);
	statusBar()->setAlignment(ID_PRE_STATUS_PAGE, AlignHCenter);
	statusBar()->insertItem("000", ID_PRE_STATUS_ANGLE);
	statusBar()->setAlignment(ID_PRE_STATUS_ANGLE, AlignHCenter);
	statusBar()->insertItem("", ID_PRE_STATUS_SENDER);
	statusBar()->setAlignment(ID_PRE_STATUS_SENDER, AlignHCenter);
	statusBar()->changeItem("", ID_PRE_STATUS_NAME);
	statusBar()->changeItem("", ID_PRE_STATUS_SIZE);
	statusBar()->changeItem("", ID_PRE_STATUS_PAGE);
	statusBar()->changeItem("", ID_PRE_STATUS_ANGLE);
}


void Preview::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Export File"), "ExportFile", CTRL+Key_E);
	keyAccel->insertItem(i18n("Mail File"), "MailFile", CTRL+Key_M);
	
	keyAccel->insertItem(i18n("Goto Prior"), "GotoPrior", ALT+Key_Left);
	keyAccel->insertItem(i18n("Goto Next"), "GotoNext", ALT+Key_Right);
	keyAccel->insertItem(i18n("Goto First"), "GotoFirst", ALT+Key_Up);
	keyAccel->insertItem(i18n("Goto Last"), "GotoLast", ALT+Key_Down);
	
	keyAccel->insertItem(i18n("Zoom In"), "ZoomIn", ALT+Key_Plus);
	keyAccel->insertItem(i18n("Zoom Out"), "ZoomOut", ALT+Key_Minus);
	keyAccel->insertItem(i18n("Rotate"), "Rotate", ALT+Key_R);
	
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);
	
	keyAccel->insertItem(i18n("Scroll Left"), "ScrollLeft", Key_Left);
	keyAccel->insertItem(i18n("Scroll Right"), "ScrollRight", Key_Right);
	keyAccel->insertItem(i18n("Scroll Up"), "ScrollUp", Key_Up);
	keyAccel->insertItem(i18n("Scroll Down"), "ScrollDown", Key_Down);
	keyAccel->insertItem(i18n("Page Up"), "PageUp", Key_PageUp);
	keyAccel->insertItem(i18n("Page Down"), "PageDown", Key_PageDown);
	keyAccel->insertItem(i18n("Home"), "Home", Key_Home);
	keyAccel->insertItem(i18n("End"), "End", Key_End);

	keyAccel->readSettings();

	keyAccel->connectItem(KAccel::Print, this, SLOT(slotFilePrint()));
	keyAccel->connectItem("ExportFile", this, SLOT(slotFileExport()));
	keyAccel->connectItem("MailFile", this, SLOT(slotFileMail()));
	keyAccel->connectItem(KAccel::Close, this, SLOT(hide()));
	
	keyAccel->connectItem("GotoPrior", this, SLOT(slotGotoPreviousPage()));
	keyAccel->connectItem("GotoNext", this, SLOT(slotGotoNextPage()));
	keyAccel->connectItem("GotoFirst", this, SLOT(slotGotoFirstPage()));
	keyAccel->connectItem("GotoLast", this, SLOT(slotGotoLastPage()));
	
	keyAccel->connectItem("ZoomIn", this, SLOT(slotViewZoomIn()));
	keyAccel->connectItem("ZoomOut", this, SLOT(slotViewZoomOut()));
	keyAccel->connectItem("Rotate", this, SLOT(slotViewRotate()));
	
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotOptionsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotOptionsShowStatusbar()));
	
	keyAccel->connectItem(KAccel::Help, kapp, SLOT(appHelpActivated()));

	keyAccel->connectItem("ScrollLeft", view, SLOT(keyLeft()));
	keyAccel->connectItem("ScrollRight", view, SLOT(keyRight()));
	keyAccel->connectItem("ScrollUp", view, SLOT(keyUp()));
	keyAccel->connectItem("ScrollDown", view, SLOT(keyDown()));
	keyAccel->connectItem("PageUp", view, SLOT(keyPageUp()));
	keyAccel->connectItem("PageDown", view, SLOT(keyPageDown()));
	keyAccel->connectItem("Home", view, SLOT(keyHome()));
	keyAccel->connectItem("End", view, SLOT(keyEnd()));
	
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_PRINT, KAccel::Print);
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_EXPORT, "ExportFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_MAIL, "MailFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_CLOSE, KAccel::Close);
	
	keyAccel->changeMenuAccel(gotoMenu, ID_PRE_GOTO_PREVIOUS, "GotoPrior");
	keyAccel->changeMenuAccel(gotoMenu, ID_PRE_GOTO_NEXT, "GotoNext");
	keyAccel->changeMenuAccel(gotoMenu, ID_PRE_GOTO_FIRST, "GotoFirst");
	keyAccel->changeMenuAccel(gotoMenu, ID_PRE_GOTO_LAST, "GotoLast");
	
	keyAccel->changeMenuAccel(viewMenu, ID_PRE_VIEW_ZOOM_IN, "ZoomIn");
	keyAccel->changeMenuAccel(viewMenu, ID_PRE_VIEW_ZOOM_OUT, "ZoomOut");
	keyAccel->changeMenuAccel(viewMenu, ID_PRE_VIEW_ROTATE, "Rotate");

	keyAccel->changeMenuAccel(optionsMenu, ID_PRE_OPTIONS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(optionsMenu, ID_PRE_OPTIONS_STATUSBAR, "ShowStatusbar");
}


void Preview::commandCallback(int id_)
{
	switch (id_) {
		case ID_PRE_FILE_PRINT:
			slotFilePrint();
			break;
		case ID_PRE_FILE_EXPORT:
			slotFileExport();
			break;
		case ID_PRE_FILE_MAIL:
			slotFileMail();
			break;
		case ID_PRE_FILE_CLOSE:
			hide();
			break;
		case ID_PRE_GOTO_PREVIOUS:
			slotGotoPreviousPage();
			break;
		case ID_PRE_GOTO_NEXT:
			slotGotoNextPage();
			break;
		case ID_PRE_GOTO_FIRST:
			slotGotoFirstPage();
			break;
		case ID_PRE_GOTO_LAST:
			slotGotoLastPage();
			break;
		case ID_PRE_VIEW_ZOOM_IN:
			slotViewZoomIn();
			break;
		case ID_PRE_VIEW_ZOOM_OUT:
			slotViewZoomOut();
			break;
		case ID_PRE_VIEW_ROTATE:
			slotViewRotate();
			break;
		case ID_PRE_OPTIONS_TOOLBAR:
			slotOptionsShowToolbar();
			break;
		case ID_PRE_OPTIONS_STATUSBAR:
			slotOptionsShowStatusbar();
			break;
		default:
			break;
	}
}


void Preview::slotFilePrint()
{
	printdlg->printFax(fname, salias);
}


void Preview::slotFileExport()
{
	exportdlg->exportFax(fname, salias);
}


void Preview::slotFileMail()
{
	maildlg->mailFax(fname, salias);
}


void Preview::slotGotoPreviousPage()
{
	if (page > 1)
		loadPage(page-1);
}


void Preview::slotGotoNextPage()
{
	if (page < pages)
		loadPage(page+1);
}


void Preview::slotGotoFirstPage()
{
	if (page > 1)
		loadPage(1);
}


void Preview::slotGotoLastPage()
{
	if (page < pages)
		loadPage(pages);
}


void Preview::slotViewZoomIn()
{
	if (zoom) {
		zoom--;
		showPage();
	}
}


void Preview::slotViewZoomOut()
{
	if (zoom < 2) {
		zoom++;
		showPage();
	}
}


void Preview::slotViewRotate()
{
	char *new_data;
	int i, w, h;

	if (angle & 1) {
		w = page_height;
		h = page_width;
	}
	else {
		w = page_width;
		h = page_height;
	}
	
	kapp->setOverrideCursor(waitCursor);

	new_data = rotatePage(w, h, page_datas[0], view->lsbFirst());
	
	for (i=0; i < 3; i++)
		free(page_datas[i]);
	
	page_datas[0] = new_data;
	page_datas[1] = shrinkPage(h, w, page_datas[0]);
	page_datas[2] = shrinkPage(h >> 1, w >> 1, page_datas[1]);

	kapp->restoreOverrideCursor();

	angle = (angle + 1) & 3;
	slotStatusAngle();

	showPage();
}


void Preview::slotOptionsShowToolbar()
{
	bool toolBarOn = !optionsMenu->isItemChecked(ID_PRE_OPTIONS_TOOLBAR);

	menuBar()->setItemChecked(ID_PRE_OPTIONS_TOOLBAR, toolBarOn);
	enableToolBar(toolBarOn ? KToolBar::Show : KToolBar::Hide);
}


void Preview::slotOptionsShowStatusbar()
{
	bool statusBarOn = !optionsMenu->isItemChecked(ID_PRE_OPTIONS_STATUSBAR);

	menuBar()->setItemChecked(ID_PRE_OPTIONS_STATUSBAR, statusBarOn);
	enableStatusBar(statusBarOn ? KStatusBar::Show : KStatusBar::Hide);
}


void Preview::slotStatusName(const QString& text)
{
	statusBar()->changeItem(text, ID_PRE_STATUS_NAME);
}


void Preview::slotStatusSize(int width_, int height_, bool normal)
{
	QString s;
	
	if (width_)
		s.sprintf("%d%d, %s", width_, height_, normal ? i18n("Normal") : i18n("Fine"));
	else
		s = "";
	statusBar()->changeItem(s, ID_PRE_STATUS_SIZE);
}


void Preview::slotStatusPage(int page_, int pages_)
{
	QString s;
	
	if (page_)
		s.sprintf(i18n("Page %d of %d"), page_, pages_);
	else
		s = "";
	statusBar()->changeItem(s, ID_PRE_STATUS_PAGE);
}


void Preview::slotStatusAngle()
{
	static const int angles[4] = { 0, 90, 180, 270 };
	QString s;

	s.sprintf("%d", angles[angle]);
	statusBar()->changeItem(s, ID_PRE_STATUS_ANGLE);
}


void Preview::slotStatusSender(const QString& text)
{
	statusBar()->changeItem(text, ID_PRE_STATUS_SENDER);
}


void Preview::hide()
{
	saveOptions();

	KTMainWindow::hide();
}


void Preview::deletePages()
{
	int i;

	page_width = page_height = 0;

	for (i=0; i < 3; i++)
		if (page_datas[i]) {
			free(page_datas[i]);
			page_datas[i] = 0;
		}
	view->deleteImage();
}


void Preview::showPage()
{
	enableCommand(ID_PRE_VIEW_ZOOM_IN, zoom > 0);
	enableCommand(ID_PRE_VIEW_ZOOM_OUT, zoom < 2);
	
	zoom_width = page_width >> zoom;
	zoom_height = page_height >> zoom;
	
	if (angle & 1)
		view->setImage(zoom_height, zoom_width, page_datas[zoom]);
	else
		view->setImage(zoom_width, zoom_height, page_datas[zoom]);
}


void Preview::loadPage(int p)
{
	page_info_t info;
	char *pd;
	int i, bpl;
	bool lsb_first = view->lsbFirst();

	angle = 0;

	slotStatusSize(0, 0, true);
	slotStatusPage(0, 0);
	slotStatusAngle();
	
	deletePages();

	page = p;

	enableCommand(ID_PRE_GOTO_PREVIOUS, page != 1);
	enableCommand(ID_PRE_GOTO_NEXT, page != pages);
	enableCommand(ID_PRE_GOTO_FIRST, page != 1);
	enableCommand(ID_PRE_GOTO_LAST, page != pages);

	if (!file.open()) {
		KMsgBox::message(0, i18n("File Error"), i18n("Cannot open facsimile file."), KMsgBox::EXCLAMATION);
		showPage();
		return;
	}
	if (!file.readPageInfo(p, &info)) {
		KMsgBox::message(0, i18n("File Error"), i18n("Cannot read facsimile page info."), KMsgBox::EXCLAMATION);
		file.close();
		showPage();
		return;
	}

	page_width = info.width;
	page_height = file.pageHeight(page);

	slotStatusSize(page_width, page_height, (info.lpi == 98));
	slotStatusPage(page, pages);
	
	kapp->setOverrideCursor(waitCursor);

	bpl = (((page_width-1) >> 4) + 1) << 1;
	if (info.lpi == 98)
		page_datas[0] = (char *)malloc(bpl * page_height * 2);
	else
		page_datas[0] = (char *)malloc(bpl * page_height);
	pd = page_datas[0];

	file.gotoPage(page);
	if (info.lpi == 98) {
		for (i=0; i < page_height; i++) {
			file.readImgLine((uchar *)pd, lsb_first);
			memcpy((void *)(pd + bpl), (void *)pd, bpl);
			pd += (bpl << 1);
		}
		page_height <<= 1;
	} else {
		for (i=0; i < page_height; i++) {
			file.readImgLine((uchar *)pd, lsb_first);
			pd += bpl;
		}
	}
	file.close();

	page_datas[1] = shrinkPage(page_width, page_height, page_datas[0]);
	page_datas[2] = shrinkPage(page_width >> 1, page_height >> 1, page_datas[1]);

	kapp->restoreOverrideCursor();

	showPage();
}


void Preview::closeEvent(QCloseEvent *)
{
	hide();
}
