#include <iostream.h>

// KDE headers
#include <ktoolbar.h>
#include <kstatusbar.h>
#include <kmenubar.h>
#include <kiconloader.h>
#include <kconfig.h>
#include <kmsgbox.h>
#include <kaccel.h>
#include <klocale.h>

// QT headers
#include <qlabel.h>
#include <qpopupmenu.h>
#include <qdatetime.h>

// Homemade headers
#include "knewmail.h"
#include "knewmailview.h"
#include "knewmailengine.h"
#include "configdialog.h"
#include "docking.h"
#include "passdialog.h"
#include "errordialog.h"

KNewMailApp::KNewMailApp(int noSplash, int upgrade)
{
  QString caption = "knewmail v";
  caption += VERSION;
  caption += " by Mike Pilone";
  setCaption(caption);

  readOptions();

  // initialize all the gui stuff
  init();

  // use the command line params
  KConfig *config = kapp->getConfig();

  config->setGroup("BASE");
  if ((!noSplash) && (!config->readBoolEntry("NeverDisplaySplash", FALSE)))
    slotDisplaySplash();

  if (upgrade)
    {
    ERROR_FORCE_DISPLAY("This version of KNewMail does not support the upgrade flag.");
    }


  config->setGroup("BASE");
  if (config->readBoolEntry("Startup", FALSE))
    slotForceCheck();


}

KNewMailApp::~KNewMailApp()
{

  // Disconnect everything
  disconnect(fEngine, SIGNAL( checkPerformed(QList<HeaderStruct>*, QList<HeaderStruct>*) ), this, SLOT( slotCheckPerformed(QList<HeaderStruct>*, QList<HeaderStruct>*) ));
  disconnect(fView, SIGNAL( rightButtonClicked(const QPoint&) ), this, SLOT( slotRightButtonClicked(const QPoint&) ));
  disconnect(fView, SIGNAL( leftButtonDoubleClicked() ), this, SLOT( slotLeftButtonDoubleClicked() ));

  delete(fToolBar);
  delete(fMenuBar);
  delete(fStatusBar);

  delete(fFilePopMenu);
  delete(fOptionsPopMenu);
  delete(fCommandsPopMenu);
  delete(fHelpPopMenu);

  delete(fView);
  delete(fEngine);
  delete(fDockWidget);

  fMailboxList.clear();

  // delete the error dialog last in case anything wants to report a final error
  delete(gErrorDialog);
  
}

void KNewMailApp::init()
{
  KConfig *config = kapp->getConfig();

  // Create the kaccel to capture key preses
  fKAccel = new KAccel(this, "Key Accel");

  // set the number of old messages
  fNumMessages = 0;

  // set the splash label to NULL;
  fSplashLabel = NULL;

  // Set the checking now bool to false.
  fCheckingNow = FALSE;

  // Set the ignore show command to false. This will be
  // set to true only if this is restored.
  config->setGroup("BASE");
  fIgnoreShowCommand = config->readBoolEntry("DockOnLaunch", FALSE);

  // create each of the main window components
  initMenu();
  initToolBar();
  initStatusBar();
  initView();

  // create the docking widget
  fDockWidget = new DockWidget(this, "DockWidget");

  // setup the engine. The engine basically keeps the time and when needed,
  // it will do a mail check. If there is new mail it will do whatever
  // the user has configured to have happen.
  initEngine();

  // set all the components to there proper settings. This includes 
  // position and if it is enabled.
  setToolBarState(fToolBarPos, fViewToolBar);
  setStatusBarState(fViewStatusBar);

  // Setup the mailbox list
  fMailboxList.setAutoDelete( TRUE );

}

void KNewMailApp::initView()
{

  fView = new KNewMailView(this, "View");
  connect(fView, SIGNAL( rightButtonClicked(const QPoint&) ), this, SLOT( slotRightButtonClicked(const QPoint&) ));
  connect(fView, SIGNAL( leftButtonDoubleClicked() ), this, SLOT( slotLeftButtonDoubleClicked() ));


  setView(fView);

  // Set the main window up
  this->setGeometry(fMainGeomRect);
  //setMinimumWidth(450);
  //setMinimumHeight(205);
}


void KNewMailApp::initToolBar()
{
  KConfig* config = kapp->getConfig();

  fToolBar = new KToolBar(this, "ToolBar");
  
  fToolBar->insertButton(Icon("knewmail_force_check.xpm"), ID_COMMANDS_FORCE_CHECK, true, i18n("Force Check"));
  fToolBar->insertButton(Icon("knewmail_launch.xpm"), ID_COMMANDS_LAUNCH_APP, true, i18n("Launch App"));
  fToolBar->insertButton(Icon("knewmail_dock.xpm"), ID_COMMANDS_DOCK, true, i18n("Dock"));
  
  config->setGroup("BASE");
  if (config->readBoolEntry("Active", FALSE))
    fToolBar->insertButton(Icon("knewmail_active.xpm"), ID_COMMANDS_ACTIVE_TOG, true, i18n("Active Toggle"));
  else
    fToolBar->insertButton(Icon("knewmail_inactive.xpm"), ID_COMMANDS_ACTIVE_TOG, true, i18n("Active Toggle"));

  CONNECT_TOOLBAR(fToolBar);

  connect(fToolBar, SIGNAL( moved(BarPosition )), this, SLOT( slotToolBarMoved()));
  addToolBar(fToolBar);
}

void KNewMailApp::initMenu()
{
  KConfig* config = kapp->getConfig();

  // -- Create all the key accelerators ------------------------------------
  fKAccel->insertItem( i18n("Toggle MenuBar"), "Toggle MenuBar", "CTRL+M");
  fKAccel->connectItem( "Toggle MenuBar", this, SLOT( slotToggleMenuBar()));
  
  fKAccel->insertItem( i18n("Configuration"), "Configuration", "CTRL+C");
  fKAccel->connectItem( "Configuration", this, SLOT( slotConfig()));

  fKAccel->insertItem( i18n("Exit"), "Exit", "CTRL+Q");
  fKAccel->connectItem( "Exit", this, SLOT( slotAppExit()));

  fKAccel->insertItem( i18n("Force Check"), "Force Check", "CTRL+F");
  fKAccel->connectItem( "Force Check", this, SLOT( slotForceCheck()));

  fKAccel->insertItem( i18n("Launch Mail App"), "Launch Mail App", "CTRL+L");
  fKAccel->connectItem( "Launch Mail App", this, SLOT( slotLaunchApp()));

  fKAccel->insertItem( i18n("Dock"), "Dock", "CTRL+D");
  fKAccel->connectItem( "Dock", this, SLOT( slotDock()));

  fKAccel->insertItem( i18n("Toggle Active"), "Toggle Active", "CTRL+A");
  fKAccel->connectItem( "Toggle Active", this, SLOT( slotToggleActive()));

  fKAccel->readSettings();

   // -- Create the menu ---------------------------------------------------
  fMenuBar = new KMenuBar(this, "MenuBar");
  
  // Set up the menu items
  fFilePopMenu = new QPopupMenu();
  fFilePopMenu->insertItem(i18n("E&xit"), ID_APP_EXIT);
  fKAccel->changeMenuAccel(fFilePopMenu, ID_APP_EXIT, "Exit");

  fOptionsPopMenu = new QPopupMenu();
  fOptionsPopMenu->insertItem(i18n("&Toggle MenuBar"), ID_OPTIONS_MENUBAR);
  fOptionsPopMenu->insertItem(i18n("&Configuration"), ID_OPTIONS_CONFIG);
  fKAccel->changeMenuAccel(fOptionsPopMenu, ID_OPTIONS_MENUBAR, "Toggle MenuBar");
  fKAccel->changeMenuAccel(fOptionsPopMenu, ID_OPTIONS_CONFIG, "Configuration");

  fCommandsPopMenu = new QPopupMenu();
  fCommandsPopMenu->insertItem(i18n("&Force Check"), ID_COMMANDS_FORCE_CHECK);
  fCommandsPopMenu->insertItem(i18n("&Launch Mail App"), ID_COMMANDS_LAUNCH_APP);
  fCommandsPopMenu->insertItem(i18n("&Dock on Panel"), ID_COMMANDS_DOCK);
  fCommandsPopMenu->insertItem(i18n("&Toggle Active"), ID_COMMANDS_ACTIVE_TOG);
  fKAccel->changeMenuAccel(fCommandsPopMenu, ID_COMMANDS_FORCE_CHECK, "Force Check");
  fKAccel->changeMenuAccel(fCommandsPopMenu, ID_COMMANDS_LAUNCH_APP, "Launch Mail App");
  fKAccel->changeMenuAccel(fCommandsPopMenu, ID_COMMANDS_DOCK, "Dock");
  fKAccel->changeMenuAccel(fCommandsPopMenu, ID_COMMANDS_ACTIVE_TOG, "Toggle Active");

  QString helpText = "KNewMail for KDE\nVersion ";
  helpText += VERSION;
  helpText += "\n\nCopyright (c) 1998 by Mike Pilone <mpilone@slac.com>\n\nThis program is licensed uner the GNU General Piblic License (GPL).\nKNewMail comes with ABSOLUTELY NO WARRANTY to the extent permitted by applicable law.";
  fHelpPopMenu = kapp->getHelpMenu(TRUE, i18n(helpText));
  
  //-- MENUBAR ----------------------------------------------
  
  fMenuBar->insertItem(i18n("&File"), fFilePopMenu);
  fMenuBar->insertItem(i18n("&Commands"), fCommandsPopMenu);
  fMenuBar->insertItem(i18n("&Options"), fOptionsPopMenu);
  fMenuBar->insertSeparator();
  fMenuBar->insertItem(i18n("&Help"), fHelpPopMenu);
  
  //-- END MENUBAR -----------------------------------------

  CONNECT_CMD(fFilePopMenu);
  CONNECT_CMD(fOptionsPopMenu);
  CONNECT_CMD(fCommandsPopMenu);

  setMenu(fMenuBar);
  
  config->setGroup("APPEARANCE");
  if (!config->readBoolEntry("ShowMenuBar", TRUE))
    {
      fMenuBar->hide();
      config->writeEntry("ShowMenuBar", FALSE);
    }
}

void KNewMailApp::initStatusBar()
{
  KConfig* config = kapp->getConfig();

  fStatusBar = new KStatusBar(this, "StatusBar");
  
 // We need to initialize everything at is largest size, then tone it down
  // to what we actually want.

  fStatusBar->insertItem("Number of new messages: 1000", 0);
  fStatusBar->changeItem("Number of new messages: -NA-", 0);
  fStatusBar->setAlignment(0, AlignCenter);
  
  fStatusBar->insertItem("Time of last check: 10:10:10  ", 1);
  fStatusBar->changeItem("Time of last check: -NA-", 1);
  fStatusBar->setAlignment(1, AlignCenter);
  
  config->setGroup("BASE");
  fStatusBar->insertItem("InActive", 2);
  fStatusBar->setAlignment(2, AlignCenter);
  
  if (config->readBoolEntry("Active", FALSE))
    fStatusBar->changeItem("Active", 2);

  setStatusBar(fStatusBar);
}

void KNewMailApp::initEngine()
{
  fEngine = new KNewMailEngine(&fMailboxList, this, "KNewMailEngine");
  connect(fEngine, SIGNAL( checkPerformed(QList<HeaderStruct>*, QList<HeaderStruct>*) ), this, SLOT( slotCheckPerformed(QList<HeaderStruct>*, QList<HeaderStruct>*) ));
}

bool KNewMailApp::queryExit()
{
  //int exit=KMsgBox::yesNo(this, i18n("Exit knewmail"), i18n("Really Quit ?"));

  //if(exit==1)
    return true;
  //else
   // return false;
}

void KNewMailApp::saveOptions()
{
  // write the config file entries
  KConfig *config = kapp->getConfig();

  fMainGeomRect = this->geometry();

  config->setGroup("APPEARANCE");
  config->writeEntry("MainGeometry", fMainGeomRect);
  config->writeEntry("ShowToolBar", fViewToolBar);
  config->writeEntry("ShowStatusBar", fViewStatusBar);
  
  // If the toolbar is floatin we will reattach it next time.
  if (fToolBarPos > 3)
      fToolBarPos = 0;

  config->writeEntry("ToolBarPos", fToolBarPos);

  fView->saveOptions();

  // Save the mailbox list in its current form. NOTE: This will change the group of
  // the config file.
  processMailboxList( MAILBOX_LIST_WRITE );

  config->sync();

}

void KNewMailApp::readProperties(KConfig* config)
{
  if (config->readBoolEntry("Docked", FALSE))
    {
    slotDock();
    fIgnoreShowCommand = TRUE;
    //KMsgBox::message(this, "YO", "docking", KMsgBox::INFORMATION, "OK");
    }
  else
    this->show();


  KTMainWindow::readProperties(config);
      
}

void KNewMailApp::saveProperties(KConfig* config)
{
  if (fDockWidget->isDocked())
    config->writeEntry("Docked", TRUE);
  else
     config->writeEntry("Docked", FALSE);

  config->sync();

  KTMainWindow::saveProperties(config);
}

void KNewMailApp::readOptions()
{
  // read the config file entries
  KConfig *config = kapp->getConfig();

  config->setGroup("APPEARANCE");
  fMainGeomRect = config->readRectEntry("MainGeometry",&QRect(0,0,500,205));
  fViewToolBar = config->readBoolEntry("ShowToolBar", TRUE );
  fViewStatusBar = config->readBoolEntry("ShowStatusBar", TRUE );
  fToolBarPos = config->readNumEntry("ToolBarPos", KToolBar::Top);

  // Load the mailbox list. NOTE: This will change the group of
  // the config file.
  processMailboxList( MAILBOX_LIST_READ );
}

void KNewMailApp::show()
{
  if (!fIgnoreShowCommand)
    {
      KTMainWindow::show();
    }

  // Since the ignore var only counts once, we will 
  // set it back to false now.
  fIgnoreShowCommand = FALSE;
}
void KNewMailApp::resizeEvent(QResizeEvent *evt)
{
    KTMainWindow::resizeEvent(evt);

    fMainGeomRect = this->geometry();
    //fView->resize(fMainGeomRect.width(), fMainGeomRect.height());
    updateRects();
}

void KNewMailApp::setToolBarState(int position, bool visible)
{
  if(visible)
    enableToolBar(KToolBar::Show);
  
  else
    enableToolBar(KToolBar::Hide);

  fViewToolBar = visible;

  fToolBar->setBarPos((KToolBar::BarPosition)position);
  fToolBarPos = position;
}

void KNewMailApp::setStatusBarState(bool visible)
{
  if (visible)
    enableStatusBar(KStatusBar::Show);

  else
    enableStatusBar(KStatusBar::Hide);

  fViewStatusBar = visible;
}

void KNewMailApp::processMailboxList(int command)
{
  KConfig *config = kapp->getConfig();
  QTime time;
  int hour, minute, second, s, count = 1, numMailboxes = 0;
  ServerStruct *serverInfo;
  char password[255], mailboxNumString[10];

  switch (command)
    {
    case MAILBOX_LIST_WRITE:
      // Write the time to the config file. Required in new config file. --     
      time.start();
      hour = time.hour();
      minute = time.minute();
      second = time.second();
      
      // Write the current time so we know when this file was created (debugging reasons only)
      config->setGroup("TIME");
      config->writeEntry("Hour", hour);
      config->writeEntry("Minute", minute);
      config->writeEntry("Second", second);
      s = second;
      
      // -- end writting time ---------------------------------------------- 
    
      serverInfo = fMailboxList.first();

      count = 1;
      while (serverInfo != NULL)
	{

	  //cout << "Processing a mailbox and writing it to config" << endl;
	 
	  sprintf(mailboxNumString, "MAILBOX_%d", count);
	  config->setGroup(mailboxNumString);
	  config->writeEntry("Address", (const char*)serverInfo->address);
	  config->writeEntry("Username", (const char*)serverInfo->username);
	  strcpy(password, ed((const char*)serverInfo->password, s, 1));
	  config->writeEntry("Password", password);
	  config->writeEntry("Folder", (const char*)serverInfo->folder);
	  config->writeEntry("Port", serverInfo->port);
	  config->writeEntry("Save", serverInfo->save);
	  config->writeEntry("Type", (int)serverInfo->type);
	  
	  serverInfo = fMailboxList.next();
	  count++;
	}
      
      config->setGroup("BASE");
      config->writeEntry("NumMailboxes", fMailboxList.count());

      config->sync();
      break;
      

    case MAILBOX_LIST_READ:   // Load the list from config
            
      fMailboxList.clear();


      config->setGroup("BASE");
      numMailboxes = config->readNumEntry("NumMailboxes", 0);
      config->setGroup("TIME");
      s = config->readNumEntry("Second", 0);

      // read the mailbox options.
      for (int i = 0; i < numMailboxes; i++)
	{
	  
	  serverInfo = new ServerStruct;
	  
	  // set the index number of the new server
	  serverInfo->index = fMailboxList.count();
	  
	  sprintf(mailboxNumString, "MAILBOX_%d", i+1);
	  
	  config->setGroup(mailboxNumString);
	  serverInfo->address = config->readEntry("Address", "ERROR");
	  serverInfo->username = config->readEntry("Username", "ERROR");
	  serverInfo->password = ed(config->readEntry("Password", "ERROR"), s, 0);
	  serverInfo->folder = config->readEntry("Folder", "INBOX");
	  serverInfo->port = config->readNumEntry("Port", 110);
	  serverInfo->save = config->readNumEntry("Save", 0);
	  serverInfo->type = (eServerType)config->readNumEntry("Type", (int)ServerTypePop3);
	  
	  fMailboxList.append(serverInfo);
	}
      break;
    }
}

char* KNewMailApp::ed(const char* p, const int s, int e)
{
  int length = strlen(p), i = 0, j = 0, k = 0, z = 0;
  char output[255], temp[5], tempA[5];
  
  if (e)
    {
      output[0] = '\0';
      for (i = 0; i < length; i++)
	{
	  sprintf(temp, "%d", (int)p[i] + s);
	  sprintf(tempA, "%d", strlen(temp));
	  strcat(output, tempA);
	  strcat(output, temp);
	}
    }
  else
    {
      i = 0;
      j = 0;
      k = 0;
      z = 0;
      output[0] = '\0';

      while (p[i] != '\0')
	{
	  tempA[0] = p[i];
	  tempA[1] = '\0';
	  z = atoi(tempA);

	  i++;
	  j = 0;
	  for (j = 0; j < z; j++) 
	    {
	    temp[j] = p[i];
	    temp[j+1] = '\0';
	    i++;
	    }
	  
	  output[k] = (char)atoi(temp) - s;
	  output[k + 1] = '\0';
	  k++;
	}
    }

  return output;
}

QString KNewMailApp::showPassDialog(QString address)
{
  QString password;
  CPassDialog* passDialog;

  passDialog = new CPassDialog(address, 0, "PassDialog");
  passDialog->setGeometry(300, 300, 200, 110);
  passDialog->exec();

  password = passDialog->password();
 
  delete (passDialog);

  return password;

}

////////////////// SLOTS ///////////////////////

void KNewMailApp::slotAppExit()
{
  ///////////////////////////////////////////////////////////////////
  // exits the Application
  // Make sure were not docked
  //fDockWidget->undock();

  if(this->queryExit())
    {
      saveOptions();
      KTMainWindow::deleteAll();
      kapp->quit();
    }

  else 
    {
      //this->show();
      return;
    }
  ///////////////////////////////////////////////////////////////////
}

void KNewMailApp::slotToolBarMoved()
{
  fToolBarPos = (int) fToolBar->barPos();
}

void KNewMailApp::slotConfig()
{
  fEngine->configDialogActivated( CONFIG_DIALOG_OPENED );

  fConfigDialog = new CConfigDialog(&fMailboxList, this, "ConfigDialog");

  connect(fConfigDialog, SIGNAL(applyButtonPressed()), this, SLOT(slotConfigDlgApply()));
  connect(fConfigDialog, SIGNAL(cancelButtonPressed()), this, SLOT(slotConfigDlgCancel()));

  fConfigDialog->exec();
   
  disconnect(fConfigDialog, SIGNAL(applyButtonPressed()), this, SLOT(slotConfigDlgApply()));
  disconnect(fConfigDialog, SIGNAL(cancelButtonPressed()), this, SLOT(slotConfigDlgCancel()));
  delete (fConfigDialog);

  fEngine->configDialogActivated( CONFIG_DIALOG_CLOSED );
}

void KNewMailApp::slotConfigDlgApply()
{
  // Trigger the recursive call in the config dialog to save
  // all the config elements
  fConfigDialog->writeOptions();

  // save the mailbox list.
  processMailboxList( MAILBOX_LIST_WRITE );

  // Refresh the view.
  fView->refresh();

  // load then update all the elements in the gui, like
  // toolbars and status bars.

   // read the config file entries
  KConfig *config = kapp->getConfig();

  config->setGroup("APPEARANCE");
  fViewToolBar = config->readBoolEntry("ShowToolBar", TRUE );
  fViewStatusBar = config->readBoolEntry("ShowStatusBar", TRUE );
  fToolBarPos = config->readNumEntry("ToolBarPos", KToolBar::Top);

  setToolBarState(fToolBarPos, fViewToolBar);
  setStatusBarState(fViewStatusBar);

  updateRects();
  resizeEvent( NULL );
}

void KNewMailApp::slotConfigDlgCancel()
{
  // Just incase the user made any changes that need to
  // be erased.
  processMailboxList( MAILBOX_LIST_READ );
}

void KNewMailApp::slotToggleMenuBar()
{
  KConfig* config = kapp->getConfig();

  if (fMenuBar->isVisible())
    {
      fMenuBar->hide();
      config->setGroup("APPEARANCE");
      config->writeEntry("ShowMenuBar", false);
    }
  
  else
    {
      fMenuBar->show();
      config->setGroup("APPEARANCE");
      config->writeEntry("ShowMenuBar", true);
    }

  config->sync();

  updateRects();
  resizeEvent( NULL );
}

void KNewMailApp::slotToggleActive()
{ 
  KConfig* config = kapp->getConfig();

  // Find out what the current state is
  config->setGroup("BASE");
  if (config->readBoolEntry("Active", FALSE ))
    {
      config->writeEntry("Active", FALSE);
      
      fToolBar->setButtonPixmap(ID_COMMANDS_ACTIVE_TOG, Icon("knewmail_inactive.xpm"));
      fStatusBar->changeItem(i18n("InActive"), 2);
    }

  else
     {
      config->writeEntry("Active", TRUE);
      
      fToolBar->setButtonPixmap(ID_COMMANDS_ACTIVE_TOG, Icon("knewmail_active.xpm"));
      fStatusBar->changeItem(i18n("Active"), 2);
    }
}

void KNewMailApp::slotForceCheck()
{
  if (!fCheckingNow)
    {
      fCheckingNow = TRUE;
      fEngine->checkMail();
    }


}

void KNewMailApp::slotLaunchApp()
{
  char systemCommand[250];
  KConfig *config = kapp->getConfig();

  config->setGroup("BASE");
  if (config->readBoolEntry("DockOnLaunch", FALSE))
    fDockWidget->dock();

  config->setGroup("NOTIFICATION");
  strcpy(systemCommand, config->readEntry("AutoLaunchPath", "klogin"));
  strcat(systemCommand, " &");

  system(systemCommand);
}

void KNewMailApp::slotPlaySound()
{
  KConfig *config = kapp->getConfig();
  char systemString[250];

  // Check if they want the system bell
  config->setGroup("NOTIFICATION");
  if (strcmp(config->readEntry("PlayerPath", "kplayaudio"), "system-bell") == 0)
    {
      kapp->beep();
      kapp->beep();
    }
  
  else       // Play the desired sound
    {
      strcpy(systemString, config->readEntry("PlayerPath", "kplayaudio"));
      strcat(systemString, " ");
      strcat(systemString, config->readEntry("SoundPath", "mail.wav"));
      strcat(systemString, " &");
      system(systemString);
    }

}

void KNewMailApp::slotShowMessage(QList<HeaderStruct>* newMsgHeaderList)
{
  HeaderStruct* headerInfo;
  QString message, from;
  int count = 0, addressStart = 0;
  QListIterator<HeaderStruct> newMsgHeaderListIter(*newMsgHeaderList);


  message = "New mail from:\n\n";

  headerInfo = newMsgHeaderListIter.toFirst();

  while ((headerInfo != NULL) && (count <= 10))
    {
      addressStart = headerInfo->from.find('<', 0, FALSE);
      
      if (addressStart == -1) // there is no name
	{
	  while (headerInfo->from[0] == ' ')
	    {
	      headerInfo->from = headerInfo->from.remove(0, 1);
	    }
	  message += headerInfo->from + "\n";
	}

      else
	{
	  from = headerInfo->from.left(addressStart);
	  while (from[0] == ' ')
	    {
	      from = from.remove(0, 1);
	    }
	  
	  if (from[0] == '\"')  // Trim quotes
	    {
	      from = from.remove(from.find('\"', 0, FALSE), 1);
	      from = from.remove(from.find('\"', 0, FALSE), 1);
	    }
	  
	  message += from + "\n"; 
	}

      headerInfo = ++newMsgHeaderListIter;
      count++;
    }
  
  if (count > 10)
    message += "...";

  KMsgBox::message(this, i18n("New Mail"), message, KMsgBox::INFORMATION, "OK");
      

}

void KNewMailApp::slotDock()
{
  fDockWidget->dock();
}

void KNewMailApp::slotDisplaySplash()
{
  if (fSplashLabel == NULL)
    {
      fSplashTimer = new QTimer(this, "SplashTimer");
      connect( fSplashTimer, SIGNAL(timeout()), this, SLOT(slotDisplaySplash()) );
  
      fSplashLabel = new QLabel(this, "SplashLabel");
      fSplashLabel->setGeometry((width()/2) - 201, (height()/2) - 50, 402, 100);
      fSplashLabel->setPixmap(Icon("knewmail_splash.xpm"));

      fSplashTimer->start(5000, TRUE);
    }

  else
    {
      disconnect( fSplashTimer, SIGNAL(timeout()), this, SLOT(slotDisplaySplash()) );
      delete(fSplashTimer);
      
      fSplashLabel->hide();
      delete(fSplashLabel);
      fSplashLabel = NULL;
    }
  
}

void KNewMailApp::slotCheckPerformed(QList<HeaderStruct> *headerList, QList<HeaderStruct> *newMsgHeaderList)
{
  QString caption;
  char numMessages[10];  // This allows for 999999999 messages.
  QTime time;
  KConfig *config = kapp->getConfig();

  sprintf(numMessages, "%d", headerList->count());

  // set the window caption
  caption = "knewmail - ";
  caption += numMessages;
  caption += " Messages";
  setCaption(caption);
 
  // set the status bar captions
  caption = "Number of new messages: ";
  caption += numMessages;
  fStatusBar->changeItem(caption, 0);
  time.start();
  caption = "Time of last check: ";
  caption += time.toString();
  fStatusBar->changeItem(caption, 1);
  
  // display the headers in the view
  fView->displayHeaderList(headerList);

  // Now we need to do all the stuff that has to be done 
  // when we check mail.
  
  // first see if there was any actual mail
  if (headerList->count() > 0)
    {
      fDockWidget->setIconAnimated();
    }
  
  else
    {
      fDockWidget->setIconNormal();
    }

  // next check to see if the number of messages is more then last time.
  // this is how we will now if we should do all the stuff the user
  // has chosen.
  if ((int)newMsgHeaderList->count() > 0)
    {
      // now check each notification option

      config->setGroup("NOTIFICATION");     
      if (config->readBoolEntry("AutoLaunch", FALSE))
	slotLaunchApp();

      config->setGroup("NOTIFICATION");     
      if (config->readBoolEntry("PlaySound", FALSE))
	slotPlaySound();

      config->setGroup("NOTIFICATION");     
      if (config->readBoolEntry("ShowHeaders", FALSE))
	fDockWidget->undock();

      config->setGroup("NOTIFICATION");     
      if (config->readBoolEntry("ShowMessage", FALSE))
	{
	  /*
	  caption = "KNewMail has detected ";
	  caption += numMessages;
	  caption += " new emails waiting.";
	  KMsgBox::message(this, i18n("New Mail!"), caption, KMsgBox::INFORMATION, "OK");
	  */
	  slotShowMessage(newMsgHeaderList);
	}
    }

  fCheckingNow = FALSE;
}

void KNewMailApp::slotRightButtonClicked(const QPoint& point)
{
  fOptionsPopMenu->popup(point);
}

void KNewMailApp::slotLeftButtonDoubleClicked()
{
  slotLaunchApp();
}

// THESE MACROS CONSTRUCT THE SWITCH FOR THE SIGNAL commandCallback(int)

BEGIN_CMD(KNewMailApp)
  ON_CMD(ID_APP_EXIT,                 slotAppExit())

  ON_CMD(ID_OPTIONS_CONFIG,           slotConfig())
  ON_CMD(ID_OPTIONS_MENUBAR,          slotToggleMenuBar())
  
  ON_CMD(ID_COMMANDS_FORCE_CHECK,     slotForceCheck())
  ON_CMD(ID_COMMANDS_LAUNCH_APP,      slotLaunchApp())
  ON_CMD(ID_COMMANDS_DOCK,            slotDock())
  ON_CMD(ID_COMMANDS_ACTIVE_TOG,      slotToggleActive())

END_CMD()


#include "knewmail.moc"
