// -*- C++ -*-
//
//  konline
//
//  Copyright (C) 1998 Christoph Neerfeld
//  email:  Christoph.Neerfeld@home.ivm.de or chris@kde.org
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#ifndef server_included
#define server_included

extern "C" {
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
}

struct users {  /* size 204 */
    unsigned long    user_id;
    char    name[41];
    char    fname[41];
    char    nname[21];
    char    date[16];
    char    email[81];
};

#include "PipeCom.h"

/// Interface for the communication with KOnserver program
/**
  This class takes care of the communication with den KOnserver program.
  It does not contain any Qt code or calls to any C++ specific library; just 
  standard C functions. This was done to make this class portable to other OS.
  It makes use of the PipeCom class which is part of this distribution.
*/

class Server
{
public:
  /// Constructor
  /**
    KOnline or any derived program needs one instance of the Server class.
    The constructor does not take any arguments.
   */
  Server() { s_pipe = 0; listensock = -1; keepsocket = -1; }

  /// Destructor
  ~Server() {}

  /// Initializes server object
  /**
    Make sure that this function is called before any other function is called.
    It takes the hostname or the IP-address of the KOnserver as first argument. In case of
    a hostname it tries to translate this into an IP-address. If this failes the function
    returns -1.
    The second argument is the portnumber used by KOnserver.
    The third argument is the password of a user account and the forth argument is the
    corresponding account Id. This information is needed to establish a connection to the
    server. Note that password and account Id are not needed if you want to create a new account.

    Beside of setting these variables init() creates a socket for connection requests made
    by the server and a socket to send UDP datagrams as keepalive packets. If the creation
    of any of these sockets fails, init() returns -1.

    It is save to call this function more than once.
   */
  int init( const char *h, unsigned short p, const char *pass = 0, unsigned long id = 0 );

  /// Sets server hostname or IP-address and portnumber
  /**
    Use this function if you want to change the server settings, but want to leave the
    account settings intact.
   */
  int setServer( const char *h, unsigned short p );

  /// Sets password and corresponding account Id
  /**
    Use this function if you want to change to another account, but don't want to change
    the server settings.
   */
  void setAccount( const char *pass, unsigned long id);

  /// Creates new account on KOnserver
  /**
    It takes name, first name, nickname, email address and password as arguments.
    The new account Id is returned in id.
    The function returns 
             -1 on a communication error
         RE_COM if the connection to the server could not be created (IP or port ?)
      RE_DUPKEY if a user with this email address does already exist
          RE_OK if the new account was created.
   */
  int create( const char *name, const char *fname, const char *nname,
	      const char *email, const char *pass, unsigned long *id);

  /// Changes account information on KOnserver
  /**
    Changes name, first name, nickname, email address and password of the
    current user account. Note that the old password and account id must be
    set correctly by a init() or setAccount() call.
    The function returns 
             -1 on a communication error
         RE_COM if the connection to the server could not be created (IP or port ?)
      RE_DUPKEY if a user with this email address does already exist
     RE_FAILURE on any other error (possibly an internal error of KOnserver)
          RE_OK if the settings to this account could be changed.
   */
  int change( const char *name, const char *fname, const char *nname,
	      const char *email, const char *new_pass );

  /// Removes the  current account
  /**
    Removes the current account from the server; the account id and passwd set by a
    previouse setAccount() call.
   */
  int remove();

  /// Searches the server database for a specific user
  /**
    Searches in the server database for a specific user. Key is one of SE_NAME (name),
    SE_NNAME (nickname), SE_EMAIL (email) and SE_ID (account id) and stands for the 
    primary search key. E.g. if you take 'name' as primary key, you have to set 'name' and use
    'fname' (first name), 'nname' and 'email' to further decrease the number of hits.
    The first hit will be returned in 'rec'.
   */
  int findUser(short key, const char *name, const char *fname, const char *nname,
	       const char *email, unsigned long id, struct users *rec);

  /// Gets the next record from a findUser() call
  /**
    findNext() returns the next hit after a findUser() call.
    Konserver will not return more than 100 records.
   */
  int findNext( struct users *rec );

  /// Adds a user to the personal watchlist
  /**
    Adds the user with the account id 'new_id' to the users watchlist.
   */
  int addWatchList( unsigned long new_id );

  /// Connects the client program to KOnserver
  /**
    Tells konserver that the user is online.
    The function returns the information of the first user in watchlist within 'rec'.
    If this user is currently online, 'act' contains 1.
   */
  int serverConnect( struct users *rec, char *act );

  /// Disconnects the client program from KOnserver
  /**
    Tells konserver that the user goes offline.
   */
  int serverDisconnect();

  /// Gets the next record from the personal watchlist
  /**
    The call serverConnect() returns the first user in the watchlist. Call nextWUserId()
    to get further records.
    The function returns RE_NOTFOUND if the end of the list was reached.
   */
  int nextWUserId( struct users *rec, char *act );

  /// Processes a message coming from KOnserver
  /**
    init() creates a server socket 'listensock' to receive messages from the server 
    like 'user is online'.
    The client main program has to check for incoming data on this socket and must call
    incomingData() whenever data arrives. Use listenSocket() to get the filedescriptor
    of 'listensock'.
    incomigData() returns the message type and copies the data into 'id', 'inbuf' and 'rec'
    according to the message type. You have to allocate the memory by yourself.
    Message types are:
      USER_ONLINE: This will set 'id' to the id of the corresponding user.
      USER_OFFLINE: This will set 'id' to the id of the corresponding user.
      SERVER_SHUTDOWN: Does not set any data and indicates that the server was shut down.
      STILL_ALIVE: Does not set any data. This message is handled internaly and you can ignore it.
      INCOMING_MESSAGE: This will copy the message into 'inbuf' and sets the structure 'rec'.
        (Note: 'id' is not changed). 'inbuf' must have a length of at least MAX_MSG_LENGTH bytes.
	If your are for any reason unable to receive messages, set 'inbuf = 0'. incomingData()
	will instruct konserver to keep the message in its queue. When you are able to 
	receive messages again call getMsg().
   */
  int incomingData( unsigned long *id, char *inbuf, struct users *rec );

  /// Returns the filedescriptor of the listensocket
  /**
    Returns the filedescriptor of the listensocket. The client program has to check for
    input on this fd and call incomingData().
   */
  int listenSocket() { return listensock; }

  /// Removes a user from the watchlist
  /**
    Removes the user with the account id 'id' from the users watchlist.
   */
  int removeWatchList( unsigned long id );

  /// Sends a message to another user
  /**
    Sends a message 'msg' to the user with account id 'rid' and with the date string 'date'.
    'date' has the format: YYYYMMDDhhmm+TZ e.g. 199806231105+01
    06/23/1998 at 11:05 localtime +01 hour from GMT.

    The message has a maximum length of MAX_MSG_LEN; but if the message has to be queued by
    konserver, konserver will truncate this message to MAX_QUEUED_MSG_LEN.
   */
  int sendMessage( unsigned long rid, const char *msg, const char *date );

  /// Checks for messages queued on KOnserver
  /**
    As stated by incomingData() the client may force konserver to keep the messages.
    Call getMsg() if you are able to accept messages again.
   */
  int getMsg();

  /// Sends a keep alive packet to KOnserver
  /**
    Konserver waits for keep alive packets send by the server.
    Call this function every KEEP_ALIVE_TIME seconds.
   */
  void sendKeepAlive();

  /// Gets the account information for a specific user id
  /**
    Gets information about user with accout id 'id'. 'pass' is the corresponding password 
    to 'id'. Password and id information inside of Server are not changed by this call.
   */
  int getInfo( unsigned long id, const char *pass, struct users *rec );
  
protected:
  int connect();
  void disconnect() { shutdown(sock, 2); close(sock); delete s_pipe; s_pipe = 0; }

  char               host[120];      // hostname of konserver
  unsigned short     port;           // port used by konserver
  struct sockaddr_in servername;
  struct sockaddr_in listenname;
  struct sockaddr_in keepname;
  int                sock;           // fd for outgoing connections
  int                listensock;     // fd for incoming connections
  int                keepsocket;     // fd for keep alive packets
  PipeCom           *s_pipe;
  PipeCom           *l_pipe;
  unsigned long      my_id;
  char               passwd[11];
};

#endif /* server_included */


