// -*- C++ -*-

//
//  konserver
//
//  Copyright (C) 1998 Christoph Neerfeld
//  email:  Christoph.Neerfeld@home.ivm.de or chris@kde.org
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//


extern "C" {
#include <sys/types.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <typhoon.h>
}

#include <iostream.h>
#include <strstream.h>
#include <iomanip.h>
#include <fstream.h>

#include "konserver.h"
#include "users.h"
#include "watch.h"
#include "message.h"

volatile sig_atomic_t cont_loop = 1;

extern int debug;

//---------------------------------------------

void Clients::insert(unsigned long id, struct sockaddr_in *a) {
  Item *pt = new Item(id, a, list);
  pt->alive = 3;
  list = pt;
}

void Clients::remove() 
{
  Item *pt = list;
  while(pt)
    {
      Item * tmp = pt;
      pt = pt->next;
      delete tmp;
    }
  list = 0;
}

void Clients::remove(unsigned long id)
{
  Item *prv;
  Item *pt = list;
  if(pt->id == id)
    {
      list = pt->next;
      delete pt;
      return;
    }
  prv = list;
  for( pt = list->next; pt != 0; pt=pt->next )
    {
      if (pt->id == id)
        {
          prv->next=pt->next;
	  delete pt;
          return;
        }
      else
        {
          prv = pt;
        }
    }
  return;
}

struct sockaddr_in *Clients::search(unsigned long id)
{
  if (list == 0) return 0;
  if (list->id == id) return &list->addr;
  Item *pt = list->next;
  for (; pt != 0; pt = pt->next)
    {
      if (pt->id == id) return &pt->addr;
    }
  return 0;
}

int Clients::sendActive(unsigned long cid, unsigned long rid, short active)
{
  if( search(rid) )
    {
      if( connect(rid) < 0 )
	return -1;
      if( active )
	*l_pipe << (short) USER_ONLINE;
      else
	*l_pipe << (short) USER_OFFLINE;
      *l_pipe << cid;
      if( l_pipe->send() < 0 )
	return -1;
      disconnect();
    }
  return 0;
}

int Clients::sendMsg(unsigned long cid, unsigned long rid, MyString msg, User *rec, MyString date)
{
  if( search(rid) )
    {
      if( connect(rid) < 0 )
	return -1;
      *l_pipe << (short) INCOMING_MESSAGE << msg
	      << rec->name
	      << rec->vname
	      << rec->nick
	      << rec->email
	      << date
	      << rec->id;
      if( l_pipe->send() < 0 )
	return -1;
      short length = 0;
      short rcode;
      if( (length = l_pipe->recv()) <= 0 || length > 1024)
	{
	  fprintf(stderr,"read error from client on listensocket\n");
	  return -1;
	}
      *l_pipe >> rcode;
      disconnect();
      return rcode;
    }
  return RE_RECV_BUF_EXCEEDED;
}

int Clients::serverShutdown()
{
  if( list == 0 )
    return 0;
  Item *pt = list;
  while(pt)
    {
      if( connect(pt->id) == 0 )
	{
	  *l_pipe << (short) SERVER_SHUTDOWN;
	  if( l_pipe->send() < 0 )
	    return -1;
	  disconnect();
	}
      pt = pt->next;
    }
  return 0;
}

int Clients::connect(unsigned long id)
{
  sock = socket (PF_INET, SOCK_STREAM, 0);
  if (sock < 0)
    {
      fprintf(stderr,"can't create socket\n");
      return -1;
    }
  if ( 0 > ::connect(sock, (struct sockaddr *) search(id), sizeof(struct sockaddr_in)) )
    {
      fprintf(stderr,"can't connect to client socket / port = %i\n", ntohs(search(id)->sin_port));
      return -1;
    }
  l_pipe = new PipeCom(sock);
  return 0;
}

int Clients::disconnect()
{
  close(sock); 
  delete l_pipe;
  l_pipe = 0;
  return 0;
}

void Clients::gotKeepAlive(unsigned long id)
{
  if( list == 0 ) 
    return;
  if( list->id == id ) 
    {    
      list->alive = 3;
      return;
    }
  Item *pt = list->next;
  for( ; pt != 0; pt = pt->next )
    {
      if( pt->id == id )
	{
	  pt->alive = 3;
	  return;
	}
    }
  return;
}

void Clients::processAliveCounter(WatchData *wdata)
{
  if( list == 0 ) 
    return;
  Item *pt = list;
  Item *temp;
  unsigned long client_id;
  unsigned long id;
  //for( ; pt != 0; pt = pt->next )
  while( pt != 0 )
    {
      if( pt->alive <= 0 )
	{ // didn't get any keep alive packages
          if( search(pt->id) )
	    {
	      if( connect(pt->id) == 0 )
		{
		  *l_pipe << (short) STILL_ALIVE;
		  if( l_pipe->send() == 0 )
		    {
		      short length = 0;
		      short rcode;
		      if( (length = l_pipe->recv()) > 0 && length <= 1024)
			{
			  *l_pipe >> rcode;
			  *l_pipe >> client_id;
			  if( client_id == pt->id )
			    {
			      disconnect();
			      pt = pt->next;
			      continue;
			    }	      
			}
		    }
		  disconnect();
		}
	    }
	  // client does not answer
	  if( debug )
	    fprintf(stderr,"lost connection to client id = %i\n", pt->id);
	  if( wdata->firstWatchId(pt->id, &id) == S_OKAY )
	    {
	      do {
		sendActive(pt->id, id, 0);
	      } while( wdata->nextWatchId( pt->id, &id) == S_OKAY );
	    }
	  temp = pt;
	  pt = pt->next;
	  remove(temp->id);
	}
      else
	{
	  pt->alive--;
	  pt = pt->next;
	}
    }
}

//--------------------------------------------------
// Konserver
//--------------------------------------------------

Konserver::Konserver(MyString config_file)
{
  if( readConfig(config_file) )
    exit(1);

  d_dbdpath((char*)databasedir.c_str());
  d_dbfpath((char*)databasedir.c_str());

  int rcode;
  rcode = d_open("konline", "x");
  if( rcode != S_OKAY )
    {
      fprintf(stderr, "Unable to open database file. Errorcode: %i\n", rcode);
      exit(1);
    }

  serversock = socket(PF_INET, SOCK_STREAM, 0);
  if(serversock < 0)
    {
      fprintf(stderr,"Konserver::Konserver: unable to create server socket\n");
      exit(1);
      return;
    }
  servername.sin_family = AF_INET;
  //servername.sin_port = htons(3000);    // set by readConfig()
  servername.sin_addr.s_addr = htonl (INADDR_ANY);
  if(bind(serversock, (struct sockaddr *) &servername, sizeof(servername)) < 0)
    {
      fprintf(stderr,"Konserver::Konserver: can't bind server socket\n");
      shutdown(serversock, 2);
      close(serversock);
      exit(1);
      return;
    }
  if(listen (serversock, 10) < 0)      // listen queue has 10 entries
    {
      fprintf(stderr,"Konserver::Konserver: can't listen on server socket\n");
      exit(1);
      return;
    }
  keepsocket = socket(PF_INET, SOCK_DGRAM, 0);
  if(keepsocket < 0)
    {
      fprintf(stderr,"Konserver::Konserver: unable to create udp keep alive socket\n");
      exit(1);
      return;
    }
  keepname.sin_family = AF_INET;
  //keepname.sin_port = htons(3001);    // set by readConfig()
  keepname.sin_addr.s_addr = htonl (INADDR_ANY);
  if( bind(keepsocket, (struct sockaddr *) &keepname, sizeof(keepname)) < 0 )
    {
      fprintf(stderr,"Konserver::Konserver: can't bind keep alive socket\n");
      shutdown(keepsocket, 2);
      close(keepsocket);
      exit(1);
      return;
    }

  udata = new UserData;
  wdata = new WatchData;
  mdata = new MessageData;
  clist = new Clients;

  // init signal handler
  if (signal(SIGTERM, (void *) &termination_handler) == SIG_IGN)   // handle SIGTERM
    { signal(SIGTERM, SIG_IGN); }
  if (signal(SIGINT, (void *) &termination_handler) == SIG_IGN)   // handle SIGINT
    { signal(SIGINT, SIG_IGN); }
  signal(SIGPIPE, SIG_IGN);                              // ignore SIGPIPE
  return;
}

Konserver::~Konserver()
{
  delete udata;
  delete clist;
  d_close();
}

int Konserver::readConfig( MyString config_file )
{
  char buf[256];
  short port = 0;
  MyString line, temp, net, mask;
  struct in_addr temp_addr;
  nr_rules = 0;
  ifstream config( config_file.c_str() );
  if( config.bad() )
    {
      fprintf(stderr,"Konserver::readConfig: Can't open config file. Giving up!\n");
      return -1;
    }
  do {
    config.getline(buf, 256);
    if( buf[0] == 0 || buf[0] == '#' )
      continue;
    line = buf;
    temp.assign(line, line.find('=')+1, line.length()-line.find('=')-1);
    if( line.substr(0, 4) == "port" )
      {
	istrstream((const char *) temp.c_str()) >> port;
	//cout << "port = " << port << endl;
      }
    else if( line.substr(0, 11) == "databasedir" )
      {
	databasedir = temp;
	//cout << "databasedir = " << databasedir << endl;
      }
    else if( line.substr(0, 7) == "ipallow")
      {
	net.assign(temp, 0, temp.find('/') );
	mask.assign(temp, temp.find('/')+1, temp.length()-temp.find('/')-1);

	//cout << "net = '" << net << "'  mask = '" << mask << "'" << endl;

#ifndef SUN_VERSION   
	if( !inet_aton( (const char *) net.c_str(), &temp_addr ) )
	  {
	    fprintf(stderr,"Syntax error in line: \n   %s\n", (const char *) line.c_str());
	    continue;
	  }
	ip_net[nr_rules] = temp_addr.s_addr; 
	if( !inet_aton( (const char *) mask.c_str(), &temp_addr ) )
	  {
	    fprintf(stderr,"Syntax error in line: \n   %s\n", (const char *) line.c_str());
	    continue;
	  }
	ip_mask[nr_rules] = temp_addr.s_addr;
#else
	ip_net[nr_rules] = inet_addr( (const char *) net );     // sun version
	ip_mask[nr_rules] = inet_addr( (const char *) mask );   // sun version
#endif
	nr_rules++;
      }
    else
      {
	fprintf(stderr, "Syntax error in config file. Giving up !\n");
	return -1;
      }
  } while( !config.eof() );
  if( !port )
    port = 3000;
  servername.sin_port = htons(port);
  keepname.sin_port = htons(port);
  if( !databasedir.length() )
    databasedir = ".";

  return 0;
}

int Konserver::termination_handler(int signum)
{
  if( signum == SIGTERM )
    {
      fprintf(stderr,"caught sigterm\n");
    }
  else
    {
      fprintf(stderr,"caught sigint\n");
    }
  cont_loop = 0;
}


int Konserver::loop()
{
  fd_set read_set;
  struct timeval timeout;
  int    retcode, allowed;
  short  bytes_read;
  size_t size = sizeof(clientname);
  int    status;
  MyString name;
  MyString passwd;
  char   buf[LL_SHORT+LL_LONG];
  cont_loop = 1;
  timeout.tv_sec = 0;
  while(cont_loop)
    {
      timeout.tv_sec = KEEP_ALIVE_TIME - timeout.tv_sec;
      timeout.tv_usec = 0;
      FD_ZERO(&read_set);
      FD_SET(serversock, &read_set);
      FD_SET(keepsocket, &read_set);
      retcode = select( FD_SETSIZE, &read_set, 0, 0, &timeout );
      if( retcode < 0 )
	{	 
	  if( errno == EINTR )
	    continue;
	  fprintf(stderr,"Konserver::loop: error in select call\n");
	  break;
	}
      if( timeout.tv_sec == 0 )
	{
	  // timeout
	  clist->processAliveCounter(wdata);
	  if( retcode == 0 )
	    continue;
	}
      if( FD_ISSET( serversock, &read_set) )
	{ // client request
	  if ( (status = accept (serversock, (struct sockaddr *) &clientname, &size)) < 0)
	    {
	      fprintf(stderr,"Konserver::loop: can't accept\n");
	      continue;
	    }
	  if( debug )
	    {
	      fprintf(stderr, "client connect from host: %s", inet_ntoa (clientname.sin_addr) );
	      fprintf(stderr, "   using port: %i\n", ntohs(clientname.sin_port) );
	    }
	  allowed = 0;
	  if( (allowed = connectClient(status)) == -1 )
	    {
	      if( debug )
		fprintf(stderr, "Konserver::loop: client not accepted\n");
	      shutdown(status, 2);
	      close(status);
	      if( debug )
		fprintf(stderr,"Konserver::loop: connection closed\n");
	      continue;
	    }
	  parseRequest(allowed);
	  shutdown(status, 2);
	  close(status);
	  if( debug )
	    fprintf(stderr, "Konserver::loop: connection closed\n");
	}
      if( FD_ISSET( keepsocket, &read_set) )
	{ // client keep alive message
	  bytes_read = recvfrom( keepsocket, buf, LL_SHORT+LL_LONG, 0, 0, 0 );
	  if( bytes_read < 0 )
	    fprintf(stderr, "Konserver::loop: read error on keep alive socket\n");
	  short type;
 	  *((char *) &type) = buf[0];
	  *(((char *) &type)+1) = buf[1];
	  type = ntohs(type);
	  switch( ntohs(*((short *) buf)) ) {
	    case KEEP_ALIVE:
	      unsigned long id;
	      *((char *) &id) = buf[2];
	      *(((char *) &id)+1) = buf[3];
	      *(((char *) &id)+2) = buf[4];
	      *(((char *) &id)+3) = buf[5];
	      id = ntohl(id);
	      if( debug )
		fprintf(stderr,"keep alive packet from %i\n", id);
              clist->gotKeepAlive(id);
	      break;
	  };
	}
    }
  clist->serverShutdown();
  shutdown(serversock, 2);
  close(serversock);
}

int Konserver::connectClient(int fd_client)
{
  // check allow list
  int i;
  short allowed = 0;
  for( i = 0; i < nr_rules; i++ )
    {
      if( (ip_mask[i] & clientname.sin_addr.s_addr) == ip_net[i] )
	{
	  allowed = 1;
	  break;
	}
    }
  // establish connection
  c_pipe = new PipeCom (fd_client);
  unsigned long id = 0;
  short length = 0;
  if( (length = c_pipe->recv()) <= 0 || length > 1024)
    {
      fprintf(stderr,"read error from client\n");
      return -1;
    }
  *c_pipe >> id;      // store client id
  if( !allowed && id == 0 )
    {
      //*c_pipe << (short) RE_ACCESS_DENIED;
      //c_pipe->send();
      cid = 0;
      return -2;
    }
  if( checkId(id) )
    {
      fprintf(stderr,"Konserver::connectClient: connect request from client with unknown id\n");
      *c_pipe << (short) RE_FAILURE;
      c_pipe->send();
      return -1;
    }
  // client accepted
  cid = id;
  return 0;
}

int Konserver::checkId(unsigned long id)
{
  if( id != 0 && udata->checkId( id ) != S_OKAY )
    return -1;
  return 0;
}

int Konserver::parseRequest(int allowed)
{
  short instruction;
  *c_pipe >> instruction;
  if( cid == 0 && instruction != NEW_CLIENT && instruction != GET_USER_INFO )
    {
      if( debug )
	cerr << "Konserver::parseRequest: client request with zero client id" << endl;
      *c_pipe << (short) RE_FAILURE;
      c_pipe->send();
      return -1;  // for unregistered clients only NEW_CLIENT and GET_USER_INFO is allowed
    }
  if( allowed == -2 && instruction == NEW_CLIENT )
    {
      if( debug )
	cerr << "Konserver::parseRequest: client request from invalid IP" << endl;
      *c_pipe << (short) RE_ACCESS_DENIED;
      c_pipe->send();
      return -1;
    }
  switch (instruction) {
  case NEW_CLIENT:
    return newClient();
    break;
  case DEL_CLIENT:
    return delClient();
    break;
  case CHANGE_CLIENT:
    return changeClient();
    break;
  case CONNECT:
    return connect();
    break;
  case DISCONNECT:
    return disconnect();
    break;
  case SEARCH_USER:
    return searchUser();
    break;
  case ADD_WATCHLIST:
    return addWatchlist();
    break;
  case REMOVE_WATCHLIST:
    return removeWatchlist();
    break;
  case SEND_MSG:
    return sendMsg();
  case GET_MSG:
    return getMsg();
  case GET_USER_INFO:
    return getUserInfo();
  };
}

int Konserver::newClient()
{
  if( cid != 0 )
    {
      *c_pipe << (short) RE_FAILURE << (short) RE_PROTOCOL;
      if( c_pipe->send() < 0 )
	return -1;
    }
  User rec;
  *c_pipe >> rec.name;
  *c_pipe >> rec.vname;
  *c_pipe >> rec.nick;
  *c_pipe >> rec.email;
  *c_pipe >> rec.passwd;
  if( udata->add(&rec) == S_OKAY )
    {
      *c_pipe << (short) RE_OK << (unsigned long) rec.id;
      if( c_pipe->send() < 0 )
	return -1;
    }
  else
    {
      *c_pipe << (short) RE_DUPKEY;
      if( c_pipe->send() < 0 )
	return -1;
    }
  return 0;
}

int Konserver::delClient()
{
  *c_pipe >> cpasswd;
fprintf(stderr,"passwd = %s", (const char *) cpasswd);
  if( udata->checkPasswd( cid, cpasswd ) != S_OKAY )
    {
      fprintf(stderr,"Konserver::delClient: delete request from client with wrong passwd\n");
      *c_pipe << (short) RE_PASSWD;
      c_pipe->send();
      return -1;
    }
  if( udata->remove(cid) == S_OKAY )
    {
      // remove watchlist
      wdata->removeAll(cid);
      // remove msglist
      mdata->removeAll(cid);
      *c_pipe << (short) RE_OK;
    }
  else
    {
      *c_pipe << (short) RE_FAILURE;
    }
  if( c_pipe->send() < 0 )
    return -1;
  return 0;
}

int Konserver::changeClient()
{
  // cerr << "changeClient" << endl;
  User rec;
  int rcode;
  *c_pipe >> cpasswd;
  if( udata->checkPasswd( cid, cpasswd ) != S_OKAY )
    {
      fprintf(stderr,"Konserver::changeClient: change request from client with wrong passwd\n");
      *c_pipe << (short) RE_PASSWD;
      c_pipe->send();
      return -1;
    }
  *c_pipe >> rec.name;
  *c_pipe >> rec.vname;
  *c_pipe >> rec.nick;
  *c_pipe >> rec.email;
  *c_pipe >> rec.passwd;
  rec.id = cid;
  if( (rcode = udata->changeRec(&rec)) == S_OKAY )
    {
      *c_pipe << (short) RE_OK;
    }
  else if( rcode == S_DUPLICATE )
    {
      *c_pipe << (short) RE_DUPKEY;
    }
  else
    {
      *c_pipe << (short) RE_FAILURE;
    }
  cerr << "sending" << endl;
  if( c_pipe->send() < 0 )
    return -1;
  return 0;
}

int Konserver::connect()
{
  unsigned long id;
  short port;
  User rec;
  *c_pipe >> cpasswd; // store client passwd
  if( udata->checkPasswd( cid, cpasswd ) != S_OKAY )
    {
      fprintf(stderr,"Konserver::connect: connect request from client with wrong passwd\n");
      *c_pipe << (short) RE_PASSWD;
      c_pipe->send();
      return -1;
    }
  *c_pipe >> port;
  if( clist->search(cid) != 0 )
    {
      clist->remove(cid);
    }
  clientname.sin_port = htons(port);
  clist->insert(cid, &clientname);
  if( wdata->firstWUserId(cid, &id) == S_OKAY )
    {
      do {
	udata->findId( &rec, id );
	*c_pipe << (short) RE_OK << rec.name
		<< rec.vname
		<< rec.nick
		<< rec.email
		<< rec.id;
	if( clist->search(id) )
	  *c_pipe << (char) 1;
	else
	  *c_pipe << (char) 0;
	if( c_pipe->send() < 0 )
	  return -1;
      } while( wdata->nextWUserId( cid, &id) == S_OKAY );
    }
  *c_pipe << (short) RE_NOTFOUND;
  if( c_pipe->send() < 0 )
    return -1;
  // send queued messages
  sendQueuedMsg();
  // notify other users
  if( wdata->firstWatchId(cid, &id) == S_OKAY )
    {
      do {
	clist->sendActive(cid, id, 1);
      } while( wdata->nextWatchId( cid, &id) == S_OKAY );
    }
  return 0;
}
 
int Konserver::disconnect()
{
  *c_pipe >> cpasswd; // store client passwd
  if( udata->checkPasswd( cid, cpasswd ) != S_OKAY )
    {
      fprintf(stderr,"Konserver::disconnect: disconnect request from client with wrong passwd\n");
      *c_pipe << (short) RE_PASSWD;
      c_pipe->send();
      return -1;
    }
  if( clist->search(cid) != 0 )
    {
      clist->remove(cid);
      *c_pipe << (short) RE_OK;
    }
  else
    {
      *c_pipe << (short) RE_FAILURE;
    }
  if( c_pipe->send() < 0 )
    return -1;
  // notify other users
  unsigned long id;
  if( wdata->firstWatchId(cid, &id) == S_OKAY )
    {
      do {
	clist->sendActive(cid, id, 0);
      } while( wdata->nextWatchId( cid, &id) == S_OKAY );
    }
  return 0;
}

int Konserver::searchUser()
{
  int rcode;
  short type;
  int key;
  MyString kvalue;
  *c_pipe >> type;
  User rec, srec;
  *c_pipe >> rec.name;
  *c_pipe >> rec.vname;
  *c_pipe >> rec.nick;
  *c_pipe >> rec.email;
  *c_pipe >> rec.id;

  short l_name = rec.name.length();
  short l_vname = rec.vname.length();
  short l_nick = rec.nick.length();
  short l_email = rec.email.length();

  switch(type) {
  case SE_NAME:
    rcode = udata->findName(&srec, (const char *) rec.name.c_str());
    key = NAME;
    kvalue = rec.name;
    break;
  case SE_NNAME:
    rcode = udata->findNick(&srec, (const char *) rec.nick.c_str());
    key = NICK;
    kvalue = rec.nick;
    break;
  case SE_EMAIL:
    rcode = udata->findEmail(&srec, (const char *) rec.email.c_str());
    key = EMAIL;
    kvalue = rec.email;
    break;
  case SE_ID:
    rcode = udata->findId(&srec, rec.id);
    key = USER_ID;
    break;
  };
  if( rcode == S_NOTFOUND )
    {
      *c_pipe << (short) RE_NOTFOUND;
      if( c_pipe->send() < 0 )
	return -1;
    }
  else
    {
      if( type == SE_ID )
	{
	  *c_pipe << (short) RE_OK << srec.name
		  << srec.vname
		  << srec.nick
		  << srec.email
		  << srec.id;
	  if( c_pipe->send() < 0 )
	    return -1;
	  *c_pipe << (short) RE_NOTFOUND;
	  if( c_pipe->send() < 0 )
	    return -1;
	}
      else
	{
	  int nr = 0;
	  do {
	    nr++;
	    if( l_name )
	      //{ if( rec.name != srec.name.before(l_name) )
	      { if( rec.name != srec.name.substr(0, l_name) )
		continue; }
	    if( l_vname )
	      //{ if( rec.vname != srec.vname.before(l_vname) )
	      { if( rec.vname !=  srec.vname.substr(0, l_vname) )
		continue; }
	    if( l_nick )
	      //{ if( rec.nick != srec.nick.before(l_nick) )
	      { if( rec.nick != srec.nick.substr(0, l_nick) )
		continue; }
	    if( l_email )
	      //{ if( rec.email != srec.email.before(l_email) )
	      { if( rec.email != srec.email.substr(0, l_email) )
		continue; }
	    *c_pipe << (short) RE_OK << srec.name
		    << srec.vname
		    << srec.nick
		    << srec.email
		    << srec.id;
	    if( c_pipe->send() < 0 )
	      return -1;
	  } while( udata->findNext(&srec, key, kvalue) == S_OKAY && nr < 101 );
	  *c_pipe << (short) RE_NOTFOUND;
	  if( c_pipe->send() < 0 )
	    return -1;
	}
    }
  return 0;
}

int Konserver::addWatchlist()
{
  short rcode;
  unsigned long new_id;
  *c_pipe >> cpasswd;
  if( udata->checkPasswd( cid, cpasswd ) != S_OKAY )
    {
      fprintf(stderr,"Konserver::addWatchList: addWatchList request with wrong passwd\n");
      *c_pipe << (short) RE_PASSWD;
      c_pipe->send();
      return -1;
    }
  *c_pipe >> new_id;
  rcode = wdata->add(cid, new_id);
  if( rcode != S_OKAY )
    *c_pipe << (short) RE_FAILURE;
  else
    *c_pipe << (short) RE_OK;
  if( c_pipe->send() < 0 )
    return -1;
  return 0;
}

int Konserver::removeWatchlist()
{
  short rcode;
  unsigned long id;
  *c_pipe >> cpasswd;
  if( udata->checkPasswd( cid, cpasswd ) != S_OKAY )
    {
      fprintf(stderr,"Konserver::removeWatchList: removeWatchList request with wrong passwd\n");
      *c_pipe << (short) RE_PASSWD;
      c_pipe->send();
      return -1;
    }
  *c_pipe >> id;
  rcode = wdata->remove(cid, id);
  if( rcode != S_OKAY )
    *c_pipe << (short) RE_FAILURE;
  else
    *c_pipe << (short) RE_OK;
  if( c_pipe->send() < 0 )
    return -1;
  return 0;
}

int Konserver::sendMsg()
{
  User rec;
  short rcode;
  unsigned long id;
  *c_pipe >> id;
  MyString msg;
  *c_pipe >> msg;
  MyString date;
  *c_pipe >> date;
  if( udata->findId( &rec, cid ) == S_OKAY )
    {
      rcode = clist->sendMsg(cid, id, msg, &rec, date);
      switch( rcode ) {
      case RE_OK:
	*c_pipe << (short) RE_MSG_DELIVERED;	    
	break;
      case RE_RECV_BUF_EXCEEDED:
	if( mdata->add(cid, id, msg.c_str(), date.c_str()) == S_OKAY )
	  *c_pipe << (short) RE_MSG_QUEUED;
	else
	  *c_pipe << (short) RE_FAILURE;
	break;
      default:
	*c_pipe << (short) RE_FAILURE;
      };
    }
  else
    *c_pipe << (short) RE_FAILURE;
  if( c_pipe->send() < 0 )
    return -1;
  return 0;
}

int Konserver::getMsg()
{
  *c_pipe << (short) RE_OK;
  if( c_pipe->send() < 0 )
    return -1;
  sendQueuedMsg();
  return 0;
}

int Konserver::sendQueuedMsg()
{
  User rec;
  MyString msg;
  MyString date;
  unsigned long id, msg_id;
  while( mdata->firstMsg( cid, &id, &msg, &msg_id, &date ) == S_OKAY)
    {
      if( udata->findId( &rec, id ) != S_OKAY )
	{
	  rec.name = "Unknown";
	  rec.vname = "";
	  rec.nick = "";
	  rec.email = "";
	  rec.id = id;
	}
      if( clist->sendMsg( id, cid, msg, &rec, date ) == RE_OK )
	{
	  if( mdata->remove(msg_id) != S_OKAY )
	    break;
	}
      else
	break;
    }
  return 0;
}

int Konserver::getUserInfo()
{
  User rec;
  unsigned long id;
  *c_pipe >> cpasswd;
  *c_pipe >> id;
  if( udata->checkPasswd( id, cpasswd ) != S_OKAY )
    {
      fprintf(stderr,"Konserver::getUserInfo: query from client with wrong passwd\n");
      *c_pipe << (short) RE_PASSWD;
      c_pipe->send();
      return -1;
    }
  cout << "id = " << id << endl;
  if( udata->findId( &rec, id ) == S_OKAY )
    {
      *c_pipe << (short) RE_OK << rec.name
	      << rec.vname
	      << rec.nick
	      << rec.email
	      << rec.id;
    }
  else
    {
      *c_pipe << (short) RE_FAILURE;
    }
  if( c_pipe->send() < 0 )
    return -1;
  return 0;
}

/*
void Konserver::exportUserDatabase()
{
  User rec;
  if( udata->getFirst(&rec) == RE_OK )
    {
      do {
	cout << rec.id << endl
	     << rec.name << endl
	     << rec.vname << endl
	     << rec.nick << endl
	     << rec.email << endl;
      } while( udata->getNext(&rec) == RE_OK );
    }
}

void Konserver::importUserDatabase()
{

}
*/

