/*
 * md5.cc
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 ***********************************************************************
 ** Copyright (C) 1990, RSA Data Security, Inc. All rights reserved.  **
 **                                                                   **
 ** License to copy and use this software is granted provided that    **
 ** it is identified as the "RSA Data Security, Inc. MD5 Message-     **
 ** Digest Algorithm" in all material mentioning or referencing this  **
 ** software or this function.                                        **
 **                                                                   **
 ** License is also granted to make and use derivative works          **
 ** provided that such works are identified as "derived from the RSA  **
 ** Data Security, Inc. MD5 Message-Digest Algorithm" in all          **
 ** material mentioning or referencing the derived work.              **
 **                                                                   **
 ** RSA Data Security, Inc. makes no representations concerning       **
 ** either the merchantability of this software or the suitability    **
 ** of this software for any particular purpose.  It is provided "as  **
 ** is" without express or implied warranty of any kind.              **
 **                                                                   **
 ** These notices must be retained in any copies of any part of this  **
 ** documentation and/or software.                                    **
 ***********************************************************************
 */

#include <iostream.h>
#include "md5.h"

// F, G, H and I are basic MD5 functions 
#define F(x, y, z) (((x) & (y)) | ((~x) & (z)))
#define G(x, y, z) (((x) & (z)) | ((y) & (~z)))
#define H(x, y, z) ((x) ^ (y) ^ (z))
#define I(x, y, z) ((y) ^ ((x) | (~z)))

// ROTATE_LEFT rotates x left n bits 
#define ROTATE_LEFT(x, n) (((x) << (n)) | ((x) >> (32-(n))))

// FF, GG, HH, and II transformations for rounds 1, 2, 3, and 4 
// Rotation is separate from addition to prevent recomputation
#define FF(a, b, c, d, x, s, ac) \
  {(a) += F ((b), (c), (d)) + (x) + (UINT32)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

#define GG(a, b, c, d, x, s, ac) \
  {(a) += G ((b), (c), (d)) + (x) + (UINT32)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

#define HH(a, b, c, d, x, s, ac) \
  {(a) += H ((b), (c), (d)) + (x) + (UINT32)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

#define II(a, b, c, d, x, s, ac) \
  {(a) += I ((b), (c), (d)) + (x) + (UINT32)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

// Constructor
MD5::MD5()
{}

// Destructor
MD5::~MD5()
{}

void MD5::hash(UINT8 *source, UINT32 len, UINT8 *dest)
{
  UINT32 total_len = len;
  
  init();
  
  while (len >= 64)
    {
      fillBuffer(source, 16);
      hashOneBlock();
   
      len -= 64;
      source += 64;
    }
  
  padding(source, len, total_len);
  fillBuffer((UINT8*) in, 16);
  hashOneBlock();
  
  fillBuffer((UINT8*) in, 16);
  memcpy(dest, buffer, 16);
};

void MD5::padding(UINT8 *source, UINT32 len, UINT32 total_len)
{
  UINT8  *input = (UINT8*) in;
  UINT32 count  = 64 - len - 1;
  
  memcpy(input, source, len);
  
  input[len] = 0x80;
  
  if (count < 8)
    {
      memset(input+len+1, 0, count);
      fillBuffer(input, 16);
      hashOneBlock();
      memset(input, 0, 56);
    }
  else
    {
      memset(input+len+1, 0, count - 8);
    }
    
  in[14] = total_len << 3 ;
  in[15] = total_len >> 29;
};

void MD5::init()
{
    buffer[0] = 0x67452301;
    buffer[1] = 0xefcdab89;
    buffer[2] = 0x98badcfe;
    buffer[3] = 0x10325476;
}

void MD5::fillBuffer(unsigned char *buf, unsigned longs)
{
#ifdef __i386__
  int i=0;
  
  do {
       in[i++] = (UINT32) ((unsigned) buf[3] << 8 | buf[2]) << 16 |
                 ((unsigned) buf[1] << 8 | buf[0]);
       buf += 4;	
  } while (i<longs);
#else
  memcpy((UINT8*) in, buf, 4*longs);
#endif
};
    
void MD5::hashOneBlock()
{
  int i;
  
  register UINT32 a = buffer[0];
  register UINT32 b = buffer[1];
  register UINT32 c = buffer[2];
  register UINT32 d = buffer[3];
  
  // Round 1
  #define S11 7
  #define S12 12
  #define S13 17
  #define S14 22
  
  FF ( a, b, c, d, in[ 0], S11, 0xD76AA478L); // 1
  FF ( d, a, b, c, in[ 1], S12, 0xE8C7B756L); // 2 
  FF ( c, d, a, b, in[ 2], S13, 0x242070DBL); // 3
  FF ( b, c, d, a, in[ 3], S14, 0xC1BDCEEEL); // 4
  FF ( a, b, c, d, in[ 4], S11, 0xF57C0FAFL); // 5
  FF ( d, a, b, c, in[ 5], S12, 0x4787C62AL); // 6
  FF ( c, d, a, b, in[ 6], S13, 0xA8304613L); // 7 
  FF ( b, c, d, a, in[ 7], S14, 0xFD469501L); // 8
  FF ( a, b, c, d, in[ 8], S11, 0x698098D8L); // 9 
  FF ( d, a, b, c, in[ 9], S12, 0x8B44F7AFL); // 10 
  FF ( c, d, a, b, in[10], S13, 0xFFFF5BB1L); // 11 
  FF ( b, c, d, a, in[11], S14, 0x895CD7BEL); // 12 
  FF ( a, b, c, d, in[12], S11, 0x6B901122L); // 13
  FF ( d, a, b, c, in[13], S12, 0xFD987193L); // 14 
  FF ( c, d, a, b, in[14], S13, 0xA679438EL); // 15 
  FF ( b, c, d, a, in[15], S14, 0x49B40821L); // 16 

  // Round 2 
  #define S21 5
  #define S22 9
  #define S23 14
  #define S24 20
  
  GG ( a, b, c, d, in[ 1], S21, 0xF61E2562L); // 17 
  GG ( d, a, b, c, in[ 6], S22, 0xC040B340L); // 18 
  GG ( c, d, a, b, in[11], S23, 0x265E5A51L); // 19 
  GG ( b, c, d, a, in[ 0], S24, 0xE9B6C7AAL); // 20 
  GG ( a, b, c, d, in[ 5], S21, 0xD62F105DL); // 21 
  GG ( d, a, b, c, in[10], S22, 0x02441453L); // 22
  GG ( c, d, a, b, in[15], S23, 0xD8A1E681L); // 23 
  GG ( b, c, d, a, in[ 4], S24, 0xE7D3FBC8L); // 24 
  GG ( a, b, c, d, in[ 9], S21, 0x21E1CDE6L); // 25 
  GG ( d, a, b, c, in[14], S22, 0xC33707D6L); // 26
  GG ( c, d, a, b, in[ 3], S23, 0xF4D50D87L); // 27 
  GG ( b, c, d, a, in[ 8], S24, 0x455A14EDL); // 28 
  GG ( a, b, c, d, in[13], S21, 0xA9E3E905L); // 29
  GG ( d, a, b, c, in[ 2], S22, 0xFCEFA3F8L); // 30 
  GG ( c, d, a, b, in[ 7], S23, 0x676F02D9L); // 31
  GG ( b, c, d, a, in[12], S24, 0x8D2A4C8AL); // 32 

  // Round 3
  #define S31 4
  #define S32 11
  #define S33 16
  #define S34 23
  
  HH ( a, b, c, d, in[ 5], S31, 0xFFFA3942L); // 33
  HH ( d, a, b, c, in[ 8], S32, 0x8771F681L); // 34 
  HH ( c, d, a, b, in[11], S33, 0x6D9D6122L); // 35 
  HH ( b, c, d, a, in[14], S34, 0xFDE5380CL); // 36 
  HH ( a, b, c, d, in[ 1], S31, 0xA4BEEA44L); // 37
  HH ( d, a, b, c, in[ 4], S32, 0x4BDECFA9L); // 38 
  HH ( c, d, a, b, in[ 7], S33, 0xF6BB4B60L); // 39
  HH ( b, c, d, a, in[10], S34, 0xBEBFBC70L); // 40 
  HH ( a, b, c, d, in[13], S31, 0x289B7EC6L); // 41 
  HH ( d, a, b, c, in[ 0], S32, 0xEAA127FAL); // 42
  HH ( c, d, a, b, in[ 3], S33, 0xD4EF3085L); // 43
  HH ( b, c, d, a, in[ 6], S34, 0x04881D05L); // 44
  HH ( a, b, c, d, in[ 9], S31, 0xD9D4D039L); // 45 
  HH ( d, a, b, c, in[12], S32, 0xE6DB99E5L); // 46
  HH ( c, d, a, b, in[15], S33, 0x1FA27CF8L); // 47 
  HH ( b, c, d, a, in[ 2], S34, 0xC4AC5665L); // 48 

  // Round 4
  #define S41 6
  #define S42 10
  #define S43 15
  #define S44 21
  
  II ( a, b, c, d, in[ 0], S41, 0xF4292244L); // 49
  II ( d, a, b, c, in[ 7], S42, 0x432AFF97L); // 50 
  II ( c, d, a, b, in[14], S43, 0xAB9423A7L); // 51 
  II ( b, c, d, a, in[ 5], S44, 0xFC93A039L); // 52 
  II ( a, b, c, d, in[12], S41, 0x655B59C3L); // 53
  II ( d, a, b, c, in[ 3], S42, 0x8F0CCC92L); // 54
  II ( c, d, a, b, in[10], S43, 0xFFEFF47DL); // 55
  II ( b, c, d, a, in[ 1], S44, 0x85845DD1L); // 56
  II ( a, b, c, d, in[ 8], S41, 0x6FA87E4FL); // 57
  II ( d, a, b, c, in[15], S42, 0xFE2CE6E0L); // 58 
  II ( c, d, a, b, in[ 6], S43, 0xA3014314L); // 59 
  II ( b, c, d, a, in[13], S44, 0x4E0811A1L); // 60 
  II ( a, b, c, d, in[ 4], S41, 0xF7537E82L); // 61
  II ( d, a, b, c, in[11], S42, 0xBD3AF235L); // 62
  II ( c, d, a, b, in[ 2], S43, 0x2AD7D2BBL); // 63
  II ( b, c, d, a, in[ 9], S44, 0xEB86D391L); // 64

  buffer[0] += a; 
  buffer[1] += b; 
  buffer[2] += c; 
  buffer[3] += d; 
};