#include "kflistbase.h"
#include <iostream.h>
#include <stdlib.h>
#include <qdragobject.h>
#include <qkeycode.h>
#include <qbitmap.h>
#include <ctype.h>
#include <kapp.h>

#include "mystrlist.h"

KFileListBase::KFileListBase(QWidget *parent, const char *name, WFlags f)
	: QTableView(parent,name,f), QDropSite(this)
{
	setBackgroundMode(PaletteBase);
	setNumRows(0);
	setNumCols(0);
	FileList = 0;
	LinkArrow = 0;
	TargetItem = 0;
	Selected = new FileInfoList;
	Selected->setAutoDelete(FALSE);
	Edit = new KFileEdit("",this);
	Edit->hide();
	CellEdited = 0;
	connect(Edit,SIGNAL(cancelEdit()),SLOT(cancelEdit()));
	connect(Edit,SIGNAL(terminateEdit()),SLOT(terminateEdit()));
	connect((QWidget*)horizontalScrollBar(),SIGNAL(valueChanged(int)),SLOT(cancelEdit()));
	connect((QWidget*)horizontalScrollBar(),SIGNAL(sliderMoved(int)),SLOT(cancelEdit()));
	connect((QWidget*)verticalScrollBar(),SIGNAL(valueChanged(int)),SLOT(cancelEdit()));
	connect((QWidget*)verticalScrollBar(),SIGNAL(sliderMoved(int)),SLOT(cancelEdit()));
	state = FILEVIEW_NORMAL;
	press_x = press_y = 0;
	setFocusPolicy(QWidget::StrongFocus);
	ToolTip = new KFileListToolTip(this);
}

KFileListBase::~KFileListBase()
{
	delete Edit;
	delete Selected;
	delete ToolTip;
}

void KFileListBase::setFileList(FileInfoList *list, bool keepSel)
{
	cancelEdit();
	FileList = list;
	if (!keepSel) Selected->clear();
	setUpdatesEnabled(false);
	updateListSize();
	setUpdatesEnabled(true);
	update();
}

void KFileListBase::mousePressEvent(QMouseEvent *e)
{
	cancelEdit();
	pressed = false;
	state = FILEVIEW_NORMAL;
	int	row = findRow(e->y()), col = findCol(e->x());
	FileInfo	*current = 0;
	bool		itemSelected;
	if (row < 0 || col < 0) itemSelected = FALSE;
	else itemSelected = getItemCell(row,col,&current,QPoint(e->x(),e->y()));
	emit selected(this);
	if (e->button() == LeftButton) {
		pressed = true;
		if (itemSelected) {
			if ((e->state() & ShiftButton) && Selected->count() > 0) {
				FileInfoList	list;
				list.setAutoDelete(false);
				FileInfo	*begin = Selected->first();
				int	i = begin->index();
				if (begin->index() < current->index())
					for (;i<=current->index();i++) list.append(FileList->at(i));
				else
					for (;i>=current->index();i--) list.append(FileList->at(i));
				setSelection(list);
			}
			else if (e->state() & ControlButton) {
				if (Selected->findRef(current) != -1) removeFromSelection(current);
				else addToSelection(current);
			}
			else {
				if (Selected->findRef(current) != -1) {
					ItemClicked = current;
					state = FILEVIEW_CANDRAG;
					if (Selected->count() == 1) state = (StateType)(state|FILEVIEW_CANEDIT);
				}
				else {
					removeSelection();
					addToSelection(current);
					state = FILEVIEW_CANDRAG;
					press_x = e->x();
					press_y = e->y();
					ItemClicked = current;
				}
			}
		}
		else if (!(e->state() & ControlButton) && !(e->state() & AltButton)) {
			if (!(e->state() & ShiftButton)) removeSelection();
			state = FILEVIEW_DORECTANGLE;
			anchorPoint = e->pos();
			anchorRect = QRect(anchorPoint,QSize(0,0));
		}
	}
	else if (e->button() == RightButton) {
		if (!itemSelected) removeSelection();
		else if (Selected->findRef(current) == -1) {
			removeSelection();
			addToSelection(current);
		}
		emit popup();
	}
	else if (e->button() == MidButton && itemSelected) {
		removeSelection();
		addToSelection(current);
		emit openItem(current);
	}
}

void KFileListBase::mouseMoveEvent(QMouseEvent *e)
{
	if ((state & FILEVIEW_CANDRAG) && Selected->count() > 0 && pressed) {
		if (abs(e->x() - press_x) + abs(e->y() - press_y) > 10) {
			FileInfoListIterator	it(*Selected);
			MyStrList		list;
			for (;it.current();++it) list.append(it.current()->absFilePath());
			QUrlDrag	*object = new QUrlDrag(list,this);
			if (Selected->count() == 1) object->setPixmap(*(Selected->first()->miniIcon()));
			else if (DNDIcon) object->setPixmap(*DNDIcon);
			state = FILEVIEW_DRAGGING;
			object->drag();
			state = FILEVIEW_NORMAL;
		}
	}
	else if (state == FILEVIEW_DORECTANGLE) {
		drawFancyRect();
		QPoint	p1(QMAX(QMIN(anchorPoint.x(),e->pos().x()),0),QMAX(QMIN(anchorPoint.y(),e->pos().y()),0));
		QPoint	p2(QMIN(QMAX(anchorPoint.x(),e->pos().x()),width()),QMIN(QMAX(anchorPoint.y(),e->pos().y()),height()));
		anchorRect = QRect(p1,p2);
		drawFancyRect();
	}
	else {
		FileInfo	*item;
		bool		itemSelected = getItem(e->pos(),&item);
		if (itemSelected && !pressed) setCursor(upArrowCursor);
		else setCursor(arrowCursor);
	}
}

void KFileListBase::mouseReleaseEvent(QMouseEvent *e)
{
	pressed = false;
	if (state == FILEVIEW_DORECTANGLE) {
		drawFancyRect();
		state = FILEVIEW_NORMAL;
		FileInfoListIterator	it(*FileList);
		for (;it.current();++it) if (anchorRect.intersects(getItemRect(it.current())) && Selected->findRef(it.current()) == -1) Selected->append(it.current());
		updateSelection();
	}
	else if (state & FILEVIEW_CANEDIT)
		startRenaming(Selected->first());
	else if (e->button() == LeftButton && e->state() == 1 && (int)(Selected->count()) == 1) {
		kapp->getConfig()->setGroup("Configuration");
		if (kapp->getConfig()->readBoolEntry("SingleClick",false)) emit openItem(Selected->first());
	}
}

void KFileListBase::mouseDoubleClickEvent(QMouseEvent *e)
{
	int	col = findCol(e->x()), row = findRow(e->y());
	FileInfo	*current = 0;
	bool		itemSelected;
	if (row < 0 || col < 0) itemSelected = FALSE;
	else itemSelected = getItemCell(row,col,&current,QPoint(e->x(),e->y()));
	if (current && itemSelected && e->button() == LeftButton) {
		emit openItem(current);
	}
}

void KFileListBase::removeSelection()
{
	if (Selected->count() == 0) return;
	FileInfo	*item = Selected->first();
	while (item) {
		Selected->remove();
		updateItem(item);
		item = Selected->first();
	}
	emit selectionChanged();
}

void KFileListBase::setLinkArrow(QPixmap *larrow)
{
	LinkArrow = larrow;
}

void KFileListBase::setDNDIcon(QPixmap *dndicon)
{
	DNDIcon = dndicon;
}

void KFileListBase::addToSelection(FileInfo *item)
{
	if (Selected->findRef(item) == -1) {
		Selected->append(item);
		updateItem(item);
		emit selectionChanged();
	}
}

void KFileListBase::removeFromSelection(FileInfo *item)
{
	Selected->removeRef(item);
	updateItem(item);
	emit selectionChanged();
}

void KFileListBase::selectAll()
{
	if (FileList != 0) {
		Selected->clear();
		FileInfoListIterator	it(*FileList);
		for ( ;it.current();++it) Selected->append(it.current());
		update();
		emit selectionChanged();
	}
}

void KFileListBase::cancelEdit()
{
	if (state == FILEVIEW_EDITING) {
		Edit->releaseKeyboard();
		int	row = 0, col = 0;
		getTablePosition(CellEdited->index(),row,col);
		Edit->hide();
		updateCell(row,col);
		state = FILEVIEW_NORMAL;
	}
}

void KFileListBase::terminateEdit()
{
	if (state == FILEVIEW_EDITING) {
		Edit->releaseKeyboard();
		emit renameItem(CellEdited,QString(Edit->text()));
		Edit->hide();
		state = FILEVIEW_NORMAL;
	}
}

void KFileListBase::setSelection(FileInfoList& list)
{
	if (Selected->isEmpty() && list.isEmpty()) return;	// nothing to do, it reduces flickering
	removeSelection();
	FileInfoListIterator	it(list);
	for (;it.current();++it) {
		Selected->append(it.current());
		updateItem(it.current());
	}
	emit selectionChanged();
}

void KFileListBase::updateSelection()
{
	FileInfoListIterator	it(*Selected);
	for (;it.current();++it) updateItem(it.current());
}

void KFileListBase::focusInEvent(QFocusEvent *)
{ emit selected(this); updateSelection();}

void KFileListBase::focusOutEvent(QFocusEvent *)
{ updateSelection();}

void KFileListBase::dropEvent(QDropEvent *e)
{
	FileInfo	*current(0);
	bool		itemSelected = getItem(e->pos(),&current);
	if (itemSelected && current->isDir()) emit dropAccepted(e,current);
	else emit dropAccepted(e,0);
	if (TargetItem) {
		FileInfo	*old = TargetItem;
		TargetItem = 0;
		updateItem(old);
	}
}

bool KFileListBase::getItem(const QPoint& p, FileInfo **item)
{
	int	row = findRow(p.y()), col = findCol(p.x());
	*item = 0;
	bool		itemSelected;
	if (row < 0 || col < 0) itemSelected = FALSE;
	else itemSelected = getItemCell(row,col,item,p);
	return itemSelected;
}

void KFileListBase::updateItem(FileInfo *item)
{
	int	row(0),col(0);
	getTablePosition(item->index(),row,col);
	updateCell(row,col);
}

void KFileListBase::keyPressEvent(QKeyEvent *e)
{
	FileInfo	*current = (Selected->count() > 0 ? Selected->last() : 0);
	FileInfo	*next(0);
	if (current) FileList->at(current->index());
	switch (e->key()) {
	   case Key_Down:
	   case Key_Up:
	   case Key_PageUp:
	   case Key_PageDown:
	   case Key_Right:
	   case Key_Left:
		next = getNextItem(current,e->key());
		break;
	   case Key_Return:
		if (current) emit openItem(current);
		return;
	   case Key_Backspace:
		emit backspacePressed();
		return;
	   case Key_F2:
		if (Selected->count() == 1) startRenaming(current);
		break;
	   default:
		// the user can press a key to jump to the first file starting with
		// that letter (i.e. selecting it). Pressing the key again jumps to
		// the next file starting with that letter. When there are no more
		// matching files, the first file is selected again. Its case insensitive.
		// TODO:
		// -find something better than isalpha()/isdigit()
		//  (special characters are not accessable right now)
		// -maybe take care of shift etc, so even strange filenames starting with ,$,& etc.
		//  can be accessed by pressing their first character?
		if( isprint(e->ascii()) ) {
			int pos_now = 0;
			if( Selected->count() > 0 )
				pos_now = current->index();
			else
				pos_now = -1;
			int ct = 0;
			int last_item_matched = 0;
			int first_item_matched = -1;
			FileInfoListIterator it(*FileList);
			for (;it.current();++it) {
				if( it.current()->fileName().find(e->key(), 0, false) == 0 ) {
					last_item_matched = ct;
					if( first_item_matched == -1 )
						first_item_matched = ct;
					if( ct > pos_now )
						break;
				}
				ct++;
			}
			if( last_item_matched < ct && first_item_matched != -1 )
				next = FileList->at(first_item_matched);
			else if (ct >= 0 && ct < (int)(FileList->count()))
				next = FileList->at(ct);
		} else {
			// return in order to react to keys like shift, ctrl, ...
			return;
		}
	   	break;
	}
	if (next != current) {
		if (!(e->state() & ShiftButton)) removeSelection();
		if (next) {
			int	i1,i2;
			if (e->state() & ShiftButton) {
				i1 = QMIN(next->index(),(current ? current->index() : 0));
				i2 = QMAX(next->index(),(current ? current->index() : 0));
			}
			else i1 = i2 = next->index();
			for (int i=i1;i<=i2;i++) addToSelection(FileList->at(i));
			scroll(next);
		}
	}
}

void KFileListBase::scroll(FileInfo *item)
{}

void KFileListBase::dragEnterEvent(QDragEnterEvent *e)
{ if (QUrlDrag::canDecode(e)) e->accept();}

void KFileListBase::dragMoveEvent(QDragMoveEvent *e)
{
	if (!QUrlDrag::canDecode(e)) { e->ignore(); return;}
	FileInfo	*oldItem = TargetItem;
	bool		itemSelected = getItem(e->pos(),&TargetItem);
	if (!itemSelected || (TargetItem && !TargetItem->isDir())) { TargetItem = 0; itemSelected = false;}
	if (TargetItem != oldItem) {
		if (oldItem) updateItem(oldItem);
		if (TargetItem) updateItem(TargetItem);
	}
	if (state == FILEVIEW_DRAGGING) {
		if (itemSelected && Selected->findRef(TargetItem) == -1 && TargetItem->isDir()) e->accept();
		else e->ignore();
	}
	else e->accept();
}

void KFileListBase::dragLeaveEvent(QDragLeaveEvent *e)
{
	if (TargetItem) {
		FileInfo	*old(TargetItem);
		TargetItem = 0;
		updateItem(old);
	}
}

void KFileListBase::clearList()
{
}

QRect KFileListBase::tip(const QPoint& pos, QString& str)
{
	return QRect(0,0,-1,-1);
}

void KFileListBase::drawFancyRect()
{
	QPainter	p(this);
	p.setRasterOp(NotXorROP);
	p.drawRect(anchorRect);
	p.end();
//	cout << anchorRect.x() << "\t" << anchorRect.y() << "\t" << anchorRect.width() << "\t" << anchorRect.height() << endl;
}

void KFileListBase::startRenaming(FileInfo *item)
{
	CellEdited = item;
	Edit->setGeometry(getItemTextRect(item));
	Edit->setText(item->fileName().data());
	Edit->grabKeyboard();
	Edit->selectAll();
	Edit->show();
	Edit->setFocus();
	state = FILEVIEW_EDITING;
}

//----------------------------------------------------------------------------------------------------

QString getTruncatedString(const QString& str, QPainter *p, int w)
{
  if (p->fontMetrics().width(str.data()) < w) return str;
  int	w2 = w - p->fontMetrics().width("...");
  int	n(0), len(0);
  for (;n<(int)(str.length()) && len<w2;n++) len += p->fontMetrics().width(str[n]);
  QString	retval;
  if (n >= (int)(str.length())) retval = str.data();
  else retval = str.left(n-1) + "...";
  return retval;
}

//----------------------------------------------------------------------------------------------------

KFileListToolTip::KFileListToolTip(QWidget *parent)
	: QToolTip(parent)
{
}

void KFileListToolTip::maybeTip(const QPoint& pos)
{
	QString	tipStr;
	QRect	r = ((KFileListBase*)parentWidget())->tip(pos,tipStr);
	if (r.isValid()) {
		tip(r,tipStr.data());
	}
}
