#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <qmessagebox.h>
#include <kapp.h>
#include <iostream.h>
#include <qtextstream.h>
#include <qfile.h>

#if HAVE_SYS_VFSTAB_H
#include <sys/vfstab.h>
#include <sys/mnttab.h>
#else
//#include <fstab.h>
#include <mntent.h>
#endif

#include "protocol.h"
#include "mountprot.h"

// file where to store the output of the mounting commands
static QString	ResultFile = "/tmp/mountresult";

QString readFile(const char *file)
{
	QFile	f(file);
	QString	Buffer;
	if (!f.exists() || !f.open(IO_ReadOnly)) {
		warning("MountProtocol : can't open or read result file");
	}
	else {
		QTextStream	t(&f);
		do { Buffer += t.readLine(); if (!t.eof()) Buffer.append("\n");}
		while (!t.eof());
		f.close();
	}
	return Buffer;
}

//-------------------------------------------------------------------------------------

MountProtocol::MountProtocol(const char *mpoint, const char *device, const char *fstype)
	: FileProtocol(mpoint)
{
	Device = device;
	FSType = fstype;
	MountPoint = mpoint;
	Type = Protocol::Mount;
	// Try to get mount point if not specified (non-root)
	if (MountPoint.isEmpty() && !getMountPoint()) {
		// couldn't find any mount point, probably non-mountable device
		Mountable = false;
		Prefix = "";
	}
	else {
		// we got a mount point
		Mountable = true;
		Prefix = "file:";
		Prefix += MountPoint;
		// put a trailing "/" at the end
		if (Prefix.right(1) != "/") Prefix += "/";
	}
	Mounted = isMounted();
}


bool MountProtocol::isMounted()
{
#if HAVE_SYS_VFSTAB_H
	FILE	*fp = fopen(MNTTAB,"r");
	if (fp == 0) return FALSE;
	int	val;
	struct mnttab	buf, spec;
	mntnull(&spec);
	spec.mnt_special = Device.data();
	if (!MountPoint.isEmpty()) spec.mnt_mountp = MountPoint.data();
	if (!FSType.isEmpty()) spec.mnt_fstype = FSType.data();
	val = getmntany(fp,&buf,&spec);
	fclose(fp);
	return (val == 0);
#else
	FILE	*fp = setmntent(MOUNTED,"r");
	if (fp == 0) return FALSE;
	struct mntent	*buf;
	bool	done(FALSE);
	while (!done && (buf = getmntent(fp)) != 0) {
		if (buf->mnt_fsname == Device) done = TRUE;
		if (!MountPoint.isEmpty() && MountPoint.find(buf->mnt_dir) == -1) done = FALSE;
		if (!FSType.isEmpty() && FSType != buf->mnt_type) done = FALSE;
	}
	endmntent(fp);
	return done;
#endif
}

bool MountProtocol::getMountPoint()
{
#if HAVE_SYS_VFSTAB_H
	FILE	*fp = fopen(VFSTAB,"r");
	if (fp == 0) return FALSE;
	struct vfstab	buf;
	int	val = getvfsspec(fp,&buf,Device.data());
	fclose(fp);
	if (val == 0) MountPoint = buf.vfs_mountp;
	return (val == 0);
#else
	FILE	*fp = setmntent(MNTTAB,"r");
	if (fp == 0) return FALSE;
	struct mntent	*buf;
	bool	done(FALSE);
	while (!done && (buf = getmntent(fp)) != 0) {
		if (Device == buf->mnt_fsname) done = TRUE;
	}
	if (done) MountPoint = buf->mnt_dir;
	endmntent(fp);
	return done;
#endif
}

bool MountProtocol::mount()
{
	sendMessage("Mounting device...");
	if (!isMountable()) {
		ErrorMsg = i18n("Device is not mountable : check parameters");
		return FALSE;
	}
	QString		cmd("mount");
	int		result;
	if (!FSType.isEmpty()) { cmd += " -t "; cmd += FSType.data();}
	cmd += (" " + Device);
	if (!MountPoint.isEmpty() && getuid() == 0) cmd += (" " + MountPoint);
	cmd += " > " + ResultFile + " 2>&1";
cout << cmd << endl;
	if ((result=system(cmd.data())) != 0) {
		ErrorMsg = i18n("Unable to mount device");
		ErrorMsg.append(" :\n\n");
		ErrorMsg += readFile(ResultFile.data());
		::remove(ResultFile.data());
	}
	else Mounted = true;
	return (result == 0);
}

bool MountProtocol::umount()
{
	sendMessage("Unmounting device...");
	QString		cmd("umount ");
	int		result;
	cmd += Device + " > " + ResultFile + " 2>&1";
	if ((result=system(cmd.data())) != 0) {
		ErrorMsg = i18n("Unable to unmount device");
		ErrorMsg.append(" :\n\n");
		ErrorMsg + readFile(ResultFile.data());
		::remove(ResultFile.data());
	}
	else Mounted = false;
	return (result == 0);
}

bool MountProtocol::init()
{ return mount();}

bool MountProtocol::finish()
{ return umount();}

bool MountProtocol::shutDown()
{
	if (isMounted()) {
		QString		msg(Device.data());
		msg += i18n(" is mounted : unmount ?");
		if (QMessageBox::information((QWidget*)0,i18n("Confirmation"),msg.data(),i18n("Yes"),i18n("No"),0,0,1) == 0)
			if (!umount()) QMessageBox::critical((QWidget*)0,i18n("Error"),ErrorMsg.data(),QMessageBox::Ok | QMessageBox::Default,0);
	}
	return TRUE;
}

bool MountProtocol::isAccesible()
{ return isMounted();}

const char* MountProtocol::menuEntry()
{ return (isMounted() ? i18n("Unmount") : i18n("Mount"));}

bool MountProtocol::check()
{
	bool	old = Mounted;
	Mounted = isMounted();
	return (old == Mounted);
}