/*
 *   kscan - a scanning program
 *   Copyright (C) 1998 Ivan Shvedunov
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qvalidator.h>
#include "kscan.h"
#include "backend.h"
#include "preferences.h"

//KScan class implementation

KScan::KScan()
{
	kconf = kapp->getConfig();
	kconf->setGroup("kscan");
	max_dpi = kconf->readNumEntry(OP_MAXDPI,600);
	QString bke = kconf->readEntry(OP_BACKEND,"hp:/dev/scanner");
	strncpy(backend,bke.data(),255);
	acquire_at_start = kconf->readNumEntry(OP_ACQUIRE,FALSE);
	ScanImage::forceBrightnessEmulation(kconf->
					    readNumEntry(OP_EMBR,FALSE));
	ScanImage::forceContrastEmulation(kconf->
					  readNumEntry(OP_EMCN,FALSE));
	if(!ScanImage::InitScanner(backend))
		QMessageBox::critical(NULL,"Error",
				      "Scanner initialization failed");
	s = new Preview(this);
	current_dpi = 75;
	s->setGamma(171);
	QPopupMenu *file = new QPopupMenu;
	file->insertItem("&Acquire Preview",s,SLOT(zoomOut()));
	file->insertItem("Final &Scan",this,SLOT(finalScan()));
	file->insertSeparator();
	file->insertItem("&Quit",this,SLOT(die()));
	QPopupMenu *view = new QPopupMenu;
	view->insertItem("Zoom &In",s,SLOT(zoomIn()));
	view->insertItem("Zoom &Out",s,SLOT(zoomOut()));
	QPopupMenu *image = new QPopupMenu;
	image->insertItem("&Enhancement",this,SLOT(enhancement()));
	//Future extensions...
	image->setItemEnabled(image->insertItem("&Hue/Saturation"),FALSE);
	QPopupMenu *options = new QPopupMenu;
	options->insertItem("&Backend",this,SLOT(selectBackend()));
	options->insertItem("&Preferences",this,SLOT(preferences()));
	QPopupMenu *help = kapp->getHelpMenu(TRUE,"kscan 0.1.1\n\n"
					     "by Ivan Shvedunov "
					     "<ivan_iv@rf-hp.npi.msu.su");
	menubar = new KMenuBar(this);
	menubar->insertItem("&File",file);
	menubar->insertItem("&View",view);
	menubar->insertItem("&Image",image);
	menubar->insertItem("&Options",options);
	menubar->insertSeparator();
	menubar->insertItem("&Help",help);
	menubar->show();
	setMenu(menubar);
	KIconLoader *loader = kapp->getIconLoader();
	toolbar = new KToolBar(this);
	toolbar->insertButton(loader->loadIcon("viewzoom.xpm"),0,
			      SIGNAL(clicked()),s,SLOT(zoomOut()),
			      TRUE,"Acquire Preview");
	toolbar->insertButton(loader->loadIcon("final_scan.xpm"),
			      ID_FINAL_SCAN,
			      SIGNAL(clicked()),this,SLOT(finalScan()),
			      TRUE,"Final Scan");
	toolbar->insertButton(loader->loadIcon("viewmag+.xpm"),
			      ID_ZOOM_IN,
			      SIGNAL(clicked()),s,SLOT(zoomIn()),
			      TRUE,"Zoom In");
	toolbar->insertButton(loader->loadIcon("viewmag-.xpm"),
			      ID_ZOOM_OUT,
			      SIGNAL(clicked()),s,SLOT(zoomOut()),
			      TRUE,"Zoom Out");
	toolbar->insertButton(loader->loadIcon("enh.xpm"),4,
			      SIGNAL(clicked()),this,SLOT(enhancement()),
			      TRUE,"Image Enhancement");
	toolbar->insertButton(loader->loadIcon("hue.xpm"),5,FALSE,
			      "Hue/Saturation");
	addToolBar(toolbar);
	QIntValidator *vld = new QIntValidator(0,32767,this,"val");
	imgsize = new KToolBar(this);
	imgsize->insertLined("",ID_XDIM,SIGNAL(textChanged(const char *)),
			     this,SLOT(changed1(const char *)),
			     TRUE,"Width");
	imgsize->insertLined("",ID_YDIM,SIGNAL(textChanged(const char *)),
			     this,SLOT(changed2(const char *)),
			     TRUE,"Height");
	imgsize->insertLined("",ID_DPI,SIGNAL(textChanged(const char *)),
			     this,SLOT(changed3(const char *)),
			     FALSE,"Dots Per Inch");
	imgsize->insertButton(loader->loadIcon("reload.xpm"),3,
			      SIGNAL(clicked()),this,SLOT(updateDpi()),
			      TRUE,"Update \"Dots Per Inch\"");
	imgsize->insertCombo("Gray",ID_TYPE,FALSE,SIGNAL(highlighted(int)),
			     this,SLOT(changeType(int)),TRUE,"Image Type",80);
	ch = 1;
	imgsize->insertComboItem(ID_TYPE,"Color",1);
	imgsize->setCurrentComboItem(ID_TYPE,s->getType()==COLOR?1:0);
	if(!s->haveGrayMode()||!s->haveColorMode())
		imgsize->setItemEnabled(ID_TYPE,FALSE);
	ch = 0;
	KLined *xdim = imgsize->getLined(ID_XDIM);
	KLined *ydim = imgsize->getLined(ID_YDIM);
	xdim->setValidator(vld);
	xdim->setMaxLength(6);
	ydim->setValidator(vld);
	ydim->setMaxLength(6);
	addToolBar(imgsize);
	imgsize->setBarPos(KToolBar::Left);
	imgsize->show();
	toolbar->setBarPos(KToolBar::Top);
	toolbar->show();
	setView(s);
	show();
	disableDim();
	displayDpi();
	ch = 0;
        connect(s,SIGNAL(noRect()),SLOT(disableDim()));
        connect(s,SIGNAL(newRect()),SLOT(enableDim()));
	if(acquire_at_start) s->zoomOut();
}

void KScan::selectBackend()
{
	BackendDialog d(kconf,backend,TRUE);
	d.exec();
}

void KScan::preferences()
{
	Preferences d(kconf,&acquire_at_start,&max_dpi);
	d.exec();
}

void KScan::closeEvent(QCloseEvent *ev)
{
	printf("CloseEvent\n");
	delete toolbar;
	delete imgsize;
	delete menubar;
	kapp->quit();
	ev->accept();
}

void KScan::enableDim()
{
	char buf[100];
	sprintf(buf,"%d",(int)(s->xInches()*(double)current_dpi));
	imgsize->setLinedText(ID_XDIM,buf);
	imgsize->setItemEnabled(ID_XDIM,TRUE);
	ch = 1;
	sprintf(buf,"%d",(int)(s->yInches()*(double)current_dpi));
	imgsize->setLinedText(ID_YDIM,buf);
	imgsize->setItemEnabled(ID_YDIM,TRUE);
	if(s->scannerPresent()) {
		toolbar->setItemEnabled(ID_FINAL_SCAN,TRUE);
		toolbar->setItemEnabled(ID_ZOOM_IN,TRUE);
		toolbar->setItemEnabled(ID_ZOOM_OUT,TRUE);
	}
	ch = 0;
	changed = 0;
	present = 1;
}

void KScan::disableDim()
{
	imgsize->setItemEnabled(ID_XDIM,FALSE);
	imgsize->setItemEnabled(ID_YDIM,FALSE);
	toolbar->setItemEnabled(ID_FINAL_SCAN,FALSE);
	toolbar->setItemEnabled(ID_ZOOM_IN,FALSE);
	toolbar->setItemEnabled(ID_ZOOM_OUT,FALSE);
	ch = 1;
	imgsize->setLinedText(ID_XDIM,"");
	imgsize->setLinedText(ID_YDIM,"");
	ch = 0;
	present = 0;
}

void KScan::changed1(const char *str)
{
	if(ch) return;
	int v = atoi(str);
	char buf[100];
	sprintf(buf,"%d",(v*s->sel()->height())/s->sel()->width());
	ch = 1;
	imgsize->setLinedText(ID_YDIM,buf);
	ch = 0;
	changed = 1;
}

void KScan::changed2(const char *str)
{
	if(ch) return;
	int v = atoi(str);
	char buf[100];
	sprintf(buf,"%d",(v*s->sel()->width())/s->sel()->height());
	ch = 1;
	imgsize->setLinedText(ID_XDIM,buf);
	ch = 0;
	changed = 1;
}

void KScan::updateDpi()
{
	if(!changed||!present) return;
	current_dpi = s->calculateDpi(atoi(imgsize->getLinedText(ID_XDIM)));
	printf("current_dpi %d\n",current_dpi);
	if(current_dpi>max_dpi) current_dpi = max_dpi; //FIXME
	else if(current_dpi<12) current_dpi = 12;
	enableDim();
	displayDpi();
}

main(int argc,char **argv)
{
	KApplication a(argc,argv,"kscan");
	new KScan;
	return a.exec();
}

#include "kscan.moc"
