/***************************************************************************
                          ksnuffle.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include	<stdio.h>
#include	<stdarg.h>
#include	<errno.h>
#include	<unistd.h>
#include	<pwd.h>

#include	<qarray.h>
#include	<qmessagebox.h>


#include	"ksnuffle.h"
#include	"knd_multi.h"
#include	"knd_setup.h"
#include	"knd_view.h"
#include	"knd_mon.h"
#include	"knd_replay.h"
#include	"resource.h"

#define	STRERR	strerror(errno)

static	KNDApp	*kndApp	;

/*  QCBFindItem	: Find item in a combobox				*/
/*  combo	: QComboBox &	: Combbox in question			*/
/*  name	: const char *	: Item required				*/
/*  defv	: int		: Result if not found			*/
/*  (returns)	: int		: Position				*/

int	QCBFindItem
	(	QComboBox	&combo,
		const	char	*name,
		int		defv
	)
{
	for (int slot = 0 ; slot < combo.count() ; slot += 1)
		if (strcmp (name, combo.text(slot)) == 0)
			return	slot ;

	return	defv ;
}

/*  QLBFindItem	: Find item in a listbox				*/
/*  list	: QListBox &	: Listbox in question			*/
/*  name	: const char *	: Item required				*/
/*  defv	: int		: Result if not found			*/
/*  (returns)	: int		: Position				*/

int	QLBFindItem
	(	QListBox	&list,
		const	char	*name,
		int		defv
	)
{
	for (unsigned int slot = 0 ; slot < list.count() ; slot += 1)
		if (strcmp (name, list.text(slot)) == 0)
			return	slot ;

	return	defv ;
}


/*  ErrExit	: Error exit						*/
/*  fmt		: char *	: Message format			*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: void		: Never returns				*/
	
void	ErrExit
	(	char	*fmt,
		...
	)
{
	va_list	  aptr 		;
	char	  buff[256]	;

	va_start (aptr, fmt) 	;
	vsprintf (buff, fmt, aptr)   ;

	QMessageBox mb ("Ksnuffle interface error",
			buff,
			QMessageBox::Critical,
			QMessageBox::Ok, 0, 0) ;
	mb.exec ( ) ;
	exit	(1) ;
}

/*  Error	: Display error popup					*/
/*  title	: const char *	: Popup box title			*/
/*  message	: const char *	: Error message				*/
/*  icon	: Icon		: Icon type				*/
/*  (returns)	: void		:					*/

void	Error
	(	const char	  *title,
		const char	  *message,
		QMessageBox::Icon icon
	)
{
	QMessageBox mb (title, message, icon, QMessageBox::Ok, 0, 0) ;
	mb.exec () ;
}


/*  upToTick	: Make sure we are up to time				*/
/*  timenow	: long		: Time in seconds			*/
/*  (returns)	: long		: Apparent time				*/

long	upToTick
	(	long	timenow
	)
{
	return	kndApp->upToTick (timenow) ;
}

/*  KNDApp								*/
/*  KNDApp	: Constructor for application object			*/
/*  (returns)	: KNDApp	:					*/

KNDApp::KNDApp ()
{
	/* Store a pointer at this object, to be used by the "upToTick"	*/
	/* routine. We also note the current time in seconds as a base	*/
	/* from which to work.						*/
	kndApp	= this ;
	second	= time (NULL) ;

	/* See if we are running with effective-uid root. If not then	*/
	/* ksnuffle is not installed setuid-root and is not bing run	*/
	/* by root. In this case we will not execute ...		*/
	if (geteuid () != 0)
	{	Error	("KSnuffle execution error",
			 "KSnuffle not being run by root nor setuid-root") ;
		exit	(1) ;
	}

	/* Get the global configuration, which contains the privileged	*/
	/* users ...							*/
	QString	confpath  = KDECONF + QString((const char *)"/ksnufflerc") ;

	userconf = new KConfig (confpath) ;
	userconf->setGroup  ("Privilege") ;

	/* If the real uid is not root then ksnuffle must be installed	*/
	/* setuid-root and is bing run by a user other than root. In	*/
	/* this case we look to see if the real user is in the list of	*/
	/* privileged users.						*/
	if (getuid () != 0)
	{
		QString	privuser  = userconf->readEntry ("users") ;
		struct	passwd *p = getpwuid (getuid()) ;

		bool	ok	  = false ;
		char	*pptr	  ;

		for (pptr = strtok (privuser, ",\n") ; pptr != NULL ; pptr = strtok (NULL, ",\n"))
			if (strcmp (p->pw_name, pptr) == 0)
			{	ok	= true ;
				break	;
			}

		if (!ok)
		{	Error	("KSnuffle execution error",
				 "You do not have permission to run KSnuffle") ;
			exit	(1) ;
		}
	}

	startNsl () ;

	/* OK, we are allowed to run. We can now get on with all the	*/
	/* mundane configuration stuff.					*/
	setMinimumSize (800, 335)  ;
	setMaximumSize (800, 335)  ;

	viewlist.resize (MAXMON)   ;
	viewlist.fill   (NULL  )   ;

	iflist	= FindAllIface	() ;
	eplist	= FindAllEProto	() ;
	iplist	= FindAllIProto	() ;
	svlist	= FindAllService() ;

	setCaption    ("Ksnuffle " VERSION);

	readOptions   () ;

	initMenuBar   () ;
	initToolBar   () ;
	initView      (userconf) ;

	if (!bViewToolbar) enableToolBar (KToolBar::Hide, 0) ;

	menuBar()->setMenuBarPos (menu_bar_pos) ;
	toolBar()->setBarPos	 (tool_bar_pos) ;

	timer.start (1000) ;
	connect	    (&timer, SIGNAL(timeout()), SLOT(timerTick())) ;
}

/*  KNDApp								*/
/*  ~KNDApp	: Destructor for application object			*/
/*  (returns)	:		:					*/

KNDApp::~KNDApp()
{
	for (int slot = 0 ; slot < MAXMON ; slot += 1)
		if (viewlist[slot] != NULL)
			delete	viewlist[slot] ;

	delete	setup	;
	delete	multi	;
	delete	userconf;
}

/*  KNDApp								*/
/*  upToTick	: Advance our time to specified value			*/
/*  timenow	: long		: Apparent time now in seconds		*/
/*  (returns)	: long		: Apparent time				*/

long	KNDApp::upToTick
	(	long	timenow
	)
{
	/* Loop until our time catches up with the apparent current	*/
	/* time. Each time round we advance our time by one second, and	*/
	/* then pass this down to the sniffers and replayers.		*/
	while (second < timenow)
	{
		second	+= 1 ;
		for (int slot = 0 ; slot < MAXMON ; slot += 1)
			if (viewlist[slot] != NULL)
				viewlist[slot]->timerTick (second) ;

		multi->timerTick (second) ;
	}

	/* Return the apparent time, which may be ahead of the value	*/
	/* with which we were called (for instance, a captured packet	*/
	/* has been delayed over a timer tick).				*/
	return	second	;
}

/*  KNDApp								*/
/*  timerTick	: Handle timer tick					*/
/*  (returns)	: void		:					*/

void	KNDApp::timerTick ()
{
	/* Ensure that we have advanced our idea of the time to be that	*/
	/* of the system.						*/
	upToTick (time (NULL)) ;
}


void	KNDApp::enableCommand
	(	int	id_
	)
{
	menuBar()->setItemEnabled (id_,  true) ;
	toolBar()->setItemEnabled (id_,  true) ;
}

void	KNDApp::disableCommand
	(	int	id_
	)
{
	menuBar()->setItemEnabled (id_, false) ;
	toolBar()->setItemEnabled (id_, false) ;
}


typedef	struct
{	char	*icon	;
	char	*text	;
	int	id	;
	int	accel	;
}	Control	;


/*  buildMenu	: Build menu from control table				*/
/*  menu	: QPopupMenu *	: Menu to construct			*/
/*  control	: Control *	: List of entries			*/
/*  (returns)	: void		:					*/

static	void	buildMenu
	(	QPopupMenu	*menu,
		Control		*control
	)
{
	Control	*cp	;

	for (cp = &control[0] ; cp->id >= 0 ; cp += 1)
		if	(cp->id   == 0)
			menu->insertSeparator () ;
		else if (cp->icon == NULL)
			menu->insertItem (i18n(cp->text), cp->id) ;
		else	menu->insertItem (Icon(cp->icon),
					  i18n(cp->text), cp->id) ;

	for (cp = &control[0] ; cp->id >= 0 ; cp += 1)
		if (cp->id > 0)
			menu->setAccel (CTRL+cp->accel, cp->id) ;
}

void KNDApp::initMenuBar()
{
	static	Control	m_monitor[] =
	{
	{	"filenew.xpm",	 "&New Sniffer",   ID_MONITOR_NEW,	Key_N	},
	{	"reload.xpm",	 "New Re&play",	   ID_MONITOR_REPLAY,	Key_P	},
	{	"delete.xpm",	 "&Close",	   ID_MONITOR_CLOSE,	Key_W	},
	{	NULL,		 NULL,		   0,			-1	},
	{	"start.xpm",	 "&Run all",	   ID_MONITOR_START,	Key_R	},
	{	"stop.xpm",	 "&Halt all",	   ID_MONITOR_STOP,	Key_H	},
	{	"down.xpm",	 "&Freeze",	   ID_MONITOR_FREEZE,	Key_F	},
	{	"up.xpm",	 "R&esume",	   ID_MONITOR_RESUME,	Key_E	},
	{	NULL,		 NULL,		   0,			-1	},
	{	"filefloppy.xpm","&Save",	   ID_MONITOR_SAVE,	Key_S	},
	{	NULL,		 "S&ave and exit", ID_MONITOR_SAVE_QUIT,Key_A	},
	{	"exit.xpm",	 "E&xit",	   ID_MONITOR_QUIT,	Key_Q	},
	{	NULL,		 NULL,		   -1,			0	}  
	}	;

	static	Control	m_view[] =
	{
	{	NULL,		 "Tool&bar",	   ID_VIEW_TOOLBAR,	-1	},
	{	NULL,		 NULL,		   0,			-1	},
	{	"configure.xpm", "&Configuration", ID_VIEW_CONFIG,	Key_C	},
	{	"graphic.xpm",	 "&Graphics",	   ID_VIEW_GRAPHIC,	Key_G	},
	{	"leftjust.xpm",	 "&Packets",	   ID_VIEW_PACKETS,	Key_P	},
	{	NULL,		 NULL,		   0,			-1	},
	{	NULL,		 "Sn&iffers",	   ID_VIEW_MONITORS,	Key_I	},
	{	"parallel.xpm",	 "Para&llel",	   ID_VIEW_MULTI,	Key_L	},
	{	"setup.xpm",	 "Set&up",	   ID_VIEW_SETUP,	Key_U	},
	{	NULL,		 NULL,		   0,			-1	},
	{	NULL,		 NULL,		   -1,			0	}  
	}	;

	buildMenu (monitor_menu = new QPopupMenu (), &m_monitor[0]) ;
	buildMenu (view_menu    = new QPopupMenu (), &m_view   [0]) ;

	view_menu->setCheckable   (true) ;
	view_menu->setItemChecked (ID_VIEW_TOOLBAR,   bViewToolbar  ) ;

  

  ///////////////////////////////////////////////////////////////////
  // menuBar entry help_menu
  help_menu = new QPopupMenu();
  help_menu = kapp->getHelpMenu(true, i18n(IDS_APP_ABOUT));


  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // set menuBar() the current menuBar and the position due to config file
  menuBar()->insertItem (i18n("&Monitor"), monitor_menu);
  menuBar()->insertItem (i18n("&View"), view_menu);

  ///////////////////////////////////////////////////////////////////
  // INSERT YOUR APPLICATION SPECIFIC MENUENTRIES HERE


  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), help_menu);

	connect (monitor_menu, SIGNAL(activated(int)), SLOT(commandCallback(int))) ;
	connect (view_menu,    SIGNAL(activated(int)), SLOT(commandCallback(int))) ;
}

/*  KNDApp								*/
/*  initToolBar	: Initialise toolbar					*/
/*  (returns)	: void		:					*/

void	KNDApp::initToolBar ()
{
	static	Control	tools[] =
	{
	{	"filenew.xpm",	 "New Sniffer",	       ID_MONITOR_NEW,	 -1	},
	{	"reload.xpm",	 "New Replay",	       ID_MONITOR_REPLAY,-1	},
	{	"delete.xpm",	 "Close monitor",      ID_MONITOR_CLOSE, -1	},
	{	NULL,		 NULL,		       0,		 -1	},
	{	"start.xpm",	 "Run all monitors",   ID_MONITOR_START, -1	},
	{	"stop.xpm",	 "Halt all monitors",  ID_MONITOR_STOP,	 -1	},
	{	"down.xpm",	 "Freeze all monitors",ID_MONITOR_FREEZE,-1	},
	{	"up.xpm",	 "Resume all monitors",ID_MONITOR_RESUME,-1	},
	{	NULL,		 NULL,		       0,		 -1	},
	{	"filefloppy.xpm","Save configuration", ID_MONITOR_SAVE,	 -1	},
	{	NULL,		 NULL,		       0,		 -1	},
	{	NULL,		 NULL,		       -1,		 0	}  
	}	;

	for (Control *cp = &tools[0] ; cp->id >= 0 ; cp += 1)
		if	(cp->id   == 0)
			toolBar()->insertSeparator () ;
		else	toolBar()->insertButton (Icon(cp->icon),
						 cp->id,
						 true,
						 i18n(cp->text)) ;

	toolBar()->insertCombo ("", ID_MONITOR_LIST, true, SIGNAL(activated(int)), this, SLOT(slotMonitorPick(int))) ; 

	monList	= toolBar()->getCombo (ID_MONITOR_LIST) ;
	QRect	rect	= monList->geometry () ;
	rect.setRight (rect.left () + 120) ;
	monList->setGeometry (rect) ;
	monList->removeItem  (   0) ;
	QToolTip::add (monList, i18n ("Select sniffer or replayer")) ;

	toolBar()->insertSeparator () ;
	toolBar()->insertButton    (Icon("help.xpm"),    ID_HELP, SIGNAL(pressed()), kapp, SLOT(appHelpActivated()), true, i18n("Help"));

	connect	(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int))) ;
}

/*  KNDApp								*/
/*  setView	: Local override fir setView				*/
/*  view	: QWidget *	: New view				*/
/*  (returns)	: void		:					*/

void	KNDApp::setView
	(	QWidget	*view
	)
{
	KTMainWindow::setView (view) ;

	/* The next two calls are needed to make things work properly	*/
	/* with KTMainWindow. Show the new monitor and then force an	*/
	/* update so that it is correctly sized.			*/
	view->show	()	;
	updateRects	()	;
}

/*  KNDApp								*/
/*  showView	: Show a selected view					*/
/*  next	: KNDView *	: Selected view to show next		*/
/*  (returns)	: void		:					*/

void	KNDApp::showView
	(	KNDView	*next
	)
{
	switch (showing)
	{
		case ShowSetup	:
			setup->hide      () ;
			break	;

		case ShowMulti	:
			multi->hide      () ;
			multi->setOnview (false) ;
			break	;

		default	:
			break	;
	}

	if ((view != next) || (showing != ShowMonitors))
	{
		if (view != NULL)
		{	view->hide	() ;
			view->setOnview (false) ;
		}

		setView		(view = next) ;
		view->setOnview (true)	;

		monList->setCurrentItem (QCBFindItem (*monList, view->getName(), 0)) ;
	}

	showing	= ShowMonitors ;
}

/*  KNDApp								*/
/*  initView	: Initialise all views (monitors)			*/
/*  userconf	: KConfig *	: Privileged users configuration	*/
/*  (returns)	: void		:					*/

void	KNDApp::initView
	(	KConfig	*userconf
	)
{
 	KConfig	*config = kapp->getConfig () ;

	config->setGroup   ("MONITORS") ;

	QString	mlist	= config->readEntry ("MonitorList") ;
	QString	mcurr	= config->readEntry ("Current", "") ;
	QString	text	;

	KNDView	*next	;
	char	*mtoks	;
	int	slot	= 0	;
	int	found	= -1	;
	KNDView	*first	= NULL	;

	view	= NULL		;
	showing	= ShowMonitors	;

	multi	= new KNDMulti	(this) ;
	setup	= new KNDSetup	(this, userconf) ;
	multi->hide () ;
	setup->hide () ;

	/* Provided that we have a monitor list to recreate, scan it	*/
	/* for entries. For simplicity, each is inserted into the list	*/
	/* at the original place. Call "showView" on each so that they	*/
	/* get correctly sized and whatnot. Also, look for the one that	*/
	/* was current last, so that it can be shown first.		*/
	if (!mlist.isNull() && !mlist.isEmpty())
		for (mtoks = strtok (mlist, ",") ; mtoks != NULL ; mtoks = strtok (NULL, ","))
		{
			text.sprintf	 ("MONITOR_%d", atoi(mtoks)) ;
			config->setGroup (text) ;

			next		= new KNDMonitor (this, slot, iflist,
								      eplist,
								      iplist,
								      svlist,
								      multi,
								      config) ;
			viewlist[slot]	= next	;
			view_menu->insertItem (next->getName(), ID_VIEW_SELECT + slot) ; 
			monList  ->insertItem (next->getName(), 0) ;

			showView (next) ;

			if (strcmp (view->getName (), mcurr) == 0)
				first	= view ;

			found		= slot	;
			slot	       += 1	;

		}

	/* We should have found and recreated the monitor that was last	*/
	/* on view, in which case make sure that it is now displayed	*/
	/* before returning.						*/
	if (first != NULL)
	{
		showView (first) ;
	}
	/* In case we failed above, show one of the monitors that we	*/
	/* have just created (unless, of course, we didn't create any!	*/
	else if (found >= 0)
	{
		showView (view = viewlist[found]) ;
		return	 ;
	}
	else
	{
		/* The fallback case covers the situation the first	*/
		/* time that the program is run, or if we exited last	*/
		/* time without saving the configuration, in which	*/
		/* case we create a skeletal monitor.			*/
		next	    = new KNDMonitor (this, 0,	iflist,
							eplist,
							iplist,
							svlist, multi) ;
		viewlist[0] = next ;
		view_menu->insertItem (next->getName(), ID_VIEW_SELECT + 0) ; 
		monList  ->insertItem (next->getName(), 0) ;
		showView (next) ;
	}
}


bool	KNDApp::queryExit ()
{
	return KMsgBox::yesNo (this, i18n ("Exit"),
				     i18n ("Really Quit ?")) == 1 ? true : false ;
}

/*  KNDApp								*/
/*  saveOptions	: Save settings						*/
/*  sm		: bool		: Save monitor configurations		*/
/*  (returns)	: void		:					*/

void	KNDApp::saveOptions
	(	bool	sm
	)
{
	KConfig *config = kapp->getConfig();
	QString	mlist	= "" ;
	QString	text	;
	int	slot	;
	bool	first	;

	/* First generate the standard appearance section which records	*/
	/* tool bar and suchlike information.				*/
	config->setGroup   ("APPEARANCE") ;
	config->writeEntry ("ShowToolbar",   toolBar  ()->isVisible()) ;
	config->writeEntry ("MenuBarPos",    (int)menuBar()->menuBarPos()) ;
	config->writeEntry ("ToolBar_Pos",   (int)toolBar()->barPos    ()) ;

	if (sm)
	{
		/* The next section contains two fields. The first is	*/
		/* a list of numbers, identifying slots in the monitor	*/
		/* table for which there are extant monitors; the	*/
		/* second is the name of the monitor which is		*/
		/* currently on view.					*/
		config->setGroup   ("MONITORS") ;
		for (slot = 0, first = true ; slot < MAXMON ; slot += 1)
			if (viewlist[slot] != NULL)
			{	if (first)
					first	= false ;
				else	mlist  += ","	;
				text.sprintf ("%d", slot) ;
				mlist	+= text ;
			}
		config->writeEntry ("MonitorList", mlist) ;
		config->writeEntry ("Current", 	   view->getName()) ;

		/* There now follow a set of sections, each called	*/
		/* MONITOR_# where each # is a number from the monitor	*/
		/* list in the previous section. These sections		*/
		/* contain all the details for a particular monitor.	*/
		for (slot = 0, first = true ; slot < MAXMON ; slot += 1)
			if (viewlist[slot] != NULL)
			{	text.sprintf ("MONITOR_%d", slot) ;
				config->setGroup    (text) ;
				viewlist[slot]->writeConfig (config) ;
			}
	}
}

/*  KNDApp								*/
/*  readOptions	: Read common display options				*/
/*  (returns)	    : void	:					*/

void	KNDApp::readOptions ()
{
	KConfig *config = kapp->getConfig() ;

	config->setGroup("APPEARANCE");
	bViewToolbar   = config->readBoolEntry ("ShowToolbar",   true) ;
	menu_bar_pos   = (KMenuBar::menuPosition)config->readNumEntry ("MenuBarPos",  KMenuBar::Top) ; 
	tool_bar_pos   = (KToolBar::BarPosition )config->readNumEntry ("ToolBar_Pos", KToolBar::Top) ;

}


/*  KNDApp	    :							*/
/*  slotMonitorNew  : Create a new monitor				*/
/*  (returns)	    : void	:					*/

void	KNDApp::slotMonitorNew ()
{
	int	slot	;
	KNDMonitor	*next	;

	/* Scan the view array for an empty slot into which the new	*/
	/* monitor will be inserted.					*/
	if ((slot = viewlist.find (NULL)) >= 0)
	{
		next		= new KNDMonitor (this, slot, iflist,
							      eplist,
							      iplist,
							      svlist, multi) ;
		viewlist[slot]	= next ;
		view_menu->insertItem (next->getName(), ID_VIEW_SELECT + slot) ; 
		monList  ->insertItem (next->getName(), 0) ;

		showView (next) ;
	}
}

/*  KNDApp	     :							*/
/*  slotMonitorReplay: Create a new replay				*/
/*  (returns)	     : void	:					*/

void	KNDApp::slotMonitorReplay ()
{
	int	slot	;
	KNDView	*next	;

	/* Scan the views array for an empty slot into which the	*/
	/* new replayer will be inserted.				*/
	if ((slot = viewlist.find (NULL)) >= 0)
	{
		next		= new KNDReplay (this, multi, slot) ;
		viewlist[slot]	= next ;
		view_menu->insertItem (next->getName(), ID_VIEW_SELECT + slot) ; 
		monList  ->insertItem (next->getName(), 0) ;
		showView (next) ;
	}
}

/*  KNDApp	    :							*/
/*  slotMonitorClose: Close the current monitor				*/
/*  (returns)	    : void	:					*/

void	KNDApp::slotMonitorClose ()
{
	KNDView	*next	;
	KNDView	*last	;
	int	oldslot	;
	int	newslot	;
	int	scan	;

	/* Scan the monitor combo list looking for the current monitor,	*/
	/* and remove it. We also remove it from the list of monitors.	*/
	for (scan = 0 ; scan < monList->count() ; scan += 1)
		if (strcmp (view->getName(), monList->text(scan)) == 0)
		{	monList->removeItem (scan) ;
			break ;
		}

	oldslot	= view->getSlot () ;
	viewlist[oldslot] = NULL   ;

	/* Scan the view array looking for another monitor to		*/
	/* activate. If none is found create a new one and allocate	*/
	/* the slot that the one we are closing occupied.		*/
	for (scan = 0, next = NULL ; scan < MAXMON ; scan += 1)
		if ((viewlist[scan] != NULL) && (viewlist[scan] != view))
		{	next	= viewlist[scan];
			break	;
		}

	if (next == NULL)
	{	newslot	= oldslot == 0 ? 1 : 0 ;
		next	= new KNDMonitor   (this, newslot, iflist,
						           eplist,
						           iplist,
						           svlist, multi) ;
		view_menu->insertItem (next->getName(), ID_VIEW_SELECT + newslot) ; 
		monList  ->insertItem (next->getName(), 0) ;
	}

	last	= view	;
	showView (next) ;
	delete	  last  ;

	view_menu->removeItem     (ID_VIEW_SELECT + oldslot) ;
	monList  ->setCurrentItem (QCBFindItem (*monList, view->getName(), 0)) ;
}

/*  KNDApp	    :							*/
/*  slotMonitorPick : Pick a monitor for display			*/
/*  slot	    : int	: Index into combo list			*/
/*  (returns)	    : void	:					*/

void	KNDApp::slotMonitorPick
	(	int	slot
	)
{
	const	char	*name	= monList->text (slot) ;
	int		scan	;

	/* Scan the list of viewlist for the required one and display	*/
	/* it. See the comments in slotMonitorNew about making this	*/
	/* work with KTMainWindow.					*/
	for (scan = 0 ; scan < MAXMON ; scan += 1)
		if ((viewlist[scan] != NULL) && (strcmp (name, viewlist[scan]->getName()) == 0))
		{
			showView (viewlist[scan]) ;
			break ;
		}
}

/*  KNDApp	:							*/
/*  rename	: Rename entry in monitor list				*/
/*  oldname	: const char *	: Old name				*/
/*  newname	: const char *	: New name				*/
/*  (returns)	: void		:					*/

void	KNDApp::rename
	(	const	char	*oldname,
		const	char	*newname
	)
{
	int	slot	;

	/* Scan the list of viewlist for the old name and remove it,	*/
	/* then insert the name name and make it current.		*/
	for (slot = 0 ; slot < monList->count() ; slot += 1)
		if (strcmp (oldname, monList->text (slot)) == 0)
		{	monList->removeItem (slot) ;
			break	;
		}

	monList->insertItem	(newname, 0) ;
	monList->setCurrentItem (0) ;
}

void	KNDApp::slotMonitorCloseWindow ()
{
	close () ;
}

/*  KNDApp								*/
/*  slotMonitorStart	: Start all idle monitors			*/
/*  (returns)		: void		:				*/

void	KNDApp::slotMonitorStart ()
{
	/* Pass the request down to all extant monitors to be handled	*/
	/* by them. Note that we have to show each in turn to ensure	*/
	/* that the controls get correctly sized.			*/
	for (int slot = 0 ; slot < MAXMON ; slot += 1)
		if (viewlist[slot] != NULL)
		{	showView(viewlist[slot])     ;
			viewlist[slot]->viewStart () ;
		}
}

/*  KNDApp								*/
/*  slotMonitorStop	: Stop all active monitors				*/
/*  (returns)		: void		:				*/

void	KNDApp::slotMonitorStop ()
{
	/* Pass the request down to all extant monitors to be handled	*/
	/* by them.							*/
	for (int slot = 0 ; slot < MAXMON ; slot += 1)
		if (viewlist[slot] != NULL)
			viewlist[slot]->viewStop () ;
}

/*  KNDApp								*/
/*  slotMonitorFreeze	: Freeze all active monitors			*/
/*  (returns)	    	: void		:				*/

void	KNDApp::slotMonitorFreeze ()
{
	/* Pass the request down to all extant monitors to be handled	*/
	/* by them.							*/
	for (int slot = 0 ; slot < MAXMON ; slot += 1)
		if (viewlist[slot] != NULL)
			viewlist[slot]->viewFreeze () ;
}

/*  KNDApp								*/
/*  slotMonitorResume	: Resume all frozen monitors			*/
/*  (returns)	    	: void		:				*/

void	KNDApp::slotMonitorResume ()
{
	/* Pass the request down to all extant monitors to be handled	*/
	/* by them.							*/
	for (int slot = 0 ; slot < MAXMON ; slot += 1)
		if (viewlist[slot] != NULL)
			viewlist[slot]->viewResume () ;
}

/*  KNDApp								*/
/*  slotMonitorSave : Save current configuration			*/
/*  (returns)	    : void		:				*/

void	KNDApp::slotMonitorSave()
{
	saveOptions (true) ;
}

/*  KNDApp								*/
/*  slotMonitorSaveQuit: Save current configuration and quit		*/
/*  (returns)	       : void		:				*/

void	KNDApp::slotMonitorSaveQuit ()
{
	if (this->queryExit ())
	{	saveOptions (true) ;
		KTMainWindow::deleteAll () ;
		kapp->quit  () ;
	}
}

/*  KNDApp								*/
/*  slotMonitorQuit : Quit						*/
/*  (returns)	    : void		:				*/

void	KNDApp::slotMonitorQuit()
{ 
	if (this->queryExit ())
	{	saveOptions (false) ;
		KTMainWindow::deleteAll () ;
		kapp->quit  () ;
	}
}

/*  KNDApp								*/
/*  slotViewToolBar: Toggle tool bar					*/
/*  (returns)	   : void	:					*/

void KNDApp::slotViewToolBar()
{
	bViewToolbar = !bViewToolbar ;
	menuBar()->setItemChecked (ID_VIEW_TOOLBAR, bViewToolbar) ;
	enableToolBar (KToolBar::Toggle, 0) ;
}

/*  KNDApp								*/
/*  slotViewConfig: View monitor configuration				*/
/*  (returns)	  : (void)		:				*/

void	KNDApp::slotViewConfig ()
{
	slotViewMonitors  () ;
	view->viewConfig  () ;
}

/*  KNDApp								*/
/*  slotViewGraphic: View monitor graphic display			*/
/*  (returns)	   : (void)		:				*/

void	KNDApp::slotViewGraphic ()
{
	slotViewMonitors  () ;
	view->viewGraphic () ;
}

/*  KNDApp								*/
/*  slotViewPackets: View monitor packet display			*/
/*  (returns)	   : (void)		:				*/

void	KNDApp::slotViewPackets ()
{
	slotViewMonitors  () ;
	view->viewPackets () ;
}

/*  KNDApp	    							*/
/*  slotViewMonitors: View monitors					*/
/*  (returns)	    : void		:				*/

void	KNDApp::slotViewMonitors ()
{
	showView (view) ;
}

/*  KNDApp	    							*/
/*  slotViewMulti   : View multiple graphs				*/
/*  (returns)	    : void		:				*/

void	KNDApp::slotViewMulti ()
{
	switch (showing)
	{
		case ShowMonitors :
			view ->hide  () ;
			view ->setOnview (false) ;
			break	;

		case ShowSetup	  :
			setup->hide  () ;
			break	;

		case ShowMulti	  :
			return	;
	}

	setView		 (multi) ;
	multi->setOnview (true)  ;
	showing	= ShowMulti	 ;
}

/*  KNDApp	    							*/
/*  slotViewSetup   : View global setup					*/
/*  (returns)	    : void		:				*/

void	KNDApp::slotViewSetup ()
{
	switch (showing)
	{
		case ShowMonitors :
			view ->hide      () ;
			view ->setOnview (false) ;
			break	;

		case ShowMulti	  :
			multi->hide      () ;
			multi->setOnview (false) ;
			setView (setup) ;
			break	;

		case ShowSetup	  :
			return	;
	}

	setView	(setup) ;
	showing	= ShowSetup ;
}

/*  KNDApp								*/
/*  showClient	: Show a specified client				*/
/*  slot	: int		: Client slot				*/
/*  (returns)	: void		:					*/

void	KNDApp::showClient
	(	int	slot
	)
{
	if (viewlist[slot] != NULL) showView (viewlist[slot]) ;
}

/*  KNDApp								*/
/*  commandCallback: Menu/toolbar command callback			*/
/*  id		   : int	: Command identifier			*/
/*  (returns)	   : void	:					*/

void	KNDApp::commandCallback
	(	int	id
	)
{
	switch (id)
	{
		ON_CMD(ID_MONITOR_NEW,              slotMonitorNew      ())
		ON_CMD(ID_MONITOR_REPLAY,           slotMonitorReplay   ())
		ON_CMD(ID_MONITOR_CLOSE,            slotMonitorClose    ())
		ON_CMD(ID_MONITOR_STOP,		    slotMonitorStop	())
		ON_CMD(ID_MONITOR_START,	    slotMonitorStart	())
		ON_CMD(ID_MONITOR_FREEZE,	    slotMonitorFreeze	())
		ON_CMD(ID_MONITOR_RESUME,	    slotMonitorResume	())
		ON_CMD(ID_MONITOR_SAVE,		    slotMonitorSave     ())
		ON_CMD(ID_MONITOR_SAVE_QUIT,	    slotMonitorSaveQuit ())
		ON_CMD(ID_MONITOR_QUIT,             slotMonitorQuit     ())

		ON_CMD(ID_VIEW_TOOLBAR,             slotViewToolBar     ())
		ON_CMD(ID_VIEW_MONITORS,	    slotViewMonitors	())
		ON_CMD(ID_VIEW_MULTI,	    	    slotViewMulti	())
		ON_CMD(ID_VIEW_SETUP,		    slotViewSetup	())
		ON_CMD(ID_VIEW_CONFIG,		    slotViewConfig	())
		ON_CMD(ID_VIEW_GRAPHIC,		    slotViewGraphic	())
		ON_CMD(ID_VIEW_PACKETS,		    slotViewPackets	())
	}

	if ((id >= ID_VIEW_SELECT) && (id < ID_VIEW_SELECT + MAXMON))
		if (viewlist[id -= ID_VIEW_SELECT] != NULL)
			showView (viewlist[id]) ;
}
