// krfbproto.cpp
// Author: Markus Wuebben <markus.wuebben@kde.org>
// This code is published under the GPL.
// March 1998

#include "krfbproto.h"
#include "krfbproto.moc"
#include <qimage.h>
#include <kapp.h>
#include <kmsgbox.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>


#define INVALID_PIXEL 0xffffffff
#define COLORMAP_SIZE 256
unsigned long BGR233ToPixel[COLORMAP_SIZE];


extern "C" {
#include "vncauth.h"
}

int endianTest = 1;


kRfbProto::kRfbProto(const char *h, int p) {

  inNormalProtocol = false;
  host = h;
  port = p;
  endianTest = 1 ;
  sock = 0L;
  memset(buffer, '\0',BUFFER_SIZE);
  readConfig();

}

kRfbProto::~kRfbProto() {

}

void kRfbProto::readConfig() {


  KConfig * config = KApplication::getKApplication()->getConfig();

  config->setGroup("Options");

  _hextile = config->readNumEntry("hextile",0);
  _corre = config->readNumEntry("corre",0);
  _rre = config->readNumEntry("rre",0);
  _raw = config->readNumEntry("raw",0);
  _copyrect = config->readNumEntry("copyrect",0);
  _deiconify = config->readNumEntry("deiconify",0);
  _restrict = config->readNumEntry("restrict",0);
  _shared = config->readNumEntry("shared",0);
  _encodings = config->readNumEntry("numEncodings",0);

  config->sync();

}



int kRfbProto::connect() {

  sock = new KSocket(host,port);
  if(sock->socket() < 0)  
    return -1;
  sock->enableRead(true);
  firstTrans = true;
  return 0;

}

void kRfbProto::close() {
  
  printf("Closing socket: %i\n",sock->socket());
  if(::close(sock->socket()) < 0) {
    cout << "Error closing socket\n";
    delete sock;
    sock = 0L;
    emit fatal();
    return;
  }
    delete sock;
    sock = 0L;

}


void kRfbProto::readVersionMsg() {

  char b[13];
  
  if(!readExact(sock->socket(),b,12)) {
    cout << "readExact in readVersionMsg\n";
    emit fatal();
    return;
  }
  if ((b[0] != 'R') || (b[1] != 'F') || (b[2] != 'B') || (b[3] != ' ')
	|| (b[4] < '0') || (b[4] > '9') || (b[5] < '0') || (b[5] > '9')
	|| (b[6] < '0') || (b[6] > '9') || (b[7] != '.')
	|| (b[8] < '0') || (b[8] > '9') || (b[9] < '0') || (b[9] > '9')
	|| (b[10] < '0') || (b[10] > '9') || (b[11] != '\n'))
    {
      cout << "Error if() in readVersionMsg()\n";
      emit fatal();
      return;
    }
  // [12]  to get a real string
  b[12] = '\0';
  QString temp;
  temp.sprintf("%s",b);
  printf("Server Version: %s",b);
  setServerVersion((const char *)temp);
  fflush(stdout);

  serverMajor = (b[4] - '0') * 100 + (b[5] - '0') * 10 + (b[6] - '0');
  serverMinor = (b[8] - '0') * 100 + (b[9] - '0') * 10 + (b[10] - '0');
  
}


void kRfbProto::writeVersionMsg() {

  char b[13];
  sprintf(b,"%s",versionMsg());

  printf("KVNCViewer versionMsg: %s",b);

  if(!writeExact(sock->socket(),b,12)) {
    cout << "writeExact writeVersionMsg() failed\n";
    emit fatal();
    return;
  }
  
}

int kRfbProto::readAuthScheme() {


  CARD32 reasonLen;
  QString temp;
  CARD32 authScheme;
  char *reason;
  if(!readExact(sock->socket(),(char *)&authScheme,4)) {
    cout << "Error read in readAuthScheme\n";
    emit fatal();
    return -1;

  }

  authScheme = Swap32IfLE(authScheme);
  cout << "Authentification: " << authScheme << endl;

  switch(authScheme) {

  case ConnFailed:
    if(!readExact(sock->socket(),(char*)&reasonLen,4)) {
      reasonLen  = Swap32IfLE(reasonLen);
      reason = (char*)malloc(reasonLen +1);
      if(!readExact(sock->socket(),(char*)&reason,reasonLen))
	return (-1);
      reason[reasonLen] = '\0';
      cout << "Connection failed: " << reason << endl;
      return (-1);
    }
  case NoAuth:
  case VncAuth:
    return authScheme;
  default:
    printf("Unknown authentication scheme from RFB server %i\n",  (int)authScheme);
    return (-1);
  }
	 
}



bool kRfbProto::writeClientInit() {

  bool i;
  if(_shared) {
    cout << "Sharing desktop!\n";
    i = true;
    if(!write(sock->socket(),&i,1)) {
      cout << "Share desktop call failed!\n";
      return false;
    }
    
  }
  else {
    cout << "*Not* Sharing desktop!\n";
    i = false;
    if(!write(sock->socket(),&i,1)) {
      cout << "(Not) Share desktop call failed!\n";
      return false;
    }
  }
  
  return true;
        

}


int kRfbProto::transmitPassword(char *passwd) {

  CARD8 tempChallenge[CHALLENGESIZE];
  CARD32 authResult;

  if(firstTrans) {
    if (!readExact(sock->socket(), (char*)tempChallenge, CHALLENGESIZE)) {
      cout << "getting challenge failed\n";
      return -1;
    }
    challenge = tempChallenge; // no good
  }
  firstTrans = false;
  
  vncEncryptBytes((unsigned char *)challenge, passwd);
  if(!writeExact(sock->socket(),(char*)challenge,CHALLENGESIZE)) {
    cout << "writing challenge failed\n";
    return -1;
  }

  /* Lose the password from memory */
  for (uint i=0; i< strlen(passwd); i++) {
    passwd[i] = '\0';
  }

  if(!readExact(sock->socket(),(char *)&authResult, 4)) {
    cout << "read authResult failed\n";
    return -1;
  }

  authResult = Swap32IfLE(authResult);
  cout << "Authentification result: " << (int)authResult << endl;
  switch (authResult) {
	case VncAuthOK:
	  cout << "VNC authentication succeeded\n";
	  return VncAuthOK;
	case VncAuthFailed:
	  cout << "VNC authentication failed\n";
	  return VncAuthFailed;
	case VncAuthTooMany:
	  cout << "VNC authentication failed - too many tries\n";
	  return VncAuthTooMany;
	default:
	  cout << "Unknown VNC authentication result " 
	       << (int)authResult << endl;
	  return VncAuthUnknown;
  }
  
  return VncAuthFailed;

}


// Well. lets see what the server prefers ;-)
bool kRfbProto::readServerInit() {


  if (!readExact(sock->socket(), (char *)&si, sz_rfbServerInitMsg)) 
    return false;
  
  si.framebufferWidth = Swap16IfLE(si.framebufferWidth);
  si.framebufferHeight = Swap16IfLE(si.framebufferHeight);
  si.format.redMax = Swap16IfLE(si.format.redMax);
  si.format.greenMax = Swap16IfLE(si.format.greenMax);
  si.format.blueMax = Swap16IfLE(si.format.blueMax);
  si.nameLength = Swap32IfLE(si.nameLength);
  

  desktopName = (char *)malloc(si.nameLength + 1);
  if (!readExact(sock->socket(), desktopName, si.nameLength)) 
    return false;
  desktopName[si.nameLength] = '\0';

  cout << "Inits:\n";
  cout << "fbWidth: " << si.framebufferWidth << endl;
  cout << "fbHeight: " << si.framebufferHeight << endl;
  cout << "redMax: " << si.format.redMax << endl;
  cout << "greenMax: " << si.format.greenMax << endl;
  cout << "blueMax: " << si.format.blueMax << endl;
  cout << "deskLen: " << si.nameLength << endl;
  cout << "desktopName: " << desktopName << endl;

  printf("depth: %i\n",si.format.bitsPerPixel);

  cout << "End inits\n";

  cout << "PixelFormat Server provided\n";
  PrintPixelFormat(&si.format);
  
  return true;
    
}

void kRfbProto::findBestVisual(QWidget *desktop) {


}

bool kRfbProto::writePixelFormat(rfbServerInitMsg *msg) {
  
  //  cout << "Entering writePixelFormat()\n";
  rfbSetPixelFormatMsg *pixelMsg;

  assert(msg);

  // We're going to be much less server friendly here and just
  // demand exactly what we want instead. (rich)

  pixelMsg = (rfbSetPixelFormatMsg *)malloc(sizeof(rfbSetPixelFormatMsg));
  CHECK_PTR(pixelMsg);
  pixelMsg->type = rfbSetPixelFormat;
  if (QImage::systemByteOrder() == QImage::BigEndian)
    pixelMsg->format.bigEndian= TRUE;
  else
    pixelMsg->format.bigEndian= FALSE;


  // Well as soon as we have other support this will change
  // We are still using our own encoding here
  
  pixelMsg->format.bitsPerPixel= 32;
  pixelMsg->format.depth= 24;
  pixelMsg->format.trueColour= TRUE;
  pixelMsg->format.redMax = Swap16IfLE(255);
  pixelMsg->format.greenMax = Swap16IfLE(255);
  pixelMsg->format.blueMax = Swap16IfLE(255);
  pixelMsg->format.redShift = 0;
  pixelMsg->format.greenShift = 8;
  pixelMsg->format.blueShift= 16;
  
  
  // Need to swap if green/red/blue max requested
  si.format = pixelMsg->format;
  
  cout << "bpp is set to: " << si.format.bitsPerPixel << endl;

  if (!writeExact(sock->socket(), (char *)pixelMsg, sz_rfbSetPixelFormatMsg))
    return false;

  cout << "Pixel format we are finally going to use:\n";
  PrintPixelFormat(&si.format);
  return true;
}
			 

uint kRfbProto::bpp() {

   if(si.format.bitsPerPixel >= 8)
     return si.format.bitsPerPixel;
   else 
     return 0;
}



bool kRfbProto::writeFramebufferUpdateRequest(int x, int y, int w, int h,
					      bool incremental) {


  rfbFramebufferUpdateRequestMsg fur;

  disconnect(sock,SIGNAL(readEvent(KSocket*)),this,SLOT(getData(KSocket*)));

  //cout << "Entering writeFramebufferUpdateRequest\n";

  fur.type = rfbFramebufferUpdateRequest;
  fur.incremental = incremental ? 1 : 0;
  fur.x = Swap16IfLE(x);
  fur.y = Swap16IfLE(y);
  fur.w = Swap16IfLE(w);
  fur.h = Swap16IfLE(h);
  
  if (!writeExact(sock->socket(), (char *)&fur, 
		  sz_rfbFramebufferUpdateRequestMsg)) {
    cout << "write in writeFrameBuffer\n";
    return false;
  }
  
  QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		   SLOT(getData(KSocket *)));


  return true;
}


bool kRfbProto::getData(KSocket *) {

  rfbServerToClientMsg msg;

  int bytesPerLine;
  int linesToRead;
  int i;
  QString temp;

  emit refresh(true);
  disconnect(sock,SIGNAL(readEvent(KSocket*)),this,SLOT(getData(KSocket*)));

  //cout << "Data pending\n";

  if(!readExact(sock->socket(),(char*)&msg,1)) {
    QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		   SLOT(getData(KSocket *)));
    return false;
  }
  
  switch(msg.type) {
    
  case rfbFramebufferUpdate: 

    rfbFramebufferUpdateRectHeader rect;
    if(!readExact(sock->socket(),((char*)&msg.fu) +1,
		  sz_rfbFramebufferUpdateMsg-1)) {
      QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		   SLOT(getData(KSocket *)));
      return false;
    }

    msg.fu.nRects = Swap16IfLE(msg.fu.nRects);

    //cout << "Pending rects: " << msg.fu.nRects << endl;

    for (i = 0; i < msg.fu.nRects; i++) {
      //cout << "Rect: " << i << endl;

      // Get rectangle's header
      if(!readExact(sock->socket(),(char*)&rect,
		    sz_rfbFramebufferUpdateRectHeader)) {
	QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		   SLOT(getData(KSocket *)));
	return false;
      }
      
      // Good...convert the crap
      rect.r.x = Swap16IfLE(rect.r.x);
      rect.r.y = Swap16IfLE(rect.r.y);
      rect.r.w = Swap16IfLE(rect.r.w);
      rect.r.h = Swap16IfLE(rect.r.h);
      rect.encoding = Swap32IfLE(rect.encoding);
      
      /*

      cout << "********Rectangle value*******\n";
      cout << "x: " << rect.r.x << endl;
      cout << "y: " << rect.r.y << endl;
      cout << "w: " << rect.r.w << endl;
      cout << "h: " << rect.r.h << endl;
      cout << "encoding: " << rect.encoding << endl;
      cout << "**************End*************\n";
      */
      // Check if valid rectangle
      if ((rect.r.x + rect.r.w > si.framebufferWidth) ||
	  (rect.r.y + rect.r.h > si.framebufferHeight)) {
	printf("Warinig: rect too large: %dx%d at (%d, %d)\n",
	       rect.r.w, rect.r.h, rect.r.x, rect.r.y);
	QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			 SLOT(getData(KSocket *)));
	return false;
      }
      
      if ((rect.r.h * rect.r.w) == 0) {
	printf("Warning: zero size rect - ignoring\n");
	continue;
      }
    
      switch(rect.encoding) {
	
      case rfbEncodingRaw:

	//printf("bitsPerPixel: %d\n",si.format.bitsPerPixel);
	bytesPerLine = rect.r.w * si.format.bitsPerPixel / 8;
	//printf("byterPerLine: %d\n",bytesPerLine);
	linesToRead = BUFFER_SIZE / bytesPerLine;
	//printf("linesToRead: %d\n",linesToRead);
	
	
	// If our buffer is capable of saving more lines then we need we 
	// only get as many as the rectangle has.
	while (rect.r.h > 0) {
	  if (linesToRead > rect.r.h) {	    
	    linesToRead = rect.r.h;
	    //cout << "linesToRead was too high is now: " << linesToRead << endl;
	  }

	  
	  // Get the rectangle's data
	  if (!readExact(sock->socket(),buffer,bytesPerLine * linesToRead)) {
	    QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			     SLOT(getData(KSocket *)));
	    return false;
	  }

	  emit sendData((CARD8*)buffer,rect.r.x, rect.r.y,
	  	linesToRead, bytesPerLine * linesToRead);
	 
	  memset(buffer,'\0',(bytesPerLine*linesToRead+1));
	  rect.r.h -= linesToRead;
	  rect.r.y += linesToRead;
	}
	// for now useless
	  emit finished();
	break;

      case rfbEncodingCopyRect:
	  rfbCopyRect cr;
	  
	  if (!readExact(sock->socket(), (char *)&cr, sz_rfbCopyRect)) {
	    QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			     SLOT(getData(KSocket *)));
	    return false;
	  }

	  cr.srcX = Swap16IfLE(cr.srcX);
	  cr.srcY = Swap16IfLE(cr.srcY);

	  emit copyRect(cr.srcX, cr.srcY, rect.r.x, rect.r.y, 
			rect.r.w, rect.r.h);
	  
	  break;


      case rfbEncodingRRE:
	{
	  rfbRREHeader hdr;
	  CARD32 pix32;
	  rfbRectangle subrect;
	  uint j;
	  
	  if (!readExact(sock->socket(), (char *)&hdr, sz_rfbRREHeader)) {
	    QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			      SLOT(getData(KSocket *)));
	    return false;
	  }
	  
	  hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
	  
	  
	  if (!readExact(sock->socket(), (char *)&pix32, 4)){
	    QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			     SLOT(getData(KSocket *)));
	    return false;
	  }
	  
	  emit fillRect(rect.r.x, rect.r.y,
			rect.r.w, rect.r.h, pix32);
	  
	  for (j = 0; j < hdr.nSubrects; j++) {
	    if (!readExact(sock->socket(), (char *)&pix32, 4)) {
	      QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			       SLOT(getData(KSocket *)));
	      return false;
	    }
	    
	    if (!readExact(sock->socket(), (char *)&subrect,
			   sz_rfbRectangle)) {
	      QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			       SLOT(getData(KSocket *)));
	      return false;
	    }
	    
	    subrect.x = Swap16IfLE(subrect.x);
	    subrect.y = Swap16IfLE(subrect.y);
	    subrect.w = Swap16IfLE(subrect.w);
	    subrect.h = Swap16IfLE(subrect.h);
	    
	    emit fillRect(rect.r.x + subrect.x, rect.r.y + subrect.y, 
			  subrect.w,subrect.h, pix32);
	  }
	  break;
	}
      case rfbEncodingCoRRE:
	
	rfbRREHeader hdr;
	CARD32 pix32;
	uint j;
	CARD8 *ptr;
	register int x, y, w, h;
	
	if (!readExact(sock->socket(), (char *)&hdr, sz_rfbRREHeader)) {
	  QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			   SLOT(getData(KSocket *)));
	  return false;
	}
	
	hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
	
	if (!readExact(sock->socket(), (char *)&pix32, 4)) {
	  QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			   SLOT(getData(KSocket *)));
	  return false;	
	}

	emit fillRect(rect.r.x, rect.r.y, rect.r.w, rect.r.h, pix32);
	
	if (!readExact(sock->socket(), buffer, hdr.nSubrects * 8)) {
	  QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			   SLOT(getData(KSocket *)));
	  return false;
	}
	
	ptr = (CARD8 *)buffer;
	
	for (j = 0; j < hdr.nSubrects; j++) {
	  pix32 = *(CARD32 *)ptr;
	  ptr += 4;
	  x = *ptr++;
	  y = *ptr++;
	  w = *ptr++;
	  h = *ptr++;
	  emit fillRect(rect.r.x + x, rect.r.y + y, w, h, pix32);
	}
	break;


      case rfbEncodingHextile:
	if(!handleHextile(rect.r.x,rect.r.y,rect.r.w,rect.r.h)) { 
	  QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
			   SLOT(getData(KSocket *)));
	  return false;
	}
	break;

      default:
	cout << "Unsupported Encoding -- sorry!\n";
	break;
      }
    
    }

    break;
  
  case rfbSetColourMapEntries: 
    
    cout << "SetColourMapEntries\n";
    break;
  
  case rfbBell: 

    cout << "Bell\n";
    emit bell();
    break;

  case rfbServerCutText: 

    cout << "ServerCutText\n";
    char *str;

    if (!readExact(sock->socket(), ((char *)&msg) + 1,
		   sz_rfbServerCutTextMsg - 1)) {
      QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		       SLOT(getData(KSocket *)));
      return false;
    }
    
    msg.sct.length = Swap32IfLE(msg.sct.length);
    
    cout << "Length: " << msg.sct.length << endl;
    if(!(str = (char *)malloc(msg.sct.length+1))) {
      QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		       SLOT(getData(KSocket *)));
      return false;
    }

    if (!readExact(sock->socket(), str, msg.sct.length)) {
      QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		       SLOT(getData(KSocket *)));
      return false;
    }
    str[msg.sct.length] = '\0';
    cout << "*************ServerCutText********** " << endl;
    cout << str << endl;
    cout << "************End ServerCutText******* " << endl;
    free(str);
    break;
    
  default:
    cout << "Unknown message from server\n";
    QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		   SLOT(getData(KSocket *)));
    return false;
  }

  emit refresh(false);
  QObject::connect(sock,SIGNAL(readEvent(KSocket *)),this,
		   SLOT(getData(KSocket *)));
  return true;

}


bool kRfbProto::handleHextile( int rx, int ry, int rw, int rh) {

  CARD32 bg, fg; 
  int x, y, w, h;                                                           
  int sx, sy, sw, sh;                                                       
  int i;    
  CARD8 *ptr;
  CARD8 subencoding;                                                        
  CARD8 nSubrects;
  


  for (y = ry; y < ry+rh; y += 16) {                                        
        for (x = rx; x < rx+rw; x += 16) {                                    
            w = h = 16;                                                       
            if (rx+rw - x < 16)                                               
                w = rx+rw - x;                                                
            if (ry+rh - y < 16)                                               
                h = ry+rh - y;                                                
                                                                              
            if (!readExact(sock->socket(), (char *)&subencoding, 1))                 
                return false;                                                 

            if (subencoding & rfbHextileRaw) {                                
                if (!readExact(sock->socket(), buffer, w * h * (32 / 8)))           
                    return false;                                             
		
                emit sendData((CARD8 *)buffer, x, y, h, w*h*(32 /8));                
                continue;                                                     
            }                                                                
                                                                             
            if (subencoding & rfbHextileBackgroundSpecified) {
                if (!readExact(sock->socket(), (char *)&bg, (32/8)))               
                    return false;              
	    }
	    emit fillRect(x,y,w,h,bg);
	                                                                                 
            if (subencoding & rfbHextileForegroundSpecified) {                 
                if (!readExact(sock->socket(), (char *)&fg, (32/8)))                
                    return false;              
		
	    }                               
	    
            if (!(subencoding & rfbHextileAnySubrects)) {                     
	      continue;                                                     
            }                                                                 
                                                                              
            if (!readExact(sock->socket(), (char *)&nSubrects, 1))                   
                return false;                                                 
                                                                              
            ptr = (CARD8 *)buffer;                                            
                                                                              
            if (subencoding & rfbHextileSubrectsColoured) {   
                if (!readExact(sock->socket(), buffer, nSubrects * (2 + (32 / 8)))) 
                    return false;                                             
                                                                              
		for (i = 0; i < nSubrects; i++) {                             
                    GET_PIXEL32(fg, ptr);                                  
                    sx = rfbHextileExtractX(*ptr);                            
                    sy = rfbHextileExtractY(*ptr);                            
                    ptr++;                                                    
                    sw = rfbHextileExtractW(*ptr);                            
                    sh = rfbHextileExtractH(*ptr);                            
                    ptr++;                                                    

		    emit fillRect(x+sx,y+sy,sw,sh,fg);
		    }                                                             
		
            } else {                                                          
	      if (!readExact(sock->socket(), buffer, nSubrects * 2))               
		return false;                                             
	      for (i = 0; i < nSubrects; i++) {                             
		sx = rfbHextileExtractX(*ptr);                            
		sy = rfbHextileExtractY(*ptr);                            
		ptr++;                                                    
		sw = rfbHextileExtractW(*ptr);                            
		sh = rfbHextileExtractH(*ptr);                            
		ptr++;                                                    
		emit fillRect(x+sx, y+sy, sw, sh, fg);      
	      }                                                             
	      
            }                                                              
        }                                                                   
  }                                                                        
                                                                              
    return true;                                

}




// Lets tell the server which encodings we want!

bool kRfbProto::writeEncodings()
{

  rfbSetEncodingsMsg encMsg;
  CARD32 encodings;

  cout << "writeEncodings: Num of encodings: " << _encodings << endl;
  if(_encodings < 1) {
    KMsgBox::message(0L,"K VNCViewer",
		     "You need to specify an encoding!\nPlease reconnect!\n");
    return false;
  }
    
  encMsg.type = rfbSetEncodings;
  encMsg.nEncodings = _encodings;
  encMsg.nEncodings = Swap16IfLE(encMsg.nEncodings);

  if(!writeExact(sock->socket(),(char*)&encMsg,sz_rfbSetEncodingsMsg))
    return false;


  if(_hextile) {
    encodings = rfbEncodingHextile;
    encodings = Swap32IfLE(encodings); 
    if(!writeExact(sock->socket(),(char*)&encodings,4)) 
      return false;
  }

  
  if(_copyrect) {
    encodings = rfbEncodingCopyRect; 
    encodings = Swap32IfLE(encodings); 
    if(!writeExact(sock->socket(),(char*)&encodings,4)) 
      return false;
  }
       
  if(_raw) {
    encodings = rfbEncodingRaw;
    encodings = Swap32IfLE(encodings); 
    if(!writeExact(sock->socket(),(char*)&encodings,4)) 
      return false;
  }

  if(_rre) {
    encodings = rfbEncodingRRE; 
    encodings = Swap32IfLE(encodings); 
    if(!writeExact(sock->socket(),(char*)&encodings,4)) 
      return false;
  }

  if(_corre) {
    encodings = rfbEncodingCoRRE; 
    encodings = Swap32IfLE(encodings); 
    if(!writeExact(sock->socket(),(char*)&encodings,4)) 
      return false;
  }
  

  return true;
  
}



bool kRfbProto::SameMachine(int sock)
{
    struct sockaddr_in peeraddr, myaddr;
    int addrlen = sizeof(struct sockaddr_in);

    cout << "Testing if running on same machine\n";
    
    getpeername(sock, (struct sockaddr *)&peeraddr, (int *)&addrlen);
    getsockname(sock, (struct sockaddr *)&myaddr, (int *)&addrlen);

    return (peeraddr.sin_addr.s_addr == myaddr.sin_addr.s_addr);
}


bool kRfbProto::writePointerEvent(CARD8 buttonMask,int x, int y) {

  rfbPointerEventMsg pe;
  
  pe.type = rfbPointerEvent;
  // Too much output (rich)
  //  printf("MASK ------------> %02x \n",buttonMask);
  pe.buttonMask = buttonMask;
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  pe.x = Swap16IfLE(x);
  pe.y = Swap16IfLE(y);
  return writeExact(sock->socket(), (char *)&pe, sz_rfbPointerEventMsg);

}


bool kRfbProto::writeKeyEvent(CARD32 key, bool down) {

    rfbKeyEventMsg ke;

    ke.type = rfbKeyEvent;
    ke.down = down ? 1 : 0;
    ke.key = Swap32IfLE(key);
    return writeExact(sock->socket(), (char *)&ke, sz_rfbKeyEventMsg);

}


void kRfbProto::PrintPixelFormat(rfbPixelFormat *format)
{
  cout << "PrintPixelFormat" << endl;
    if (format->bitsPerPixel == 1) {
	fprintf(stderr,"Single bit per pixel.\n");
	fprintf(stderr,
		"%s significant bit in each byte is leftmost on the screen.\n",
		(format->bigEndian ? "Most" : "Least"));
    } else {
	fprintf(stderr,"%d bits per pixel.\n",format->bitsPerPixel);
	if (format->bitsPerPixel != 8) {
	    fprintf(stderr,"%s significant byte first in each pixel.\n",
		    (format->bigEndian ? "Most" : "Least"));
	}
	if (format->trueColour) {
	    fprintf(stderr,"True colour: max red %d green %d blue %d\n",
		    format->redMax, format->greenMax, format->blueMax);
	    fprintf(stderr,"            shift red %d green %d blue %d\n",
		    format->redShift, format->greenShift, format->blueShift);
	} else {
	    fprintf(stderr,"Uses a colour map (not true colour).\n");
	}
    }
}



bool kRfbProto::readExact(int sock, char *buf, int n)
{
    int i = 0;
    int j;

    if(sock < 1) {
      printf("socket failure\n");
      emit fatal();
      return false;
    }
      
    while (i < n) {
      //printf("offset: 0x%08x\n",buf);
      j = read(sock, buf + i, (n - i));
      //printf("read %i bytes to offset: 0x%08x\n",j,(buf+i));
      if (j <= 0) {
	if (j < 0) {
	  perror(": read");
	  emit fatal();
	}
	return false;
	emit fatal();
	}
      i += j;
    }
    //printf("Leaving readSocket\n");
    return true;
}



/*
 * Write an exact number of bytes, and don't return until you've sent them.
 */

bool kRfbProto::writeExact(int sock, char *buf, int n)
{
    int i = 0;
    int j;
    
    if(sock <= 0) {
      cout << "Socket <= 0!\n";
      emit fatal();
      return false;
    }

    while (i < n) {
	j = write(sock, buf + i, (n - i));
	if (j <= 0) {
	    if (j < 0) {
		perror(": write");
		emit fatal();
	    } 
	    return false;
	    emit fatal();
	}
	i += j;
    }
    return true;
}













