/***************************************************************************
                          arraydlg.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfileinfo.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qtextstream.h>
#include <kapp.h>
#if (KDE_VERSION_MAJOR < 3) || (KDE_VERSION_MINOR < 9)
#include <qmultilineedit.h>
#endif
#include <klocale.h>
#include <kurl.h>
#include <kfiledialog.h>
#include <kcolorbtn.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include "arraydlg.h"
#include "kpldoc.h"
#include "symboldlg.h"
#include "arrayitem.h"
#include "kpldoubleedit.h"
#include "kplspinbox.h"
#include "editdlg.h"
#include "utils.h"

ArrayDlg::ArrayDlg(QWidget* _parent, KplDoc* model, ArrayItem* _ad) :
 KDialogBase(Plain, i18n("Array"), Help | Ok | Apply | Cancel, Ok, _parent, 0,
             true, true),
 m(model), ad(_ad), adt(0)
{
  QFrame* frame = plainPage();
  QGridLayout* grid = new QGridLayout(frame, 8, 6, spacingHint());
  QPushButton* b = new QPushButton(i18n("&File"), frame);
  grid->addWidget(b, 0, 0);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  QHBoxLayout* hbox = new QHBoxLayout(spacingHint());
  hbox->addWidget(fileName = new QLabel(frame));
  if (!ad->internal)
    fileName->setText(ad->url.isLocalFile() ?  ad->url.path() : ad->url.url());
  fileName->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  fileName->setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding,
                                      QSizePolicy::Minimum));
  hbox->addWidget(bReload = new QPushButton(frame));
  bReload->setPixmap(BarIcon("reload"));
  bReload->setDisabled(fileName->text().isEmpty());
  connect(bReload, SIGNAL(clicked()), SLOT(slotReload()));
  grid->addMultiCellLayout(hbox, 0, 0, 1, 5);
  grid->addMultiCellWidget(internal = new QCheckBox(i18n("Internal data"),
                                                    frame), 1, 1, 1, 2);
  internal->setChecked(ad->internal);
  grid->addMultiCellWidget(edit = new QPushButton(i18n("&Edit"), frame),
                           1, 1, 4, 5);
  edit->setEnabled(ad->internal);
  connect(edit, SIGNAL(clicked()), SLOT(slotEdit()));
  grid->addWidget(new QLabel(i18n("x column"), frame), 2, 0);
  grid->addWidget(ix = new KplSpinBox(0, 0, 1, frame), 2, 1);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 2, 3);
  grid->addWidget(new QLabel(i18n("y column"), frame), 2, 4);
  grid->addWidget(iy = new KplSpinBox(0, 0, 1, frame), 2, 5);
  grid->addWidget(new QLabel(i18n("Error column"), frame), 3, 0);
  grid->addWidget(ie = new KplSpinBox(0, 0, 1, frame), 3, 1);
  ie->setEnabled(ad->errbars);
  grid->addMultiCellWidget(err = new QCheckBox(i18n("Error bar"), frame),
                           3, 3, 2, 5);
  err->setChecked(ad->errbars);
  grid->addWidget(new QLabel(i18n("Start index"), frame), 4, 0);
  grid->addWidget(iStart = new KplSpinBox(0, 0, 1, frame), 4, 1);
  grid->addWidget(new QLabel(i18n("Number of points"), frame), 4, 4);
  grid->addWidget(n = new KplSpinBox(0, 0, 1, frame), 4, 5);
  grid->addWidget(new QLabel(i18n("x normalization"), frame), 5, 0);
  grid->addWidget(efx = new KplDoubleEdit(ad->fx, frame), 5, 1);
  grid->addWidget(new QLabel(i18n("y normalization"), frame), 5, 4);
  grid->addWidget(efy = new KplDoubleEdit(ad->fy, frame), 5, 5);
  grid->addWidget(new QLabel(i18n("Symbol"), frame), 6, 0);
  grid->addWidget(symb = new KplSpinBox(-17, 9, 1, frame), 6, 1);
  symb->setValue(ad->symb);
  SymbolButton *bSymb = new SymbolButton(frame, symb);
  bSymb->setFixedWidth(50);
  grid->addWidget(bSymb, 6, 2);
  grid->addWidget(new QLabel(i18n("Size"), frame), 7, 0);
  grid->addWidget(eRelSize = new KplSpinBox(10, 800, 1, frame), 7, 1);
  eRelSize->setValue(qRound(100 * ad->relsiz));
  eRelSize->setSuffix(" %");
  grid->addWidget(new QLabel(i18n("Color"), frame), 7, 4);
  grid->addWidget(colData = new KColorButton(ad->color, frame), 7, 5);
  colData->setMinimumWidth(50);
  setSpinBoxes(ad);
  Utils::setSize(this, "ArrayDialog");
#if KDE_VERSION == 303
  connect(this, SIGNAL(helpClicked()), SLOT(slotHelp2()));
#else
  setHelp("SEC-ARRAY");
#endif
  connect(internal, SIGNAL(toggled(bool)), SLOT(slotInternal(bool)));
  connect(err, SIGNAL(toggled(bool)), ie, SLOT(setEnabled(bool)));
  connect(iStart, SIGNAL(valueChanged(int)), SLOT(slotStartChanged(int)));
  connect(n, SIGNAL(valueChanged(int)), SLOT(slotNChanged(int)));
}

ArrayDlg::~ArrayDlg()
{
  Utils::saveSize(this, "ArrayDialog");
  delete adt;
}

void ArrayDlg::setSpinBoxes(ArrayItem* _ad)
{
  int iColMax = QMAX(0, _ad->ncols - 1);
  ix->setRange(0, iColMax);
  ix->setValue(_ad->ix);
  iy->setRange(0, iColMax);
  iy->setValue(_ad->iy);
  ie->setRange(0, iColMax);
  ie->setValue(_ad->ie);
  iStart->setRange(0, QMAX(0, _ad->nrows - 1));
  iStart->setValue(_ad->istart);
  n->setRange(0, _ad->nrows);
  n->setValue(_ad->n);
}

bool ArrayDlg::load(KURL &url)
{
  bool success = !url.isEmpty();
  if (success) {
    delete adt;
    adt = new ArrayItem;
    adt->url = url;
    success = readData();
  }
  return success;
}

void ArrayDlg::getValues(bool ok)
{
  if (adt) {
    *ad = *adt;
    delete adt;
    adt = 0;
  }
  ad->fx = efx->value();
  ad->fy = efy->value();
  ad->color = colData->color().rgb();
  ad->ix = ix->interpretedValue();
  ad->iy = iy->interpretedValue();
  ad->ie = ie->interpretedValue();
  ad->symb = symb->interpretedValue();
  ad->relsiz = 0.01 * eRelSize->interpretedValue();
  ad->istart = iStart->interpretedValue();
  ad->url = fileName->text();
  ad->errbars = err->isChecked();
  ad->internal = internal->isChecked();
  ad->istart = QMAX(ad->istart, 0);
  ad->n = QMIN(n->interpretedValue(), ad->nrows - ad->istart);
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
  else
    setSpinBoxes(ad);
}

void ArrayDlg::slotStartChanged(int ia)
{
  int nmax = ad->nrows - ia;
  if (n->interpretedValue() > nmax)
    n->setValue(nmax);
}

void ArrayDlg::slotNChanged(int na)
{
  int imin = ad->nrows - na;
  if (iStart->interpretedValue() > imin)
    iStart->setValue(imin);
}

void ArrayDlg::slotFile()
{
  KURL url = KFileDialog::getOpenURL(m->currentDir(), "*.dat *.DAT\n*");
  if (load(url)) {
    fileName->setText(url.isLocalFile() ? url.path() : url.url());
    m->setCurrentDir(url);
    bReload->setDisabled(fileName->text().isEmpty());
    disconnect(internal, SIGNAL(toggled(bool)), this,
               SLOT(slotInternal(bool)));
    internal->setChecked(false);
    connect(internal, SIGNAL(toggled(bool)), SLOT(slotInternal(bool)));
  }
}

void ArrayDlg::slotReload()
{
  KURL url(fileName->text());
  load(url);
}

void ArrayDlg::slotInternal(bool state)
{
  if (state) {
    ArrayItem* a2 = 0;
    if (adt)
      a2 = new ArrayItem(*adt);
    delete adt;
    adt = new ArrayItem;
    ArrayItem* src = a2 ? a2 : ad;
    adt->url = KURL();
    adt->nrows = src->nrows;
    adt->ncols = 3;
    adt->istart = iStart->interpretedValue();
    adt->n = n->interpretedValue();
    int dim = adt->ncols * adt->nrows;
    if (dim) {
      double* x1 = new double[dim];
      memset(x1, 0, dim * sizeof(double));
      adt->x = new double*[adt->ncols];
      for (int j = 0; j < adt->ncols; j++)
        adt->x[j] = &x1[j * adt->nrows];
      if (src->ncols > 3) {
        int siz = adt->nrows * sizeof(double);
        memcpy(adt->x[0], src->x[ix->interpretedValue()], siz);
        memcpy(adt->x[1], src->x[iy->interpretedValue()], siz);
        memcpy(adt->x[2], src->x[ie->interpretedValue()], siz);
        adt->iy = 1;
        adt->ie = 2;
      } else {
        memcpy(adt->x[0], src->x[0], src->ncols * src->nrows * sizeof(double));
        adt->ix = ix->interpretedValue();
        adt->iy = iy->interpretedValue();
        adt->ie = ie->interpretedValue();
      }
    }
    setSpinBoxes(adt);
    fileName->clear();
    delete a2;
  } else {
    ArrayItem* src = adt ? adt : ad;
    KURL url;
    if (Utils::getWriteURL(this, url, "*.dat\n*", m)) {
      QFile f(url.isLocalFile() ? url.path() : m->tmpFile());
      bool success = f.open(IO_WriteOnly);
      if (success) {
        QTextStream t(&f);
        for (int i = 0; i < src->nrows; i++)
          for (int j = 0; j < src->ncols; j++) {
            t << QString::number(src->x[j][i]);
            t << (j < (src->ncols - 1) ? " " : "\n");
          }
        f.close();
        fileName->setText(url.isLocalFile() ? url.path() : url.url());
        m->setCurrentDir(url);
        if (!url.isLocalFile())
          m->copyTmp(url);
      } else {
        KMessageBox::error(this, i18n("while trying to open file"));
        internal->setChecked(true);
        return;
      }
    } else {
      internal->setChecked(true);
      return;
    }
  }
  edit->setEnabled(state);
  bReload->setDisabled(fileName->text().isEmpty());
}

void ArrayDlg::slotEdit()
{
  QTextStream t(&buf, IO_WriteOnly);
  ArrayItem* src = adt ? adt : ad;
  for (int i = 0; i < src->nrows; i++)
    for (int j = 0; j < src->ncols; j++)
      t << m->number(src->x[j][i])
        << (j < (src->ncols - 1) ? m->separator() : QString("\n"));
#if (KDE_VERSION_MAJOR > 2) || (KDE_VERSION_MINOR > 8)
  EditDlg dlg(this, &buf, m->options()->prec + 4);
#else
  int sav = QMultiLineEdit::defaultTabStop();
  QMultiLineEdit::setDefaultTabStop(m->options()->prec + 4);
  EditDlg dlg(this, &buf);
#endif
  connect(&dlg, SIGNAL(applyClicked()), SLOT(slotEditApply()));
  if (dlg.exec()) {
    delete adt;
    adt = new ArrayItem;
    readData(true);
  }
#if (KDE_VERSION_MAJOR < 3) && (KDE_VERSION_MINOR < 9)
  QMultiLineEdit::setDefaultTabStop(sav);
#endif
}

bool ArrayDlg::readData(bool fromBuffer)
{
  if (fromBuffer)
    adt->nrows = ArrayItem::readFile(&buf, &adt->ncols, &adt->x);
  else
    adt->nrows = ArrayItem::readFile(adt->url, &adt->ncols, &adt->x);
  if (adt->nrows) {
    int maxCol = adt->ncols - 1;
    adt->ix = QMIN(ad->ix, maxCol);
    adt->iy = QMIN(ad->iy, maxCol);
    adt->ie = QMIN(ad->ie, maxCol);
    adt->istart = QMIN(ad->istart, adt->nrows - 1);
    adt->n = adt->nrows - adt->istart;
    if (ad->n && (!fromBuffer))
      adt->n = QMIN(ad->n, adt->n);
    setSpinBoxes(adt);
    return adt->nrows;
  } else {
    delete adt;
    adt = 0;
    KMessageBox::error(this, i18n("while reading the data"));
    return 0;
  }
}

void ArrayDlg::slotEditApply()
{
  delete adt;
  adt = new ArrayItem;
  if (readData(true))
    slotApply();
}

void ArrayDlg::slotOk()
{
  getValues(true);
}

void ArrayDlg::slotApply()
{
  getValues(false);
}

void ArrayDlg::slotHelp2()
{
#if KDE_VERSION == 303
  kapp->startServiceByDesktopName("konqueror", QString("help:/kpl/array.html"),
                                  0, 0, 0, "", true);
#endif
}
