/***************************************************************************
                          funitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Aug 29 1999
    copyright            : (C) 2003 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qurl.h>
#include <klocale.h>
#include <ksimpleconfig.h>
#include <kmessagebox.h>
#include "funitem.h"
#include "funcdlg.h"
#include "kpldoc.h"
#include "kplgraph.h"
#include "kplchecklistitem.h"
#include "utils.h"

FunItem::FunItem() : logxo(false), tmin(0.0), tmax(0.0), dt(0.0),
 tmino(0.0), tmaxo(0.0), dto(0.0), fkty(0), fktyo(0), hmody(0)
{
  init();
}

FunItem::FunItem(const FunItem& f) :
 ScaledItem(f), logxo(f.logxo), tmin(f.tmin), tmax(f.tmax), dt(f.dt),
 tmino(f.tmino), tmaxo(f.tmaxo), dto(f.dto), fkty(f.fkty), fktyo(f.fktyo),
 tv(f.tv), yv(f.yv), namey(f.namey), pathy(f.pathy), hmody(f.hmody)
{
  lt_dlinit();
  memcpy(py, f.py, sizeof(py));
  memcpy(pyo, f.pyo, sizeof(pyo));
  tv.detach();
  yv.detach();
  if (f.hmody)
    getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

FunItem::FunItem(Kpl::AutoStruct* aut) : ScaledItem(aut), logxo(false),
 tmin(0.0), tmax(0.0), dt(0.0), tmino(0.0), tmaxo(0.0), dto(0.0),
 fkty(0), fktyo(0), hmody(0)
{
  init();
}

FunItem::FunItem(KSimpleConfig* plo, Kpl::AutoStruct* aut, const KURL& uPlo) :
 ScaledItem(plo, aut), logxo(false), tmino(0.0), tmaxo(0.0), dto(0.0),
 fktyo(0)
{
  lt_dlinit();
  tmin = plo->readDoubleNumEntry("xfmin");
  tmax = plo->readDoubleNumEntry("xfmax", tmin + 1.0);
  dt = plo->readDoubleNumEntry("dx");
  memset(pyo, 0, sizeof(pyo));
  QStringList list = plo->readListEntry("p", ' ');
  int cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    py[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  namey = plo->readEntry("name", "");
  QString s = plo->readEntry("path", "");
  pathy = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

FunItem::FunItem(bool act, int sym, const QString& col,
                 double xn, double yn, double xmin, double xmax, double dx,
                 const QString& name, const KURL& path, double relSize)
 : ScaledItem(act, sym, col, xn, yn, relSize), logxo(false), tmin(xmin),
 tmax(xmax), dt(dx), tmino(0.0), tmaxo(0.0), dto(0.0), fkty(0), fktyo(0),
 namey(name), pathy(path), hmody(0)
{
  init();
  getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

FunItem::~FunItem()
{
  if (hmody)
    lt_dlclose(hmody);
  lt_dlexit();
}

FunItem& FunItem::operator=(const FunItem& f)
{
  if (this != &f) {
    *(ScaledItem*)this = f;
    logxo = f.logxo;
    tmin = f.tmin;
    tmax = f.tmax;
    dt = f.dt;
    tmino = f.tmino;
    tmaxo = f.tmaxo;
    dto = f.dto;
    fkty = f.fkty;
    fktyo = f.fktyo;
    memcpy(py, f.py, sizeof(py));
    memcpy(pyo, f.pyo, sizeof(pyo));
    tv = f.tv;
    tv.detach();
    yv = f.yv;
    yv.detach();
    namey = f.namey;
    pathy = f.pathy;
    if (hmody)
      lt_dlclose(hmody);
    hmody = 0;
    if (f.hmody)
      getFuncAddr(pathy.path(), namey, &hmody, &fkty);
  }
  return *this;
}

KplItem::ItemTypes FunItem::iType() const
{
  return Function;
}

void FunItem::draw(KplGraph* g)
{
  if (fkty && active) {
    setProperties(g);
    double sav = g->setRelSize(relsiz);
    g->setRelSize(relsiz * sav);
    int n = calcTable(g->logx);
    g->plArray(tv.data(), yv.data(), fx, fy, n);
    g->setRelSize(sav);
  }
}

void FunItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                       KplDoc* m) const
{
  plo->writeEntry("Type", "FUNITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  char frm = m->options()->format;
  int prec = m->options()->prec;
  plo->writeEntry("xfmin", tmin, true, false, frm, prec);
  plo->writeEntry("xfmax", tmax, true, false, frm, prec);
  plo->writeEntry("dx", dt, true, false, frm, prec);
  QStrList list;
  for (int i = 0; i < KPL_NPMAX; i++)
    list.insert(i, m->number(py[i]));
  plo->writeEntry("p", list, ' ');
  plo->writeEntry("name", namey);
  plo->writeEntry("path", Utils::relPath(url, pathy, _abs));
}

void FunItem::setText(KplCheckListItem* it, bool*, bool* funcs) const
{
  *funcs = true;
  it->setText(1, i18n("Function"));
  QFileInfo fi(pathy.path());
  it->setText(2, QString("y = ") + fi.baseName() + "." + namey + "(x)");
}

int FunItem::editItem(QWidget* parent, KplDoc* m, int)
{
  FuncDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* FunItem::copy() const
{
  return new FunItem(*this);
}

void FunItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (fkty) {
    Utils::expo(QMAX(fabs(tmin), fabs(tmax)), iext, fxt);
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void FunItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (fkty) {
    *xmi = tmin;
    *xma = tmax;
    int n = calcTable(logxo);
    Utils::minMaxFile(ymi, yma, yv, n);
  }
}

void FunItem::setPar(int i, double value, bool yFun)
{
  if (yFun)
    py[i] = value;
}

void FunItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  int n = calcTable(logxo);
  for (int i = 0; i < n; i++)
    ts << m->number(fx * tv[i]) << m->separator()
       << m->number(fy * yv[i]) << "\n";
}

bool FunItem::getFuncAddr(const QString& path, const QString& name,
                          lt_dlhandle* hmod,
                          double (**fkt)(double, const double*))
{
  *hmod = 0;
  *fkt = 0;
  if (path.isEmpty() || name.isEmpty())
    KMessageBox::sorry(0, i18n("no library or no function specified!"));
  else
    if (!(*hmod = lt_dlopen(path)))
      KMessageBox::error(0, lt_dlerror());
    else
      if ((*fkt = (double (*)(double, const double *)) lt_dlsym(*hmod, name)))
        return false;
      else {
        KMessageBox::error(0, lt_dlerror());
        lt_dlclose(*hmod);
        *hmod = 0;
      }
  return true;
}

void FunItem::init()
{
  lt_dlinit();
  memset(py, 0, sizeof(py));
  memset(pyo, 0, sizeof(pyo));
}

int FunItem::calcTable(bool logx) const
{
  bool newv = tv.isNull();
  double d;
  int n;
  if (dt) {
    n = int(fabs((tmax - tmin) / dt)) + 1;
    d = logx ? (log10(tmax / tmin) / (n - 1)) : dt;
  } else {
    n = 101;
    d = 0.01 * (logx ? log10(tmax / tmin) : (tmax - tmin));
  }
  int i;
  if (newv || (tmin != tmino) || (tmax != tmaxo) || (d != dto) ||
      (logx != logxo)) {
    tv.detach();
    tv.resize(n);
    for (i = 0; i < n; i++)
      tv[i] = logx ? pow(10.0, log10(tmin) + i * d) : (tmin + i * d);
  }
  bool pchanged = false;
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (py[i] != pyo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fkty != fktyo) || (logx != logxo)) {
    yv.detach();
    yv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fkty != fktyo) || (logx != logxo))
    for (i = 0; i < n; i++)
      yv[i] = ((*fkty)(tv[i], py));
  if (pchanged)
    for (i = 0; i < KPL_NPMAX; i++)
      pyo[i] = py[i];
  if (fkty != fktyo)
    fktyo = fkty;
  if (tmin != tmino)
    tmino = tmin;
  if (tmax != tmaxo)
    tmaxo = tmax;
  if (d != dto)
    dto = d;
  if (logx != logxo)
    logxo = logx;
  return n;
}

ParFunItem::ParFunItem(): fktx(0), fktxo(0), hmodx(0)
{
  init();
}

ParFunItem::ParFunItem(const ParFunItem& f) :
 FunItem(f), fktx(f.fktx), fktxo(f.fktxo), xv(f.xv),
 namex(f.namex), pathx(f.pathx), hmodx(f.hmodx)
{
  memcpy(px, f.px, sizeof(px));
  memcpy(pxo, f.pxo, sizeof(pxo));
  xv.detach();
  if (f.hmodx)
    getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
}

ParFunItem::ParFunItem(Kpl::AutoStruct* aut) :
 FunItem(aut), fktx(0), fktxo(0), hmodx(0)
{
  init();
}

ParFunItem::ParFunItem(KSimpleConfig* plo, Kpl::AutoStruct* aut,
                       const KURL& uPlo) : fktx(0), fktxo(0), hmodx(0)
{
  active = plo->readBoolEntry("active", true);
  ScaledItem::readPlo(plo, aut);
  tmin = plo->readDoubleNumEntry("tmin");
  tmax = plo->readDoubleNumEntry("tmax", tmin + 1.0);
  dt = plo->readDoubleNumEntry("dt");
  QStringList list = plo->readListEntry("px", ' ');
  int cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    px[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  list = plo->readListEntry("py", ' ');
  cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    py[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  namex = plo->readEntry("namex", "");
  namey = plo->readEntry("namey", "");
  QString s = plo->readEntry("pathx", "");
  pathx = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  s = plo->readEntry("pathy", "");
  pathy = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
  getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

ParFunItem::ParFunItem(bool act, int sym, const QString& col,
                       double xn, double yn, double tmi, double tma, double d,
                       const QString& name_x, const KURL& path_x,
                       const QString& name_y, const KURL& path_y,
                       double relSize) :
 FunItem(act, sym, col, xn, yn, tmi, tma, d, name_y, path_y, relSize), fktxo(0),
 namex(name_x), pathx(path_x)
{
  init();
  getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
}

ParFunItem::~ParFunItem()
{
  if (hmodx)
    lt_dlclose(hmodx);
}

ParFunItem& ParFunItem::operator=(const ParFunItem& f)
{
  if (this != &f) {
    *(FunItem*)this = f;
    fktx = f.fktx;
    fktxo = f.fktxo;
    memcpy(px, f.px, sizeof(px));
    memcpy(pxo, f.pxo, sizeof(pxo));
    xv = f.xv;
    xv.detach();
    namex = f.namex;
    pathx = f.pathx;
    if (hmodx)
      lt_dlclose(hmodx);
    hmodx = 0;
    if (f.hmodx)
      getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
  }
  return *this;
}

KplItem::ItemTypes ParFunItem::iType() const
{
  return ParFunction;
}

void ParFunItem::draw(KplGraph* g)
{
  if (fkty && active) {
    setProperties(g);
    double sav = g->setRelSize(relsiz);
    g->setRelSize(relsiz * sav);
    int n = calcTable();
    g->plArray(xv.data(), yv.data(), fx, fy, n);
    g->setRelSize(sav);
  }
}

void ParFunItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                          KplDoc* m) const
{
  plo->writeEntry("Type", "PARFUNITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  char frm = m->options()->format;
  int prec = m->options()->prec;
  plo->writeEntry("tmin", tmin, true, false, frm, prec);
  plo->writeEntry("tmax", tmax, true, false, frm, prec);
  plo->writeEntry("dt", dt, true, false, frm, prec);
  QStrList lx, ly;
  for (int i = 0; i < KPL_NPMAX; i++) {
    lx.insert(i, m->number(px[i]));
    ly.insert(i, m->number(py[i]));
  }
  plo->writeEntry("px", lx, ' ');
  plo->writeEntry("py", ly, ' ');
  plo->writeEntry("namex", namex);
  plo->writeEntry("namey", namey);
  plo->writeEntry("pathx", Utils::relPath(url, pathx, _abs));
  plo->writeEntry("pathy", Utils::relPath(url, pathy, _abs));
}

void ParFunItem::setText(KplCheckListItem* it, bool*, bool*) const
{
  it->setText(1, i18n("Par. function"));
  QFileInfo fi1(pathx.path());
  QFileInfo fi2(pathy.path());
  it->setText(2, QString("x = ") + fi1.baseName() + "." + namex
              + "(t), y = " + fi2.baseName() + "." + namey + "(t)");
}

int ParFunItem::editItem(QWidget* parent, KplDoc* m, int)
{
  ParFuncDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* ParFunItem::copy() const
{
  return new ParFunItem(*this);
}

void ParFunItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (fktx && fkty) {
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void ParFunItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (fktx && fkty) {
    int n = calcTable();
    Utils::minMaxFile(xmi, xma, xv, n);
    Utils::minMaxFile(ymi, yma, yv, n);
  }
}

void ParFunItem::setPar(int i, double value, bool yFun)
{
  if (yFun)
    py[i] = value;
  else
    px[i] = value;
}

void ParFunItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  int n = calcTable();
  for (int i = 0; i < n; i++)
    ts << m->number(fx * xv[i]) << m->separator()
       << m->number(fy * yv[i]) << "\n";
}

void ParFunItem::init()
{
  memset(px, 0, sizeof(px));
  memset(pxo, 0, sizeof(pxo));
}

int ParFunItem::calcTable() const
{
  bool newv = tv.isNull();
  double d;
  int n;
  if (dt) {
    n = int(fabs((tmax - tmin) / dt)) + 1;
    d = dt;
  } else {
    n = 101;
    d = 0.01 * (tmax - tmin);
  }
  int i;
  if (newv || (tmin != tmino) || (tmax != tmaxo) || (d != dto)) {
    tv.detach();
    tv.resize(n);
    for (i = 0; i < n; i++)
      tv[i] = tmin + i * d;
  }
  bool pchanged = false;
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (py[i] != pyo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fkty != fktyo)) {
    yv.detach();
    yv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fkty != fktyo))
    for (i = 0; i < n; i++)
      yv[i] = ((*fkty)(tv[i], py));
  if (pchanged) {
    for (i = 0; i < KPL_NPMAX; i++)
      pyo[i] = py[i];
    pchanged = false;
  }
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (px[i] != pxo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fktx != fktxo)) {
    xv.detach();
    xv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fktx != fktxo))
    for (i = 0; i < n; i++)
      xv[i] = ((*fktx)(tv[i], px));
  if (pchanged)
    for (i = 0; i < KPL_NPMAX; i++)
      pxo[i] = px[i];
  if (fktx != fktxo)
    fktxo = fktx;
  if (fkty != fktyo)
    fktyo = fkty;
  if (tmin != tmino)
    tmino = tmin;
  if (tmax != tmaxo)
    tmaxo = tmax;
  if (d != dto)
    dto = d;
  return n;
}
