/***************************************************************************
 *
 * knetworkmanager-vpnconnectionsdialog.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <stdlib.h>
#include <klistview.h>
#include <kpushbutton.h>
#include <qbuttongroup.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qwidgetstack.h>
#include <klineedit.h>
#include <qlabel.h>
#include <qlayout.h>
#include <kactivelabel.h>
#include <klocale.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kstddirs.h>
#include <kiconloader.h>

#include "knetworkmanager-vpnconnectionsdialog.h"
#include "knetworkmanager-vpn.h"
#include "knetworkmanager-vpn_dbus.h"
#include "knetworkmanager-storage.h"
#include "unknownservice.h"

/*
 * class VPNEditConnectionDialog
 *
 */
VPNEditConnectionDialog::VPNEditConnectionDialog(VPN* vpn, VPNConnection* conn, QWidget* parent, const char* name, bool modal, WFlags fl)
	: EditConnectionDialog(parent, name, modal, fl)
{
	bool newcon = !conn;
	_vpn = vpn;
	_vpnConnection = conn;
	_vpnConfig = NULL;

	// setup our dialog
	this->setCaption(conn ? i18n("Edit VPN connection") : i18n("Create new VPN connection") );
	this->setIcon(SmallIcon("encrypted", QIconSet::Automatic));
	
	// nice icons
	pushOK->setIconSet(SmallIcon("button_ok", QIconSet::Automatic));
	pushCancel->setIconSet(SmallIcon("button_cancel", QIconSet::Automatic));

	// decide wether to create a new connetion or to edit an already existing one
	if (newcon)
		labelInformation->setText(i18n("Please select the VPN service you want to create"));
	else
		labelInformation->setText("");

	// create the info-widget
	_widgetInfo = new QWidget(this->widgetStack);
	this->widgetStack->addWidget(_widgetInfo);
	this->widgetStack->raiseWidget(_widgetInfo);

	// create the "unknown service widget"
	_widgetUnknownService = new UnknownServiceWidget(this->widgetStack);
	if (!_vpn->hasGnomeVPNConfig())
	{
		_widgetUnknownService->groupBox->setTitle(i18n("No configuration interface found"));
		_widgetUnknownService->labelInfoText->setText(i18n("There is no configuration interface for the VPN service %1 installed. Please check your installation."));
	}
	else
	{
		_widgetUnknownService->groupBox->setTitle(i18n("No native configuration interface found"));
		_widgetUnknownService->labelInfoText->setText(i18n("You have the GNOME configuration applet installed and can use this to configure the VPN connection. Once done you can import the VPN configuration using KNetworkManager. Press OK to run the GNOME configuration applet."));
	}
	this->widgetStack->addWidget(_widgetUnknownService);
		
	// add all possible services to the combobox
	int index = 0;
	QStringList services = vpn->getVPNServices();
	for (QStringList::iterator i = services.begin(); i != services.end(); ++i)
	{
		QString icon;
		QString disp_name;

		// get service icon and display name
		VPNService* service = _vpn->getVPNService(*i);
		if (service)
		{
			icon = service->getIcon();
			disp_name = service->getDisplayName();
		}

		// no special icon for this service, use default
		if (icon.isEmpty())
			icon = "encrypted";

		if (newcon)
		{
			cboServices->insertItem(SmallIcon(icon, QIconSet::Automatic), disp_name, index);
			_mapServiceCombo.insert(index++, *i);
		}
		else
			if (conn->getServiceName() == *i)
			{
				cboServices->insertItem(SmallIcon(icon, QIconSet::Automatic), disp_name, index);
				_mapServiceCombo.insert(index++, *i);
			}
	}

	// if a connection should be edited show the right widget
	if (!newcon)
	{
		editName->setText(conn->getName());
		cboServices->setEnabled(false);
		VPNConfigWidget* conf = this->showVPNWidget(conn->getServiceName());
		if (conf)
			conf->setVPNData(_vpnConnection->getRoutes(), _vpnConnection->getData());
	}
	else
	{
		// select the first VPN Service
		showVPNWidget(services.first());
	}

	this->resize(minimumSizeHint());
}

VPNEditConnectionDialog::~VPNEditConnectionDialog()
{

}

void VPNEditConnectionDialog::done(int r)
{
	emit done();
	EditConnectionDialog::done(r);
}

bool VPNEditConnectionDialog::isValid(QStringList& err_msg)
{
	bool retval = true;
	if (editName->text() == "")
	{
		err_msg.append(i18n("You have to provide a name for this VPN Connection"));
		retval = false;
	}
	return retval;
}

void VPNEditConnectionDialog::cancel()
{
	this->done(1);
}

void VPNEditConnectionDialog::ok()
{
	bool newConn = false;
	bool valid = true;
	QStringList err_msg;

	if (_vpnConfig )
	{
		// Name etc. valid?
		valid = this->isValid(err_msg);

		// we need to save data only if it changed
		if (_vpnConfig->hasChanged())
		{
			// is the configuration Valid?
			valid = _vpnConfig->isValid(err_msg) && valid;
			if ( valid )
			{
				if (!_vpnConnection)
				{
					// ah, a new connection, lets create it
					_vpnConnection = _vpn->newVPNConnection();
					_vpnConnection->setServiceName(_mapServiceCombo[cboServices->currentItem()]);
					newConn = true;
				}

				// set VPN attribs
				_vpnConnection->setName(editName->text());
				_vpnConnection->setData(_vpnConfig->getVPNProperties());
				_vpnConnection->setRoutes(_vpnConfig->getVPNRoutes());

				// save connection
				_vpnConnection->save();
				if (newConn)
					if (!_vpn->appendVPNConnection(_vpnConnection))
					{
						delete _vpnConnection;
						_vpnConnection = NULL;
					}
			}
			else
			{
				err_msg.push_front(i18n("Configuration Error"));
				QMessageBox::critical(this, i18n("Configuration Error"), err_msg.join("\n"));
				return;
			}
		}
	}
	else if (_widgetUnknownService == widgetStack->visibleWidget())
	{
		_vpn->startGnomeVPNConfig();
	}
	this->done(0);
}

void VPNEditConnectionDialog::serviceSelected(int index)
{
	this->showVPNWidget(_mapServiceCombo[index]);
}

VPNConfigWidget* VPNEditConnectionDialog::showVPNWidget(const QString& service)
{
	QMap<QString, VPNConfigWidget*>::Iterator it;
	VPNConfigWidget* vpnConfig = NULL;

	if (service == "")
		// show dummy widget
		this->widgetStack->raiseWidget(1);
	else
	{
		// look if we have already a widget loaded to handle this service
		if ((it = _serviceWidgetMap.find(service)) != _serviceWidgetMap.end())
		{
			// yeahha, here is it
			vpnConfig = dynamic_cast<VPNConfigWidget*>(*it);
		}
		else
		{
			// we have to load the widget
			VPNService* vpnService = _vpn->getVPNService(service);
			if(vpnService)
			{
				VPNPlugin* vpnPlugin = vpnService->getVPNPlugin();
				if (vpnPlugin)
					if ( (vpnConfig = vpnPlugin->CreateConfigWidget(this->widgetStack)) )
					{
						// store widgetpointer so we dont need to load it next time
						_serviceWidgetMap.insert(service, vpnConfig);
					}
			}
		}

		if (vpnConfig)
		{
			this->widgetStack->raiseWidget(vpnConfig);
			this->editName->setEnabled(true);
			pushOK->setEnabled(true);
		}
		else
		{
			if (!_vpn->hasGnomeVPNConfig())
			{
				_widgetUnknownService->labelInfoText->setText(i18n("There is no configuration interface for the VPN service %1 installed. Please check your installation.").arg(service));
				pushOK->setEnabled(false);
			}
			else
				pushOK->setEnabled(true);
			this->widgetStack->raiseWidget(_widgetUnknownService);
			this->editName->setEnabled(false);
		}

	}
	_vpnConfig = vpnConfig;
	return vpnConfig;
}



/*
 * class VPNConnectionsDialog
 *
 */
VPNConnectionsDialog::VPNConnectionsDialog( KNetworkManager* ctx, QWidget* parent, const char* name, bool modal , WFlags fl )
 : ManageConnectionsDialog( parent, name, modal, fl )
{
	_ctx = ctx;
	_vpn = _ctx->getVPN();
	this->setCaption(i18n("Manage VPN Connections"));
	this->setIcon(SmallIcon("encrypted", QIconSet::Automatic));

	// put nice icons on the buttons
	pushEdit->setIconSet(SmallIcon("edit", QIconSet::Automatic));
	pushRemove->setIconSet(SmallIcon("remove", QIconSet::Automatic));
	pushClose->setIconSet(SmallIcon("button_ok", QIconSet::Automatic));
	pushAdd->setIconSet(SmallIcon("add", QIconSet::Automatic));
	pushImport->setIconSet(SmallIcon("fileimport", QIconSet::Automatic));

	// show all connections
	this->refreshList();

	connect(_vpn, SIGNAL(vpnConnectionsUpdated()), this, SLOT(refreshList()));
}

VPNConnectionsDialog::~VPNConnectionsDialog()
{

}

void VPNConnectionsDialog::refreshList()
{
	getAvailableConnections();

	// are there any GConf Connections which are not imported yet
	QStringList gconfconns = _vpn->listGConfConnections();
	if (gconfconns.empty())
		pushImport->hide();
	else
		pushImport->show();
}

void VPNConnectionsDialog::getAvailableConnections()
{
	// get selected connection
	QListViewItem* old_lvi = klvConnections->selectedItem();
	QString conn_name;
	if (old_lvi)
		conn_name = old_lvi->text(0);

	// retrieve a list of available vpn connections
	klvConnections->clear();
	VPNList *vpnList = _vpn->getVPNList();
	QListViewItem* lvi = NULL; 
	// append all known connections to our listview
	for (VPNList::iterator i = vpnList->begin (); i != vpnList->end (); ++i) {
		QString icon = "encrypted";
		// get icon for this connection
		VPNService* service = (*i)->getVPNService();
		if (service)
			icon = service->getIcon();
		lvi = new QListViewItem(this->klvConnections);
		lvi->setPixmap(0, SmallIcon(icon, QIconSet::Automatic));
		lvi->setText(0, (*i)->getName());
		lvi->setText(1, service->getDisplayName());
		// restore last selected or the first one 
		if ((*i)->getName() == conn_name )
			klvConnections->setSelected(lvi, true);
	}
	if (klvConnections->selectedItem() == NULL)
		klvConnections->setSelected(klvConnections->firstChild(), true);

}

void VPNConnectionsDialog::connectionSelected(QListViewItem* lvi)
{
	VPNConnection* conn = _vpn->getVPNConnection(lvi->text(0));
	if(conn)
	{
		VPNService* service = conn->getVPNService();
		VPNPlugin* plugin = NULL;
		if (service) plugin = service->getVPNPlugin();

		bool readonly = conn->isReadonly();
		pushImport->setEnabled(readonly && plugin);
		pushRemove->setEnabled(!readonly);
	}
}

void VPNConnectionsDialog::addConnection()
{
	// open dialog for creating a new connection
	VPNEditConnectionDialog* ecd = new VPNEditConnectionDialog(_vpn, NULL, this, 0, FALSE, WDestructiveClose);
	if (ecd->result() == 0)
	{
		connect(ecd, SIGNAL(done()), this, SLOT(refreshList()));
		ecd->show();
	}
}

void VPNConnectionsDialog::editConnection()
{
	QListViewItem* lvi = this->klvConnections->selectedItems().first();
	if (lvi)
	{
		QString conn_name = lvi->text(0);
		// get the connection
		VPNConnection* conn = _vpn->getVPNConnection (conn_name);
		if (conn)
		{
			VPNService* service = conn->getVPNService();
			VPNPlugin* plugin = NULL;
			if (service) plugin = service->getVPNPlugin();
			
			// if no plugin is available ask to start the gnome vpn manager
			if (!plugin && conn->isReadonly() && _vpn->hasGnomeVPNConfig())
			{
				if (KMessageBox::Yes == KMessageBox::questionYesNo(this, i18n("The selected connection is read from an external file and no plugin for KNetworkManager is available. Do you wish to start the GNOME configuration applet?")))
					_vpn->startGnomeVPNConfig();
			}
			// if the connection is readonly we have to import it for editing
			else if (plugin)
			{
				if (conn->isReadonly()) 
				{
					if (KMessageBox::No == KMessageBox::questionYesNo(this, i18n("The selected connection is read from an external file. Do you wish to import it now?")))
						return;
					else
						importVPNConnection(conn_name);
				}
			
				// open the editdialog
				VPNEditConnectionDialog* ecd = new VPNEditConnectionDialog(_vpn, conn, this, 0, FALSE, WDestructiveClose);
				if (ecd->result() == 0)
				{	
					connect(ecd, SIGNAL(done()), this, SLOT(refreshList()));
					ecd->show();
				}
			}
			else
			{
				KMessageBox::sorry(this, i18n("There is no configuration interface for the VPN service %1 installed. Please check your installation.").arg(service->getDisplayName()));
			}
		}
	}
}

void VPNConnectionsDialog::removeConnection()
{
	QListViewItem* lvi = this->klvConnections->selectedItems().first();
	if (!lvi) return;

	if (KMessageBox::Yes == KMessageBox::questionYesNo(this, QString("Do you really want to delete the VPN Connection \"%1\"").arg(lvi->text(0))))
	{
		QString conn_name = lvi->text(0);
		// delete the connection
		_vpn->deleteVPNConnection(conn_name);
		// remove item from listview
		klvConnections->takeItem(lvi);
		delete lvi;
	}
}

void VPNConnectionsDialog::close()
{
	this->done(0);
}

void VPNConnectionsDialog::done(int n)
{
	disconnect(_vpn, SIGNAL(vpnConnectionsUpdated()), this, SLOT(refreshList()));
	emit(done());
	ManageConnectionsDialog::done(n);
}

void VPNConnectionsDialog::apply()
{
	emit(done());
}

void VPNConnectionsDialog::importConnection()
{
	QListViewItem* lvi = this->klvConnections->selectedItems().first();
	if (!lvi) return;

	QString connectionName = lvi->text(0);
	importVPNConnection(connectionName);
}

void VPNConnectionsDialog::importVPNConnection(QString& connectionName)
{
	// import the connetion and refresh our list
	_vpn->importVPNConnection(connectionName);
	this->getAvailableConnections();
}

#include "knetworkmanager-vpnconnectionsdialog.moc"

