/***************************************************************************
                          kpl.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <qcursor.h>
#include <qcheckbox.h>
#include <qdir.h>
#include <qimage.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <kaccel.h>
#include <kaction.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kedittoolbar.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kimageio.h>
#include <kkeydialog.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#if KDE_VERSION >= 0x030100
#include <knotifydialog.h>
#endif
#include <kprocess.h>
#include <kstatusbar.h>
#include <kstdaction.h>
#include <kstandarddirs.h>
#include <kprinter.h>
#include "arcitem.h"
#include "arrayitem.h"
#include "ellipseitem.h"
#include "fitdlg.h"
#include "frameitem.h"
#include "funitem.h"
#include "imageitem.h"
#include "itemdlg.h"
#include "kpl.h"
#include "kpldoc.h"
#include "kplview.h"
#include "legenditem.h"
#include "lineitem.h"
#include "psgraph.h"
#include "refreshdlg.h"
#include "scalebaritem.h"
#include "settingsdlg.h"
#include "splfitdlg.h"
#include "splineitem.h"
#include "textitem.h"
#include "threeditem.h"
#include "utils.h"
#include "zoomdlg.h"

Kpl::Kpl() : KMainWindow(0, "Kpl"), DCOPObject("KplIface"),
 doc(new KplDoc(this)), config(kapp->config()), timer(new QTimer(this)),
 plot(false), tRefresh(5)
{
  KImageIO::registerFormats();
  setMinimumSize(120, 80);
  doc->newDocument();
  actionCollection()->setHighlightingEnabled(true);
  connect(actionCollection(), SIGNAL(actionStatusText(const QString&)),
          this, SLOT(slotStatusHelpMsg(const QString&)));
  KAction* a = KStdAction::openNew(this, SLOT(slotFileNew()),
                                   actionCollection());
  a->setToolTip(i18n("Creates a new plot"));
  a = new KAction(i18n("&Open Plot File..."), "fileopen", KStdAccel::open(),
                  this, SLOT(slotFileOpenPlot()), actionCollection(),
                  "open_plot");
  a->setToolTip(i18n("Opens an existing plot file"));
  a = new KAction(i18n("Open &Data File..."), "fileopen", 0, this,
                  SLOT(slotFileOpenData()), actionCollection(), "open_data");
  a->setToolTip(i18n("Opens an existing data file"));
  recentFiles = KStdAction::openRecent(this,
                                       SLOT(slotOpenRecent(const KURL&)),
                                       actionCollection());
  fileSave = KStdAction::save(this, SLOT(slotFileSave()), actionCollection());
  fileSave->setToolTip(i18n("Saves the current plot"));
  fileSaveAs = KStdAction::saveAs(this, SLOT(slotFileSaveAs()),
                                  actionCollection());
  fileSaveAs->setToolTip(i18n("Saves the plot under a different name"));
  fileClose = KStdAction::close(this, SLOT(slotFileClose()),
                                actionCollection());
  fileClose->setToolTip(i18n("Closes the current plot"));
  filePlot = new KAction(i18n("Display P&lot"), 0, this, SLOT(slotFilePlot()),
                         actionCollection(), "plot");
  filePlot->setToolTip(i18n("Displays the current plot"));
  filePSP = new KAction(i18n("&Portrait..."), 0, this, SLOT(slotFilePSP()),
                        actionCollection(), "postscript_p");
  filePSP->setToolTip(i18n("Generates PostScript file (portrait)"));
  filePSL = new KAction(i18n("&Landscape..."), 0, this, SLOT(slotFilePSL()),
                        actionCollection(), "postscript_l");
  filePSL->setToolTip(i18n("Generates PostScript file (landscape)"));
  filePreviewP = new KAction(i18n("&Portrait..."), 0, this,
                             SLOT(slotFilePreviewP()), actionCollection(),
                             "preview_p");
  filePreviewP->setToolTip(i18n("Displays PostScript preview (portrait)"));
  filePreviewL = new KAction(i18n("&Landscape..."), 0, this,
                             SLOT(slotFilePreviewL()), actionCollection(),
                             "preview_l");
  filePreviewL->setToolTip(i18n("Displays PostScript preview (landscape)"));
  filePrint = KStdAction::print(this, SLOT(slotFilePrint()),
                                actionCollection());
  filePrint->setToolTip(i18n("Prints the current plot"));
  a = new KAction(i18n("New &Window"), QIconSet(kapp->miniIcon()), 0, this,
                  SLOT(slotFileNewWindow()), actionCollection(), "new_window");
  a->setToolTip(i18n("Opens a new application window"));
  a = new KAction(i18n("Close W&indow"), 0, this, SLOT(slotFileCloseWindow()),
                  actionCollection(), "close_window");
  a->setToolTip(i18n("Closes the current window"));
  a = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
  a->setToolTip(i18n("Exits the program"));
  editUndo = KStdAction::undo(this, SLOT(slotEditUndo()), actionCollection());
  editUndo->setToolTip(i18n("Undoes the last change"));
  editRedo = KStdAction::redo(this, SLOT(slotEditRedo()), actionCollection());
  editRedo->setToolTip(i18n("Redoes the last change"));
  a = new KAction(i18n("&Items..."), 0, this, SLOT(slotEditItems()),
                  actionCollection(), "items");
  a->setToolTip(i18n("Edits plot items"));
  a = KStdAction::zoomIn(this, SLOT(slotViewZoomIn()), actionCollection());
  a->setToolTip(i18n("Increases magnification by 20 %"));
  a = KStdAction::zoomOut(this, SLOT(slotViewZoomOut()), actionCollection());
  a->setToolTip(i18n("Decreases magnification by 20 %"));
  a = KStdAction::zoom(this, SLOT(slotViewZoom()), actionCollection());
  a->setToolTip(i18n("Sets magnification"));
  viewReload = KStdAction::redisplay(this, SLOT(slotViewReload()),
                                     actionCollection());
  viewReload->setShortcut(KStdAccel::reload());
  viewReload->setToolTip(i18n("Reloads file"));
  a = new KAction(i18n("Reload &Periodically..."), 0, this,
                  SLOT(slotViewAutoReload()), actionCollection(),
                  "autoreload");
  a->setToolTip(i18n("Reloads file periodically"));
#if KDE_VERSION < 0x030100
  showToolbar = KStdAction::showToolbar(this, SLOT(slotOptionsToolBar()),
                                        actionCollection());
  showToolbar->setToolTip(i18n("Enables / disables the toolbar"));
#else
  setStandardToolBarMenuEnabled(true);
#endif
  showStatusbar = KStdAction::showStatusbar(this, SLOT(slotOptionsStatusBar()),
                                            actionCollection());
  showStatusbar->setToolTip(i18n("Enables / disables the statusbar"));
  autoPlot = new KToggleAction(i18n("Aut&oplot"), 0, this,
                               SLOT(slotOptionsAutoPlot()),
                               actionCollection(), "autoplot");
  autoPlot->setToolTip(i18n("Enables / disables automatic plotting"));
  addData = new KToggleAction(i18n("Add &Files"), Key_Insert, this,
                              SLOT(slotOptionsAddData()),
                              actionCollection(), "add_data");
  addData->setToolTip(i18n("Enables / disables adding of data and plot files"));
  showSource = new KToggleAction(i18n("Show Function Sou&rce"), 0, this,
                                 SLOT(slotOptionsShowSource()),
                                 actionCollection(), "show_source");
  showSource->setToolTip(i18n("Enables / disables display of "
                              "function source"));
  boundingBox = new KToggleAction(i18n("Calculate &PS Bounding Box"), 0,
                                 actionCollection(), "bounding_box");
  boundingBox->setToolTip(i18n("Enables / disables calculation of "
                               "PS bounding box"));
  printPS = new KToggleAction(i18n("Print PS Output"), 0,
                              actionCollection(), "print_ps");
  printPS->setToolTip(i18n("Enables / disables printing of "
                           "PostScript output files"));
  saveAbsPath = new KToggleAction(i18n("Save Absol&ute Paths"), 0,
                              actionCollection(), "save_abs_path");
  saveAbsPath->setToolTip(i18n("Saves absolute paths of data "
                               "and library files in plot files"));
  warning = new KToggleAction(i18n("Unsaved Changes &Warning"), 0,
                              actionCollection(), "warning");
  warning->setToolTip(i18n("Enables / disables warning on "
                           "unsaved changes"));
  saveAtEnd = new KToggleAction(i18n("Save Settings At &End"), 0,
                                actionCollection(), "save_at_end");
  saveAtEnd->setToolTip(i18n("Enables / disables saving settings at end"));
  a = KStdAction::keyBindings(this, SLOT(slotOptionsKeys()),
                              actionCollection());
  a->setToolTip(i18n("Displays key bindings dialog"));
  a = KStdAction::configureToolbars(this, SLOT(slotOptionsToolbars()),
                                    actionCollection());
  a->setToolTip(i18n("Displays toolbar configuration dialog"));
#if KDE_VERSION >= 0x030100
  a = KStdAction::configureNotifications(this,
                                         SLOT(slotOptionsNotifications()),
                                         actionCollection());
  a->setToolTip(i18n("Displays notification configuration dialog"));
#endif
  a = KStdAction::preferences(this, SLOT(slotOptionsConfigure()),
                              actionCollection());
  a->setToolTip(i18n("Displays the Kpl settings dialog"));
  a = KStdAction::saveOptions(this, SLOT(slotOptionsSave()),
                              actionCollection());
  a->setToolTip(i18n("Saves settings"));
  itemDlg = new ItemDlg(doc);
  config->setGroup(kapp->instanceName());
  config->setDollarExpansion();
  resize(config->readNumEntry("Width", 640),
         config->readNumEntry("Height", 512));
  recentFiles->loadEntries(config);
  showStatusbar->setChecked(config->readBoolEntry("ShowStatusbar", true));
  if (!showStatusbar->isChecked())
    statusBar()->hide();
  itemDlg->appendItems->setChecked(config->readBoolEntry("AppendItems", true));
  if (config->readBoolEntry("ShowItemDialog"))
    slotEditItems();
  toolBarPos = (KToolBar::BarPosition) config->readNumEntry("ToolBarPos",
                                                            KToolBar::Top);
  QString kplDir = KGlobal::dirs()->findResourceDir("appdata", "demo1.plo");
  doc->setCurrentDir(config->readEntry("Directory", kplDir) + "x");
  doc->aut.showSource = config->readBoolEntry("ShowSource");
  showSource->setChecked(doc->aut.showSource);
  addData->setChecked(config->readBoolEntry("AddData"));
  autoPlot->setChecked(config->readBoolEntry("AutoPlot", true));
  boundingBox->setChecked(config->readBoolEntry("BoundingBox"));
  printPS->setChecked(config->readBoolEntry("PrintPS"));
  saveAbsPath->setChecked(config->readBoolEntry("SaveAbsPath", true));
  warning->setChecked(config->readBoolEntry("Warning", true));
  saveAtEnd->setChecked(config->readBoolEntry("SaveAtEnd"));
  doc->aut.ixAuto = config->readNumEntry("AutoIndexX");
  doc->aut.iyAuto = config->readNumEntry("AutoIndexY", 1);
  doc->aut.ieAuto = config->readNumEntry("AutoIndexE", 2);
  doc->aut.autoNorm = config->readBoolEntry("AutoNorm", true);
  doc->aut.autoErr = config->readBoolEntry("AutoErrBar");
  doc->aut.autoLogX = config->readBoolEntry("AutoLogX");
  doc->aut.autoLogY = config->readBoolEntry("AutoLogY");
  doc->aut.autoGrid = config->readNumEntry("AutoGrid",
                                           KplGraph::AxesWithLabels);
  doc->aut.autohPath = config->readBoolEntry("AutoPathHeader", true);
  doc->aut.autoFit = config->readBoolEntry("AutoFit");
  doc->aut.autoSymb = config->readNumEntry("AutoSymbol", 1);
  doc->aut.x0Auto = config->readDoubleNumEntry("AutoX0", 4.0);
  doc->aut.y0Auto = config->readDoubleNumEntry("AutoY0", 3.0);
  doc->aut.xlAuto = config->readDoubleNumEntry("AutoXl", 15.0);
  doc->aut.ylAuto = config->readDoubleNumEntry("AutoYl", 10.0);
  doc->aut.relSize = config->readDoubleNumEntry("RelSize", 1.0);
  doc->aut.err.fitErrCol = config->readBoolEntry("FitErrorColumn");
  doc->aut.fitNonLin = config->readBoolEntry("FitNonLinear", true);
  doc->aut.fitShowDlg = config->readBoolEntry("FitShowDialog");
  doc->aut.fitSavePar = config->readBoolEntry("FitSaveParam");
  doc->aut.fitFollow = config->readBoolEntry("FitFollow");
  doc->aut.fitMaxIt = config->readNumEntry("FitMaxIterations", 100);
  doc->aut.fitTol = config->readDoubleNumEntry("FitTolerance", 1e-14);
  doc->aut.fitSymb = config->readNumEntry("FitSymbol", 1);
  doc->aut.fitColor = KplGraph::rgbQt(config->readUnsignedNumEntry("FitColor"));
  doc->aut.fitPath = config->readEntry("FitPath", kplDir + "fkt.so");
  doc->aut.fitName = config->readEntry("FitName", "polynom");
  doc->aut.err.errModPath = config->readEntry("ErrorModelPath",
                                              kplDir + "fkt.so");
  doc->aut.err.errModName = config->readEntry("ErrorModelName", "polynom");
  QStringList list = config->readListEntry("FitEnableParam", ' ');
  int cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    doc->aut.fitPar[i] = (i < cnt) ? list[i].toInt() : false;
  list = config->readListEntry("FitParam", ' ');
  cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    doc->aut.pFit[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  list = config->readListEntry("ErrorModelParam", ' ');
  cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    doc->aut.err.pErrMod[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  if (!cnt)
    doc->aut.err.pErrMod[0] = 0.01;
  doc->aut.err.errModArg = config->readNumEntry("ErrorModelArgument", 1);
  doc->aut.xf = config->readDoubleNumEntry("FormatX", 21.0);
  doc->aut.yf = config->readDoubleNumEntry("FormatY", 29.7);
  doc->aut.colFrame = KplGraph::rgbQt(config->readUnsignedNumEntry("ColorFrame"));
  doc->aut.colGrid = KplGraph::rgbQt(config->readUnsignedNumEntry("ColorGrid"));
  doc->aut.colData = KplGraph::rgbQt(config->readUnsignedNumEntry("ColorData"));
  doc->aut.format = *(config->readEntry("OutputFormat", "g").latin1());
  doc->aut.iDec = config->readNumEntry("DecimalSymbol");
  doc->aut.prec = config->readNumEntry("Precision", 6);
  doc->aut.iSep = config->readNumEntry("Separator");
  fz = config->readDoubleNumEntry("Zoom", 1.0);
  config->setDollarExpansion(false);
  doc->aut.sxAuto = config->readEntry("AutoLetX", "x");
  doc->aut.syAuto = config->readEntry("AutoLetY", "y");
  doc->aut.shAuto = config->readEntry("AutoLetH", "");
  view = new KplView(this, fz);
  setCentralWidget(view);
  new KAction(i18n("Scroll Left"), Key_Left, view, SLOT(slotScrollLeft()),
              actionCollection(), "cursor_left");
  new KAction(i18n("Scroll Right"), Key_Right, view, SLOT(slotScrollRight()),
              actionCollection(), "cursor_right");
  new KAction(i18n("Scroll Up"), Key_Up, view, SLOT(slotScrollUp()),
              actionCollection(), "cursor_up");
  new KAction(i18n("Scroll Down"), Key_Down, view, SLOT(slotScrollDown()),
              actionCollection(), "cursor_down");
  new KAction(i18n("Scroll Page Up"), Key_PageUp, view,
              SLOT(slotScrollPageUp()), actionCollection(),
              "page_up");
  new KAction(i18n("Scroll Page Down"), Key_PageDown, view,
              SLOT(slotScrollPageDown()), actionCollection(),
              "page_down");
  new KAction(i18n("Scroll Page Left"), CTRL + Key_PageUp, view,
              SLOT(slotScrollPageLeft()), actionCollection(),
              "page_left");
  new KAction(i18n("Scroll Page Right"), CTRL + Key_PageDown, view,
              SLOT(slotScrollPageRight()), actionCollection(),
              "page_right");
  new KAction(i18n("Scroll to Left Margin"), CTRL + Key_Home, view,
              SLOT(slotScrollHomeLeft()), actionCollection(),
              "margin_left");
  new KAction(i18n("Scroll to Bottom Margin"), Key_Home, view,
              SLOT(slotScrollHomeDown()), actionCollection(),
              "margin_bottom");
  new KAction(i18n("Scroll to Right Margin"), CTRL + Key_End, view,
              SLOT(slotScrollEndRight()), actionCollection(),
              "margin_right");
  new KAction(i18n("Scroll to Top Margin"), Key_End, view,
              SLOT(slotScrollEndUp()), actionCollection(),
              "margin_top");
  createGUI();
#if KDE_VERSION >= 0x030100
  showToolbar = (KToggleAction*) toolBarMenuAction();
#endif
  showToolbar->setChecked(config->readBoolEntry("ShowToolbar", true));
  slotOptionsToolBar();
  toolBar("mainToolBar")->setBarPos(toolBarPos);
  calcStatusBar();
  initStatusBar();
  updWin();
  connect(timer, SIGNAL(timeout()), SLOT(slotWatchFile()));
  connect(kapp, SIGNAL(lastWindowClosed()), kapp, SLOT(quit()));
  connect(doc, SIGNAL(modelChanged(bool, bool)), SLOT(slotUpdate(bool, bool)));
  connect(view, SIGNAL(mouseMoved(const QPoint&, bool)),
          SLOT(updStatus(const QPoint&, bool)));
  connect(view, SIGNAL(mouseLeaved()), SLOT(clearStatus()));
  connect(view, SIGNAL(rightButtonPressed()), SLOT(showPopupMenu()));
  connect(view, SIGNAL(urlsDropped(KURL::List*)),
          SLOT(slotOpenList(KURL::List*)));
  connect(itemDlg, SIGNAL(displayMessage(const QString&)),
          SLOT(slotStatusMsg(const QString&)));
}

Kpl::~Kpl()
{
  for (unsigned i = 0; i < previews.count(); i++)
    QFile::remove(previews[i]);
  delete itemDlg;
}

KplDoc* Kpl::getDocument() const
{
  return doc;
}

void Kpl::openList(KURL::List* list, int idec, int type)
{
  statusBar()->message(i18n("Opening file..."));
  if (idec < 0)
    idec = doc->aut.iDec;
  for (unsigned i = 0; i < list->count(); i++)
    if ((!i) || doc->aut.addData) {
      if (doc->aut.addData || (warning->isChecked() ? queryClose() : true))
        doc->openDocument((*list)[i], idec, (KplDoc::FileType) type);
    } else {
      Kpl* kpl = new Kpl;
      if (kpl->doc->openDocument((*list)[i], idec, (KplDoc::FileType) type))
        kpl->show();
      else
        delete kpl;
    }
  ready();
}

double Kpl::xStep()
{
  double step = 1.0 / (view->xscal * view->fxn);
  if (view->logx)
    step = pow(10, step);
  return step;
}

double Kpl::yStep()
{
  double step = -1.0 / (view->yscal * view->fyn);
  if (view->logy)
    step = pow(10, step);
  return step;
}

void Kpl::openDocumentURL(const KURL& url)
{
  statusBar()->message(i18n("Opening file..."));
  if (doc->aut.addData || (warning->isChecked() ? queryClose() : true))
    doc->openDocument(url, doc->aut.iDec, KplDoc::Unknown);
  ready();
}

void Kpl::slotOpenList(KURL::List* list)
{
  openList(list, doc->aut.iDec, 0);
}

void Kpl::calcStatusBar()
{
  xPrec = QMIN(QMAX(2, Utils::int9(log10(QMAX(fabs(view->xmin),
                                              fabs(view->xmax)) / view->fxn
                                   + 1.0e-37) + 0.999999) +
               Utils::int9(log10(view->xscal * view->fxn + 1.0e-300) +
                           0.999999)), 16);
  yPrec = QMIN(QMAX(2, Utils::int9(log10(QMAX(fabs(view->ymin),
                                              fabs(view->ymax)) / view->fyn
                                   + 1.0e-37) + 0.999999) +
               Utils::int9(log10(-view->yscal * view->fyn + 1.0e-300) +
                           0.999999)), 16);
}

void Kpl::initStatusBar()
{
  statusBar()->insertItem("", StatusAdd, 0, true);
  statusBar()->insertFixedItem(" 10.00 cm,  15.00 cm ", StatusPos, true);
  double x = -10.0e-300 / 3.0;
  statusBar()->insertFixedItem("  x = " + QString::number(x, 'g', xPrec) +
                               ",  y = " + QString::number(x, 'g', yPrec),
                               StatusVal, true);
  statusBar()->insertItem(i18n("Ready."), StatusMsg);
  statusBar()->changeItem("", StatusVal);
  statusBar()->changeItem("", StatusPos);
  slotOptionsAddData();
}

void Kpl::newStatusBar()
{
  int xold = xPrec;
  int yold = yPrec;
  calcStatusBar();
  if ((xPrec != xold) || (yPrec != yold)) {
    double x = -10.0e-300 / 3.0;
    statusBar()->changeItem("  x = " +
                            QString::number(x, 'g', xPrec) + ",  y = " +
                            QString::number(x, 'g', yPrec), StatusVal);
    statusBar()->setItemFixed(StatusVal);
    statusBar()->changeItem("", StatusVal);
  }
}

void Kpl::saveOptions(bool saveAll)
{
  recentFiles->saveEntries(config);
  Utils::saveGeometry(itemDlg, "ItemDialog");
  config->setGroup(kapp->instanceName());
  if (saveAll) {
    config->writeEntry("ShowToolbar", showToolbar->isChecked());
    config->writeEntry("ShowStatusbar", showStatusbar->isChecked());
    config->writeEntry("ShowItemDialog", itemDlg->isVisible());
    config->writeEntry("AppendItems", itemDlg->appendItems->isChecked());
    config->writeEntry("ToolBarPos", (int) toolBar("mainToolBar")->barPos());
    config->writeEntry("Directory", doc->currentDir());
    config->writeEntry("Width", width());
    config->writeEntry("Height", height());
    config->writeEntry("AutoPlot", autoPlot->isChecked());
    config->writeEntry("AddData", addData->isChecked());
    config->writeEntry("ShowSource", showSource->isChecked());
    config->writeEntry("BoundingBox", boundingBox->isChecked());
    config->writeEntry("PrintPS", printPS->isChecked());
    config->writeEntry("SaveAbsPath", saveAbsPath->isChecked());
    config->writeEntry("Warning", warning->isChecked());
    config->writeEntry("SaveAtEnd", saveAtEnd->isChecked());
    config->writeEntry("AutoIndexX", doc->aut.ixAuto);
    config->writeEntry("AutoIndexY", doc->aut.iyAuto);
    config->writeEntry("AutoIndexE", doc->aut.ieAuto);
    config->writeEntry("AutoNorm", doc->aut.autoNorm);
    config->writeEntry("AutoErrBar", doc->aut.autoErr);
    config->writeEntry("AutoLogX", doc->aut.autoLogX);
    config->writeEntry("AutoLogY", doc->aut.autoLogY);
    config->writeEntry("AutoGrid", doc->aut.autoGrid);
    config->writeEntry("AutoPathHeader", doc->aut.autohPath);
    config->writeEntry("AutoFit", doc->aut.autoFit);
    config->writeEntry("AutoSymbol", doc->aut.autoSymb);
    config->writeEntry("AutoX0", doc->aut.x0Auto);
    config->writeEntry("AutoY0", doc->aut.y0Auto);
    config->writeEntry("AutoXl", doc->aut.xlAuto);
    config->writeEntry("AutoYl", doc->aut.ylAuto);
    config->writeEntry("RelSize", doc->aut.relSize);
    config->writeEntry("FitErrorColumn", doc->aut.err.fitErrCol);
    config->writeEntry("FitNonLinear", doc->aut.fitNonLin);
    config->writeEntry("FitShowDialog", doc->aut.fitShowDlg);
    config->writeEntry("FitSaveParam", doc->aut.fitSavePar);
    config->writeEntry("FitFollow", doc->aut.fitFollow);
    config->writeEntry("FitMaxIterations", doc->aut.fitMaxIt);
    config->writeEntry("FitTolerance", doc->aut.fitTol);
    config->writeEntry("FitSymbol", doc->aut.fitSymb);
    config->writeEntry("FitColor", KplGraph::rgbQt1(doc->aut.fitColor));
    config->writeEntry("FitPath", doc->aut.fitPath);
    config->writeEntry("FitName", doc->aut.fitName);
    config->writeEntry("ErrorModelPath", doc->aut.err.errModPath);
    config->writeEntry("ErrorModelName", doc->aut.err.errModName);
    QStringList s1, s2, s3;
    for (int i = 0; i < KPL_NPMAX; i++) {
      s1.append(QString::number((int) doc->aut.fitPar[i]));
      s2.append(doc->number(doc->aut.pFit[i]));
      s3.append(doc->number(doc->aut.err.pErrMod[i]));
    }
    config->writeEntry("FitEnableParam", s1, ' ');
    config->writeEntry("FitParam", s2, ' ');
    config->writeEntry("ErrorModelParam", s3, ' ');
    config->writeEntry("ErrorModelArgument", doc->aut.err.errModArg);
    config->writeEntry("FormatX", doc->aut.xf);
    config->writeEntry("FormatY", doc->aut.yf);
    config->writeEntry("ColorFrame", KplGraph::rgbQt1(doc->aut.colFrame));
    config->writeEntry("ColorGrid", KplGraph::rgbQt1(doc->aut.colGrid));
    config->writeEntry("ColorData", KplGraph::rgbQt1(doc->aut.colData));
    config->writeEntry("OutputFormat", QString(QChar(doc->aut.format)));
    config->writeEntry("DecimalSymbol", doc->aut.iDec);
    config->writeEntry("Precision", doc->aut.prec);
    config->writeEntry("Separator", doc->aut.iSep);
    config->writeEntry("Zoom", view->zoom());
    config->writeEntry("AutoLetX", doc->aut.sxAuto);
    config->writeEntry("AutoLetY", doc->aut.syAuto);
    config->writeEntry("AutoLetH", doc->aut.shAuto);
  }
  config->sync();
}

void Kpl::openRead(int type)
{
  int idec = doc->aut.iDec;
  KURL::List list = (type == KplDoc::Data ?
                     doc->getReadURLs(idec) :
                     KFileDialog::getOpenURLs(doc->currentDir(),
                                              "*.plo *.PLO\n*"));
  if (!list.isEmpty())
    openList(&list, idec, type);
}

bool Kpl::save(bool current)
{
  bool success = false;
  if ((doc->URL().fileName() != i18n("Untitled")) && current &&
      (doc->fileType() == KplDoc::Plot)) {
    success = doc->saveDocument(doc->URL(), doc->items(),
                                saveAbsPath->isChecked());
    if (success) {
      doc->setModified(false);
      doc->setFileType(KplDoc::Plot);
      updWin();
    }
  } else {
    KURL url;
    if (doc->getWriteURL(this, url, "*.plo\n*"))
      success = saveAs(url);
  }
  return success;
}

bool Kpl::saveAs(const KURL& url)
{
  bool success = doc->saveDocument(url, doc->items(),
                                   saveAbsPath->isChecked());
  if (success) {
    doc->setURL(url);
    doc->setModified(false);
    doc->setFileType(KplDoc::Plot);
    slotUpdate(false, true);
  }
  return success;
}

bool Kpl::queryClose()
{
  if (doc->isModified())
    do {
      int i = KMessageBox::warningYesNoCancel(this,
        i18n("The current plot has been changed.\nSave file?"));
      if (i == KMessageBox::Cancel)
        return false;
      if (i == KMessageBox::No)
        break;
    } while (!save(true));
  return true;
}

bool Kpl::queryExit()
{
  saveOptions(saveAtEnd->isChecked());
  return true;
}

void Kpl::saveProperties(KConfig* cfg)
{
  bool untitled = (doc->URL().fileName() == i18n("Untitled"));
  if (untitled && !doc->isModified())
    return;
  QString fileName;
  if (untitled)
    fileName = QDir::homeDirPath() + "/" + i18n("Untitled");
  else
    fileName = doc->URL().url();
  cfg->writeEntry("filename", fileName);
  cfg->writeEntry("modified", doc->isModified());
  if (doc->isModified())
    doc->saveDocument(kapp->tempSaveName(fileName), doc->items());
}

void Kpl::readProperties(KConfig* cfg)
{
  QString filename = cfg->readEntry("filename", "");
  KURL u = KURL(filename);
  bool recovered = false;
  if (cfg->readBoolEntry("modified")) {
    bool canRecover;
    QString tempname = kapp->checkRecoverFile(filename, canRecover);
    if (canRecover) {
      doc->openDocument(tempname, doc->aut.iDec, KplDoc::Plot);
      QFile::remove(tempname);
      doc->setModified();
      recovered = true;
    }
  } else
    if (!filename.isEmpty()) {
      doc->openDocument(u, doc->aut.iDec, KplDoc::Unknown);
      recovered = true;
    }
  if (recovered)
    doc->setURL(u);
  updWin();
}

void Kpl::ready()
{
  slotStatusMsg(i18n("Ready."));
}

void Kpl::shrink()
{
  view->layout();
  if ((QMAX(view->contentsWidth(), 120) < view->frameRect().width()) ||
      (view->contentsHeight() < view->frameRect().height())) {
    int w = QMIN(QMAX(view->contentsWidth(), 120), view->frameRect().width()) -
            view->frameRect().width() + width();
    resize(w, QMIN(view->contentsHeight(), view->frameRect().height()) -
           view->frameRect().height() + height() +
           menuBar()->heightForWidth(w) - menuBar()->height());
    newStatusBar();
  }
  view->eraseWidget();
  slotPlotCond();
}

void Kpl::drawPS(PSGraph* g)
{
  KplWidget::drawItems(g, doc->items());
  g->psExit();
}

bool Kpl::savePSURL(const KURL& url, bool orientation)
{
  QFile f((url.isEmpty() || (!url.isLocalFile())) ?
          doc->tmpFile() : url.path());
  bool success = f.open(IO_WriteOnly);
  if (success) {
    view->QScrollView::viewport()->setCursor(waitCursor);
    PSGraph g;
    QTextStream ts(&f);
    g.psInit(&ts, doc->aut.xf, doc->aut.yf, orientation);
    drawPS(&g);
    f.close();
    if (boundingBox->isChecked()) {
      QString tmpPBM = doc->tmpFile();
      QCString s;
      KProcess p;
      p << "gs" << "-dNOPAUSE" << "-dBATCH" << "-dSAFER" << "-dPARANOIDSAFER"
        << "-q" << "-sDEVICE=pbmraw" << ("-sOutputFile=" + tmpPBM) << "-r72"
        << s.sprintf("-g%ix%i", qRound(PSGraph::pts *
          (orientation ? doc->aut.yf : doc->aut.xf)),
          qRound(PSGraph::pts * (orientation ? doc->aut.xf : doc->aut.yf)))
        << f.name();
      p.start(KProcess::Block);
      QImage image;
      if (image.load(tmpPBM, "PBMRAW")) {
        int x, y, llx = 0, ury = 0;
        int urx = image.width();
        int lly = image.height();
        for (y = 0; y < image.height(); y++)
          for (x = 0; x < image.width(); x++)
            if (!(image.pixel(x, y) & 0xffffff)) {
              ury = y;
              llx = urx = x;
              goto yo_found;
            }
yo_found:
        for (y = image.height() - 1; y > ury; y--)
          for (x = 0; x < image.width(); x++)
            if (!(image.pixel(x, y) & 0xffffff)) {
              lly = y;
              llx = QMIN(llx, x);
              urx = QMAX(urx, x);
              goto yu_found;
            }
yu_found:
        for (y = ury; y <= lly; y++){
          for (x = 0; x < llx; x++)
            if (!(image.pixel(x, y) & 0xffffff)) {
              llx = x;
              break;
            }
          for (x = image.width() - 1; x > urx; x--)
            if (!(image.pixel(x, y) & 0xffffff)) {
              urx = x;
              break;
            }
        }
        llx = QMAX(llx - 2, 0);
        urx = QMIN(urx + 2, image.width() - 1);
        lly = QMAX(image.height() - lly - 2, 0);
        ury = QMIN(image.height() - ury + 2, image.height() - 1);
        QFile::remove(tmpPBM);
        f.open(IO_WriteOnly);
        g.psInit(&ts, doc->aut.xf, doc->aut.yf, orientation,
                 llx, lly, urx, ury);
        drawPS(&g);
        f.close();
      } else
        KMessageBox::error(this,
          i18n("Bitmap file for bounding box calculation not loaded!"));
    }
    if (url.isEmpty()) {
      previews << f.name();
      KApplication::startServiceByDesktopName("kghostview", f.name());
    } else {
      doc->setCurrentDir(url);
      if (printPS->isChecked()) {
        KProcess p;
        p << "kprinter" << f.name();
        p.start(KProcess::Block);
      }
      if (!url.isLocalFile())
        doc->copyTmp(f.name(), url);
    }
    view->QScrollView::viewport()->setCursor(crossCursor);
  } else
    KMessageBox::sorry(this, i18n("Writing to this file not possible."));
  return success;
}

void Kpl::slotFileNew()
{
  if (queryClose())
    newPlot();
  ready();
}

void Kpl::slotFileOpenPlot()
{
  statusBar()->message(i18n("Opening file..."));
  openRead(KplDoc::Plot);
  ready();
}

void Kpl::slotFileOpenData()
{
  statusBar()->message(i18n("Opening file..."));
  openRead(KplDoc::Data);
  ready();
}

void Kpl::slotOpenRecent(const KURL& url)
{
  statusBar()->message(i18n("Opening file..."));
  if (doc->aut.addData || (warning->isChecked() ? queryClose() : true))
    if (!doc->openDocument(url, doc->aut.iDec, KplDoc::Unknown))
      recentFiles->removeURL(url);
  ready();
}

void Kpl::slotFileSave()
{
  statusBar()->message(i18n("Saving file..."));
  save(true);
  ready();
}

void Kpl::slotFileSaveAs()
{
  statusBar()->message(i18n("Saving file under different name..."));
  save(false);
  ready();
}

void Kpl::slotFileClose()
{
  statusBar()->message(i18n("Closing file..."));
  if (queryClose()) {
    doc->newDocument();
    doc->backupItems();
  }
  ready();
}

void Kpl::slotFilePlot()
{
  statusBar()->message(i18n("Plotting..."));
  plot = doc->items()->count();
  slotPlotCond();
  ready();
}

void Kpl::slotPlotCond()
{
  if (plot)
    view->paintWidget();
  newStatusBar();
}

void Kpl::slotFilePrint()
{
  statusBar()->message(i18n("Printing..."));
  KPrinter printer(true, QPrinter::HighResolution);
  if (printer.setup(this))
    view->print(&printer);
  ready();
}

void Kpl::slotFilePSP()
{
  slotFilePS(false);
}

void Kpl::slotFilePSL()
{
  slotFilePS(true);
}

void Kpl::slotFilePS(bool orientation)
{
  statusBar()->message(i18n("Generating PostScript file..."));
  KURL u;
  if (doc->getWriteURL(this, u, "*.ps *.eps\n*"))
    if (!savePSURL(u, orientation))
      KMessageBox::sorry(this, i18n("Writing to this file not possible."));
  ready();
}

void Kpl::slotFilePreviewP()
{
  slotFilePreview(false);
}

void Kpl::slotFilePreviewL()
{
  slotFilePreview(true);
}

void Kpl::slotFilePreview(bool orientation)
{
  statusBar()->message(i18n("Generating PostScript preview..."));
  KURL u;
  savePSURL(u, orientation);
  ready();
}

void Kpl::slotFileNewWindow()
{
  (new Kpl)->show();
  ready();
}

void Kpl::slotFileCloseWindow()
{
  close();
}

void Kpl::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  if (memberList)
    for (KMainWindow* w = memberList->first(); w; w = memberList->next())
      if (!w->close()) {
        ready();
        break;
      }
}

void Kpl::slotEditUndo()
{
  doc->restoreItems();
}

void Kpl::slotEditRedo()
{
  doc->restoreItems(false);
}

void Kpl::slotEditItems()
{
  itemDlg->show();
  itemDlg->raise();
}

void Kpl::slotViewZoomIn()
{
  view->setZoom(0.01 * (QMIN(qRound(100 * view->zoom()) + 20, 2000)));
  shrink();
  ready();
}

void Kpl::slotViewZoomOut()
{
  view->setZoom(0.01 * (QMAX(qRound(100 * view->zoom()) - 20, 1)));
  shrink();
  ready();
}

void Kpl::slotViewZoom()
{
  statusBar()->message(i18n("Setting magnification..."));
  int i = qRound(100 * view->zoom());
  ZoomDlg dlg(this, &i);
  if (dlg.exec()) {
    view->setZoom(0.01 * i);
    shrink();
  }
  ready();
}

void Kpl::slotWatchFile()
{
  if (doc->URL().isLocalFile()) {
    QFileInfo fi(doc->URL().path());
    if (fi.lastModified() == doc->URLTime())
      return;
  }
  slotViewReload();
}

void Kpl::slotViewReload()
{
  statusBar()->message(i18n("Reloading file..."));
  autoPlot->setChecked(true);
  bool sav = doc->aut.addData;
  doc->aut.addData = false;
  doc->openDocument(KURL(doc->URL()), doc->aut.iDec, doc->fileType());
  doc->aut.addData = sav;
  ready();
}

void Kpl::slotViewAutoReload()
{
  statusBar()->message(i18n("Reloading file periodically..."));
  bool act = timer->isActive();
  int tRef = tRefresh;
  RefreshDlg dlg(this, &act, &tRef);
  if (dlg.exec()) {
    if (timer->isActive() && ((tRef != tRefresh) || (act != timer->isActive())))
      timer->stop();
    if (act && ((tRef != tRefresh) || (act != timer->isActive()))) {
      timer->start(1000 * tRef);
    }
    tRefresh = tRef;
  }
  ready();
}

void Kpl::slotOptionsToolBar()
{
  toolBar("mainToolBar")->setShown(showToolbar->isChecked());
}

void Kpl::slotOptionsStatusBar()
{
  statusBar()->setShown(showStatusbar->isChecked());
}

void Kpl::slotOptionsAutoPlot()
{
  if (autoPlot->isChecked())
    slotFilePlot();
}

void Kpl::slotOptionsAddData()
{
  doc->aut.addData = addData->isChecked();
  statusBar()->changeItem(i18n(doc->aut.addData ? "Add" : "Replace"),
                          StatusAdd);
}

void Kpl::slotOptionsShowSource()
{
  doc->aut.showSource = showSource->isChecked();
}

void Kpl::slotOptionsKeys()
{
  statusBar()->message(i18n("Key bindings..."));
  KKeyDialog::configure(actionCollection());
  ready();
}

void Kpl::slotOptionsToolbars()
{
  statusBar()->message(i18n("Toolbar configuration..."));
  KEditToolbar dlg(actionCollection());
  connect(&dlg, SIGNAL(newToolbarConfig()), this,
          SLOT(slotNewToolbarConfig()));
  dlg.exec();
  ready();
}

void Kpl::slotNewToolbarConfig()
{
  toolBarPos = toolBar("mainToolBar")->barPos();
  createGUI();
  toolBar("mainToolBar")->setBarPos(toolBarPos);
}

void Kpl::slotOptionsNotifications()
{
#if KDE_VERSION >= 0x030100
  statusBar()->message(i18n("Notification configuration..."));
  KNotifyDialog::configure(this);
  ready();
#endif
}

void Kpl::slotOptionsConfigure()
{
  statusBar()->message(i18n("Kpl settings..."));
  SettingsDlg dlg(this, doc);
  dlg.exec();
  ready();
}

void Kpl::slotOptionsSave()
{
  saveOptions(true);
  ready();
}

void Kpl::slotStatusMsg(const QString &text)
{
  statusBar()->clear();
  statusBar()->changeItem(text, StatusMsg);
}

void Kpl::slotStatusHelpMsg(const QString &text)
{
  statusBar()->message(text, 3000);
}

void Kpl::slotUpdate(bool updPlot, bool updList)
{
  updWin();
  if (updPlot)
    if (doc->items()->count())
      slotPlotCond();
    else
      view->eraseWidget();
  if (updList)
    itemDlg->updItemList();
}

void Kpl::updWin()
{
  if (doc->URL().fileName() == i18n("Untitled"))
    setCaption(doc->URL().fileName(), doc->isModified());
  else {
    recentFiles->addURL(doc->URL());
    if (doc->URL().isLocalFile())
      setCaption(doc->URL().path(), doc->isModified());
    else
      setCaption(doc->URL().url(), doc->isModified());
  }
  recentFiles->setEnabled(recentFiles->items().count());
  bool nItems = doc->items()->count();
  fileSaveAs->setEnabled(nItems);
  fileClose->setEnabled(nItems);
  filePlot->setEnabled(nItems);
  filePSP->setEnabled(nItems);
  filePSL->setEnabled(nItems);
  filePreviewP->setEnabled(nItems);
  filePreviewL->setEnabled(nItems);
  filePrint->setEnabled(nItems);
  fileSave->setEnabled(doc->isModified());
  editUndo->setEnabled(doc->undoAllowed());
  editRedo->setEnabled(doc->redoAllowed());
  viewReload->setEnabled((doc->URL().fileName() != i18n("Untitled")));
  plot = autoPlot->isChecked() && nItems;
}

void Kpl::updStatus(const QPoint& p, bool showValues)
{
  if (statusBar()->isVisible()) {
    double xcm, ycm, xr, yr;
    if ((!view->i2cm(p, &xcm, &ycm)) && plot && showValues) {
      view->i2r(p, view->fxn, view->fyn, &xr, &yr);
      statusBar()->changeItem("  x = " + QString::number(xr, 'g', xPrec) +
                              ",  y = " + QString::number(yr, 'g', yPrec),
                              StatusVal);
    } else
      statusBar()->changeItem("", StatusVal);
    QCString s;
    if ((xcm < doc->aut.xf) && (ycm >= 0.0))
      s.sprintf("%6.2f cm, %6.2f cm", xcm, ycm);
    statusBar()->changeItem(s, StatusPos);
  }
}

void Kpl::clearStatus()
{
  if (statusBar()->isVisible()) {
    statusBar()->changeItem("", StatusVal);
    statusBar()->changeItem("", StatusPos);
  }
}

void Kpl::showPopupMenu()
{
  ((QPopupMenu*) factory()->container("popup", this))->exec(QCursor::pos());
}

void Kpl::newPlot()
{
  doc->newDocument();
  doc->backupItems();
}

int Kpl::openPlot(const QString& url)
{
  return doc->openDocument(KURL(url), 0, KplDoc::Plot) ? 1 : 0;
}

int Kpl::openData(const QString& url)
{
  return doc->openDocument(KURL(url), doc->aut.iDec, KplDoc::Data) ? 1 : 0;
}

void Kpl::doPlot()
{
  slotFilePlot();
}

int Kpl::savePlot(const QString& url)
{
  if (doc->items()->count())
    return saveAs(KURL(url)) ? 1 : 0;
  else
    return 0;
}

int Kpl::savePS(const QString& url, int orientation)
{
  if (doc->items()->count())
    return savePSURL(KURL(url), orientation) ? 1 : 0;
  else
    return 0;
}

int Kpl::printPlot(const QString& printerName, const QString& fileName,
                   int orientation, int numCopies)
{
  if (doc->items()->count()) {
    KPrinter p;
    p.setPrinterName(printerName);
    p.setOutputFileName(fileName);
    p.setOrientation(orientation ? KPrinter::Landscape : KPrinter::Portrait);
    p.setNumCopies(numCopies);
    view->print(&p);
    return 1;
  } else
    return 0;
}

void Kpl::setAutoplotOption(int setting)
{
  autoPlot->setChecked(setting);
}

void Kpl::setAddFilesOption(int setting)
{
  addData->setChecked(setting);
  slotOptionsAddData();
}

void Kpl::setPathOption(int setting)
{
  saveAbsPath->setChecked(setting);
}

void Kpl::setBoundingBoxOption(int setting)
{
  boundingBox->setChecked(setting);
}

void Kpl::setPageFormat(double w, double h)
{
  doc->aut.xf = w;
  doc->aut.yf = h;
  shrink();
}

void Kpl::setColors(const QString& colFrame, const QString& colGrid,
                    const QString& colData)
{
  doc->aut.colFrame = KplGraph::rgbQt(colFrame);
  doc->aut.colGrid = KplGraph::rgbQt(colGrid);
  doc->aut.colData = KplGraph::rgbQt(colData);
}

void Kpl::setOutputFormat(const QString& pres, int prec)
{
  doc->aut.format = pres[0].latin1();
  doc->aut.prec = prec;
}

void Kpl::setAutoplot(double x0, double w, double y0, double h,
                      int gridmode, const QString& sx, const QString& sy,
                      const QString& sh, int pathHeading, int ix, int iy,
                      int ie, int logx, int logy, int errbars, int symb,
                      int autoNorm, int autoFit)
{
  doc->aut.x0Auto = x0;
  doc->aut.xlAuto = w;
  doc->aut.y0Auto = y0;
  doc->aut.ylAuto = h;
  doc->aut.autoGrid = gridmode;
  doc->aut.sxAuto = sx;
  doc->aut.syAuto = sy;
  doc->aut.shAuto = sh;
  doc->aut.autohPath = pathHeading;
  doc->aut.ixAuto = ix;
  doc->aut.iyAuto = iy;
  doc->aut.ieAuto = ie;
  doc->aut.autoLogX = logx;
  doc->aut.autoLogY = logy;
  doc->aut.autoErr = errbars;
  doc->aut.autoSymb = symb;
  doc->aut.autoNorm = autoNorm;
  doc->aut.autoFit = autoFit;
}

void Kpl::setAutofit(const QString& path, const QString& _name,
                     int nonLin, int errCol,
                     const QString& errModPath, const QString& errModName,
                     int itMax, double tol, int symb, const QString& color,
                     int showDlg, int savePar, int follow)
{
  doc->aut.fitPath = path;
  doc->aut.fitName = _name;
  doc->aut.fitNonLin = nonLin;
  if (nonLin) {
    doc->aut.fitMaxIt = itMax;
    doc->aut.fitTol = tol;
  }
  doc->aut.err.fitErrCol = errCol;
  if (!errCol) {
    doc->aut.err.errModPath = errModPath;
    doc->aut.err.errModName = errModName;
  }
  doc->aut.fitSymb = symb;
  doc->aut.fitColor = KplGraph::rgbQt(color);
  doc->aut.fitShowDlg = showDlg;
  doc->aut.fitSavePar = savePar;
  doc->aut.fitFollow = follow;
}

int Kpl::loadAutofitPar(const QString& url)
{
  return doc->loadPar(KURL(url), this, doc->aut.pFit, doc->aut.iDec) ? 1 : 0;
}

int Kpl::setAutofitPar(int iPar, double value)
{
  if ((iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  doc->aut.pFit[iPar] = value;
  return 1;
}

int Kpl::enableFitPar(int iPar, int fit)
{
  if ((iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  doc->aut.fitPar[iPar] = fit;
  return 1;
}

int Kpl::loadErrModPar(const QString& url, int iArr)
{
  int success = 0;
  if (iArr && (unsigned(iArr) >= arr.count()))
    return success;
  double p[KPL_NPMAX];
  success = doc->loadPar(KURL(url), this, p, doc->aut.iDec) ? 1 : 0;
  if (success) {
    QString s;
    s.sprintf("DataErrors%i", iArr);
    config->setGroup(s);
    QStringList l;
    for (int i = 0; i < KPL_NPMAX; i++) {
      l.append(doc->number(p[i]));
      if (!iArr)
        doc->aut.err.pErrMod[i] = p[i];
    }
    config->writeEntry("ErrorModelParam", l, ' ');
  }
  return success;
}

int Kpl::setErrModPar(int iPar, double value, int iArr)
{
  if ((iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  if (iArr && (unsigned(iArr) >= arr.count()))
    return 0;
  QString s;
  s.sprintf("DataErrors%i", iArr);
  config->setGroup(s);
  QStringList l = config->readListEntry("ErrorModelParam", ' ');
  int cnt = l.count();
  double p[KPL_NPMAX];
  for (int i = 0; i < KPL_NPMAX; i++)
    p[i] = (i < cnt) ? l[i].toDouble() : 0.0;
  p[iPar] = value;
  QStringList l2;
  for (int i = 0; i < KPL_NPMAX; i++)
    l2.append(doc->number(p[i]));
  config->writeEntry("ErrorModelParam", l2, ' ');
  if (!iArr)
    doc->aut.err.pErrMod[iPar] = value;
  return 1;
}

int Kpl::deleteItem(int iItem)
{
  if (unsigned(iItem) < doc->items()->count()) {
    doc->deleteItem(iItem);
    return 1;
  } else
    return 0;
}

int Kpl::moveItem(int is, int id)
{
  if ((unsigned(is) < doc->items()->count()) &&
      (unsigned(id) < doc->items()->count()) && (is != id)) {
    doc->moveItem(is, id);
    return 1;
  } else
    return 0;
}

int Kpl::activateItem(int iItem, int active)
{
  if (unsigned(iItem) < doc->items()->count()) {
    doc->items()->at(iItem)->setActive(active);
    doc->setModified();
    doc->backupItems();
    return 1;
  } else
    return 0;
}

void Kpl::newFrameItem(double x0, double w, double y0, double h,
                       double xmin, double xmax, double ymin, double ymax,
                       int logx, int logy, int ndigx, int ndigy,
                       const QString& colFrame, const QString& colGrid,
                       double xtic, double ytic, int mticx, int mticy,
                       int gridmode, int iex, int iey,
                       const QString& sx, const QString& sy,
                       const QString& sh, double relSize, int active)
{
  doc->items()->append(new FrameItem(active, logx, logy, ndigx, ndigy, mticx,
                                     mticy, gridmode, iex, iey, colFrame,
                                     colGrid, x0, w, y0, h, xmin, xmax, ymin,
                                     ymax, xtic, ytic, relSize, sx, sy, sh,
                                     &doc->aut));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newAutoFrameItem()
{
  doc->items()->append(new FrameItem(&doc->aut));
  doc->setModified();
  doc->backupItems();
}

int Kpl::newArrayItem(const QString& path, int ix, int iy, int ie,
                      int istart, int n, int errbars, int symb,
                      const QString& color, double fx, double fy,
                      int active)
{
  doc->items()->append(new ArrayItem(active, 0, symb, color, fx, fy,
                                     ix, iy, ie, istart, n, errbars, path));
  ArrayItem* it = (ArrayItem*) doc->items()->current();
  if (it->nrows) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

int Kpl::newFunItem(const QString& path, const QString& _name,
                    double xmin, double xmax, double dx,
                    int symb, const QString& color,
                    double fx, double fy, int active)
{
  doc->items()->append(new FunItem(active, 0, symb, color, fx, fy,
                                   xmin, xmax, dx, _name, path));
  FunItem* it = (FunItem*) doc->items()->current();
  if (it->liby) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

int Kpl::newParFunItem(const QString& pathx, const QString& namex,
                       const QString& pathy, const QString& namey,
                       double tmin, double tmax, double dt,
                       int symb, const QString& color,
                       double fx, double fy, int active)
{
  doc->items()->append(new ParFunItem(active, 0, symb, color, fx, fy, tmin,
                                      tmax, dt, namex, pathx, namey, pathy));
  ParFunItem* it = (ParFunItem*) doc->items()->current();
  if (it->libx || it->liby) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

void Kpl::newSplineItem(int deriv, double low, double xmin, double xmax,
                        double dx, int symb, const QString& color,
                        double relSize, double fx, double fy, int active)
{
  doc->items()->append(new SplineItem(active, 0, symb, color, fx, fy, xmin,
                                      xmax, dx, 3, deriv, low, relSize));
  doc->setModified();
  doc->backupItems();
}

int Kpl::newArray3DItem(double x0, double w, double y0, double h,
                        double xmin, double xmax, double ymin, double ymax,
                        double zmin, double zmax, int logx, int logy,
                        int logz, int ndigx, int ndigy, int ndigz,
                        const QString& colFrame, const QString& colGrid,
                        const QString& colData, double xtic, double ytic,
                        double ztic, int mticx, int mticy, int mticz,
                        int frame, int gridmode, int gridmode3D,
                        double xBox, double yBox, double zBox, double xRef,
                        double yRef, double phi, double theta, int iex,
                        int iey, int iez, const QString& sx,
                        const QString& sy, const QString& sz,
                        const QString& sh, double relSize,
                        const QString& path, int ix, int iy, int iz, int ie,
                        int istart, int n, int errbars, double smf,
                        double dx, double dz, double fx, double fy,
                        double fz, int active)
{
  doc->items()->append(new Array3DItem(active, logx, logy, logz, frame, ndigx,
                                       ndigy, ndigz, mticx, mticy, mticz,
                                       gridmode, gridmode3D, iex, iey, iez,
                                       colFrame, colGrid, colData, x0, w, y0,
                                       h, xmin, xmax, ymin, ymax, zmin, zmax,
                                       dx, dz, xtic, ytic, ztic, relSize, phi,
                                       theta, xBox, yBox, zBox, xRef, yRef, fx,
                                       fy, fz, sx, sy, sz, sh, ix, iy, iz, ie,
                                       istart, n, errbars, smf, path,
                                       &doc->aut));
  Array3DItem* it = (Array3DItem*) doc->items()->current();
  if (it->nrows) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

int Kpl::newFun3DItem(double x0, double w, double y0, double h,
                      double xmin, double xmax, double ymin, double ymax,
                      double zmin, double zmax, int logx, int logy,
                      int logz, int ndigx, int ndigy, int ndigz,
                      const QString& colFrame, const QString& colGrid,
                      const QString& colData, double xtic, double ytic,
                      double ztic, int mticx, int mticy, int mticz,
                      int frame, int gridmode, int gridmode3D, double xBox,
                      double yBox, double zBox, double xRef, double yRef,
                      double phi, double theta, int iex, int iey, int iez,
                      const QString& sx, const QString& sy,
                      const QString& sz, const QString& sh, double relSize,
                      const QString& path, const QString& _name,
                      double dx, double dz, double fx, double fy,
                      double fz, int active)
{
  doc->items()->append(new Fun3DItem(active, logx, logy, logz, frame, ndigx,
                                     ndigy, ndigz, mticx, mticy, mticz,
                                     gridmode, gridmode3D, iex, iey, iez,
                                     colFrame, colGrid, colData, x0, w, y0, h,
                                     xmin, xmax, ymin, ymax, zmin, zmax, dx,
                                     dz, xtic, ytic, ztic, relSize, phi, theta,
                                     xBox, yBox, zBox, xRef, yRef, fx, fy, fz,
                                     sx, sy, sz, sh, _name, path, &doc->aut));
  Fun3DItem* it = (Fun3DItem*) doc->items()->current();
  if (it->liby) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else {
    doc->items()->remove(it);
    return 0;
  }
}

void Kpl::newImageItem(const QString& path, double x, double y, double fx,
                       double fy, int active)
{
  doc->items()->append(new ImageItem(active, x, y, fx, fy, path));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newLegendItem(int symb, const QString& colSymb, double x,
                        double y, const QString& text,
                        const QString& colText, double xoff, double yoff,
                        double relSize, int active)
{
  doc->items()->append(new LegendItem(active, symb, colSymb, colText, x, y,
                                      xoff, yoff, relSize, text));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newTextItem(const QString& text, const QString& color, int align,
                      double x, double y, double dir, double relSize,
                      int active)
{
  doc->items()->append(new TextItem(active, align, color, x, y, dir, relSize,
                                    text));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newLineItem(int symb, const QString& color, double x, double y,
                      double dir, double len, double relSize, int active)
{
  doc->items()->append(new LineItem(active, symb, color, x, y, dir, len,
                                    relSize));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newArrowItem(const QString& color, double x, double y,
                       double dir, double len, double relSize, int active)
{
  doc->items()->append(new ArrowItem(active, color, x, y, dir, len, relSize));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newArcItem(const QString& color, double x, double y, double w,
                     double h, double a, double alen, double relSize,
                     double ang, int active)
{
  doc->items()->append(new ArcItem(active, color, x, y, w, h, a, alen,
                                   relSize, ang));
  doc->setModified();
  doc->backupItems();
}
                  
void Kpl::newRectangleItem(const QString& color, double x, double y,
                           double w, double h, double relSize, int fill,
                           int active)
{
  doc->items()->append(new RectItem(active, fill, color, x, y, w, h, relSize));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newEllipseItem(const QString& color, double x, double y,
                         double w, double h, double relSize, int fill,
                         int active)
{
  doc->items()->append(new EllipseItem(active, fill, color, x, y, w, h,
                                       relSize));
  doc->setModified();
  doc->backupItems();
}

void Kpl::newScaleBarItem(const QString& str, const QString& color,
                          double x, double y, double len, double relSize,
                          int o, int active)
{
  doc->items()->append(new ScaleBarItem(active, color, str, x, y, o, len,
                                        relSize));
  doc->setModified();
  doc->backupItems();
}

int Kpl::autoScaleFrame(int iItem, int autoNorm)
{
  if (unsigned(iItem) >= doc->items()->count())
    return 0;
  if (doc->items()->at(iItem)->iType() != KplItem::Frame)
    return 0;
  if (((FrameItem*) doc->items()->at(iItem))->autoScale(autoNorm,
                                                        doc->items())) {
    doc->setModified();
    doc->backupItems();
    return 1;
  } else
    return 0;
}

int Kpl::loadPar(const QString& url, int iItem, int yFun)
{
  int success = false;
  if (unsigned(iItem) >= doc->items()->count())
    return success;
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Function) && (ityp != KplItem::ParFunction) &&
      (ityp != KplItem::Function3D))
    return success;
  double *p;
  if ((ityp == KplItem::Function) || (ityp == KplItem::Function3D))
    if (!yFun)
      return success;
    else
      if (ityp == KplItem::Function)
        p = ((FunItem*) doc->items()->at(iItem))->py;
      else
        p = ((Fun3DItem*) doc->items()->at(iItem))->py;
  else
    p = yFun ? ((ParFunItem*) doc->items()->at(iItem))->py :
               ((ParFunItem*) doc->items()->at(iItem))->px;
  success = doc->loadPar(KURL(url), this, p, doc->aut.iDec) ? 1 : 0;
  if (success) {
    doc->setModified();
    doc->backupItems();
  }
  return success;
}

int Kpl::setPar(int iItem, int iPar, double value, int yFun)
{
  if ((unsigned(iItem) >= doc->items()->count()) ||
      (iPar < 0) || (iPar >= KPL_NPMAX))
    return 0;
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Function) && (ityp != KplItem::ParFunction) &&
      (ityp != KplItem::Function3D))
    return 0;
  if (((ityp == KplItem::Function) || (ityp == KplItem::Function3D)) &&
      (!yFun))
    return 0;
  doc->items()->at(iItem)->setPar(iPar, value, yFun);
  doc->setModified();
  doc->backupItems();
  return 1;
}

int Kpl::setArrayRange(int iItem, int iStart, int n)
{
  if (unsigned(iItem) >= doc->items()->count())
    return 0;
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Array) && (ityp != KplItem::Array3D))
    return 0;
  if (ityp == KplItem::Array) {
    ArrayItem* it = (ArrayItem*) doc->items()->at(iItem);
    if ((iStart < 0) || (iStart > (it->nrows - 1)) || (n < 1) ||
        ((iStart + n) > it->nrows))
      return 0;
    it->istart = iStart;
    it->n = n;
  } else {
    Array3DItem* it = (Array3DItem*) doc->items()->at(iItem);
    if ((iStart < 0) || (iStart > (it->nrows - 1)) || (n < 1) ||
        ((iStart + n) > it->nrows))
      return 0;
    it->istart = iStart;
    it->n = n;
  }
  doc->setModified();
  doc->backupItems();
  return 1;
}

int Kpl::addFitItems(int erase, int iArr, int iFun, int errCol,
                     const QString& errModPath, const QString& errModName)
{
  if ((unsigned(iArr) >= doc->items()->count()) ||
      (unsigned(iFun) >= doc->items()->count()))
    return 0;
  if ((doc->items()->at(iArr)->iType() != KplItem::Array) ||
      (doc->items()->at(iFun)->iType() != KplItem::Function))
    return 0;
  if (erase) {
    arr.clear();
    fun.clear();
  }
  QString s;
  s.sprintf("DataErrors%i", arr.count());
  config->setGroup(s);
  config->writeEntry("FitErrorColumn", bool(errCol));
  if (!errCol) {
    config->writeEntry("ErrorModelPath", errModPath);
    config->writeEntry("ErrorModelName", errModName);
  }
  arr.append((ArrayItem*) doc->items()->at(iArr));
  fun.append((FunItem*) doc->items()->at(iFun));
  return 1;
}

int Kpl::fit(int nonLin, int savePar, int follow)
{
  doc->chisq = 0.0;
  if (!arr.count())
    return 0;
  doc->aut.fitNonLin = nonLin;
  int mode = 0;
  if (savePar)
    mode |= FitDlg::SavePar;
  if (follow)
    mode |= FitDlg::Follow;
  FitDlg dlg(this, doc, &arr, &fun, mode);
  dlg.fit();
  dlg.getValues(false);
  return 1;
}

int Kpl::splineFit(int iArr, int iSpl, int deg, double smf, double xmin,
                   double xmax, int errCol, const QString& errModPath,
                   const QString& errModName)
{
  if ((doc->items()->at(iArr)->iType() != KplItem::Array) ||
      (doc->items()->at(iSpl)->iType() != KplItem::Spline))
    return 0;
  ArrayItem* ad = (ArrayItem*) doc->items()->at(iArr);
  if (errCol && (ad->ie >= ad->ncols))
    return 0;
  doc->chisq = 0.0;
  QPtrList<SplineItem> spl;
  spl.append((SplineItem*) doc->items()->at(iSpl));
  SplFitDlg dlg(this, doc, (ArrayItem*) doc->items()->at(iArr),
                &spl, 0);
  KplNamespace::DataErrorStruct err = (*doc->options()).err;
  err.fitErrCol = errCol;
  err.errModPath = errModPath;
  err.errModName = errModName;
  if (dlg.fit(deg, smf, xmin, xmax, &err) < 4) {
    dlg.getValues(false);
    return 1;
  } else
    return 0;
}

double Kpl::chiSquare()
{
  return doc->chisq;
}

int Kpl::saveFitPar(const QString& url)
{
  return doc->saveParameter(this, KURL(url), doc->pLast, doc->pErr) ? 1 : 0;
}

int Kpl::exportValues(const QString& url, int iItem)
{
  KplItem::ItemTypes ityp = doc->items()->at(iItem)->iType();
  if ((ityp != KplItem::Function) && (ityp != KplItem::ParFunction) &&
      (ityp != KplItem::Spline) &&(ityp != KplItem::Function3D) &&
      (ityp != KplItem::Array3D))
    return 0;
  KURL u = KURL(url);
  QFile f(u.isLocalFile() ? u.path() : doc->tmpFile());
  int success = f.open(IO_WriteOnly) ? 1 : 0;
  if (success) {
    QTextStream ts(&f);
    doc->items()->at(iItem)->exportTable(ts, doc);
    f.close();
    if (!u.isLocalFile())
      doc->copyTmp(f.name(), u);
  } else
    KMessageBox::sorry(this, i18n("Writing to this file not possible."));
  return success;
}
