/***************************************************************************
                          splfitdlg.cpp  -  description
                             -------------------
    begin                : Fri Mar 1 2002
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "arrayitem.h"
#include "fitdlg.h"
#include "frameitem.h"
#include "funitem.h"
#include "kpldoc.h"
#include "kpldoubleedit.h"
#include "kplgraph.h"
#include "kplspinbox.h"
#include "residualsdlg.h"
#include "rootdlg.h"
#include "splfitdlg.h"
#include "utils.h"

SplFitDlg::SplFitDlg(QWidget* _parent, KplDoc* model, ArrayItem* ad0,
                     QPtrList<SplineItem>* fd0, int mode) :
 KDialogBase(Plain, i18n("Smoothing spline fit"),
             Help | Ok | Apply | Cancel | User1 | User2 | User3, Ok, _parent,
             0, true, true, i18n("&Start"), i18n("&Residuals"), i18n("R&oots")),
             running(false), dlgMode(mode), m(model), ad(ad0), fd(fd0)
{
  spl = *(fd->first());
  err = (*m->options()).err;
  if (ad->ie >= ad->ncols)
    err.fitErrCol = false;
  Utils::minMaxFile(&xmin, &xmax, &ad->x[ad->ix][ad->istart], ad->n);
  for (unsigned i = 0; i < fd->count(); i++) {
    SplineItem* spl2 = fd->at(i);
    xmin = QMIN(spl2->xmin, xmin);
    xmax = QMAX(spl2->xmax, xmax);
  }
  if (dlgMode & FitDlg::ShowDlg) {
    Utils::setSize(this, "SplineFitDialog");
    QFrame* frame = plainPage();
    QVBoxLayout* vbox = new QVBoxLayout(frame, 11, spacingHint());
    QHBoxLayout* hbox = new QHBoxLayout(vbox);
    QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Spline"), frame);
    hbox->addWidget(g);
    QGridLayout* grid = new QGridLayout(g->layout(), 5, 2, spacingHint());
    grid->addWidget(new QLabel(i18n("Degree"), g), 0, 0);
    grid->addWidget(sDeg = new KplSpinBox(1, 5, 1, g), 0, 1);
    sDeg->setValue(spl.kdeg);
    grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 0, 2);
    grid->addWidget(new QLabel(i18n("Smoothing factor"), g), 0, 3);
    grid->addWidget(eFactor = new KplDoubleEdit(1.0, g), 0, 4);
    grid->addWidget(new QLabel("xmin", g), 1, 0);
    grid->addWidget(exMin = new KplDoubleEdit(xmin, -1.0e300, xmin, g, 'g', 6),
                    1, 1);
    grid->addWidget(new QLabel("xmax", g), 1, 3);
    grid->addWidget(exMax = new KplDoubleEdit(xmax, xmax, 1.0e300, g, 'g', 6),
                    1, 4);
    g = new QGroupBox(0, Qt::Horizontal, i18n("Data errors"), frame);
    QHBoxLayout* hbox2 = new QHBoxLayout(g->layout(), spacingHint());
    hbox2->addWidget(new QLabel(i18n("Array"), g));
    QLabel* l = new QLabel(ad->url.fileName() + " " + QString::number(ad->ix) +
                           " " + QString::number(ad->iy) + " " +
                           QString::number(ad->ie), g);
    hbox2->addWidget(l);
    l->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox2->addWidget(errCol = new QCheckBox(i18n("Error column"), g));
    errCol->setChecked(err.fitErrCol);
    errCol->setEnabled(ad->ie < ad->ncols);
    hbox2->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding));
    hbox2->addWidget(errMod = new QPushButton(i18n("Model"), g));
    errMod->setEnabled(!errCol->isChecked());
    hbox->addWidget(g);
    vbox->addWidget(results = new QListBox(frame));
    results->setMinimumHeight(50);
    enableButton(User2, false);
    enableButton(User3, spl.nk && (spl.kdeg == 3));
    setHelp("SEC-SPLINE-FIT");
    connect(errMod, SIGNAL(clicked()), SLOT(slotErrMod()));
    connect(errCol, SIGNAL(toggled(bool)), SLOT(errColToggled(bool)));
  }
}

SplFitDlg::~SplFitDlg()
{
  if (dlgMode & FitDlg::ShowDlg)
    Utils::saveSize(this, "SplineFitDialog");
}

int SplFitDlg::cmpasc(const void *e1, const void *e2)
{
  double a1 = *((double*) e1);
  double a2 = *((double*) e2);
  if (a1 < a2)
    return -1;
  if (a1 > a2)
    return 1;
  return 0;
}

void SplFitDlg::getValues(bool ok)
{
  *(fd->first()) = spl;
  for (unsigned i = 1; i < fd->count(); i++) {
    SplineItem* spl2 = fd->at(i);
    spl2->nk = spl.nk;
    spl2->xmin = QMAX(fd->at(i)->xmin, xmin);
    spl2->xmax = QMIN(fd->at(i)->xmax, xmax);
    spl2->t.duplicate(spl.t, spl.nk);
    spl2->c.duplicate(spl.c, spl.nk - spl.kdeg - 1);
  }
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
}

void SplFitDlg::errColToggled(bool state)
{
  errMod->setEnabled(!state);
  enableButton(User2, false);
}

void SplFitDlg::slotOk()
{
  getValues(true);
}

void SplFitDlg::slotApply()
{
  getValues(false);
}

void SplFitDlg::slotUser1()
{
  enableButton(User2, false);
  err.fitErrCol = errCol->isChecked();
  fit(sDeg->interpretedValue(), eFactor->value(), exMin->value(),
  exMax->value(), &err);
}

int SplFitDlg::fit(int k, double f, double xmi, double xma,
                   KplNamespace::DataErrorStruct* er)
{
  spl.kdeg = k;
  int nk;
  double avgErr;
  int ier = spl.fit(ad, xmi, xma, &m->chisq, &nk, f, er, &avgErr);
  if (dlgMode & FitDlg::ShowDlg) {
    QString s;
    results->clear();
    if (ier > 0) {
      switch (ier) {
        case 1:
          s = i18n("curfit: The required storage space exceeds the available "
                   "storage space");
          break;
        case 2:
          s = i18n("curfit: A theoretically impossible result was found during "
                   "the iteration process");
          break;
        case 3:
          s = i18n("curfit: The maximal number of iterations maxit has been "
                   "reached");
          break;
        case 4:
          s = i18n("Data error <= 0. No output generated");
          break;
        case 10:
          s = i18n("Input error. No output generated");
      }
      results->insertItem(s);
    }
    if (ier < 4) {
      results->insertItem(s.sprintf(i18n("Number of knots = %i    "
                                         "chi-square = %.4g    "
                                         "Average error = %.3g"),
                                          nk, m->chisq, avgErr));
      enableButton(User2, true);
      enableButton(User3, spl.kdeg == 3);
    }
  }
  return ier;
}

void SplFitDlg::slotErrMod()
{
  ErrModDlg dlg(this, m, &err);
  if (dlg.exec())
    enableButton(User2, false);
}

void SplFitDlg::slotUser2()
{
  QPtrList<KplItem> items;
  items.setAutoDelete(true);
  FrameItem* fd = new FrameItem(true, false, false, -1, -1, 5, 2,
                                KplGraph::GridWithLabels, 0, 0, "0", "0",
                                4.0, 15.0, 3.0, 10.0, 0.0, 15.0, 0.0, 10.0,
                                5.0, 2.0, 1.0, "x", i18n("Residuals"), "",
                                m->options());
  items.append(fd);
  ArrayItem* ar = new ArrayItem(true, false, ad->symb, "0", 1.0, 1.0, 0, 1, 2,
                                0, ad->n, 1, "", true);
  ar->color = ad->color;
  SplineItem::sortedArrays(&ad->x[ad->ix][ad->istart],
                           &ad->x[ad->iy][ad->istart],
                           &ad->x[ad->ie][ad->istart], ad->n, &err,
                           ar->x[0], ar->x[1], ar->x[2]);
  QMemArray<double> fv(ad->n);
  spl.calcValues(ar->x[0], fv.data(), ad->n);
  for (int i = 0; i < ad->n; i++)
    ar->x[1][i] -= fv[i];
  items.append(ar);
  fd->autoScale(true, &items);
  ResidualsDlg dlg(this, m, &items);
  dlg.exec();
}

void SplFitDlg::slotUser3()
{
  RootDlg dlg(this, m, &spl);
  dlg.exec();
}
