/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a generic for board widget.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgboardwidget.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgzoomselector.h"

#include <QToolButton>
#include <QPushButton>
#include <kmenu.h>

SKGBoardWidget::SKGBoardWidget(SKGDocument* iDocument, const QString& iTitle, bool iZoomable)
    : SKGWidget(iDocument), m_title(iTitle), m_menu(NULL), m_zoomMenu(NULL), m_zoomRatio(1.0)
{
    SKGTRACEIN(10, "SKGBoardWidget::SKGBoardWidget");

    //Created widgets
    QHBoxLayout* horizontalLayout = new QHBoxLayout(this);
    horizontalLayout->setSpacing(0);
    horizontalLayout->setContentsMargins(0, 0, 0, 0);
    m_frame = new QFrame(this);
    m_frame->setObjectName(QString::fromUtf8("frame"));
    m_frame->setFrameShape(QFrame::StyledPanel);
    m_frame->setFrameShadow(QFrame::Raised);
    m_gridLayout = new QGridLayout(m_frame);
    m_gridLayout->setSpacing(2);
    m_gridLayout->setContentsMargins(0, 0, 0, 0);

    m_toolButton = new QToolButton(m_frame);
    m_toolButton->setIconSize(QSize(16, 16));
    m_toolButton->setMaximumSize(QSize(22, 22));
    m_toolButton->setPopupMode(QToolButton::InstantPopup);
    m_toolButton->setAutoRaise(true);
    m_toolButton->hide();

    m_gridLayout->addWidget(m_toolButton, 0, 0, 1, 1);

    m_Title = new QLabel(m_frame);
    QFont font = m_Title->font();
    font.setBold(true);
    m_Title->setFont(font);
    m_Title->setAlignment(Qt::AlignCenter);
    m_Title->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed));

    getDragWidget()->setCursor(QCursor(Qt::OpenHandCursor));

    m_gridLayout->addWidget(m_Title, 0, 1, 1, 1);

    m_line = new QFrame(m_frame);
    m_line->setFrameShape(QFrame::HLine);
    m_line->setFrameShadow(QFrame::Sunken);

    m_gridLayout->addWidget(m_line, 1, 0, 1, 2);

    horizontalLayout->addWidget(m_frame);

    //Add default actions
    QWidget* w = new QWidget(this);
    QHBoxLayout* hlayoutMove = new QHBoxLayout(w);
    hlayoutMove->setSpacing(2);
    hlayoutMove->setContentsMargins(0, 0, 0, 0);

    QPushButton* pbFirst = new QPushButton(w);
    pbFirst->setToolTip(i18nc("Move tooltip", "Move first"));
    pbFirst->setIcon(KIcon("go-first-view"));
    pbFirst->setMaximumSize(QSize(22, 22));
    pbFirst->setFlat(true);
    connect(pbFirst, SIGNAL(clicked(bool)), this, SLOT(requestMoveFirst()));
    hlayoutMove->addWidget(pbFirst);

    QPushButton* pbBefore = new QPushButton(w);
    pbBefore->setToolTip(i18nc("Move tooltip", "Move before"));
    pbBefore->setIcon(KIcon("go-previous-view"));
    pbBefore->setMaximumSize(QSize(22, 22));
    pbBefore->setFlat(true);
    connect(pbBefore, SIGNAL(clicked(bool)), this, SLOT(requestMoveBefore()));
    hlayoutMove->addWidget(pbBefore);

    QPushButton* pbDelete = new QPushButton(w);
    pbDelete->setToolTip(i18nc("Move tooltip", "Delete"));
    pbDelete->setIcon(KIcon("edit-delete"));
    pbDelete->setMaximumSize(QSize(22, 22));
    pbDelete->setFlat(true);
    connect(pbDelete, SIGNAL(clicked(bool)), this, SIGNAL(requestRemove()));
    hlayoutMove->addWidget(pbDelete);

    QPushButton* pbAfter = new QPushButton(w);
    pbAfter->setToolTip(i18nc("Move tooltip", "Move after"));
    pbAfter->setIcon(KIcon("go-next-view"));
    pbAfter->setMaximumSize(QSize(22, 22));
    pbAfter->setFlat(true);
    connect(pbAfter, SIGNAL(clicked(bool)), this, SLOT(requestMoveAfter()));
    hlayoutMove->addWidget(pbAfter);

    QPushButton* pbLast = new QPushButton(w);
    pbLast->setToolTip(i18nc("Move tooltip", "Move last"));
    pbLast->setIcon(KIcon("go-last-view"));
    pbLast->setMaximumSize(QSize(22, 22));
    pbLast->setFlat(true);
    connect(pbLast, SIGNAL(clicked(bool)), this, SLOT(requestMoveLast()));
    hlayoutMove->addWidget(pbLast);

    QWidgetAction* moveWidget = new QWidgetAction(this);
    moveWidget->setDefaultWidget(w);
    addAction(moveWidget);

    if (iZoomable) {
        m_zoomMenu = new SKGZoomSelector(this);
        m_zoomMenu->setResetValue(-10);
        m_zoomMenu->setValue(-10, false);
        connect(m_zoomMenu, SIGNAL(changed(int)), this, SLOT(onZoom(int)));

        QWidgetAction* zoomWidget = new QWidgetAction(this);
        zoomWidget->setDefaultWidget(m_zoomMenu);
        addAction(zoomWidget);
    }

    KAction* sep = new KAction(this);
    sep->setSeparator(true);
    addAction(sep);

    //Set main title
    setMainTitle(iTitle);

    //Set default icon
    m_toolButton->setIcon(KIcon("configure"));
}

SKGBoardWidget::~SKGBoardWidget()
{
    SKGTRACEIN(10, "SKGBoardWidget::~SKGBoardWidget");
}

QWidget* SKGBoardWidget::getDragWidget()
{
    return m_Title;
}

double SKGBoardWidget::getZoomRation()
{
    return m_zoomRatio;
}

void SKGBoardWidget::onZoom(int iZoom)
{
    setZoomRation((iZoom + 15.0) / 5.0);
}

void SKGBoardWidget::requestMoveAfter()
{
    Q_EMIT requestMove(1);
}

void SKGBoardWidget::requestMoveBefore()
{
    Q_EMIT requestMove(-1);
}

void SKGBoardWidget::requestMoveFirst()
{
    Q_EMIT requestMove(-100000);
}

void SKGBoardWidget::requestMoveLast()
{
    Q_EMIT requestMove(100000);
}

void SKGBoardWidget::setZoomRation(double iRation)
{
    if (m_zoomMenu) {
        if (m_zoomRatio == 1.0) {
            //Memorize initial size
            m_initialSize = minimumSize();
        }

        //Move zoom widget
        m_zoomRatio = iRation;
        if (m_zoomRatio < 1.0) m_zoomRatio = 1.0;
        else if (m_zoomRatio > 5.0) m_zoomRatio = 5.0;
        m_zoomMenu->setValue(5.0 * iRation - 15.0, false);

        //Resize widget
        QSize newSize(m_initialSize.width()*iRation, m_initialSize.height()*iRation);
        setMinimumSize(newSize);
    }
}

void SKGBoardWidget::setState(const QString& iState)
{
    Q_UNUSED(iState)
}

QString SKGBoardWidget::getState()
{
    return "";
}

void SKGBoardWidget::setMainWidget(QWidget* iWidget)
{
    iWidget->setParent(m_frame);
    m_gridLayout->addWidget(iWidget, 2, 0, 1, 2);
}

void SKGBoardWidget::insertAction(int iPos, QAction* iAction)
{
    m_menu->insertAction(m_menu->actions()[iPos], iAction);

    //Change icon
    if (!iAction->isCheckable() && !iAction->isSeparator()) m_toolButton->setIcon(KIcon("run-build-configure"));
}

void SKGBoardWidget::addAction(QAction* iAction)
{
    if (!m_menu) {
        m_menu = new KMenu(this);
        m_toolButton->show();
        m_toolButton->setMenu(m_menu);
    }
    m_menu->addAction(iAction);

    //Change icon
    if (!iAction->isCheckable() && !iAction->isSeparator()) m_toolButton->setIcon(KIcon("run-build-configure"));
}

void SKGBoardWidget::hideTitle()
{
    m_toolButton->hide();
    m_Title->hide();
    m_line->hide();
}

void SKGBoardWidget::setMainTitle(const QString& iTitle)
{
    m_Title->setText(iTitle);
}

QString SKGBoardWidget::getMainTitle()
{
    return m_Title->text();
}

QString SKGBoardWidget::getOriginalTitle()
{
    return m_title;
}

#include "skgboardwidget.h"
