/*  
    Copyright (C) 1998 Pietro Iglio (iglio@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
 */

#include "kdbm.h"

//////////////////////////////////////////////////////////////////////////////

bool
KGDbm::open(char* file, enum KDbmOpenMode mode) 
{
  assert(mode != read || this->fastMode == false);
  int flags;
  
  if (mode == read) {
    flags = GDBM_READER;
    assert(this->force == false);
  }
  else {
    assert(mode == readWrite);
    if (this->force)
      flags = GDBM_WRCREAT;
    else
      flags = GDBM_WRITER;
    
    flags |= this->fastMode;
  }

  this->dbf = gdbm_open(file,
                        this->blockSize,
                        flags,
                        this->accessMode,
                        0);

  return this->isOpen();
}


//////////////////////////////////////////////////////////////////////////////

bool
KGDbm::create(char* file)
{
    assert(!this->isOpen());

    this->dbf = gdbm_open(file, 
                          this->blockSize,
                          GDBM_NEWDB | this->fastMode,
                          this->accessMode,
                          0);

    return this->isOpen();
}

//////////////////////////////////////////////////////////////////////////////

void
KGDbm::enableBuffer(int _initSize)
{
    disableBuffer();
    this->buffer = new char[_initSize];
    this->bufferSize = _initSize;
}

//////////////////////////////////////////////////////////////////////////////

void
KGDbm::disableBuffer() 
{
    if (this->buffer) {
      delete [] buffer;
      this->buffer = 0;
      this->bufferSize = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////

int
KGDbm::resizeBuffer(int _newSize) 
{
    if (_newSize == 0)
      return this->bufferSize;
    else {
      this->buffer = (char*) realloc(this->buffer, _newSize);
      this->bufferSize = _newSize;
      return _newSize;
    }
}

//////////////////////////////////////////////////////////////////////////////

void
KGDbm::setFastMode(bool flag) 
{
    fastMode = flag;
    int arg = (int) fastMode;

    if (this->isOpen()) {
      int ret = gdbm_setopt(this->dbf, GDBM_FASTMODE, &arg, sizeof(int));
      assert(ret == 0);
    }
}

//////////////////////////////////////////////////////////////////////////////

char*
KGDbm::fetch(datum& _key, int* _psize) 
{
    assert(this->isOpen());
    datum content = gdbm_fetch(this->dbf, _key);

    if (_psize)
      *_psize = content.dsize;

    if (content.dptr == 0) {
      this->foundLastItem = false;
      return 0;
    }
  
    this->foundLastItem = true;

    if (bufferIsEnabled()) {
      copyInBuffer(content.dptr, content.dsize);
      free(content.dptr);
      return this->buffer;
    }
    else
      return content.dptr;
}

//////////////////////////////////////////////////////////////////////////////



// Template specialization for char* values
int
KDbm<char>::insert(char* _key, char* _val, const int /* _size */)
{
  datum key = STR_DATUM(_key);
  datum val = STR_DATUM(_val);

  return gdbm_store(dbf, key, val, GDBM_INSERT);
}

// Template specialization for char* values
int
KDbm<char>::replace(char* key, char* val, const int /* _size */)
{
  datum _key = STR_DATUM(key);
  datum _val = STR_DATUM(val);

  return gdbm_store(dbf, _key, _val, GDBM_REPLACE);
}


// Template override for "char*" values
int
KIntDbm<char>::insert(const long _key, char* _val, const int /* _size */)
{
  datum key = LONG_DATUM(_key);
  datum val = STR_DATUM(_val);

  return gdbm_store(dbf, key, val, GDBM_INSERT);
}

// Template override for <code>char*</code> values
int
KIntDbm<char>::replace(const long key, char* val, const int /* _size */)
{
  datum _key = LONG_DATUM(key);
  datum _val = STR_DATUM(val);

  return gdbm_store(dbf, _key, _val, GDBM_REPLACE);
}
