/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "kllabel.h"
#include "kllabelpriv.h"
#include "qpixmap.h"
#include "qpushbt.h"
#include "stdlib.h"
#include "klbutton.h"
#include "kltools.h"
#include "klhelper.h"
#include "kfiledialog.h"

IMPLEMENT_KLMETA_STANDALONE(KLTextPixmapMixer,KLWidgetBase,"Text/Image Mixer");

KLTextPixmapMixer::KLTextPixmapMixer(WidgetModes widgetMode, const char *text=0, const char *filename=0, KLTextPixmapMixer::DisplayModes displayMode = Text)
{
  a_widgetMode = widgetMode;
  a_displayMode = displayMode;
  a_filename = filename;
  a_text = text;
  if( displayMode == Image )
    setWeight(0);
  image = 0;
}

bool KLTextPixmapMixer::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  if( !KLWidgetBase::klAskMinMax(minMaxSizes) )
    return false;
  HASSERT(a_widget);
  QSize size = a_widget->sizeHint();
  minMaxSizes->minX += size.height();
  minMaxSizes->minY += size.width();
  minMaxSizes->defX += size.height();
  minMaxSizes->defY += size.width();
  minMaxSizes->maxX += size.height();
  minMaxSizes->maxY += size.width();
  minMaxSizes->maxY += KLMaxSize;
  return true;
}

bool KLTextPixmapMixer::klSetup( KLSetupInfo *setupInfo )
{
  if( !KLWidgetBase::klSetup(setupInfo) )
    return false;
  if( a_displayMode != Text) {
    HASSERT( image == 0 ); // Imagepointer MUST be NULL
    image = new QPixmap();
    if( !image->load(a_filename) ) 
      WARN((const char *)("Can't load image " + a_filename + "!"));
    if( a_widgetMode == PushButton ) {
      HASSERT( a_widget->inherits("QPushButton") );
      ((QPushButton *)a_widget)->setPixmap(*image);
    }
    else {
      HASSERT( a_widget->inherits("QLabel") );
      ((QLabel *)a_widget)->setPixmap(*image);
    }
  }
  else {
    if( a_widgetMode == PushButton ) {
      HASSERT( a_widget->inherits("QPushButton") );
      ((QPushButton *)a_widget)->setText(a_text);
    }
    else {
      HASSERT( a_widget->inherits("QLabel") );
      ((QLabel *)a_widget)->setText(a_text);
    }
  } 
  if( a_widgetMode == Label ) 
    a_widget->setFocusPolicy(QWidget::NoFocus);
  return true;
}


void KLTextPixmapMixer::setDisplayMode( KLTextPixmapMixer::DisplayModes displayMode )
{
  a_displayMode = displayMode;
  if( a_state == SI_Showed )
    klHide();
  if( a_state == SI_SetupDone ) {
    klCleanup();
    klSetup(&setupInfo());
  }
  doRelayout(true);
}

void KLTextPixmapMixer::klCleanup()
{
  if( image ) 
    delete image;
  image = 0;
  KLWidgetBase::klCleanup();
}


KLTextPixmapMixer::DisplayModes KLTextPixmapMixer::displayMode()
{
  return a_displayMode;
}


void KLTextPixmapMixer::setText(const char *text)
{
  a_text = text;
  if( a_displayMode != Image ) {
    if( a_state == SI_Showed )
      klHide();
    if( a_state == SI_SetupDone ) {
      klCleanup();
      klSetup(&setupInfo());
    }
    if( a_widgetMode == PushButton ) {
      HASSERT( a_widget->inherits("QPushButton") );
      ((QPushButton *)a_widget)->setText(a_text);
    }
    else {
      HASSERT( a_widget->inherits("QLabel") );
      ((QLabel *)a_widget)->setText(a_text);
    }
    doRelayout(true); 
  }
}


const char *KLTextPixmapMixer::text()
{
  return a_text;
}

void KLTextPixmapMixer::setFilename(const char *filename)
{
  a_filename = filename;
  if( a_displayMode != Text ) {
    if( a_state == SI_Showed )
      klHide();
    if( a_state == SI_SetupDone ) {
      klCleanup();
      klSetup(&setupInfo());
    }
    doRelayout(true); 
  }
}


const char *KLTextPixmapMixer::filename()
{
  return a_filename;
}


bool KLTextPixmapMixer::MyChildMeta::dumpObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLTextPixmapMixer *child = (KLTextPixmapMixer *)_child;
  dev.writeEntry("displaymode",helpStr.setNum((ulong)child->displayMode()));
  dev.writeEntry("text",child->text());
  dev.writeEntry("filename",child->filename());
  return true; 
}


void KLTextPixmapMixer::MyChildMeta::restoreObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLTextPixmapMixer *child = (KLTextPixmapMixer *)_child;
  child->setDisplayMode((DisplayModes)atol((dev.readEntry("displaymode",helpStr.setNum((ulong)child->displayMode())))));
  child->setText(dev.readEntry("text",child->text()));
  child->setFilename(dev.readEntry("filename",child->filename()));
}


KLChild *KLTextPixmapMixer::MyChildMeta::createObjectEdit(KLChild *child,bool) const
{
  HASSERT(child->metaInherits("KLTextPixmapMixer"));
  KLTextPixmapMixer *mixer = (KLTextPixmapMixer *)child;
  KLTextPixmapMixerEditGroup *edit = new KLTextPixmapMixerEditGroup("Display Modes");
  *edit << new KLRadioButton("Text",&edit->rm,0) << new KLRadioButton("Image",&edit->rm,1);

  KLButton *button;
  edit->mixer = mixer;
  KLLineEdit *textedit;
  KLGridGroup *gridgroup = (KLGridGroup *)KLChildMeta::createObjectEdit(child,false);
  *gridgroup << klHVSpace()
             << edit
             << (*klLabel("Text") << ::setWeight << 1)
             << (textedit = klLineEdit())
             << (*klLabel("Image") << ::setWeight << 1)
             << (klHGroup()
                << (edit->fileedit = klLineEdit())
                << (*(button = klButton("...")) << ::setWeight << 0));
  
  QObject::connect(&edit->rm,SIGNAL(clicked(int)),edit,SLOT(onMode(int)));
  QObject::connect(textedit,SIGNAL(textChanged(const char*)),edit,SLOT(onText(const char *)));
  QObject::connect(edit->fileedit,SIGNAL(textChanged(const char*)),edit,SLOT(onFile(const char *)));
  QObject::connect(button,SIGNAL(clicked()),edit,SLOT(onFileSelect()));


  QString helpStr;
  edit->rm.setChecked((int)mixer->displayMode());
  textedit->setText(mixer->text());
  edit->fileedit->setText(mixer->filename());

  return gridgroup;
}

void KLTextPixmapMixerEditGroup::onMode(int n)
{
  mixer->setDisplayMode((KLTextPixmapMixer::DisplayModes)n);
}

void KLTextPixmapMixerEditGroup::onText(const char *text)
{
  mixer->setText(text);
}

void KLTextPixmapMixerEditGroup::onFile(const char *file)
{
  mixer->setFilename(file);
}

void KLTextPixmapMixerEditGroup::onFileSelect()
{
  QString filename = KFileDialog::getOpenFileName(0,0,0,mixer->filename());
  if( filename.isEmpty() )
    return;
  mixer->setFilename(filename);
  fileedit->setText(filename);
}



//////////////////////////////////////////////////////////////////
// KLLabel
//////////////////////////////////////////////////////////////////

IMPLEMENT_KLMETA_STANDALONE(KLLabel,KLTextPixmapMixer,"Label");

KLLabel::KLLabel(const char *text, const char *filename, KLTextPixmapMixer::DisplayModes displayMode = Text) 
: QLabel((QWidget *)0,0), 
  KLTextPixmapMixer(KLTextPixmapMixer::Label,text,filename,displayMode) 
{
  setKLWidget(this); 
  setWeight(0);
}


#include "kllabel.moc"
#include "kllabelpriv.moc"

