/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#ifndef KLLABEL_H
#define KLLABEL_H

#include "klwidgetbase.h"
#include "qlabel.h"

/**
* Text/Pixmap Mixer. This is an internal toolclass
* to mix text and pixmaps on a button or label.
*/
class KLTextPixmapMixer : public KLWidgetBase {
public:
  /**
  * Display mix modes
  *
  * <UL>
  * <LI> Text:       Display text only </LI>
  * <LI> Image:      Display image only </LI>
  * <LI> HTextImage: Horionzontal: Text left / Image right (not yet) </LI>
  * <LI> HImageText: Horionzontal: Image left / Text right (not yet) </LI>
  * <LI> VTextImage: Vertical: Text top / Image bottom     (not yet) </LI>
  * <LI> VImageText: Vertical: Image top / Text bottom     (not yet) </LI>
  * </UL>
  */
  enum DisplayModes { Text, 
                      Image, 
                      HTextImage, 
                      HImageText, 
                      VTextImage, 
                      VImageText };
public:

  /**
  * Changes the text
  *
  * Changing this will force a relayout.
  */
  void setText(const char *);  

  /**
  * @return the text
  */
  const char *text();

  /**
  * Changes the image filename
  *
  * Changing this will force a relayout.
  */
  void setFilename(const char *);  

  /**
  * @return the image filename
  */
  const char *filename();

  /**
  * If true, the text() is interpreted as a
  * filename of an image. 
  *
  * Changing this will force a relayout.
  */
  void setDisplayMode( DisplayModes displayMode );
  /**
  * @return true if this label is an image
  */
  DisplayModes displayMode();

  virtual bool klAskMinMax(KLMinMaxSizes *minMaxSizes);
  virtual bool klSetup( KLSetupInfo *setupInfo );
  virtual void klCleanup();

/**
* Macro to delegate KLLabel/KLButton funcs to this baseclass
*
* Write in your class definetion:
*
* public:
* KLTPM_DELEGATION();
*
*/
#define KLTPM_DELEGATION()                                 \
  virtual bool klAskMinMax(KLMinMaxSizes *minMaxSizes)     \
  { return KLTextPixmapMixer::klAskMinMax(minMaxSizes); }; \
  virtual bool klSetup( KLSetupInfo *setupInfo )           \
  { return KLTextPixmapMixer::klSetup(setupInfo); };       \
  virtual void klCleanup()                                 \
  { KLTextPixmapMixer::klCleanup(); };                     \
  void setText(const char *text)                           \
  { KLTextPixmapMixer::setText(text); }                    \
  const char *text()                                       \
  { return KLTextPixmapMixer::text(); };                   \
  void setFilename(const char *filename)                   \
  { KLTextPixmapMixer::setFilename(filename); };           \
  const char *filename()                                   \
  { return KLTextPixmapMixer::filename(); }                \
  void setDisplayMode( DisplayModes displayMode )          \
  { KLTextPixmapMixer::setDisplayMode(displayMode ); };    \
  DisplayModes displayMode()                                \
  { return KLTextPixmapMixer::displayMode(); }             

protected:
  /**
  * The constructor is only for use with derivided classes 
  *
  * Note: If the constructor is called with displayMode == Image, it
  * will set the widget weight to 0.
  */
  enum WidgetModes { Label, PushButton };
  KLTextPixmapMixer( WidgetModes widgetMode = Label, const char *text=0, const char *filename=0, DisplayModes displayMode = Text);

  DECLARE_KLMETA_STANDALONE();

private:
  QString  a_filename;
  QString  a_text;
  WidgetModes a_widgetMode;
  DisplayModes a_displayMode;
  QPixmap *image;
};

class KLLabel : public QLabel, public KLTextPixmapMixer {
  Q_OBJECT
public:
  /**
  * Constructs a KLLabel object.
  */
  KLLabel(const char *text=0, const char *filename=0, DisplayModes displayMode = Text);
  KLTPM_DELEGATION();
  DECLARE_KLMETA_STANDALONE_SIMPLE();
  KLWDIGET_DELEGATES(KLLabel,QLabel);
};

#endif 

