/*  This file is part of the KDE libraries
    Copyright (C) 1998 Jrgen Hochwald (juergen.hochwald@privat.kkf.net

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; see the file COPYING.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.*/



#include "ksplineedit.h"
#include "ksplinedlg.h"
#include "ksplineedit.moc"
#include "kiconloader.h"
#define BTNH 24          // Hhe der Buttons
#define BTNW 24          // Breite der Buttons
#define STATH 15         // Hhe der Statuszeile

KSplineEdit::KSplineEdit(SplineCurve* Spl, QWidget* parent=0, const char* name=0,WFlags f=0,bool allowLines=true) 
     :QFrame (parent, name, f, allowLines)
{
//printf("KSplineEdit()\n");
   Spline = Spl;
   setFrameStyle( Panel | Sunken );
   setLineWidth( 1 );
   
   setFocusPolicy(QWidget::StrongFocus);
   AddBtn = new QPushButton("Add", this, "AddBtn");
   DelBtn = new QPushButton("Del", this, "DelBtn");
   NewBtn = new QPushButton("New", this, "NewBtn");
   ConfBtn= new QPushButton("Conf",this,"ConfBtn");
   UndoBtn= new QPushButton("Undo",this,"UndoBtn");
   ZoomInBtn= new QPushButton("Z+",this,"ZoomInBtn");
   ZoomOutBtn= new QPushButton("Z-",this,"ZoomOutBtn");
   PosNegBtn= new QPushButton("+-",this,"PosNegBtn");

   AddBtn->setPixmap(kapp->getIconLoader()->loadIcon("spl_addpnt.xpm"));
   DelBtn->setPixmap(kapp->getIconLoader()->loadIcon("spl_delpnt.xpm"));
   NewBtn->setPixmap(kapp->getIconLoader()->loadIcon("spl_new.xpm"));
   ConfBtn->setPixmap(kapp->getIconLoader()->loadIcon("configure.xpm"));
   UndoBtn->setPixmap(kapp->getIconLoader()->loadIcon("spl_undo.xpm"));
   ZoomInBtn->setPixmap(kapp->getIconLoader()->loadIcon("viewmag+.xpm"));
   ZoomOutBtn->setPixmap(kapp->getIconLoader()->loadIcon("viewmag-.xpm"));

   // Buttons haben keinen Focus
   AddBtn->setFocusPolicy(QWidget::NoFocus);
   DelBtn->setFocusPolicy(QWidget::NoFocus);
   NewBtn->setFocusPolicy(QWidget::NoFocus);
   ConfBtn->setFocusPolicy(QWidget::NoFocus);
   UndoBtn->setFocusPolicy(QWidget::NoFocus);
   ZoomInBtn->setFocusPolicy(QWidget::NoFocus);
   ZoomOutBtn->setFocusPolicy(QWidget::NoFocus);
   PosNegBtn->setFocusPolicy(QWidget::NoFocus);
   connect(AddBtn, SIGNAL(clicked()), this, SLOT(AddBtnClick()));
   connect(DelBtn, SIGNAL(clicked()), this, SLOT(DelBtnClick()));
   connect(NewBtn, SIGNAL(clicked()), this, SLOT(NewBtnClick()));
   connect(ConfBtn,SIGNAL(clicked()), this, SLOT(ConfBtnClick()));
   connect(UndoBtn,SIGNAL(clicked()), this, SLOT(UndoBtnClick()));
   connect(ZoomInBtn,SIGNAL(clicked()), this, SLOT(ZoomInBtnClick()));
   connect(ZoomOutBtn,SIGNAL(clicked()), this, SLOT(ZoomOutBtnClick()));
   connect(PosNegBtn,SIGNAL(clicked()), this, SLOT(PosNegBtnClick()));
   
   ZoomLab = new QLabel("Zoom", this, "ZoomLab");
   YLab = new QLabel("Y", this, "YLab");
   XLab = new QLabel("X", this, "XLab");
   Paint  = new QPainter();
   //SplArea = new SplineArea(this, "SplArea");
   Spline->calcSpline();
   if (Spline->numPoints()==0)
      CurPoint=-1;
   else
      CurPoint = 0;
   Save();  // Sichern
   Mod = FALSE;
   Zoom = 1.0;
   PosNeg=FALSE;
   GridSizeY = 1.0;
}

void KSplineEdit::cancel(void) {
// nderungen verwerfen
int i;

   Spline->clear();
   for (i=0; i<MAXPNTS; i++) {
      Spline->add(XSave[i],YSave[i]);
   }
   Spline->setY1(yp1);
   Spline->setYn(ypn);
   Spline->setNatural(ParaSave);
   Mod = FALSE;
}

void KSplineEdit::paintEvent( QPaintEvent * e) {
//printf("KSplineEdit::paintEvent\n");

   QFrame::paintEvent(e);
   Plot();
}

void KSplineEdit::resizeEvent( QResizeEvent * e) {
int fw;
//printf("KSplineEdit::resizeEvent\n");
   QFrame::resizeEvent(e);
   fw = frameWidth();
   AddBtn->setGeometry(width()-fw-BTNW,fw,BTNW,BTNH);
   DelBtn->setGeometry(width()-fw-BTNW,BTNH+fw,BTNW,BTNH);
   NewBtn->setGeometry(width()-fw-BTNW,2*BTNH+fw,BTNW,BTNH);
   ConfBtn->setGeometry(width()-fw-BTNW,3*BTNH+fw,BTNW,BTNH);
   UndoBtn->setGeometry(width()-fw-BTNW,4*BTNH+fw,BTNW,BTNH);
   ZoomInBtn->setGeometry(width()-fw-BTNW,5*BTNH+fw,BTNW,BTNH);
   ZoomOutBtn->setGeometry(width()-fw-BTNW,6*BTNH+fw,BTNW,BTNH);
   PosNegBtn->setGeometry(width()-fw-BTNW,7*BTNH+fw,BTNW,BTNH);

   ZoomLab->setGeometry(fw,height()-STATH+1,90,STATH-fw-1);
   XLab->setGeometry(fw+91,height()-STATH+1,90,STATH-fw-1);
   YLab->setGeometry(fw+182,height()-STATH+1,90,STATH-fw-1);
   
   SplX = width() - 2*fw-BTNW -3;
   SplY = height() - 2*fw-STATH -3;
   //NullX = 0;
   Ofs = fw+1;
   CalcD();
}

void KSplineEdit::CalcD () {
//printf("CalcD %d\n",SplX);
   dx = 1.0/SplX;
   dy = Zoom/SplY;
   if (PosNeg)
      dy *= 2;
   if (PosNeg)
      NullX = SplY>>1;
   else
      NullX=0;
}

void KSplineEdit::keyPressEvent (QKeyEvent*kev) {
double x,y;

   //printf("keypress\n");
   x = Spline->getX(CurPoint);
   y = Spline->getY(CurPoint);
   
   switch (kev->key()) {
      case Key_Insert : AddBtnClick(); break;
      case Key_Delete : DelBtnClick(); break;
      case Key_N : NewBtnClick(); break;
      case Key_C : ConfBtnClick(); break;
      case Key_X : UndoBtnClick(); break;
      case Key_Left : if (CurPoint>=0) {
                         MovePoint(x-dx,y);
                      }
                      break;
      case Key_Right: if (CurPoint>=0) {
                         MovePoint(x+dx,y);
                      }
                      break;
      case Key_Up   : if (CurPoint>=0) {
                         MovePoint(x,y+dy);
                      }
                      break;
      case Key_Down : if (CurPoint>=0) {
                         MovePoint(x,y-dy);
                      }
                      break;
      case Key_Next : if (CurPoint<Spline->numPoints()) {
                         CurPoint++;
                         update(Ofs,Ofs, SplX+1, SplY+1);
                      }
                      break;
      case Key_Prior: if (CurPoint > 0) {
                         CurPoint--;
                         update(Ofs,Ofs, SplX+1, SplY+1);
                      }
                      break;
      default: kev->ignore();
   }
}

void KSplineEdit::focusInEvent ( QFocusEvent * ) {
//printf("focusin\n");
   Plot();
}

void KSplineEdit::focusOutEvent ( QFocusEvent * ) {
//printf("fosucout\n");
   Plot();
}

void KSplineEdit::MovePoint(int x, int y) {
// den aktiven Punkt verschieben
double rx,ry;
int xl,xr;  // Bereich fr den X-Wert

   x=x-Ofs;
   y=SplY-y+Ofs-NullX;

   if (x<0) x=0; else
   if (x>SplX) x=SplX;
   if (PosNeg) {
      if (y<-NullX) y=-NullX; else
      if (y>NullX) y=NullX+1;
   } else {
      if (y<0) y=0; else
      if (y>SplY) y=SplY;
   }
   
   if (CurPoint==0)  // linker nicht nderbar
      rx = Spline->getX(0);
   else
   if (CurPoint==Spline->numPoints()-1)  // rechts genauso
      rx = Spline->getX(Spline->numPoints()-1);
   else {  // punkt in der Mitte
      xl = (int)(Spline->getX(CurPoint-1)/dx);
      xr = (int)(Spline->getX(CurPoint+1)/dx);
      if (x<=xl) x=xl+1; else
      if (x>=xr) x=xr-1;
      rx = dx*x;
   }   
   
   
   ry = dy*y;
   Spline->modify(CurPoint,rx,ry);
   update(Ofs,Ofs, SplX+1,SplY+1);
   Mod = TRUE;  // nderung
}

void KSplineEdit::MovePoint(double x, double y) {
// den aktiven Punkt verschieben
double rx,ry;
   
   if (CurPoint == 0) rx = Spline->getX(0);  // linker Rand
   else
   if (CurPoint == Spline->numPoints()-1)   // rechter Rand
      rx = Spline->getX(CurPoint); else
   {
      rx = x;

      if (rx-dx <= Spline->getX(CurPoint-1))
         rx = Spline->getX(CurPoint-1)+dx;  // Begrenzung links
      else if (rx+dx >= Spline->getX(CurPoint+1)) {
         rx = Spline->getX(CurPoint+1)-dx;
      }
   }
   
   ry=y;
   if (PosNeg) {
      if (ry<-Zoom) ry=-Zoom; else
      if (ry>Zoom) ry=Zoom;
   } else {
      if (ry<0.0) ry=0.0; else
      if (ry>Zoom) ry=Zoom;
   }
   
   Spline->modify(CurPoint,rx,ry);
   update(Ofs,Ofs, SplX+1,SplY+1);
   Mod = TRUE;
}

void KSplineEdit::mouseMoveEvent (QMouseEvent*mev) {
//printf("mousemove %d\n",mev->x());

   if (CurPoint>=0) {   // es ist ein Punkt aktiv
      MovePoint(mev->x(), mev->y());
   }
}

void KSplineEdit::mousePressEvent (QMouseEvent*mev) {
int i,px,py;

   CurPoint = -1;
   if (mev->button() == LeftButton) 
   {   // Bestimmung des gewhlten Punktes
      for (i=0; i<Spline->numPoints(); i++) {
         px = (int)(Spline->getX(i)/dx);
         py = (int)(Spline->getY(i)/dy);
         px = px+Ofs;
         py = SplY-py+Ofs-NullX;
         if (abs(mev->x()-px)<3)
         if (abs(mev->y()-py)<3) {
            CurPoint=i;
            //printf("CurPoint=%d\n",CurPoint);
            break;
         }
      }
   }
   update(Ofs,Ofs,SplX+1,SplY+1);
}

void KSplineEdit::Plot (void) {
int i,x,y,ay,b,h;
double ry,yu,yo;
//printf("Plot()\nSplX=%d  SplY=%d\nNullX=%d  PosNeg=%d\nZoom=%7.2f\n",SplX,SplY,NullX,PosNeg,Zoom);
   
   // Grid, Linien in bestimmten Y-Abstnden
   Paint->begin(this);
   Paint->setPen(QRgb(0x00A0A0A0));
   
   i =((int)(log10(Zoom)+100.0))-100;
//   if (i<0) i--;
   GridSizeY = pow(10.0,i);
   //printf("i=%d  Grid=%0.5f\n",i,GridSizeY);
   
   if (PosNeg)  ry = Zoom+Zoom;
   else ry=Zoom;
/*   i = (int)(ry/GridSizeY+0.5);
   if (i>10) GridSizeY *= 10;   // Linienabstnde anpassen
   else
   if (i<=1) GridSizeY *= 0.1;
*/
   yo = Zoom - fmod(Zoom, GridSizeY);
   if (PosNeg)  yu = -yo;
   else yu=0.0;
   for (ry=yu; ry<(yo+GridSizeY); ry+=GridSizeY) {
      y = (int)(ry/dy-0.5);
      y = SplY - y - NullX + Ofs;
      Paint->drawLine(Ofs,y,SplX+Ofs,y);
      //printf("%d\n",y);
   }
   
   // Nulllinie X
   Paint->setPen(QRgb(0x00808080));
   Paint->drawLine(Ofs,SplY+Ofs-NullX,SplX+Ofs,SplY+Ofs-NullX);

   // Labels aktualisieren
   PrintLabels();
   
   // Rahmen ausgeben
   if (hasFocus()) 
      Paint->setPen(QRgb(0x00000000));
   else
      Paint->setPen(QRgb(0x808080));
   Paint->drawRect(frameWidth(),frameWidth(),
                    width()-2*frameWidth()-BTNW,height()-2*frameWidth()-STATH);

   // Spline ausgeben
    //  Paint->setPen(QRgb(0xff4040));
    //  Paint->drawLine(Ofs,10,Ofs+SplX,10);
    //  Paint->drawLine(10,Ofs,10,Ofs+SplY);


   ay = (int)(Spline->spline(0)/dy);
   ay = SplY - ay;
   for (i=1; i<=SplX;i++) {
      y = (int)(Spline->spline(dx*i)/dy);
      if (PosNeg) {
         if (y<-NullX) y=-NullX; else
         if (y>NullX) y=NullX+1;
      } else {
         if (y<0) y=0; else
         if (y>SplY) y=SplY;
      }
      y = SplY - y;
      Paint->drawLine(i+Ofs-1,ay+Ofs-NullX,i+Ofs,y+Ofs-NullX);
      ay=y;
   }
  
   // Sttzstelle ausgeben
   for (i=0; i<Spline->numPoints(); i++) {
      b=h=5;
      x = (int)(Spline->getX(i)/dx);
      if (x<=1) {  // linker Rand
         b = b-2+x;
         x = 2;
      } else
      if (x>=SplX-1) {   // rechter Rand
         b = b-2-(SplX-x);
         //printf("hier %d\n",b);
      }
      y = (int)(Spline->getY(i)/dy);
      y = SplY - y - NullX;
      if (y<=1) {  // oberer Rand
         h = h-2+y;
         y = 2;
      } else
      if (y>=SplY-1) {   // unter Rand
         h = h-2-(SplY-y);
         //printf("hier %d\n",h);
      }
      if (i==CurPoint)
         Paint->fillRect(x-2+Ofs,y-2+Ofs,b,h,QBrush(QRgb(0xff)));
      else
         Paint->fillRect(x-2+Ofs,y-2+Ofs,b,h,QBrush(QRgb(0)));
   }
   
   Paint->end();
}

void KSplineEdit::AddBtnClick() {
double m,v;

   //printf("addbtn\n");
   if (CurPoint==-1) // kein Punkt selektiert,
      CurPoint=1;    // dann zwischen 1. und 2. Punkt einfgen
   if (CurPoint > 0) {
      m = (Spline->getX(CurPoint-1) + Spline->getX(CurPoint))*0.5;
      v = Spline->spline(m);
      //if (v<0.0) v=0.0; else
      //if (v>1.0) v=1.0;
      Spline->add(m, v);
      Spline->calcSpline();
      //CurPoint++;
      update(Ofs,Ofs, SplX+1,SplY+1);
      Mod = TRUE;
   }
}

void KSplineEdit::DelBtnClick() {
  //printf("DELBTNCLICK\n");
  if ((CurPoint > 0) && (CurPoint < Spline->numPoints()-1)) {
     Spline->del(CurPoint);
     update(Ofs,Ofs, SplX+1, SplY+1);
     Mod = TRUE;
  }
}
void KSplineEdit::NewBtnClick() {
//printf("NewBtnClick\n");
   Spline->defaultInit();
   update(Ofs,Ofs, SplX+1, SplY+1);
   Mod = TRUE;
}

void KSplineEdit::ConfBtnClick() {
//printf("ConfBtnClick\n");
KSplineDlg Kspldlg(Spline, this, "Kspldlg");

   if (Kspldlg.exec()) {
      Kspldlg.setParams();
      update(Ofs,Ofs, SplX+1,SplY+1);
      Mod = TRUE;
   }
}

void KSplineEdit::UndoBtnClick() {
//printf("UndoBtnClick\n");
   cancel();
   update(Ofs,Ofs, SplX+1,SplY+1);
}

void KSplineEdit::Save(void) {
// Alle Werte des Splines fr einen evtl. "Cancel" speichern
int i;

   for (i=0; i<MAXPNTS; i++) {
      XSave[i] = Spline->getX(i);
      YSave[i] = Spline->getY(i);
   }
   yp1 = Spline->getY1();
   ypn = Spline->getYn();
   ParaSave = Spline->natural();
   Mod = FALSE;  // Undo-Puffer leer
}

void KSplineEdit::setSpline(SplineCurve* Spl) {
   Spline = Spl;
   Save();   // Undo-Informationen speichern
   update(Ofs,Ofs, SplX+1,SplY+1);
   Mod = FALSE;
}

int KSplineEdit::modified(void) {
   return Mod;
}

void KSplineEdit::ZoomInBtnClick() {
   Zoom *= 0.5;
   CalcD();
   update(Ofs,Ofs, SplX+1,SplY+1);
}

void KSplineEdit::ZoomOutBtnClick() {
   Zoom *= 2.0;
   CalcD();
   update(Ofs,Ofs, SplX+1,SplY+1);
}

void KSplineEdit::PosNegBtnClick() {
//printf("PosNegBtnClick()\n");
//printf("PosNeg=%d\n",PosNeg);
//printf("Zoom=%7.2f\n",Zoom);
   PosNeg = 1-PosNeg; //toggle
   CalcD();
   if (PosNeg) NullX = SplY>>1;   // Position der X-Nulllinie
   else NullX=0;
   update(Ofs,Ofs, SplX+1,SplY+1);
}

int KSplineEdit::posNeg() {
   return PosNeg;
}

void KSplineEdit::setPosNeg(int pn) {
   PosNeg = pn;
   CalcD();
   if (PosNeg) NullX = SplY>>1;   // Position der X-Nulllinie
   else NullX=0;
   update(Ofs,Ofs, SplX+1,SplY+1);
//printf("setPosNeg()\n");
//printf("PosNeg=%d\nNullX=%d %d\n",PosNeg,NullX,SplX);
}

double KSplineEdit::zoom(void) {
   return 1.0/Zoom;
}

void KSplineEdit::setZoom(double z) {
   z = 1.0/z;
   if (z>0.0)
      Zoom = z;
   CalcD();
   update();
//printf("setZoom()\n");
//printf("PosNeg=%d\nNullX=%d %d\n",PosNeg,NullX,SplX);
}

void KSplineEdit::PrintLabels(void) {
char Strg[20];

   sprintf(Strg,"Z %10.4f", 1.0/Zoom);
   ZoomLab->setText(Strg);
   if (CurPoint>=0) {
      sprintf(Strg,"X %10.4f", Spline->getX(CurPoint));
      XLab->setText(Strg);
      sprintf(Strg,"Y %10.4f", Spline->getY(CurPoint));
      YLab->setText(Strg);
   }
}
