/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <qglobal.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qwhatsthis.h>
#include <qfile.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kstdaction.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>

#include "kmlocfg.h"
#include "global.h"
#include "configure.h"
#include "mloloader.h"
#include "setupdlg.h"
#include "tamsetupdlg.h"



StatusProgress::StatusProgress(QWidget *parent = 0, const char *name = 0): QFrame(parent, name)
{
	QBoxLayout *box = new QHBoxLayout(this, 0, 0);

	abort = new QPushButton("X", this);
	abort->setMinimumSize(20, fontMetrics().height());
	box->addWidget(abort);

	progress = new KProgress(this);
	progress->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
	progress->setLineWidth(0);
	progress->setMargin(0);
	progress->setBackgroundMode(QWidget::PaletteBackground);
	progress->setFixedSize(80, fontMetrics().height());
	box->addWidget(progress);

	box->activate();

	connect(abort, SIGNAL(clicked()), SLOT(abortSlot()));

	setFixedSize(sizeHint());
}


StatusProgress::~StatusProgress()
{
}


void StatusProgress::setEnabled(bool enable)
{
	if (enable) {
		abort->show();
		progress->show();
	} else {
		abort->hide();
		progress->hide();
		setProgress(0);
	}
}


void StatusProgress::abortSlot()
{
	emit aborted();
}



KMLOCfg::KMLOCfg(QWidget *parent, const char *name): KMainWindow(parent, name)
{
	config = kapp->config();

	has_config = false;
	has_dev_config = false;
	
	initMenuBar();
	initToolBar();
	initStatusBar();
	initKeyAccel();
	initWhatsThis();
	
	loader = new MLOLoader(this, "loader");
	CHECK_PTR(loader);

	view = new SetupWidget(MODEL_UNKNOWN, this, "setupwidget");
	view->setMinimumSize(view->sizeHint());
	setCentralWidget(view);

	enableCommand(ID_FILE_WRITE, false);
	enableCommand(ID_EXTRAS_DEVCONFIG, false);
	
	readOptions();

	connect(loader, SIGNAL(totalSteps(int)), progress, SLOT(setTotalSteps(int)));
	connect(loader, SIGNAL(progress(int)), progress, SLOT(setProgress(int)));
	connect(loader, SIGNAL(message(const QString &)), SLOT(slotStatusMsg(const QString &)));
	connect(loader, SIGNAL(done(int, bool)), SLOT(slotFetchDone(int, bool)));
	
	resize(sizeHint());
}


KMLOCfg::~KMLOCfg()
{
}


void KMLOCfg::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void KMLOCfg::readOptions()
{
	config->setGroup("General Options");

	bool toolBarOn = config->readBoolEntry("Show Toolbar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		toolBar()->hide();
	bool statusBarOn = config->readBoolEntry("Show Statusbar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		statusBar()->hide();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolbarPos", (int)KToolBar::Top));
}


void KMLOCfg::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "mainfilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(SmallIcon("download"), i18n("&Read Configuration"), ID_FILE_READ);
	fileMenu->insertItem(SmallIcon("upload"), i18n("&Write Configuration"), ID_FILE_WRITE);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("exit"), i18n("&Quit"), ID_FILE_QUIT);

	extrasMenu = new QPopupMenu(0, "extrasmenu");
	CHECK_PTR(extrasMenu);
	extrasMenu->insertItem(SmallIcon("devcfg"), i18n("Configure &Devices..."), ID_EXTRAS_DEVCONFIG);
	
	settingsMenu = new QPopupMenu(0, "mainsettingsmenu");
	CHECK_PTR(settingsMenu);
	settingsMenu->setCheckable(true);
	settingsMenu->insertItem(i18n("Show &Toolbar"), ID_SETTINGS_TOOLBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, true);
	settingsMenu->insertItem(i18n("Show &Statusbar"), ID_SETTINGS_STATUSBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, true);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Save &Options"), ID_SETTINGS_SAVEOPTIONS);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(SmallIcon("configure"), i18n("&Configure KMLOCfg..."), ID_SETTINGS_CONFIGURE);
	
	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Extras"), extrasMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu());

	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(extrasMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(settingsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void KMLOCfg::initToolBar()
{
	toolBar()->insertButton(BarIcon("download"), ID_FILE_READ, true, i18n("Read Configuration"));
	toolBar()->insertButton(BarIcon("upload"), ID_FILE_WRITE, true, i18n("Write Configuration"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("devcfg"), ID_EXTRAS_DEVCONFIG, true, i18n("Configure Devices"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("configure"), ID_SETTINGS_CONFIGURE, true, i18n("Configure KMLOCfg"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	toolBar()->insertButton(BarIcon("exit"), ID_FILE_QUIT, true, i18n("Quit"));
	toolBar()->alignItemRight(ID_FILE_QUIT, true);

	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}

	
void KMLOCfg::initStatusBar()
{
	progress = new StatusProgress(statusBar());	
	
	statusBar()->addWidget(progress, 0, true);
	
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);

	progress->setEnabled(false);
}


void KMLOCfg::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Read Configuration"), "ReadConfig", CTRL+Key_R);
	keyAccel->insertItem(i18n("Write Configuration"), "WriteConfig", CTRL+Key_W);
	keyAccel->insertItem(i18n("Configure Devices"), "DevConfig", CTRL+Key_D);
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);

	keyAccel->readSettings();
	
	keyAccel->connectItem("ReadConfig", this, SLOT(slotFileRead()));
	keyAccel->connectItem("WriteConfig", this, SLOT(slotFileWrite()));
	keyAccel->connectItem(KStdAccel::Quit, this, SLOT(slotFileQuit()));
	keyAccel->connectItem("DevConfig", this, SLOT(slotExtrasDevConfig()));
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotSettingsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotSettingsShowStatusbar()));
	
	keyAccel->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));

	keyAccel->changeMenuAccel(fileMenu, ID_FILE_READ, "ReadConfig");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_WRITE, "WriteConfig");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_QUIT, KStdAccel::Quit);
	keyAccel->changeMenuAccel(extrasMenu, ID_EXTRAS_DEVCONFIG, "DevConfig");
	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_STATUSBAR, "ShowStatusbar");
}


void KMLOCfg::initWhatsThis()
{
	QWhatsThis::add(toolBar()->getButton(ID_FILE_READ),
		i18n("Click this button to read the configuration from your modem.<br><br>" \
		"You can also select the command <b>Read Configuration</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_WRITE),
		i18n("Click this button to write the configuration to your modem.<br><br>" \
		"You can also select the command <b>Write Configuration</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_EXTRAS_DEVCONFIG),
		i18n("Click this button to configure the answering machines and fax devices of your modem.<br><br>" \
		"You can also select the command <b>Configure Devices</b> from the \"Extras\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_SETTINGS_CONFIGURE),
		i18n("Opens the Configure Dialog.<br><br>" \
		"You can also select the command <b>Configure KMLOCfg</b> from the \"Settings\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_HELP_CONTENTS),
		i18n("Invokes the KDE help system with the KMLOCfg help pages displayed.<br><br>" \
		"You can also select the command <b>Contents</b> from the \"Help\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_QUIT),
		i18n("Click this button to quit KMLOCfg.<br><br>" \
		"You can also select the command <b>Quit</b> from the \"File\" menu."));
}


void KMLOCfg::commandCallback(int id_)
{
	switch (id_) {
		case ID_FILE_READ:
			slotFileRead();
			break;
		case ID_FILE_WRITE:
			slotFileWrite();
			break;
		case ID_FILE_QUIT:
			slotFileQuit();
			break;
		case ID_EXTRAS_DEVCONFIG:
			slotExtrasDevConfig();
			break;
		case ID_SETTINGS_TOOLBAR:
			slotSettingsShowToolbar();
			break;
		case ID_SETTINGS_STATUSBAR:
			slotSettingsShowStatusbar();
			break;
		case ID_SETTINGS_SAVEOPTIONS:
			slotSettingsSaveOptions();
			break;
		case ID_SETTINGS_CONFIGURE:
			slotSettingsPreferences();
			break;
		default:
			break;
	}
}


void KMLOCfg::slotFileRead()
{
	if (loader->startRead()) {
		progress->setEnabled(true);
		slotStatusMsg(i18n("Reading modem configuration..."));
		enableCommand(ID_FILE_READ, false);
		enableCommand(ID_FILE_WRITE, false);
		enableCommand(ID_EXTRAS_DEVCONFIG, false);
		view->setEnabled(false);
	}
}


void KMLOCfg::slotFileWrite()
{
	loader->setConfig(view->getConfig());
	if (loader->startWrite()) {
		progress->setEnabled(true);
		slotStatusMsg(i18n("Writing modem configuration..."));
		enableCommand(ID_FILE_READ, false);
		enableCommand(ID_FILE_WRITE, false);
		enableCommand(ID_EXTRAS_DEVCONFIG, false);
		view->setEnabled(false);
	}
}


void KMLOCfg::slotFileQuit()
{
	close();
}


void KMLOCfg::slotExtrasDevConfig()
{
	if (loader->startFileRead()) {
		progress->setEnabled(true);
		slotStatusMsg(i18n("Reading device configuration..."));
		enableCommand(ID_FILE_READ, false);
		enableCommand(ID_FILE_WRITE, false);
		enableCommand(ID_EXTRAS_DEVCONFIG, false);
	}
}


void KMLOCfg::slotSettingsShowToolbar()
{
	bool toolBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_TOOLBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (toolBarOn)
		toolBar()->show();
	else
		toolBar()->hide();

	setFixedSize(minimumSizeHint());
}


void KMLOCfg::slotSettingsShowStatusbar()
{
	bool statusBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_STATUSBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (statusBarOn)
		statusBar()->show();
	else
		statusBar()->hide();
	
	setFixedSize(minimumSizeHint());
}


void KMLOCfg::slotSettingsSaveOptions()
{
	config->setGroup("General Options");
	
	config->writeEntry("Show Toolbar", toolBar()->isVisible());
	config->writeEntry("Show Statusbar", statusBar()->isVisible());
	config->writeEntry("ToolbarPos", (int)toolBar()->barPos());
}


void KMLOCfg::slotSettingsPreferences()
{
	Configure configure(this, "configure");
	if (configure.exec())
		view->setToolTips();
}


void KMLOCfg::slotStatusMsg(const QString &text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KMLOCfg::slotStatusHelpMsg(const QString &text)
{
	statusBar()->message(text, 2000);
}


void KMLOCfg::slotFetchDone(int action, bool success)
{
	bool enable = true;

	if (success)
		slotStatusMsg(i18n("Ready."));

	progress->setEnabled(false);
	view->setEnabled(true);

	if ((action == LOADER_ACTION_CFG_READ) && success) {
		const mlo_config_t *c = loader->getConfig();

		has_config = true;
//		has_dev_config = (c->model == MODEL_OFFICE_II) || (c->model == MODEL_ISDN_OFFICE);
		has_dev_config = (c->model == MODEL_ISDN_OFFICE);
	
		if (view->model() != c->model) {
			view->setToolTips(false);
			SetupWidget *nview = new SetupWidget(c->model, this, "setupwidget");
			nview->setMinimumSize(nview->sizeHint());
			setCentralWidget(nview);
			delete view;
			view = nview;
			view->show();
		}
		view->setConfig(c);
	}
	
	if ((action == LOADER_ACTION_CFG_WRITE) && success) {
		view->resetTime();
	}

	if ((action == LOADER_ACTION_FILE_READ) && success) {
		TAMSetupDlg setupdlg(view->model(), this, "tamsetupdlg");
		if (setupdlg.exec()) {
			if (loader->startFileWrite()) {
				progress->setEnabled(true);
				slotStatusMsg(i18n("Writing device configuration..."));
				enable = false;
			}
		} else
			QFile::remove(expandPath("config.cfg"));
	}

	if (action == LOADER_ACTION_FILE_WRITE) {
		QFile::remove(expandPath("config.cfg"));
	}
	
	enableCommand(ID_FILE_READ, enable);
	enableCommand(ID_FILE_WRITE, enable && has_config);
	enableCommand(ID_EXTRAS_DEVCONFIG, enable && has_dev_config);
}
