/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <qglobal.h>

#include <kapp.h>
#include <klocale.h>

#include "mloloader.h"
#include "global.h"


#define BOOL_TO_INT(x) ((x) ? 1 : 0)


MLOLoader::MLOLoader(QWidget *parent, const char *name): QObject(parent, name)
{
	config = kapp->config();

	modem = new Modem(this, "modem");
	CHECK_PTR(modem);

	init();
	action = LOADER_ACTION_NONE;
	
	connect(modem, SIGNAL(gotLine(const QCString &)), SLOT(fetchModemLine(const QCString &)));
	connect(modem, SIGNAL(xmodemStatus(int)), SLOT(fetchXModemStatus(int)));
	connect(modem, SIGNAL(xmodemDone(bool)), SLOT(fetchXModemDone(bool)));
	connect(modem, SIGNAL(timeout()), SLOT(fetchTimeout()));
}


MLOLoader::~MLOLoader()
{
	reset();
}


void MLOLoader::setDefaultConfig()
{
	c.firmware[0] = 0;
	c.date_time = QDateTime::currentDateTime();			// AT$JDATE, AT$JTIME
	c.sync_timer = false;

	if (c.model == MODEL_OFFICE) {
		c.mod.off.country = 0x04;
		c.mod.off.mem_free = -1;
		c.mod.off.fax_takeover = true;								// AT$JCFGF
		c.mod.off.accept_full = true;
		c.mod.off.announce_system = true;							// AT$JCFGM
		c.mod.off.announce_number = true;
		c.mod.off.announce_time = true;
		c.mod.off.outgoing_message = 0;
		c.mod.off.fax_operation = true;								// AT$JCFGT
		c.mod.off.voice_operation = true;
		c.mod.off.voice_recording = true;
		c.mod.off.hook_config = false;
		c.mod.off.keyboard_config = false;
		c.mod.off.accept_dtr = false;
		c.mod.off.max_rec_time = 0;										// AT$JCFGV
		c.mod.off.min_rec_time = 0;
		qstrcpy(c.mod.off.rec_quality, "ADPCM4-7");
		c.mod.off.rec_speaker = true;
		c.mod.off.rec_monitor = true;
		c.mod.off.fax_id[0] = 0;											// AT$JFLI
		c.mod.off.remote_query = false;								// AT$JPWD
		c.mod.off.remote_config = false;
		qstrcpy(c.mod.off.remote_pw, "9999");
		c.mod.off.ring_number = 4;										// AT$JRING
		c.mod.off.ring_signal = true;
		c.mod.off.suppress_ri = false;
		c.mod.off.accept_early = true;
		c.mod.off.rec_vol = 128;											// AT$JVGR
		c.mod.off.play_vol = 128;											// AT$JVGT
		c.mod.off.micro_vol = 255;										// AT$JVGM
		c.mod.off.speaker_vol = 255;									// AT$JVGS
	} else if (c.model == MODEL_OFFICE_II) {
		int i;
		c.mod.offii.country = 0x04;										// AT+GCI
		for (i=0; i < 3; i++)
			c.mod.offii.in_number[i][0] = 0;						// AT$JSPRN
	} else if (c.model == MODEL_ISDN_OFFICE) {
		int i;
		for (i=CONF_PORT_ISDN; i <= CONF_PORT_AB2; i++) {
			c.mod.isdnoff.msn_trans[i][0] = 0;					// AT$ICLI
			c.mod.isdnoff.msn_do_trans[i] = true;
			c.mod.isdnoff.msn_acc[i][0][0] = 0;					// AT$IMSN
			c.mod.isdnoff.msn_acc[i][1][0] = 0;
			c.mod.isdnoff.msn_acc[i][2][0] = 0;
			c.mod.isdnoff.eaz[i] = 127;									// AT$IEAZ
			c.mod.isdnoff.in_ind[i][0] = true;					// AT$ISCI
			c.mod.isdnoff.in_ind[i][1] = true;
			c.mod.isdnoff.in_ind[i][2] = true;
			c.mod.isdnoff.in_ind[i][3] = true;
			c.mod.isdnoff.out_ind[i] = 1;								// AT$ISCO
			c.mod.isdnoff.knock_enable[i] = false;			// AT$IBIS
			c.mod.isdnoff.accept_enable[i] = true;
			c.mod.isdnoff.auto_dial_num[i][0] = 0;			// AT$IAD
			c.mod.isdnoff.auto_dial[i] = false;
			c.mod.isdnoff.tot_charge[i] = 0;						// AT$ICI
			c.mod.isdnoff.last_charge[i] = 0;
			c.mod.isdnoff.reset_charge[i] = false;
		}
	
		c.mod.isdnoff.country = 49;										// AT$IPCS
		c.mod.isdnoff.exchange_hook[CONF_AB_PORT_1] = false;	// AT$IEL;
		c.mod.isdnoff.exchange_hook[CONF_AB_PORT_2] = false;
		c.mod.isdnoff.exchange_flash[CONF_AB_PORT_1] = false;
		c.mod.isdnoff.exchange_flash[CONF_AB_PORT_2] = false;
		qstrcpy(c.mod.isdnoff.d_chn_prot, "AUTO");			// AT$IDP
	}
}


const mlo_config_t *MLOLoader::getConfig()
{
	return &c;
}


void MLOLoader::setConfig(const mlo_config_t *conf)
{
	c = *conf;
}


bool MLOLoader::initModem()
{
	static int speeds[] = { 19200, 38400, 57600, 115200, 230400 };
	int speed;
	bool use_usb;

	if (status)
		return false;

	model = MODEL_UNKNOWN;

	config->setGroup(ID_PREF_GROUP_MODEM);
	use_usb = config->readBoolEntry(ID_PREF_MOD_USB, PREF_MOD_USB);
	if (use_usb) {
		modem->setDevice(use_usb, config->readNumEntry(ID_PREF_MOD_USB_DEVICE, PREF_MOD_USB_DEVICE));
	} else {
		modem->setDevice(use_usb, config->readNumEntry(ID_PREF_MOD_SER_DEVICE, PREF_MOD_SER_DEVICE));
		speed = config->readNumEntry(ID_PREF_MOD_SER_SPEED, PREF_MOD_SER_SPEED);
		if ((speed < 0) || (speed > 4))
			speed = PREF_MOD_SER_SPEED;
		modem->setSpeed(speeds[speed]);
	}

	modem->setLineParams(8, 'N', 1);

	if (!modem->open()) {
		emit message(i18n("Cannot open modem device."));
		return false;
	}
	if (!use_usb && !modem->dsrOn()) {
		emit message(i18n("Modem is off."));
		modem->close();
		return false;
	}
	if (!modem->ctsOn()) {
		emit message(i18n("Modem is busy."));
		modem->close();
		return false;
	}

	modem->writeLine("");
	usleep(250000);
	modem->flush();

	return true;
}

	
bool MLOLoader::startRead()
{
	if (!initModem())
		return false;

	action = LOADER_ACTION_CFG_READ;

	writeCommand("AT&F");

	emit totalSteps(35);

	status = 1;

	return true;
}
	
	
bool MLOLoader::startWrite()
{
	if (!initModem())
		return false;

	action = LOADER_ACTION_CFG_WRITE;
	
	writeCommand("AT&F");

	emit totalSteps(1);

	status = 1;

	return true;
}
	
	
bool MLOLoader::startFileRead()
{
	if (!initModem())
		return false;

	action = LOADER_ACTION_FILE_READ;
	
	writeCommand("AT&F");

	emit totalSteps(1000);

	status = 900;

	return true;
}
	
	
bool MLOLoader::startFileWrite()
{
	if (!initModem())
		return false;

	action = LOADER_ACTION_FILE_WRITE;

	writeCommand("AT&F");

	emit totalSteps(1000);

	status = 900;

	return true;
}
	
	
void MLOLoader::cancel()
{
	if ((action == LOADER_ACTION_FILE_READ) || (action == LOADER_ACTION_FILE_WRITE))
		modem->abortXModem();
	else {
		reset();
		emit done(action, false);
	}
}


void MLOLoader::fetchModemLine(const QCString &line)
{
	if ((action == LOADER_ACTION_CFG_READ) || (action == LOADER_ACTION_FILE_READ))
		fetchModemLineDown(line);
	if ((action == LOADER_ACTION_CFG_WRITE) || (action == LOADER_ACTION_FILE_WRITE))
		fetchModemLineUp(line);
}


void MLOLoader::fetchModemLineDown(const QCString &line)
{
	int d1, d2, d3, d4, d5, d6, p;
	char str1[41], str2[41], str3[41];

	if (line == command)	// Discard command echo
		return;

	if (line == "ERROR") {
		modem->timerStop();
		emit message(i18n("Error after modem command '%1'.").arg(command));
		reset();
		emit done(action, false);
		return;
	}

	if ((status >= 100) && (status < 900))
		emit progress(status % 100 + 1);

	switch (status) {
		case 1:
			if (line == "OK") {
				model = MODEL_UNKNOWN;
				writeCommand("ATI6");
				status++;
			}
			break;
		case 2:
			if (line == ID_MODEL_OFFICE)
				model = MODEL_OFFICE;
			else if (line == ID_MODEL_OFFICE_II)
				model = MODEL_OFFICE_II;
			else if (line == ID_MODEL_ISDN_OFFICE)
				model = MODEL_ISDN_OFFICE;
			else if (line == "OK") {
				modem->timerStop();
				if (model == MODEL_UNKNOWN) {
					reset();
					emit message(i18n("Unknown modem model."));
					emit done(action, false);
					return;
				}
				c.model = model;
				setDefaultConfig();
				writeCommand("ATI3");
				status++;
			}
			break;
		case 3:
			if (line.left(7) == "Version") {
				double vernum;
				qstrncpy(c.firmware, line.data(), 40);
				c.firmware[40] = 0;
				sscanf(c.firmware, "Version %lf", &vernum);
				if (model == MODEL_OFFICE && vernum < 1.22) {
					reset();
					emit message(i18n("Insufficient modem firmware."));
					emit done(action, false);
					return;
				}
				status++;
			}
			break;
		case 4:
			if (line == "OK") {
				writeCommand("AT$JDATE?");
				status++;
			}
			break;
		case 5:
			if (sscanf(line.data(), "$JDATE: %d,%d,%d", &d1, &d2, &d3) == 3) {
				c.date_time.setDate(QDate(d1, d2, d3));
				status++;
			}
			break;
		case 6:
			if (line == "OK") {
				writeCommand("AT$JTIME?");
				status++;
			}
			break;
		case 7:
			if (sscanf(line.data(), "$JTIME: %d,%d,%d", &d1, &d2, &d3) == 3) {
				c.date_time.setTime(QTime(d1, d2, d3, 0));
				switch (c.model) {
					case MODEL_OFFICE:
						emit totalSteps(28);
						status = 100;
						break;
					case MODEL_OFFICE_II:
						emit totalSteps(4);
						status = 200;
						break;
					case MODEL_ISDN_OFFICE:
						emit totalSteps(24);
						status = 300;
						break;
					default:
						break;
				}
			}
			break;
		case 100:
			if (line == "OK") {
				writeCommand("AT+GCI?");
				status++;
			}
			break;
		case 101:
			if (sscanf(line.data(), "+GCI: %x", &c.mod.off.country) == 1) {
				status++;
			}
			break;
		case 102:
			if (line == "OK") {
				writeCommand("AT$JMEM=Free");
				status++;
			}
			break;
		case 103:
			if (sscanf(line.data(), "$JMEM: %d", &c.mod.off.mem_free) == 1) {
				status++;
			}
			break;
		case 104:
			if (line == "OK") {
				writeCommand("AT$JCFGF?");
				status++;
			}
			break;
		case 105:
			if (sscanf(line.data(), "$JCFGF: %d,%d", &d1, &d2) > 0) {
				c.mod.off.fax_takeover = d1 != 0;
				c.mod.off.accept_full = d2 != 0;
				status++;
			}
			break;
		case 106:
			if (line == "OK") {
				writeCommand("AT$JCFGM?");
				status++;
			}
			break;
		case 107:
			if (sscanf(line.data(), "$JCFGM: %d,%d,%d,%d", &d1, &d2, &d3, &c.mod.off.outgoing_message) > 0) {
				c.mod.off.announce_system = d1 != 0;
				c.mod.off.announce_number = d2 != 0;
				c.mod.off.announce_time = d3 != 0;
				status++;
			}
			break;
		case 108:
			if (line == "OK") {
				writeCommand("AT$JCFGT?");
				status++;
			}
			break;
		case 109:
			if (sscanf(line.data(), "$JCFGT: %d,%d,%d,%d,%d,%d", &d1, &d2, &d3, &d4, &d5, &d6) > 0) {
				c.mod.off.fax_operation = d1 != 0;
				c.mod.off.voice_operation = d2 != 0;
				c.mod.off.voice_recording = d3 != 0;
				c.mod.off.hook_config = d4 != 0;
				c.mod.off.keyboard_config = d5 != 0;
				c.mod.off.accept_dtr = d6 != 0;
				status++;
			}
			break;
		case 110:
			if (line == "OK") {
				writeCommand("AT$JCFGV?");
				status++;
			}
			break;
		case 111:
			if (sscanf(line.data(), "$JCFGV: %d,%d,%*[^,],%[^,],%d,%d", &c.mod.off.max_rec_time, &c.mod.off.min_rec_time, c.mod.off.rec_quality, &d1, &d2) > 0) {
				c.mod.off.rec_speaker = d1 != 0;
				c.mod.off.rec_monitor = d2 != 0;
				status++;
			}
			break;
		case 112:
			if (line == "OK") {
				writeCommand("AT$JFLI?");
				c.mod.off.fax_id[0] = 0;
				status++;
			}
			break;
		case 113:
			if (line.left(8) == "$JFLI: \"") {
				c.mod.off.fax_id[0] = 0;
				sscanf(line, "$JFLI: \"%[^\"]\"", c.mod.off.fax_id);
				status++;
			}
			break;
		case 114:
			if (line == "OK") {
				writeCommand("AT$JPWD?");
				status++;
			}
			break;
		case 115:
			if (sscanf(line.data(), "$JPWD: %d,%d,\"%[^\"]\"", &d1, &d2, c.mod.off.remote_pw) > 0) {
				c.mod.off.remote_query = d1 != 0;
				c.mod.off.remote_config = d2 != 0;
				status++;
			}
			break;
		case 116:
			if (line == "OK") {
				writeCommand("AT$JRING?");
				status++;
			}
			break;
		case 117:
			if (sscanf(line.data(), "$JRING: %d,%d,%d,%d", &c.mod.off.ring_number, &d1, &d2, &d3) > 0) {
				c.mod.off.ring_signal = d1 != 0;
				c.mod.off.suppress_ri = d2 != 0;
				c.mod.off.accept_early = d3 != 0;
				status++;
			}
			break;
		case 118:
			if (line == "OK") {
				writeCommand("AT$JVGR?");
				status++;
			}
			break;
		case 119:
			if (sscanf(line.data(), "$JVGR: %d", &c.mod.off.rec_vol) > 0) {
				status++;
			}
			break;
		case 120:
			if (line == "OK") {
				writeCommand("AT$JVGT?");
				status++;
			}
			break;
		case 121:
			if (sscanf(line.data(), "$JVGT: %d", &c.mod.off.play_vol) > 0) {
				status++;
			}
			break;
		case 122:
			if (line == "OK") {
				writeCommand("AT$JVGM?");
				status++;
			}
			break;
		case 123:
			if (sscanf(line.data(), "$JVGM: %d", &c.mod.off.micro_vol) > 0) {
				status++;
			}
			break;
		case 124:
			if (line == "OK") {
				writeCommand("AT$JVGS?");
				status++;
			}
			break;
		case 125:
			if (sscanf(line.data(), "$JVGS: %d", &c.mod.off.speaker_vol) > 0) {
				status++;
			}
			break;
		case 126:
			if (line == "OK") {
				reset();
				emit progress(28);
				emit done(action, true);
			}
			break;
		case 200:
			if (line == "OK") {
				writeCommand("AT+GCI?");
				status++;
			}
			break;
		case 201:
			if (sscanf(line.data(), "+GCI: %x", &c.mod.offii.country) == 1) {
				status++;
			}
			break;
		case 202:
			if (line == "OK") {
				writeCommand("AT$JSPRN?");
				status++;
			}
			break;
		case 203:
			if (sscanf(line.data(), "$JSPRN: %d,%s", &d1, str1) == 2) {
				if ((d1 >= 0) && (d1 <= 2)) {
					d2 = qstrlen(str1)-2;
					if (d2 < 0)
						d2 = 0;
					if (d2 > 20)
						d2 = 20;
					strncpy(c.mod.offii.in_number[d1], str1+1, d2);
					c.mod.offii.in_number[d1][d2] = 0;
					if (d1 == 2)
						status++;
				}
			}
			break;
		case 204:
			if (line == "OK") {
				reset();
				emit progress(6);
				emit done(action, true);
			}
			break;
		case 300:
			if (line == "OK") {
				writeCommand("AT$ICLI?");
				status++;
			}
			break;
		case 301:
			if (sscanf(line.data(), "$ICLI: %d,%[^,],%d", &p, str1, &d1) == 3) {
				if ((p >= 0) && (p <= 2)) {
					d2 = qstrlen(str1)-2;
					if (d2 < 0)
						d2 = 0;
					if (d2 > 20)
						d2 = 20;
					strncpy(c.mod.isdnoff.msn_trans[p], str1+1, d2);
					c.mod.isdnoff.msn_trans[p][d2] = 0;
					c.mod.isdnoff.msn_do_trans[p] = (d1 != 0);
					if (p == 2)
						status++;
				}
			}
			break;
		case 302:
			if (line == "OK") {
				writeCommand("AT$IMSN?");
				status++;
			}
			break;
		case 303:
			if (sscanf(line.data(), "$IMSN: %d,%[^,],%[^,],%s", &p, str1, str2, str3) == 4) {
				if ((p >= 0) && (p <= 2)) {
					d1 = qstrlen(str1)-2;
					if (d1 < 0)
						d1 = 0;
					if (d1 > 20)
						d1 = 20;
					strncpy(c.mod.isdnoff.msn_acc[p][0], str1+1, d1);
					c.mod.isdnoff.msn_acc[p][0][d1] = 0;
					d2 = qstrlen(str2)-2;
					if (d2 < 0)
						d2 = 0;
					if (d2 > 20)
						d2 = 20;
					strncpy(c.mod.isdnoff.msn_acc[p][1], str2+1, d2);
					c.mod.isdnoff.msn_acc[p][1][d2] = 0;
					d3 = qstrlen(str3)-2;
					if (d3 < 0)
						d3 = 0;
					if (d3 > 20)
						d3 = 20;
					strncpy(c.mod.isdnoff.msn_acc[p][2], str3+1, d3);
					c.mod.isdnoff.msn_acc[p][2][d3] = 0;
					if (p == 2)
						status++;
				}
			}
			break;
		case 304:
			if (line == "OK") {
				writeCommand("AT$IEAZ?");
				status++;
			}
			break;
		case 305:
			if (sscanf(line.data(), "$IEAZ: %d,%s", &p, str1) == 2) {
				if ((p >= 0) && (p <= 2)) {
					if (str1[1] == '"')
						c.mod.isdnoff.eaz[p] = 127;
					else {
						if (str1[2] == '"')
							str1[2] = 0;
						else
							str1[3] = 0;
						c.mod.isdnoff.eaz[p] = (char)atoi(&str1[1]);
					}
					if (p == 2)
						status++;
				}
			}
			break;
		case 306:
			if (line == "OK") {
				writeCommand("AT$ISCI?");
				status++;
			}
			break;
		case 307:
			if (sscanf(line.data(), "$ISCI: %d,%d,%d,%d,%d", &p, &d1, &d2, &d3, &d4) == 5) {
				if ((p >= 0) && (p <= 2)) {
					c.mod.isdnoff.in_ind[p][0] = (d1 != 0);
					c.mod.isdnoff.in_ind[p][1] = (d2 != 0);
					c.mod.isdnoff.in_ind[p][2] = (d3 != 0);
					c.mod.isdnoff.in_ind[p][3] = (d4 != 0);
					if (p == 2)
						status++;
				}
			}
			break;
		case 308:
			if (line == "OK") {
				writeCommand("AT$ISCO?");
				status++;
			}
			break;
		case 309:
			if (sscanf(line.data(), "$ISCO: %d,%d", &p, &d1) == 2) {
				if ((p >= 0) && (p <= 2)) {
					c.mod.isdnoff.out_ind[p] = (d1 != 0);
					if (p == 2)
						status++;
				}
			}
			break;
		case 310:
			if (line == "OK") {
				writeCommand("AT$IBIS?");
				status++;
			}
			break;
		case 311:
			if (sscanf(line.data(), "$IBIS: %d,%d,%d", &p, &d1, &d2) == 3) {
				if ((p >= 0) && (p <= 2)) {
					c.mod.isdnoff.knock_enable[p] = (d1 != 0);
					c.mod.isdnoff.accept_enable[p] = (d2 != 0);
					if (p == 2)
						status++;
				}
			}
			break;
		case 312:
			if (line == "OK") {
				writeCommand("AT$IAD?");
				status++;
			}
			break;
		case 313:
			if (sscanf(line.data(), "$IAD: %d,%[^,],%d", &p, str1, &d1) == 3) {
				if ((p >= 0) && (p <= 2)) {
					d2 = qstrlen(str1)-2;
					if (d2 < 0)
						d2 = 0;
					if (d2 > 20)
						d2 = 20;
					strncpy(c.mod.isdnoff.auto_dial_num[p], str1+1, d2);
					c.mod.isdnoff.auto_dial_num[p][d2] = 0;
					c.mod.isdnoff.auto_dial[p] = (d1 != 0);
					if (p == 2)
						status++;
				}
			}
			break;
		case 314:
			if (line == "OK") {
				writeCommand("AT$ICI?");
				status++;
			}
			break;
		case 315:
			if (sscanf(line.data(), "$ICI: %d,%d, %d", &p, &d1, &d2) == 3) {
				if ((p >= 0) && (p <= 2)) {
					c.mod.isdnoff.tot_charge[p] = d1;
					c.mod.isdnoff.last_charge[p] = d2;
					if (p == 2)
						status++;
				}
			}
			break;
		case 316:
			if (line == "OK") {
				writeCommand("AT$IPCS?");
				status++;
			}
			break;
		case 317:
			if (sscanf(line.data(), "$IPCS: %d", &d1) == 1) {
				c.mod.isdnoff.country = d1;
				status++;
			}
			break;
		case 318:
			if (line == "OK") {
				writeCommand("AT$IEL?");
				status++;
			}
			break;
		case 319:
			if (sscanf(line.data(), "$IEL: %d,%d, %d", &p, &d1, &d2) == 3) {
				if ((p >= 1) && (p <= 2)) {
					c.mod.isdnoff.exchange_hook[p-1] = (d1 != 0);
					c.mod.isdnoff.exchange_flash[p-1] = (d2 != 0);
					if (p == 2)
						status++;
				}
			}
			break;
		case 320:
			if (line == "OK") {
				writeCommand("AT$IDP?");
				status++;
			}
			break;
		case 321:
			if (sscanf(line.data(), "$IDP: %s", str1) == 1) {
				strncpy(c.mod.isdnoff.d_chn_prot, str1, 7);
				c.mod.isdnoff.d_chn_prot[7] = 0;
				status++;
			}
			break;
		case 322:
			if (line == "OK") {
				reset();
				emit progress(24);
				emit done(action, true);
			}
			break;
		case 900:
			if (line == "OK") {
				emit progress(1);
				cfg_file = false;
				writeCommand("AT$JDIR", 3000);
				status++;
			}
			break;
		case 901:
			char buf[81], *p, *type, *flags, *datetime, *sizes, *clip;

			if (line == "OK") {
				modem->timerStop();
				if (cfg_file) {
					status++;
					downloadFile();
				} else {
					reset();
					createFile();
					emit message(i18n("Ready."));
					emit done(action, true);
				}
				break;
			}

			strncpy(buf, line, 80);
			buf[80] = 0;

			if (!(type = strchr(buf, ',')))
				break;
			p = type;
			type++;
			while ((p > buf) && (*(p-1) == ' '))
				p--;
			*p = 0;

			if (qstrcmp(buf, "config.cfg"))
				break;

			while (*type == ' ')
				type++;

			if (!(flags = strchr(type, ',')))
				break;
			*flags = 0;
			flags++;
			
			if (!(datetime = strchr(flags, ',')))
				break;
			*datetime = 0;
			datetime++;
			
			if (!(p = strchr(datetime, ',')))
				break;
			p++;
			
			if (!(sizes = strchr(p, ',')))
				break;
			*sizes = 0;
			sizes++;
			
			if ((clip = strchr(sizes, ','))) {
				*clip = 0;
				clip++;
			}

			if (sscanf(sizes, " %d", &cfg_file_size) != 1)
				break;

			cfg_file = true;

			break;
		case 902:
			if (line == "OK") {
				reset();
				emit progress(cfg_file_size+1);
				emit done(action, true);
			}
			break;
		default:
			break;
	}
}


void MLOLoader::fetchModemLineUp(const QCString &line)
{
	QCString com;

	if (line == command)	// Discard command echo
		return;

	if (line == "ERROR") {
		modem->timerStop();
		emit message(i18n("Error after modem command '%1'.").arg(command));
		reset();
		emit done(action, false);
		return;
	}

	if ((status >= 100) && (status < 900))
		emit progress(status % 100 + 1);

	switch (status) {
		case 1:
			if (line == "OK") {
				model = MODEL_UNKNOWN;
				writeCommand("ATI6");
				status++;
			}
			break;
		case 2:
			if (line == ID_MODEL_OFFICE)
				model = MODEL_OFFICE;
			else if (line == ID_MODEL_OFFICE_II)
				model = MODEL_OFFICE_II;
			else if (line == ID_MODEL_ISDN_OFFICE)
				model = MODEL_ISDN_OFFICE;
			else if (line == "OK") {
				modem->timerStop();
				if (model == MODEL_UNKNOWN) {
					reset();
					emit message(i18n("Unknown modem model."));
					emit done(action, false);
					return;
				}
				if (model != c.model) {
					reset();
					emit message(i18n("Modem model mismatch."));
					emit done(action, false);
					return;
				}
				usleep(250000);

				switch (c.model) {
					case MODEL_OFFICE:
						emit totalSteps(15);
						com.sprintf("AT+GCI=%x", c.mod.off.country);
						writeCommand(com);
						status = 100;
						break;
					case MODEL_OFFICE_II:
						emit totalSteps(7);
						com.sprintf("AT+GCI=%x", c.mod.offii.country);
						writeCommand(com);
						status = 200;
						break;
					case MODEL_ISDN_OFFICE:
						emit totalSteps(27);
						com.sprintf("AT$ICLI=0,\"%s\",%d", c.mod.isdnoff.msn_trans[CONF_PORT_ISDN], BOOL_TO_INT(c.mod.isdnoff.msn_do_trans[CONF_PORT_ISDN]));
						writeCommand(com);
						status = 300;
						break;
					default:
						break;
				}
			}
			break;
		case 100:
			if (line == "OK") {
				com.sprintf("AT$JCFGF=%d,%d", BOOL_TO_INT(c.mod.off.fax_takeover), BOOL_TO_INT(c.mod.off.accept_full));
				writeCommand(com);
				status++;
			}
			break;
		case 101:
			if (line == "OK") {
				com.sprintf("AT$JCFGM=%d,%d,%d,%d", BOOL_TO_INT(c.mod.off.announce_system), BOOL_TO_INT(c.mod.off.announce_number), BOOL_TO_INT(c.mod.off.announce_time), c.mod.off.outgoing_message);
				writeCommand(com);
				status++;
			}
			break;
		case 102:
			if (line == "OK") {
				com.sprintf("AT$JCFGT=%d,%d,%d,%d,%d,%d", BOOL_TO_INT(c.mod.off.fax_operation), BOOL_TO_INT(c.mod.off.voice_operation), BOOL_TO_INT(c.mod.off.voice_recording), BOOL_TO_INT(c.mod.off.hook_config), BOOL_TO_INT(c.mod.off.keyboard_config), BOOL_TO_INT(c.mod.off.accept_dtr));
				writeCommand(com);
				status++;
			}
			break;
		case 103:
			if (line == "OK") {
				com.sprintf("AT$JCFGV=%d,%d,,%s,%d,%d", c.mod.off.max_rec_time, c.mod.off.min_rec_time, c.mod.off.rec_quality, BOOL_TO_INT(c.mod.off.rec_speaker), BOOL_TO_INT(c.mod.off.rec_monitor));
				writeCommand(com);
				status++;
			}
			break;
		case 104:
			if (line == "OK") {
				com.sprintf("AT$JFLI=\"%s\"", c.mod.off.fax_id);
				writeCommand(com);
				status++;
			}
			break;
		case 105:
			if (line == "OK") {
				com.sprintf("AT$JPWD=%d,%d,\"%s\"", BOOL_TO_INT(c.mod.off.remote_query), BOOL_TO_INT(c.mod.off.remote_config), c.mod.off.remote_pw);
				writeCommand(com);
				status++;
			}
			break;
		case 106:
			if (line == "OK") {
				com.sprintf("AT$JRING=%d,%d,%d,%d", c.mod.off.ring_number, BOOL_TO_INT(c.mod.off.ring_signal), BOOL_TO_INT(c.mod.off.suppress_ri), BOOL_TO_INT(c.mod.off.accept_early));
				writeCommand(com);
				status++;
			}
			break;
		case 107:
			if (line == "OK") {
				com.sprintf("AT$JVGR=%d", c.mod.off.rec_vol);
				writeCommand(com);
				status++;
			}
			break;
		case 108:
			if (line == "OK") {
				com.sprintf("AT$JVGT=%d", c.mod.off.play_vol);
				writeCommand(com);
				status++;
			}
			break;
		case 109:
			if (line == "OK") {
				com.sprintf("AT$JVGM=%d", c.mod.off.micro_vol);
				writeCommand(com);
				status++;
			}
			break;
		case 110:
			if (line == "OK") {
				com.sprintf("AT$JVGS=%d", c.mod.off.speaker_vol);
				writeCommand(com);
				if (c.sync_timer)
					status++;
				else
					status = 113;
			}
			break;
		case 111:
			if (line == "OK") {
				c.date_time = QDateTime::currentDateTime();
				com.sprintf("AT$JTIME=%d,%d,%d", c.date_time.time().hour(), c.date_time.time().minute(), c.date_time.time().second());
				writeCommand(com);
				status++;
			}
			break;
		case 112:
			if (line == "OK") {
				com.sprintf("AT$JDATE=%d,%d,%d", c.date_time.date().year(), c.date_time.date().month(), c.date_time.date().day());
				writeCommand(com);
				status++;
			}
			break;
		case 113:
			if (line == "OK") {
				reset();
				emit progress(15);
				emit done(action, true);
			}
			break;
		case 200:
			if (line == "OK") {
				com.sprintf("AT$JSPRN=0,\"%s\"", c.mod.offii.in_number[0]);
				writeCommand(com);
				status++;
			}
			break;
		case 201:
			if (line == "OK") {
				com.sprintf("AT$JSPRN=1,\"%s\"", c.mod.offii.in_number[1]);
				writeCommand(com);
				status++;
			}
			break;
		case 202:
			if (line == "OK") {
				com.sprintf("AT$JSPRN=2,\"%s\"", c.mod.offii.in_number[2]);
				writeCommand(com);
				if (c.sync_timer)
					status++;
				else
					status = 205;
			}
			break;
		case 203:
			if (line == "OK") {
				c.date_time = QDateTime::currentDateTime();
				com.sprintf("AT$JTIME=%d,%d,%d", c.date_time.time().hour(), c.date_time.time().minute(), c.date_time.time().second());
				writeCommand(com);
				status++;
			}
			break;
		case 204:
			if (line == "OK") {
				com.sprintf("AT$JDATE=%d,%d,%d", c.date_time.date().year(), c.date_time.date().month(), c.date_time.date().day());
				writeCommand(com);
				status++;
			}
			break;
		case 205:
			if (line == "OK") {
				reset();
				emit progress(7);
				emit done(action, true);
			}
			break;
		case 300:
			if (line == "OK") {
				com.sprintf("AT$ICLI=1,\"%s\",%d", c.mod.isdnoff.msn_trans[CONF_PORT_AB1], BOOL_TO_INT(c.mod.isdnoff.msn_do_trans[CONF_PORT_AB1]));
				writeCommand(com);
				status++;
			}
			break;
		case 301:
			if (line == "OK") {
				com.sprintf("AT$ICLI=2,\"%s\",%d", c.mod.isdnoff.msn_trans[CONF_PORT_AB2], BOOL_TO_INT(c.mod.isdnoff.msn_do_trans[CONF_PORT_AB2]));
				writeCommand(com);
				status++;
			}
			break;
		case 302:
			if (line == "OK") {
				com.sprintf("AT$IMSN=0,\"%s\",\"%s\",\"%s\"", c.mod.isdnoff.msn_acc[CONF_PORT_ISDN][0], c.mod.isdnoff.msn_acc[CONF_PORT_ISDN][1], c.mod.isdnoff.msn_acc[CONF_PORT_ISDN][2]);
				writeCommand(com);
				status++;
			}
			break;
		case 303:
			if (line == "OK") {
				com.sprintf("AT$IMSN=1,\"%s\",\"%s\",\"%s\"", c.mod.isdnoff.msn_acc[CONF_PORT_AB1][0], c.mod.isdnoff.msn_acc[CONF_PORT_AB1][1], c.mod.isdnoff.msn_acc[CONF_PORT_AB1][2]);
				writeCommand(com);
				status++;
			}
			break;
		case 304:
			if (line == "OK") {
				com.sprintf("AT$IMSN=2,\"%s\",\"%s\",\"%s\"", c.mod.isdnoff.msn_acc[CONF_PORT_AB2][0], c.mod.isdnoff.msn_acc[CONF_PORT_AB2][1], c.mod.isdnoff.msn_acc[CONF_PORT_AB2][2]);
				writeCommand(com);
				status++;
			}
			break;
		case 305:
			if (line == "OK") {
				if (c.mod.isdnoff.eaz[CONF_PORT_ISDN] > 19)
					com.sprintf("AT$IEAZ=0,\"\"");
				else
					com.sprintf("AT$IEAZ=0,\"%d\"", c.mod.isdnoff.eaz[CONF_PORT_ISDN]);
				writeCommand(com);
				status++;
			}
			break;
		case 306:
			if (line == "OK") {
				if (c.mod.isdnoff.eaz[CONF_PORT_AB1] > 19)
					com.sprintf("AT$IEAZ=1,\"\"");
				else
					com.sprintf("AT$IEAZ=1,\"%d\"", c.mod.isdnoff.eaz[CONF_PORT_AB1]);
				writeCommand(com);
				status++;
			}
			break;
		case 307:
			if (line == "OK") {
				if (c.mod.isdnoff.eaz[CONF_PORT_AB2] > 19)
					com.sprintf("AT$IEAZ=2,\"\"");
				else
					com.sprintf("AT$IEAZ=2,\"%d\"", c.mod.isdnoff.eaz[CONF_PORT_AB2]);
				writeCommand(com);
				status++;
			}
			break;
		case 308:
			if (line == "OK") {
				com.sprintf("AT$ISCO=0,%d", BOOL_TO_INT(c.mod.isdnoff.out_ind[CONF_PORT_ISDN]));
				writeCommand(com);
				status++;
			}
			break;
		case 309:
			if (line == "OK") {
				com.sprintf("AT$ISCO=1,%d", BOOL_TO_INT(c.mod.isdnoff.out_ind[CONF_PORT_AB1]));
				writeCommand(com);
				status++;
			}
			break;
		case 310:
			if (line == "OK") {
				com.sprintf("AT$ISCO=2,%d", BOOL_TO_INT(c.mod.isdnoff.out_ind[CONF_PORT_AB2]));
				writeCommand(com);
				status++;
			}
			break;
		case 311:
			if (line == "OK") {
				com.sprintf("AT$IBIS=0,%d,%d", BOOL_TO_INT(c.mod.isdnoff.knock_enable[CONF_PORT_ISDN]), BOOL_TO_INT(c.mod.isdnoff.accept_enable[CONF_PORT_ISDN]));
				writeCommand(com);
				status++;
			}
			break;
		case 312:
			if (line == "OK") {
				com.sprintf("AT$IBIS=1,%d,%d", BOOL_TO_INT(c.mod.isdnoff.knock_enable[CONF_PORT_AB1]), BOOL_TO_INT(c.mod.isdnoff.accept_enable[CONF_PORT_AB1]));
				writeCommand(com);
				status++;
			}
			break;
		case 313:
			if (line == "OK") {
				com.sprintf("AT$IBIS=2,%d,%d", BOOL_TO_INT(c.mod.isdnoff.knock_enable[CONF_PORT_AB2]), BOOL_TO_INT(c.mod.isdnoff.accept_enable[CONF_PORT_AB2]));
				writeCommand(com);
				status++;
			}
			break;
		case 314:
			if (line == "OK") {
				com.sprintf("AT$IAD=0,\"%s\",%d", c.mod.isdnoff.auto_dial_num[CONF_PORT_ISDN], BOOL_TO_INT(c.mod.isdnoff.auto_dial[CONF_PORT_ISDN]));
				writeCommand(com);
				status++;
			}
			break;
		case 315:
			if (line == "OK") {
				com.sprintf("AT$IAD=1,\"%s\",%d", c.mod.isdnoff.auto_dial_num[CONF_PORT_AB1], BOOL_TO_INT(c.mod.isdnoff.auto_dial[CONF_PORT_AB1]));
				writeCommand(com);
				status++;
			}
			break;
		case 316:
			if (line == "OK") {
				com.sprintf("AT$IAD=2,\"%s\",%d", c.mod.isdnoff.auto_dial_num[CONF_PORT_AB2], BOOL_TO_INT(c.mod.isdnoff.auto_dial[CONF_PORT_AB2]));
				writeCommand(com);
				status++;
			}
			break;
		case 317:
			if (line == "OK") {
				status++;
				if (c.mod.isdnoff.reset_charge[CONF_PORT_ISDN]) {
					com.sprintf("AT$ICI=0,0");
					writeCommand(com);
					break;
				}
			}
		case 318:
			if (line == "OK") {
				status++;
				if (c.mod.isdnoff.reset_charge[CONF_PORT_AB1]) {
					com.sprintf("AT$ICI=1,0");
					writeCommand(com);
					break;
				}
			}
		case 319:
			if (line == "OK") {
				status++;
				if (c.mod.isdnoff.reset_charge[CONF_PORT_AB2]) {
					com.sprintf("AT$ICI=2,0");
					writeCommand(com);
					break;
				}
			}
		case 320:
			if (line == "OK") {
				com.sprintf("AT$IEL=1,%d,%d", BOOL_TO_INT(c.mod.isdnoff.exchange_hook[CONF_AB_PORT_1]), BOOL_TO_INT(c.mod.isdnoff.exchange_flash[CONF_AB_PORT_1]));
				writeCommand(com);
				status++;
			}
			break;
		case 321:
			if (line == "OK") {
				com.sprintf("AT$IEL=2,%d,%d", BOOL_TO_INT(c.mod.isdnoff.exchange_hook[CONF_AB_PORT_2]), BOOL_TO_INT(c.mod.isdnoff.exchange_flash[CONF_AB_PORT_2]));
				writeCommand(com);
				status++;
			}
			break;
		case 322:
			if (line == "OK") {
				com.sprintf("AT$IDP=%s", c.mod.isdnoff.d_chn_prot);
				writeCommand(com);
				if (c.sync_timer)
					status++;
				else
					status = 325;
			}
			break;
		case 323:
			if (line == "OK") {
				c.date_time = QDateTime::currentDateTime();
				com.sprintf("AT$JTIME=%d,%d,%d", c.date_time.time().hour(), c.date_time.time().minute(), c.date_time.time().second());
				writeCommand(com);
				status++;
			}
			break;
		case 324:
			if (line == "OK") {
				com.sprintf("AT$JDATE=%d,%d,%d", c.date_time.date().year(), c.date_time.date().month(), c.date_time.date().day());
				writeCommand(com);
				status++;
			}
			break;
		case 325:
			if (line == "OK") {
				reset();
				emit progress(27);
				emit done(action, true);
			}
			break;
		case 900:
			if (line == "OK") {
				emit progress(1);
				status++;
				uploadFile();
			}
			break;
		case 901:
			if (line == "OK") {
				reset();
				emit progress(cfg_file_size+1);
				emit done(action, true);
			}
			break;
		default:
			break;
	}
}


void MLOLoader::fetchXModemStatus(int pos)
{
	emit progress(pos);
}


void MLOLoader::fetchXModemDone(bool success)
{
	modem->timerStart(1000);	// Work-around for too fast OK response
}


void MLOLoader::fetchTimeout()
{
	if (action == LOADER_ACTION_FILE_READ && status == 902) {	// Work-around for too fast OK response
		modem->flush();
		fetchModemLineDown("OK");
		return;
	}
	
	if (action == LOADER_ACTION_FILE_WRITE && status == 901) {	// Work-around for too fast OK response
		modem->flush();
		fetchModemLineUp("OK");
		return;
	}

	reset();
	emit message(i18n("Modem response timeout."));
	emit done(action, false);
}


void MLOLoader::init()
{
	status = 0;
	model = MODEL_UNKNOWN;
	command = 0;
}


void MLOLoader::reset()
{
	usleep(100000);
	modem->close();

	init();
}


void MLOLoader::downloadFile()
{
	modem->timerStop();

	emit totalSteps(cfg_file_size+1);
	writeCommand("AT$JDNL=\"config.cfg\"", 3000);
	modem->receiveXModem(expandPath("config.cfg"), cfg_file_size, false);	// firmware problem (no crc allowed with isdn yet)!
}


void MLOLoader::uploadFile()
{
	QFile file;

	modem->timerStop();
	
	file.setName(expandPath("config.cfg"));
	if (!file.open(IO_ReadOnly))
		return;

	cfg_file_size = file.size();
	file.close();

	emit totalSteps(cfg_file_size+1);
	usleep(100000);
	modem->writeLine("AT$JDEL=\"config.cfg\"");
	usleep(250000);
	modem->flush();
	writeCommand("AT$JUPL=\"config.cfg\"", 3000);
	modem->sendXModem(expandPath("config.cfg"));
}


void MLOLoader::createFile()
{
	static const char *generic_file = "[COMMON]\r\nCountryConfigured=1\r\nTamActive=1\r\n";
	QFile file;

	file.setName(expandPath("config.cfg"));
	if (!file.open(IO_WriteOnly | IO_Truncate))
		return;

	file.writeBlock(generic_file, qstrlen(generic_file));
	file.close();
}


void MLOLoader::writeCommand(const QCString &com, int msec)
{
	command = com;
	usleep(10000);
	modem->writeLine(command, msec);
}
