/*
 * KMLOFax
 *
 * A utility to process facsimiles received with the ELSA
 * MicroLink(tm) Office or MicroLink(tm) ISDN Office modem.
 *
 * Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen.
 * PostScript(R) is a registered trademark of Adobe Systems Incorporated.
 */

#include <stdlib.h>
#include <string.h>

#include <qdatetime.h>

#include <kapp.h>
#include <klocale.h>

#include "global.h"
#include "senderaliases.h"
#include "filters.h"



MLOFilter::MLOFilter()
{
	init();
}


MLOFilter::MLOFilter(const QString& infile)
{
	setFile(infile);
}


MLOFilter::~MLOFilter()
{
}


void MLOFilter::setFile(const QString& infile)
{
	init();
	mlofile.setName(infile);
}


void MLOFilter::setTime(int year, int month, int day, int hour, int minute, int second)
{
	m_time.setDate(QDate(year, month, day));
	m_time.setTime(QTime(hour ,minute, second));
}


void MLOFilter::init()
{
	m_first = 0;
	m_last = 0;
	m_pages = 0;
	m_time = QDateTime::currentDateTime();
}




MLO2TiffFilter::MLO2TiffFilter(): MLOFilter()
{
}


MLO2TiffFilter::MLO2TiffFilter(const QString& infile): MLOFilter(infile)
{
}


MLO2TiffFilter::~MLO2TiffFilter()
{
}


bool MLO2TiffFilter::convertFile(const QString& outfile)
{
	int page;
	
	if (!mlofile.open(IO_ReadOnly)) {
		return false;
	}

	m_pages = mlofile.pages();
	
	if (m_first == 0) {
		m_first = 1;
		m_last = m_pages;
	}

	if ((m_first > m_last) || (m_first > m_pages)) {
		mlofile.close();
		return false;
	}

	if (m_last > m_pages)
		m_last = m_pages;
	
	m_pages = m_last - m_first + 1;

	tifffile.setName(outfile);
	
	if (!tifffile.open(IO_ReadWrite | IO_Truncate)) {
		mlofile.close();
		return false;
	}

	tifffile.setSender(mlofile.sender());
	tifffile.setTime(m_time);

	for (page = m_first; page <= m_last; page++)
		convertPage(page);

	tifffile.close();
	mlofile.close();

	return true;
}


bool MLO2TiffFilter::convertPage(int page)
{
	int h, i;
	short buffer[MAX_RLE_BUFF];

	if (!mlofile.gotoPage(page))
		return false;
	
	h = mlofile.pageHeight();

	tifffile.addPage(page, m_pages, mlofile.pageWidth(), h, mlofile.fineResolution());

	for (i=0; i < h; i++) {
		mlofile.readRleLine(buffer);
		tifffile.writeRleLine(buffer);
	}

	tifffile.finishPage();

	return true;
}



TiffFilter::TiffFilter()
{
	init();
}


TiffFilter::TiffFilter(const QString& infile)
{
	init();
	setFile(infile);
}


TiffFilter::~TiffFilter()
{
}


void TiffFilter::setFile(const QString& infile)
{
	tifffile.setName(infile);
}


bool TiffFilter::convertFile(FILE *stream)
{
	f = stream;

	return true;
}


void TiffFilter::init()
{
	m_first = 0;
	m_last = 0;
	m_pages = 0;
	f = 0;
}



Tiff2TiffFilter::Tiff2TiffFilter(): TiffFilter()
{
}


Tiff2TiffFilter::~Tiff2TiffFilter()
{
}


bool Tiff2TiffFilter::convertFile(const QString& outfile)
{
	int page;
	
	if (!tifffile.open(IO_ReadOnly)) {
		return false;
	}

	m_pages = tifffile.pages();
	
	if (m_first == 0) {
		m_first = 1;
		m_last = m_pages;
	}

	if ((m_first > m_last) || (m_first > m_pages)) {
		tifffile.close();
		return false;
	}

	if (m_last > m_pages)
		m_last = m_pages;
	
	m_pages = m_last - m_first + 1;

	tiffout.setName(outfile);

	if (!tiffout.open(IO_ReadWrite | IO_Truncate)) {
		tifffile.close();
		return false;
	}

	tiffout.setSender(tifffile.sender());
	tiffout.setTime(tifffile.time());

	for (page = m_first; page <= m_last; page++)
		convertPage(page);

	tiffout.close();
	tifffile.close();

	return true;
}


bool Tiff2TiffFilter::convertPage(int page)
{
	int h, i;
	short buffer[MAX_RLE_BUFF];

	if (!tifffile.gotoPage(page))
		return false;
	
	h = tifffile.pageHeight();

	tiffout.addPage(page, m_pages, tifffile.pageWidth(), h, tifffile.fineResolution());

	for (i=0; i < h; i++) {
		tifffile.readRleLine(buffer);
		tiffout.writeRleLine(buffer);
	}

	tiffout.finishPage();

	return true;
}



Tiff2PSFilter::Tiff2PSFilter(): TiffFilter()
{
	init();
}


Tiff2PSFilter::Tiff2PSFilter(const QString& infile): TiffFilter(infile)
{
	init();
}


Tiff2PSFilter::~Tiff2PSFilter()
{
}


void Tiff2PSFilter::setFormat(int format, bool landscp)
{
	/* A4, B4, B5, Letter, Legal, Executive */
	static int papers_x[] = { 595,  729, 516, 612,  612, 540 };
	static int papers_y[] = { 842, 1032, 729, 792, 1008, 720 };

	landscape = landscp;

	if (landscape) {
		paper_x = papers_y[format];
		paper_y = papers_x[format];
	}
	else {
		paper_x = papers_x[format];
		paper_y = papers_y[format];
	}
}


bool Tiff2PSFilter::convertFile(FILE *stream)
{
	int page, sub_page;
	int *size_y_buff;
	int maxlines;
	int ps_pages;
	QString s;

	f = stream;

	/* Size of viewport (in pixel, 1/72 dpi) */
	view_x = paper_x - (int)((double)(l_margin + r_margin) * 2.8346);
	view_y = paper_y - (int)((double)(t_margin + b_margin) * 2.8346);
	
	/* Bottom-left border (in pixel, 1/72 dpi) */
	trans_x = (int)((double)l_margin * 2.8346);
	trans_y = (int)((double)b_margin * 2.8346);
	
	/* Calculate real number of pages */
	if (!tifffile.open(IO_ReadOnly)) {
		return false;
	}

	m_pages = tifffile.pages();
	
	if (m_first == 0) {
		m_first = 1;
		m_last = m_pages;
	}

	if ((m_first > m_last) || (m_first > m_pages)) {
		tifffile.close();
		return false;
	}

	if (m_last > m_pages)
		m_last = m_pages;

	m_pages = 0;
	size_y_buff = (int *)malloc((m_last - m_first + 1) * sizeof(int));
	for (page = m_first; page <= m_last; page++) {
		tifffile.gotoPage(page);

		/* Horizontal size of image (in pixel, 204 dpi) */
		size_x = tifffile.pageWidth();
		/* Vertical size of image (in pixel, 98 or 196 lpi) */
		size_y = tifffile.pageUsedHeight();
		size_y_buff[page-m_first] = size_y;

		scale_x = size_x * 72 / 204;
		
		maxlines = view_y * (tifffile.fineResolution() ? 196 : 98) / 72;

		if (scale_x > view_x)
			maxlines = (maxlines * scale_x) / view_x;

		while (size_y > maxlines) {
			size_y -= maxlines;
			m_pages++;
		}
		m_pages++;
	}

	fprintf(f, "%%!PS-Adobe-3.0\n");
	if (ribbons)
		fprintf(f, "%%%%BoundingBox: %d 0 %d %d\n", trans_x, trans_x + view_x, paper_y);
	else
		fprintf(f, "%%%%BoundingBox: %d %d %d %d\n", trans_x, trans_y, trans_x + view_x, trans_y + view_y);
	fprintf(f, "%%%%Creator: KMLOFax Version " VERSION "\n");
	fprintf(f, "%%%%CreationDate: %s\n", tifffile.time().toString().latin1());
	fprintf(f, "%%%%DocumentData: Clean7Bit\n");
	if (level2)
		fprintf(f, "%%%%LanguageLevel: 2\n");
	fprintf(f, "%%%%Orientation: %s\n", landscape ? "Landscape" : "Portrait");
	fprintf(f, "%%%%Pages: %d\n", m_pages);
	fprintf(f, "%%%%PageOrder: Ascend\n");
	fprintf(f, "%%%%Title: ");
	s = SENDER_ALIAS(tifffile.sender());
	if (s.isEmpty())
		fprintf(f, i18n("Facsimile").latin1());
	else
		fprintf(f, i18n("Facsimile from %1").arg(s).latin1());
	fprintf(f, "\n%%%%DocumentFonts: Helvetica-Bold\n");
	fprintf(f, "%%%%EndComments\n");
	if (ribbons) {
		fprintf(f, "%%%%BeginProlog\n");
		fprintf(f, "/Ribbon {\n");
		fprintf(f, " gsave\n dup\n 0.7 setgray\n newpath\n 0 moveto\n");
		fprintf(f, " 0 %d rlineto\n", paper_y);
		fprintf(f, " 15 0 rlineto\n 0 -%d rlineto\n", paper_y);
		fprintf(f, " closepath\n fill\n 1 setgray\n /Helvetica-Bold findfont\n 12 scalefont\n setfont\n");
		fprintf(f, " 12 add\n -40 moveto\n 90 rotate\n");
		fprintf(f, " %d {40 0 rmoveto gsave (FAX) show grestore} repeat\n", (paper_y + 39) / 40);
		fprintf(f, " grestore\n");
		fprintf(f, "} bind def\n");
	}
	fprintf(f, "%%%%EndProlog\n");
	fprintf(f, "%%%%BeginSetup\n");
	if (level2)
		fprintf(f, "/DeviceGray setcolorspace\n");
	fprintf(f, "/#copies %d def\n", doc_copies);
	fprintf(f, "%%%%EndSetup\n");

	ps_pages = 0;
	for (page = m_first; page <= m_last; page++) {
		tifffile.gotoPage(page);

		size_x = tifffile.pageWidth();
		size_y = size_y_buff[page-m_first];

		scale_x = (size_x * 72) / 204;
		scale_y = (size_y * 72) / (tifffile.fineResolution() ? 196 : 98);
		maxlines = (view_y * (tifffile.fineResolution() ? 196 : 98)) / 72;

		if (scale_x > view_x) {
			scale_y = (scale_y * view_x) / scale_x;
			maxlines = (maxlines * scale_x) / view_x;
			scale_x = view_x;
		}

		sub_page = 0;
		while (size_y > maxlines) {
			if (!convertPage(page, ++sub_page, ++ps_pages, maxlines, view_y)) {
				tifffile.close();
				return false;
			}
			size_y -= maxlines;
			scale_y -= view_y;
		}
		if (!convertPage(page, ++sub_page, ++ps_pages, size_y, scale_y)) {
			tifffile.close();
			return false;
		}
	}

	free(size_y_buff);

	fprintf(f, "%%%%Trailer\n%%%%EOF\n");

	tifffile.close();

	return true;
}


void Tiff2PSFilter::init()
{
	setFormat(PAPER_A4, false);
	level2 = true;
	interpolate = false;
	ribbons = true;
	doc_copies = 1;
	l_margin = 0;
	r_margin = 0;
	t_margin = 0;
	b_margin = 0;

	resetImageData();
}


bool Tiff2PSFilter::convertPage(int tiff_page, int sub_page, int ps_page, int lines, int strech_y)
{
	static const char n_hex[16] = {
		'f', '7', 'b', '3', 'd', '5', '9', '1',
		'e', '6', 'a', '2', 'c', '4', '8', '0'
	};
	int bpl, i, j;
	uchar img_in_buff[MAX_IMG_BUFF], huf_in_buff[MAX_HUF_BUFF], c;
	char out_buff[66];

	bpl = size_x >> 3;

	if (fprintf(f, "%%%%Page: %d.%d %d\n", tiff_page, sub_page, ps_page) <= 0)
		return false;

	fprintf(f, "%%%%BeginPageSetup\n/pagelevel save def\n%%%%EndPageSetup\n");

	if (ribbons)
		fprintf(f, "40 Ribbon\n%d Ribbon\n", paper_x - 55);
	
	fprintf(f, "%d %d translate\n", trans_x, trans_y + view_y - strech_y);
	fprintf(f, "%d %d scale\n", scale_x, strech_y);

	if (level2) {
		fprintf(f, "{\n");
		fprintf(f, "/G3DICT <<\n");
		fprintf(f, " /Columns %d\n", size_x);
		fprintf(f, " /K 0\n");
		fprintf(f, " /Uncompressed false\n");
		fprintf(f, " /EndOfLine true\n");
		fprintf(f, " /EndOfBlock true\n");
		fprintf(f, " /BlackIs1 false\n");
		fprintf(f, ">> def\n");
		fprintf(f, "/Data1 currentfile /ASCII85Decode filter def\n");
		fprintf(f, "/Data2 Data1 G3DICT /CCITTFaxDecode filter def\n");
		fprintf(f, "<<\n");
		fprintf(f, " /ImageType 1\n");
		fprintf(f, " /Width %d\n", size_x);
		fprintf(f, " /Height %d\n", lines);
		fprintf(f, " /BitsPerComponent 1\n");
		fprintf(f, " /Decode [ 0 1 ]\n");
		fprintf(f, " /ImageMatrix [ %d 0 0 %d 0 %d ]\n", size_x, -lines, lines);
		fprintf(f, " /Interpolate %s\n", interpolate ? "true" : "false");
		fprintf(f, " /DataSource Data2\n");
		if (ribbons)
			fprintf(f, ">> imagemask\n");
		else
			fprintf(f, ">> image\n");
		fprintf(f, "Data1 flushfile\n");
		fprintf(f, "} exec\n");
	}
	else {
		fprintf(f, "/bpl %d string def\n", bpl);
		fprintf(f, "%d %d 1 [ %d 0 0 %d 0 %d ]\n", size_x, lines, size_x, -lines, lines);
		fprintf(f, "{currentfile bpl readhexstring pop}\nimage\n");	
	}
	
	if (level2) {
		writeBase85(0x00);
		writeBase85(0x01);

		while (lines--) {
			bpl = tifffile.readHuffLine(huf_in_buff);
			for (i=0; i < bpl; i++)
				writeBase85(huf_in_buff[i]);
		}

		writeBase85(0x00); /* Write End-Of-Block */
		writeBase85(0x01);
		writeBase85(0x00);
		writeBase85(0x10);
		writeBase85(0x01);
		writeBase85(0x00);
		writeBase85(0x10);
		writeBase85(0x01);

		flushBase85();
		flushImageData();
		resetImageData();
		
		fprintf(f, "~>\n");
	}
	else {
		while (lines--) {
			tifffile.readImgLine(img_in_buff, true);
			i = 0;
			for (j=0; j < bpl; j++) {
				c = img_in_buff[j];
				out_buff[i++] = n_hex[c & 0x0f];
				out_buff[i++] = n_hex[c >> 4];
				if (i == 64) {
					out_buff[i++] = '\n';
					out_buff[i] = 0;
					fprintf(f, out_buff);
					i = 0;
				}
			}
			if (i) {
				out_buff[i++] = '\n';
				out_buff[i] = 0;
				fprintf(f, out_buff);
			}
		}
	}

	fprintf(f, "pagelevel restore\nshowpage\n");

	return true;
}


void Tiff2PSFilter::writeBase85(uchar c)
{
	base85_buff[base85_ind] = c;

	if (base85_ind)
		base85_ind--;
	else
		flushBase85();
}


void Tiff2PSFilter::flushBase85()
{
	static ulong power85[5] = { 85*85*85*85, 85*85*85, 85*85, 85, 1 };
	ulong v, c;
	int i;

	if (base85_ind == 3)
		return;

	v = *((ulong *)base85_buff);

	if (v)
		for (i=0; i < 5; i++) {
			c = v / power85[i];
			writeImageData('!' + (char)c);
			v -= c * power85[i];
		}
	else
		writeImageData('z');

	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
}


void Tiff2PSFilter::writeImageData(char c)
{
	image_buff[image_ind] = c;

	if (++image_ind == 253)
		flushImageData();
}


void Tiff2PSFilter::flushImageData()
{
	if (!image_ind)
		return;

	image_buff[image_ind] = 0;
	
	if ((image_buff[0] == '%') && (image_buff[1] == '%'))
		fprintf(f, "%% %s\n", &image_buff[1]);
	else
		fprintf(f, "%s\n", image_buff);

	image_ind = 0;
}


void Tiff2PSFilter::resetImageData()
{
	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
	
	image_ind = 0;
}
