/***************************************************************************
                          cmapwidget.h
                      -------------------
    description          : Map Display Widget
    begin                : Wed Oct 20 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef CMAPWIDGET_H
#define CMAPWIDGET_H

#include <kapp.h>
#include <klocale.h>

#include <qscrollview.h>
#include <qpixmap.h>
#include <qlist.h>
#include <qpopupmenu.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qcursor.h>

#include "resource.h"
#include "kmuddoc.h"
#include "cmapelement.h"
#include "cmaptext.h"
#include "cmappath.h"
#include "cmaproom.h"
#include "cmaptoolbase.h"
#include "cmapzone.h"
#include "cmapdata.h"
#include "cmapundoelement.h"

class CMapToolBase;

struct selectedTyp
{
	CMapElement *element;
	int x;
	int y;
	int mouseX;
	int mouseY;
};

/**This class is used to display dynamic tool tips
  *for the maps. This enables the map widget to show
  *room labels as tool tips.
  *@author John-Paul Stanford
  */
class CMapToolTip : public QToolTip
{
public:
    CMapToolTip( QWidget * parent );

protected:
    void maybeTip( const QPoint & );
};

/**This is the widget used to display the map
  *@author John-Paul Stanford
  */
class CMapWidget : public QWidget
{
	Q_OBJECT
public:
	/** The map widget constructor */
	CMapWidget(CMapData *mapData,KmudDoc *document,QString mudTitle,CMudProfile* mudPro,QScrollView *view,QWidget *parent=0);
	/** The Map widget deconstructor */
	~CMapWidget();

	/** Used to get the tool tip for a given location passed in p */
	QString CMapWidget::getTip(const QPoint p);
	
	/** These methods are used to set the colour values */
	void setElementsColour(QColor loPath,QColor defPath,QColor hiPath,QColor loRoom,QColor defRoom,
	                       QColor hiRoom,QColor login,QColor sel,QColor defText,QColor specialCol,
	                       QColor edit, QColor current);

	/** Used to set the default path type */
	void setDefaultPathTwoWay(bool set);
	/** Used to return the default path type */
	bool getDefaultPathTwoWay(void);
	/** Used to create a new level of the map */
	void createLevel(direction dir,CMapZone *intoZone);	
	/** This checks the given size and resizes if needed */		
	void checkSize(int x,int y);
	/** Used to get the current level number */
	int getCurrentLevel(void);
	/** Used to export a map to a file */
	void exportMap(QString filename);
	/** Used to import a map a file */
	int importMap(QString filename);
	/** Used to import a map a file from file-version 1_3*/
	int importMap_1_3(QFile* file);
	/** Used to set  the current tool */
	void setTool(CMapToolBase *tool);
	/** Returns the visable state of the grid */
	bool GridVisable(void);
	/** Turns the grid on or off */
	void setGridVisable(bool visable);	
	/** Set overview visable */
	void setViewOverview(bool visiable);
	/** Returns the current room that the player is in */
	CMapRoom *getCurrentRoom(void);	
	/** This method is used to load a map */
	void loadMap(void);
	/** This method is used to save a map */
	void saveMap(void);
	/** This method is used to move the map by the given vector */
	void moveMap(signed int x,signed int y);
	/** This method is used to erase a map from memory */
	void eraseMap(void);
	/** the showLevel method is used to display a level on the map */
	void showLevel(int startx, int starty, signed int level,CMapZone *zone);
	/** move the player relative to the current position of the player
	  * This command has support for special paths, but can only move
	  * anlong exsiting special paths. In other words, it is unable to
	  * create special paths.
	  */
	void movePlayerBy(direction dir,bool create,QString specialCmd);
	/** move the player relative to the current position of the player
	  * This command has support for special paths, but can only move
	  * anlong exsiting special paths. In other words, it is unable to
	  * create special paths.
	  *
	  * This is simlar to the above method but uses differe params.
	  */
	void movePlayerBy(QString dirCmd,bool create);
	/** Covert a Direction command to a direction */
	direction cmdToDir(QString dirCmd);
	/** Check to see if a string is a valid move command */
	bool validMoveCmd(QString dirCmd);
	/** Create a new map */
	void createNewMap();
	/** Turn on and off the display of the higher map */
	void setViewHigherMap(bool visiable);
	/** Turn on and off the display of the lower map */
	void setViewLowerMap(bool visiable);
	/** Get the totals for the map */
	void getTotals(int *lvl,int *room, int *path, int *text,int *zone);
	/** Get the login room */
	CMapRoom *getLoginRoom(void);
	/** Used to store the max x,y location of the rooms */
	int xMax, yMax;
	/** This method is used to find a room at a given location in the current level of the map.
	  * Null is retured if the room can't be found otherwise a pointer is returned to the room.
	  */
	CMapRoom *findRoomAt(int x,int y,signed int level,CMapZone *zone);
	/** This method is used to find a elment at a given location. Null is retured if
	  * the element can't be found otherwise a pointer is returned to the element.
	  */	
	CMapElement *findElementAt(int x,int y,signed int level,CMapZone *zone);	
	/** Set if the speed walk will be aborted when it reaches a number of steps */
	void setSpeedWalkAbortActive(bool set);
	/** Return if the speed walk will be aborted when it reaches a number of steps */
	bool getSpeedWalkAbortActive(void);
	/** Set the number of steps before the speed walking will abort */
	void setSpeedWalkLimit(int limit);
	/** get the number of steps before the speed walking will abort */
	int getSpeedWalkLimit(void);
	/** Redraw the map */
	void redraw(void);
	QScrollView *getMapView(void);
	/** Used to find out if the overview is visible */
	bool getViewOverview(void);
	/** Used to change the text of the status bar message field */
	void changeStatusBar(QString msg);
	/** Used to find out if the ctrl key is pressed */
	bool getCtrlPressed(void);
	/** Used to unslected all the elements on the map */
	bool unselectAll(void);
	/** This method is used to create a new room on the map */
	CMapRoom *createRoom(int x,int y,signed int level,CMapZone *zone);
	/** This method is used to create a new zone on the map */
	CMapZone *createZone(int x,int y,signed int level,CMapZone *intoZone,bool doExsitsCheck);
	/** Used to get a pointer to the current room list */
	QList<CMapRoom> *getCurrentMap(void);
	/** Used to calulate and set the cords of a path */
	void setPathCords(CMapPath *path);
	void delElement(CMapElement *element);
	/** Used to create a text element */
	CMapText *createText(QString str,int x,int y,QFont font,QColor col,int level,CMapZone *zone,bool doSizeCheck);
	/** Return a list of the elements in current level */
	QList<CMapElement> *getElementList(void);
	/** Used to make a path two way */
	CMapPath *makePathTwoWay(CMapPath *path);
	/** Used to make a path one way */
	void makePathOneWay(CMapPath *path);
	/** Delete a level in the zone */
	void deleteLevel(int lvlNum,CMapZone *zone,bool recreate);
	/** Get the current zone */
	CMapZone *getCurrentZone(void);
	/** Export a map to a KConfig file */
 	void exportKCMap(QString filename);
	/** Import a map from a KConfig file */
	int importKCMap(QString filename);

	void setCurrentPos(CMapRoom* newPos);
	/** Used to speedwalk a player to a given room */
	void walkPlayerTo(CMapRoom *room1);
	/** Used to display and set the room properties */
	void roomProperties(CMapRoom *room);
	/** Used to set the zone properties */
	void zoneProperties(CMapZone *zone1);

public slots:
	/** switch argument for slot selection by menu or toolbar ID */
	void commandCallback(int id_);
	/** switch argument for Statusbar help entries on slot selection */
	void statusCallback(int id_);
	/** change the status message of the whole statusbar temporary */
	void slotStatusHelpMsg(const char *text);
	/** This slot is called when the copy menu option is selected */
	void slotCopy(void);
	/** This slot is called when the paste menu option is selected */
	void slotPaste(void);
	/** This slot is called when the cut menu option is selected */
	void slotCut(void);
	/** This slot is called when the delete menu option is selected */
	void slotDelete(void);
	/** This slot is used to select all elements */
	void slotSelectAll(void);
	/** This slot is used to deselect all elements */
	void slotDeselectAll(void);
	/** This slot is used to invert the elements selected */
	void slotInvertSelection(void);
	/** Used to view the next level up */
	void slotLevelUp(void);
	/** Used to view the next level down */
	void slotLevelDown(void);
	/** Edit the bends of a path */
	void slotPathEditBends(void);
	/** Used to display the parent zone of the current zone */
	void slotZoneUp(void);
	
protected slots:
	/** Used to recersivly move the play along a speedwalk path */
	void slotWalkPlayerAlongPath(void);
	/** Used to change the properties of the text elements */
	void slotTextProperties(void);
	/** Used to change a path to one way */	
	void slotPathSetOneWay(void);
	/** Used to change a path to a twoway path */
	void slotPathSetTwoWay(void);
	/** Used display the path properties */
	void slotPathProperties(void);
	/** Move the player to the selected position on the map */
	void slotSetCurrentPos(void);
	/** Called when the user selects delete from the room menu */
	void slotDelElement(void);
	/** This is used to display the contex menu for map elements */
	void slotContexMenu(QMouseEvent *e);
	/** Used to speed walk a player */
	void slotMovePlayerTo(void);
	/** Used to set the login position for a character */
	void slotSetLogin(void);
	/** This slot is used to set the properties of a room */
	void slotRoomProperties(void);
	/** Add a bend to a path */
	void slotPathAddBend(void);
	/** Add a bend to a path */	
	void slotPathDelBend(void);
	/** Used to display a zones contents */
	void slotZoneOpen(void);
	/** Used to set the properties of a zone */
	void slotZoneProperties(void);
	/** Used to add a room to the speed walk list */
	void slotRoomAddToSpeedwalk(void);
	
protected:
	CMapZone *pasteZone(CMapZone *intoZone,CMapZone *clipZone,int inc);
	void copyZone(CMapZone *parent,CMapZone *orgZone);
	/** A pointer to the mud profile */
	CMudProfile* mudProfile;
 	/** true when the map is being moved by mouse */
 	bool bMouseDrag;
 	/** y position of last mouse drag event */
 	int nMouseDragPosY;
 	/** x position of last mouse drag event */
 	int nMouseDragPosX;
 	/** The Cursor used when move draging the map */
 	QCursor* mouseDragCursor;
	/** draw the map widget */
	void paintEvent(QPaintEvent *pe);
	/** The mouse release event */	
	void mouseReleaseEvent(QMouseEvent *e);
	/** The mouse press event */
	void mousePressEvent(QMouseEvent *e);
	/** Called when the mouse is being moved */
	void mouseMoveEvent(QMouseEvent *e);
	/** Called when a key is pressed */
	void keyPressEvent(QKeyEvent *e);
	/** Called when a key is released */
	void keyReleaseEvent(QKeyEvent *e);		
	
private:
	CMapZone *copiedZone;
	CMapData *mapperData;
	/** A flag to tell the mapper if it needs to redraw the contents of the map */
	bool bNeedsCreate;
	/** Pointer to the last drawn map */
	QPixmap *buffer;
	/** Pointer to last drawn overvire of the map */
	QPixmap *overviewBuffer;
	/** number of moves before speed walk is aborted  */
	int speedWalkAbortLimit;
	/** Tells the mapper if speed walk move limit is active */
	bool speedWalkAbortActive;
	QCursor *deleteCursor;
	QCursor *pathStartCursor;
	QCursor *pathEndCursor;
	CMapRoom *pathStartRoom;
	int pathToolMode;
	/** Used to tell the mapper to create two way paths by default */
	bool defaultTwoWayPath;	
	/** Used to tell the mapper if the overview pane needs to be displayed */	
	bool bOverview;
	/** A pointer to the scroll view of the map */
	QScrollView *mapView;
	/** This is used to tell if movePlayerBy has to wait for the previous move to finish */
	bool moveEnabled;
	/** The dynamic tool tip class */
	CMapToolTip *tip;
	/** The kmud document */
	KmudDoc *doc;
	/** The filename of the map */
	QString mapName;

private:
	/** Create a zoomed version of the map */
	void createZoomedPixmap(QPixmap *newPixmap, QPixmap pm,int w,int h);
	/** Used to read a string from a file */
	QString readStr(QFile *f);	
	/** Used to write a string to a file */
	void writeStr(QFile *f,QString str);
	/** Used to read a int from a file */
	int readInt(QFile *f);
	/** Used to write a int to a file */
	void writeInt(QFile *f,int i);
	/** Delete a text element */
	void deleteText(CMapText *text,int level,CMapZone *fromZone);
	/** This method is used to convert a direction into a offset */
	void directionToCord(direction dir, int distance,signed int *x,signed int *y);
	/** Draw the grid if it's visable */
	void drawGrid(QPainter *p);
	/** This method is used to delete a room on the map */
	void deleteRoom(CMapRoom *room,int level,CMapZone *fromZone);
	/** This method is used to delete a zone on the map */
	void deleteZone(CMapZone *zone,int level,CMapZone *fromZone);
	/** Delete a path */
	void deletePath(CMapPath *path);
	/** This method is used to create a path between to rooms */
	CMapPath *createPath (CMapRoom *srcRoom,CMapRoom *destRoom,direction srcDir,direction destDir);
	/** This method is used to create a path between to rooms */
	CMapPath *createPath(int srcX,int srcY,signed int srcLevel,CMapZone *srcZone,direction srcDir,int destX,int destY,signed int destLevel,CMapZone *destZone, direction destDir);
	/** This points to the room that the character is standing in */
	CMapRoom *currentRoom;
	/** This is a pointer to the room that is the login room */
	CMapRoom *loginRoom;
	/** This is a list of all the rooms on the current map */
	QList<CMapRoom> *currentMap;
	/** This is a pointer ot the Current level list */
	QList<CMapLevelList> *currentLevelList;
	/** This is a pointer to the current zone */
	CMapZone *currentZone;
	/** Used to store the current level number of the map */
	int currentLevelNum;
	/** The spaceing of the gird on the Y axis */
	int YGridSpace;
	/** The spaceing of the gird on the X axis */
	int XGridSpace;
	/** Used to store the on/off state of the grid */
	int hasGrid;
	/** The context menu for the rooms */
	QPopupMenu *room_menu;
	/** The context menu for the zones */	
	QPopupMenu *zone_menu;	
	/** The context menu for the paths */
	QPopupMenu *path_menu;
	/** The context menu for the text elements */
	QPopupMenu *text_menu;
	/** Used to regiester when the ctrl key is being held down */
	bool bCtrlPressed;
	/* This is a list of pointers pointing to all the elements on the map
	 * can be used to iterate through all the elements
	 */
	QList<CMapElement> elementList;
	/** Same as the elementList but used to paint the level above the current one */
	QList<CMapElement> elementListUpper;
	/** Same as the elementList but used to paint the level below the current one */
	QList<CMapElement> elementListLower;
	/* The element clipboard. This is used in cut/copy/paste operations */
	QList<CMapElement> elementClipboard;

	struct selectedTyp selected;

	/** The current tool that is being used */
	CMapToolBase *currentTool;
	/** Used to flag if the lower map is to be displayed */
	bool bViewLowerMap;
	/** Used to flag if the upper map is to be displayed */
	bool bViewHigherMap;

		
signals:
	/*** will be emmited when the speedwalk list needs changing */
	void updateSpeedwalkList(CMapRoom *);
	/** will be emmited when the mapper wants to move the player */
	void movePlayer(QString dirCmd);
	/** Used to inform of a change in level */
	void levelChange(signed int level);
	/** Used to inform when the help status bar message needs to be changed */
	void statusMsg(const char *text);
	/** Used to inform when the status bar message needs to be changed */
	void statusHelpMsg(const char *text);
	/** Used to inform when the zone changes */
	void zoneChanged(QString name);
	/** emitted when elements are created by the mapper itself */
	void elementsAutoCreated(CMapUndoElement* element);
	/** Used to inform tabar is zone up should be enabled */
	void enableZoneUp(bool);
	/** Used to inform tabar is level down should be enabled */
	void enableLevelDown(bool);
	/** Used to inform tabar is level up should be enabled */
	void enableLevelUp(bool);
  /** signal emitted when mouse button is pressed */
  void mousePressed();
};

#endif
