/*****************************************g**********************************
                          cmapwindow.cpp
                      -------------------
    description          : Auto Mapper Window Class
    begin                : Fri Oct 22 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmapwindow.h"

#include <qmessagebox.h>
#include <qlayout.h>

#include <kfiledialog.h>
#include <kwm.h>

#include "cmaptooldel.h"
#include "cmaptoolroom.h"
#include "cmaptoolpath.h"
#include "cmaptoolselect.h"
#include "cmaptooltext.h"
#include "cmaptoolzone.h"

#include "dialogs/dlgmapzoneproperties.h"

/** Used to add a map to the map view */
void CMapView ::addMap(CMapWidget *mapWidget)
{
	map = mapWidget;
	addChild(map);
}

/** the resize event which has been over riddent to risze the map correctly */
void CMapView::resizeEvent(QResizeEvent *e)
{
	QScrollView::resizeEvent(e);
	map->checkSize(map->xMax,map->yMax);
}


CMapWindow::CMapWindow(QString mudTitle,QString mudLogin,KmudDoc *document, KmudView *mView)
{
	setCaption(i18n("Kmud Mapper"));

	mapperData = new CMapData();
	doc = document;
	mud_Title = mudTitle;
	mud_Login = mudLogin;
	mudView = mView;
	accelerators = new KAccel(this);

	///////////////////////////////////////////////////////////////////
	// read the config file options
	// read them _before_ the inits so we init the right
	readOptions();

	this->resize(mapperWindowWidth,mapperWindowHeight);

	initMenubar();
	initStatusBar();
	initToolBars();
  initAccelerators();
		
	if (bViewSpeedwalk)
		split = new KPanner(this,"panner",KPanner::O_VERTICAL, speedwalk_width );
	else
		split = new KPanner(this,"panner",KPanner::O_VERTICAL, 0 );
		

	connect(split,SIGNAL(positionChanged()),SLOT(slotSplitMove()));

	mapView = new CMapView(split->child1(),"mapview");
	QGridLayout *gl = new QGridLayout( split->child1(), 1, 1 );
	gl->addWidget( mapView, 0, 0 );
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);
	
		map = new CMapWidget(mapperData,doc,mud_Title,mudPro,mapView,mapView->viewport());
	}
	else
	{
		map = new CMapWidget(mapperData,doc,mud_Title,NULL,mapView,mapView->viewport());	
	}

	speedwalk = new DlgMapSpeedwalk (map,mapperData,split->child0(),"speedwalk");

	gl = new QGridLayout(split->child0(), 1, 1 );
	gl->addWidget( speedwalk, 0, 0 );
	
	connect(map,SIGNAL(updateSpeedwalkList(CMapRoom *)),speedwalk,SLOT(updateSpeedwalkList(CMapRoom *)));
	connect(map,SIGNAL(enableLevelUp(bool)),SLOT(slotEnableLevelUp(bool)));
	connect(map,SIGNAL(enableLevelDown(bool)),SLOT(slotEnableLevelDown(bool)));
	connect(map,SIGNAL(enableZoneUp(bool)),SLOT(slotEnableZoneUp(bool)));
	
	mapView->addMap(map);				
	mapView->setVScrollBarMode(QScrollView::AlwaysOn);
	mapView->setHScrollBarMode(QScrollView::AlwaysOn);	
	map->setUpdatesEnabled(false);
	map->setGridVisable(bGridShow);
	map->setViewHigherMap(bViewHigherMap);
	map->setViewLowerMap(bViewLowerMap);
	map->setViewOverview(bViewOverview);
	map->setElementsColour(mapperData->lowerPathColour,mapperData->defaultPathColour,mapperData->higherPathColour,
                         mapperData->lowerRoomColour,mapperData->defaultRoomColour,mapperData->higherRoomColour,mapperData->loginColour,mapperData->selectedColour,
                         mapperData->defaultTextColour,mapperData->specialColour,mapperData->editColour,mapperData->currentColour);
 	map->setDefaultPathTwoWay(bTwoWay);
	initTools();
	map->loadMap();
	map->checkSize(map->xMax,map->yMax);
	map->setSpeedWalkLimit(speedWalkLimit);
	map->setSpeedWalkAbortActive(bLimitSpeedWalk);
	map->setUpdatesEnabled(true);
	
	CMapRoom *room=map->getCurrentRoom();									

	mapView->center(room->getLowX()+HALF_ROOM_SIZE,room->getLowY()+HALF_ROOM_SIZE);
	


	if (!bViewStatusbar)
		enableStatusBar(KStatusBar::Hide);
	
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		if (mudPro->getFollowMode())
		{
		        options_menu->setItemChecked(ID_MAPPER_OPTIONS_FOLLOW,mudPro->getFollowMode());
						modesToolbar->setButton(ID_MAPPER_OPTIONS_FOLLOW,mudPro->getFollowMode());
		}
	
		if (mudPro->getCreateMode())
		{
		        options_menu->setItemChecked(ID_MAPPER_OPTIONS_CREATE,mudPro->getCreateMode());;
						modesToolbar->setButton(ID_MAPPER_OPTIONS_CREATE,mudPro->getCreateMode());
		}
	}
	
	toolsToolbar->setButton(ID_MAPPER_TOGGLE_GRID,bGridShow);

	setView(split,false);
	show();
	
	mapView->center(room->getLowX(),room->getLowY());

	connect(map,SIGNAL(movePlayer(QString)),SLOT(slotMovePlayer(QString)));
	connect(map,SIGNAL(levelChange(signed int)),SLOT(slotLevelChanged(signed int)));
	connect(map,SIGNAL(statusMsg(const char *)),SLOT(slotStatusMsg(const char *)));
	connect(map,SIGNAL(statusHelpMsg(const char *)),SLOT(slotStatusHelpMsg(const char *)));
	connect(map,SIGNAL(zoneChanged(QString)),SLOT(slotZoneChanged(QString)));

	if (bStayOnTop)	
		KWM::setDecoration(winId(), KWM::normalDecoration | KWM::staysOnTop);	
		
	mapUndo = new CMapCreationUndo(map, 10);
  connect(map, SIGNAL(mousePressed()), this, SLOT(slotClearUndo()));
  connect(map, SIGNAL(elementsAutoCreated(CMapUndoElement*)), this, SLOT(slotUndoPossible()));
}

CMapWindow::~CMapWindow()
{
	delete file_menu;
	delete view_menu;
	delete options_menu;
	delete tools_menu;				
	delete level_menu;
	delete zone_menu;

	delete mainToolbar;
	delete toolsToolbar;
	delete modesToolbar;
		
	delete map;
	delete mapView;	
	delete split;	
	
	delete mapUndo;
	delete accelerators;
}

void CMapWindow::readOptions(void)
{                           	
	///////////////////////////////////////////////////////////////////
	// read the config file entries
	KConfig *config = kapp->getConfig();
	config->setGroup("MAPPER APPEARANCE");
	mapperWindowWidth = config->readNumEntry("WindowWidth",600);
	mapperWindowHeight = config->readNumEntry("WindowHeight",400);
	bViewSpeedwalk = config->readBoolEntry("ShowSpeedwalk",true);
	speedwalk_width = config->readNumEntry("SpeedwalkWidth",30);
	bViewStatusbar = config->readBoolEntry("ShowStatusbar", true);
	bViewToolsToolbar = config->readBoolEntry("ShowToolsToolbar",true);
	bViewModesToolbar = config->readBoolEntry("ShowModesToolbar",true);
	bViewMainToolbar = config->readBoolEntry("ShowMainToolbar",true);
	bViewHigherMap = config->readBoolEntry("ShowHigherMap",true);
	bViewLowerMap = config->readBoolEntry("ShowLowerMap",false);
	bViewOverview = config->readBoolEntry("ShowOverview",false);
	bGridShow = config->readBoolEntry("GridShow",true);
	bTwoWay = config->readBoolEntry("DefaltPathTwoWay",true);
	bStayOnTop = config->readBoolEntry("StayOnTop",false);
	
	tools_toolbar_pos = (KToolBar::BarPosition)config->readNumEntry("Mapper_ToolBar_Tools_Pos", KToolBar::Left);
	modes_toolbar_pos = (KToolBar::BarPosition)config->readNumEntry("Mapper_ToolBar_Modes_Pos", KToolBar::Top);
	main_toolbar_pos = (KToolBar::BarPosition)config->readNumEntry("Mapper_ToolBar_Main_Pos", KToolBar::Top);
	
	config->setGroup("GENERAL");	
	bLimitSpeedWalk = config->readBoolEntry("SpeedWalkLimitActive",true);
	speedWalkLimit = config->readNumEntry("SpeedWalkLimit",1000);
	
	mapperData->readData();
}

void CMapWindow::saveOptions(void)
{
	if (bViewSpeedwalk)
		speedwalk_width = split->getSeparator();
		
	///////////////////////////////////////////////////////////////////
	// save the global map options to the config file.
	KConfig *config = kapp->getConfig();
	
	config->setGroup("MAPPER APPEARANCE");
	config->writeEntry("WindowWidth",this->width());
	config->writeEntry("WindowHeight",this->height());
	config->writeEntry("ShowStatusbar",statusBar()->isVisible());
	config->writeEntry("ShowToolsToolbar",bViewToolsToolbar);
	config->writeEntry("ShowModesToolbar",bViewModesToolbar);		
	config->writeEntry("ShowSpeedwalk",bViewSpeedwalk);
	config->writeEntry("SpeedwalkWidth",speedwalk_width);
	
	config->writeEntry("ShowMainToolbar",bViewMainToolbar);	
	config->writeEntry("ShowHigherMap",bViewHigherMap);
	config->writeEntry("ShowLowerMap",bViewLowerMap);
	config->writeEntry("GridShow",bGridShow);
	config->writeEntry("Mapper_ToolBar_Tools_Pos",(int)toolsToolbar->barPos());
	config->writeEntry("Mapper_ToolBar_Modes_Pos",(int)modesToolbar->barPos());
	config->writeEntry("Mapper_ToolBar_Main_Pos",(int)mainToolbar->barPos());
	config->writeEntry("ShowOverview",	bViewOverview);
	config->writeEntry("DefaltPathTwoWay", map->getDefaultPathTwoWay());
	config->writeEntry("StayOnTop",bStayOnTop);

	config->setGroup("GENERAL");	
	config->writeEntry("SpeedWalkLimitActive", map->getSpeedWalkAbortActive());
	config->writeEntry("SpeedWalkLimit",map->getSpeedWalkLimit());
	

	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		doc->getMudProfile(entry->mudfile)->writeData();
	}
	
	mapperData->writeData();
}

void CMapWindow::initMenubar(void)
{
	///////////////////////////////////////////////////////////////////
	// menuBar entry file_menu
	file_menu = new QPopupMenu();
	file_menu->insertItem(Icon("filenew.xpm"),i18n("&New Map"), ID_MAPPER_FILE_NEW );
	file_menu->insertSeparator();
	file_menu->insertItem(Icon("fileopen.xpm"),i18n("&Import Map..."), ID_MAPPER_FILE_IMPORT );
	QPopupMenu* export_menu = new QPopupMenu();
	export_menu->insertItem(i18n("Export to &native binary format..."), ID_MAPPER_FILE_EXPORT );
	export_menu->insertItem(i18n("Export to &config style format..."), ID_MAPPER_FILE_EXPORT_KCMFILE );
	file_menu->insertItem(Icon("filefloppy.xpm"), i18n("&Export Map"), export_menu );
	file_menu->insertSeparator();
	file_menu->insertItem(i18n("In&formation..."), ID_MAPPER_FILE_INFO );
	file_menu->insertSeparator();	
	file_menu->insertItem(i18n("C&lose Window"), ID_MAPPER_FILE_CLOSE);

	// file_menu key accelerators
	file_menu->setAccel(CTRL+Key_N, ID_MAPPER_FILE_NEW);
	file_menu->setAccel(CTRL+Key_W, ID_MAPPER_FILE_CLOSE);

	///////////////////////////////////////////////////////////////////
	// menuBar entry file_menu
	edit_menu = new QPopupMenu();
	edit_menu->insertItem(Icon("editcut.xpm"),i18n("C&ut"),ID_MAPPER_EDIT_CUT);
	edit_menu->insertItem(Icon("editcopy.xpm"),i18n("&Copy"),ID_MAPPER_EDIT_COPY);
	edit_menu->insertItem(Icon("editpaste.xpm"),i18n("&Paste"),ID_MAPPER_EDIT_PASTE);
	edit_menu->insertItem(i18n("&Delete"),ID_MAPPER_EDIT_DELETE);
	edit_menu->insertSeparator();
	edit_menu->insertItem(i18n("&Select All"),ID_MAPPER_EDIT_SELECTALL);
	edit_menu->insertItem(i18n("D&eselect All"),ID_MAPPER_EDIT_DESELECTALL);
	edit_menu->insertItem(i18n("&Invert Selection"),ID_MAPPER_EDIT_INVERTSEL);
	edit_menu->insertSeparator();
  edit_menu->insertItem(i18n("Undo Autocreation"), ID_MAPPER_EDIT_UNDO_AUTO);
  edit_menu->setItemEnabled(ID_MAPPER_EDIT_UNDO_AUTO, false);
	
	// edit_menu key accelerators
	edit_menu->setAccel(CTRL+Key_X,ID_MAPPER_EDIT_CUT);
	edit_menu->setAccel(CTRL+Key_C,ID_MAPPER_EDIT_COPY);
	edit_menu->setAccel(CTRL+Key_V,ID_MAPPER_EDIT_PASTE);
	edit_menu->setAccel(Key_Delete,ID_MAPPER_EDIT_DELETE);
	edit_menu->setAccel(CTRL+Key_A,ID_MAPPER_EDIT_SELECTALL);	

	///////////////////////////////////////////////////////////////////
	// menuBar entry view_menu
	view_menu = new QPopupMenu();
	view_menu->setCheckable(true);
	view_menu->insertItem(i18n("&Always on top"), ID_MAPPER_VIEW_ALWAYS_ON_TOP);
	view_menu->insertSeparator();
	view_menu->insertItem(i18n("Show &Lower Map"), ID_MAPPER_VIEW_LOWER );
	view_menu->insertItem(i18n("Show &Higher Map"), ID_MAPPER_VIEW_HIGHER );
	view_menu->insertItem(Icon("kmud_grid.xpm"), i18n("Show &Grid"), ID_MAPPER_TOGGLE_GRID );	
	view_menu->insertSeparator();
	view_menu->insertItem(i18n("&Edit toolbar"), ID_MAPPER_VIEW_MAIN );
	view_menu->insertItem(i18n("&Tools toolbar"), ID_MAPPER_VIEW_TOOLS );
	view_menu->insertItem(i18n("&Modes toolbar"), ID_MAPPER_VIEW_MODES );
	view_menu->insertSeparator();
	view_menu->insertItem(i18n("Show &Overview"), ID_MAPPER_VIEW_OVERVIEW );
	view_menu->insertItem(i18n("Show S&peedwalk List"), ID_MAPPER_VIEW_SPEEDWALK );
	view_menu->insertSeparator();
	view_menu->insertItem(i18n("&Statusbar"), ID_MAPPER_VIEW_STATUSBAR );

	view_menu->setItemChecked(ID_MAPPER_VIEW_ALWAYS_ON_TOP, bStayOnTop);	
	view_menu->setItemChecked(ID_MAPPER_VIEW_STATUSBAR, bViewStatusbar);
	view_menu->setItemChecked(ID_MAPPER_VIEW_MODES, bViewModesToolbar);
	view_menu->setItemChecked(ID_MAPPER_VIEW_TOOLS, bViewToolsToolbar);
	view_menu->setItemChecked(ID_MAPPER_VIEW_MAIN, bViewMainToolbar);
	view_menu->setItemChecked(ID_MAPPER_TOGGLE_GRID, bGridShow);
	view_menu->setItemChecked(ID_MAPPER_VIEW_HIGHER, bViewHigherMap);
	view_menu->setItemChecked(ID_MAPPER_VIEW_LOWER, bViewLowerMap);
	view_menu->setItemChecked(ID_MAPPER_VIEW_OVERVIEW, bViewOverview);	
	view_menu->setItemChecked(ID_MAPPER_VIEW_SPEEDWALK,bViewSpeedwalk);
	
	view_menu->setAccel(CTRL+Key_O, ID_MAPPER_VIEW_OVERVIEW);	
	view_menu->setAccel(CTRL+Key_G, ID_MAPPER_TOGGLE_GRID);		
	view_menu->setAccel(CTRL+Key_L, ID_MAPPER_VIEW_LOWER);			
	view_menu->setAccel(CTRL+Key_H, ID_MAPPER_VIEW_HIGHER);			
  view_menu->setAccel(CTRL+Key_S, ID_MAPPER_VIEW_SPEEDWALK);
	
	///////////////////////////////////////////////////////////////////
	// menuBar entry tools_menu
	tools_menu = new QPopupMenu();
	tools_menu->setCheckable(true);
	
	level_menu = new QPopupMenu();	
	zone_menu = new QPopupMenu();		

	///////////////////////////////////////////////////////////////////
	// menuBar entry options_menu
	options_menu = new QPopupMenu();
	options_menu->setCheckable(true);
	options_menu->insertItem(Icon("kmud_follow.xpm"),i18n("&Follow Player"),ID_MAPPER_OPTIONS_FOLLOW);
	options_menu->insertItem(Icon("kmud_create.xpm"),i18n("&Create New Rooms"),ID_MAPPER_OPTIONS_CREATE);
	options_menu->insertSeparator();
	options_menu->insertItem(i18n("&Preferences..."),ID_MAPPER_OPTIONS_PREFS);

	///////////////////////////////////////////////////////////////////
	// MENUBAR CONFIGURATION
	// set menuBar() the current menuBar and the position due to config file
	KMenuBar *menu = new KMenuBar(this);
	menu->insertItem(i18n("&File"), file_menu,-1,-1);
	menu->insertItem(i18n("&Edit"), edit_menu);
	menu->insertItem(i18n("&View"), view_menu);
	menu->insertItem(i18n("&Tools"), tools_menu);
	menu->insertItem(i18n("&Options"), options_menu);
	setMenu(menu);

	CONNECT_CMD(file_menu);	
	CONNECT_CMD(export_menu);
	CONNECT_CMD(view_menu);
	CONNECT_CMD(options_menu);
	CONNECT_CMD(edit_menu);
	CONNECT_CMD(level_menu);
	CONNECT_CMD(zone_menu);		
}

void CMapWindow::initStatusBar(void)
{
	statusBar()->setInsertOrder(KStatusBar::RightToLeft);
	statusBar()->insertItem(i18n("Level : 1"), ID_MAPPER_STATUS_LEVEL);
	statusBar()->insertItem(i18n("Zone : Unamed Zone"), ID_MAPPER_STATUS_ZONE);		
	statusBar()->insertItem(i18n("Ready."), ID_MAPPER_STATUS_MSG );

}

void CMapWindow::initToolBars(void)
{


	mainToolbar = new KToolBar(this);
	addToolBar(mainToolbar);	

	mainToolbar->insertButton(Icon("filenew.xpm"), ID_MAPPER_FILE_NEW, true, i18n("New Map"));
	mainToolbar->insertSeparator();
	mainToolbar->insertButton(Icon("editcut.xpm"), ID_MAPPER_EDIT_CUT, true, i18n("Cut"));
	mainToolbar->insertButton(Icon("editcopy.xpm"), ID_MAPPER_EDIT_COPY, true, i18n("Copy"));
	mainToolbar->insertButton(Icon("editpaste.xpm"), ID_MAPPER_EDIT_PASTE, true, i18n("Paste"));
	mainToolbar->insertSeparator();

	mainToolbar->setFullWidth(false);

	CONNECT_TOOLBAR(mainToolbar);

	if (bViewMainToolbar)
	{
		enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MAIN);
	}
	else
	{
		enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MAIN);
	}

 	mainToolbar->setBarPos(main_toolbar_pos);

	toolsToolbar = new KToolBar(this);
	addToolBar(toolsToolbar);

	toolsToolbar->setFullWidth(false);

	if (bViewToolsToolbar)
	{

		enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_TOOLS);
	}
	else
	{
		enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_TOOLS);
	}

	toolsToolbar->setBarPos(tools_toolbar_pos);

	modesToolbar = new KToolBar(this);
	addToolBar(modesToolbar);

	modesToolbar->insertButton(Icon("kmud_follow.xpm"), ID_MAPPER_OPTIONS_FOLLOW, true, i18n("Follow Mode"));
	modesToolbar->insertButton(Icon("kmud_create.xpm"), ID_MAPPER_OPTIONS_CREATE, true, i18n("Create Mode"));
	modesToolbar->setToggle(ID_MAPPER_OPTIONS_FOLLOW,true);
	modesToolbar->setToggle(ID_MAPPER_OPTIONS_CREATE,true);

	modesToolbar->setFullWidth(false);

	CONNECT_TOOLBAR(modesToolbar);

	if (bViewModesToolbar)
		enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MODES);
	else
		enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MODES);
	

}

void CMapWindow::initTools(void)
{
	toolList.setAutoDelete(true);

	toolList.append(new CMapToolSelect(mapperData,map,0,"Select",Icon("kmud_pointer.xpm"),toolsToolbar,tools_menu));
	toolList.append(new CMapToolRoom(mapperData,map,1,"Create Room",Icon("kmud_room.xpm"),toolsToolbar,tools_menu));	
	toolList.append(new CMapToolPath(mapperData,map,2,"Create Path",Icon("kmud_path.xpm"),toolsToolbar,tools_menu));	
	toolList.append(new CMapToolText(mapperData,map,3,"Create Text",Icon("kmud_text.xpm"),toolsToolbar,tools_menu));			
	toolList.append(new CMapToolZone(mapperData,map,4,"Create Zone",Icon("kmud_zone.xpm"),toolsToolbar,tools_menu));			
	toolList.append(new CMapToolDel(mapperData,map,5,"Erase",Icon("kmud_eraser.xpm"),toolsToolbar,tools_menu));		

	toolsToolbar->insertSeparator();
	toolsToolbar->insertButton(Icon("kmud_grid.xpm"), ID_MAPPER_TOGGLE_GRID, true, i18n("Hide/Show Grid"));
	toolsToolbar->setToggle(ID_MAPPER_TOGGLE_GRID,true);
	toolsToolbar->insertSeparator();
	toolsToolbar->insertButton(Icon("kmud_zoneup.xpm"), ID_MAPPER_TOOL_ZONE_UP, true, i18n("Display Parent Zone"));
	toolsToolbar->insertButton(Icon("kmud_lvlup.xpm"), ID_MAPPER_LEVEL_UP, true, i18n("Level Up"));
	toolsToolbar->insertButton(Icon("kmud_lvldown.xpm"), ID_MAPPER_LEVEL_DOWN, true, i18n("Level Down"));

	level_menu->insertItem(Icon("kmud_lvlup.xpm"), i18n("Level Up"), ID_MAPPER_LEVEL_UP);
	level_menu->insertItem(Icon("kmud_lvldown.xpm"),i18n("Level Down"), ID_MAPPER_LEVEL_DOWN );
	level_menu->insertSeparator();	
	level_menu->insertItem(i18n("Create New Top Level"), ID_MAPPER_TOOL_NEW_TOP );		
	level_menu->insertItem(i18n("Create New Bottom Level"), ID_MAPPER_TOOL_NEW_BOTTOM );
	level_menu->insertItem(i18n("Delete Current Level"), ID_MAPPER_LEVEL_DEL );	

	zone_menu->insertItem(Icon("kmud_zoneup.xpm"),i18n("Dsiplay &Parent Zone"), ID_MAPPER_TOOL_ZONE_UP );	
	zone_menu->insertItem(i18n("Edit Curent Zone Properties"),ID_MAPPER_TOOL_ZONE_PROP);
	
	tools_menu->insertItem(i18n("View Curent &Position"),ID_MAPPER_TOOL_VIEW_CURENT);
	tools_menu->setAccel(CTRL+Key_P, ID_MAPPER_TOOL_VIEW_CURENT);	
	tools_menu->insertSeparator();
	tools_menu->insertItem(i18n("&Level Tools"),level_menu);
	tools_menu->insertItem(i18n("&Zone Tools"),zone_menu);	


	connect (toolsToolbar,SIGNAL(clicked(int)),this,SLOT(slotToolsToolbarClicked(int)));
	connect (tools_menu,SIGNAL(activated(int)),this,SLOT(slotToolsToolbarClicked(int)));

	map->setTool(toolList.at(0));	
}

void CMapWindow::movePlayerBy(QString dirCmd)
{
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{		
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		if (mudPro->getFollowMode())
		{
			map->movePlayerBy(dirCmd,mudPro->getCreateMode());
			CMapRoom *room=map->getCurrentRoom();									
			mapView->center(room->getLowX()+HALF_ROOM_SIZE,room->getLowY()+HALF_ROOM_SIZE);
		}
	}
}

/** Check to see if a string is a valid move command */
bool CMapWindow::validMoveCmd(QString dirCmd)
{
	return map->validMoveCmd(dirCmd);
}

/** Execute the before enter command of the next room */
void CMapWindow::executeBeforeEnterCommand(QString dirCmd)
{

	QString specialCmd = "";	
	direction dir = map->cmdToDir(dirCmd);
	if (dir == SPECIAL)
		specialCmd = dirCmd;
	
	CMapPath *path = map->getCurrentRoom()->getPathDirection(dir,specialCmd);
	
	if (path)
	{
		CMapRoom *room = path->getDestRoom();
		QString command = room->getBeforeEnterCommand();
		
		if (command.stripWhiteSpace()!="")
			mudView->slotInputReturnPressed(command);			
	}
}

/** Execute the after enter command of the current room */
void CMapWindow::executeAfterEnterCommand(QString dirCmd)
{
	QString specialCmd = "";	
	direction dir = map->cmdToDir(dirCmd);
	if (dir == SPECIAL)
		specialCmd = dirCmd;
	
	CMapPath *path = map->getCurrentRoom()->getPathDirection(dir,specialCmd);
	
	if (path)
	{
		CMapRoom *room = path->getDestRoom();		
		QString command = room->getAfterEnterCommand();
	
		if (command.stripWhiteSpace()!="")
			mudView->slotInputReturnPressed(command);	
	}
}

/** Execute the before exiting command of the current room */
void CMapWindow::executeBeforeExitCommand(void)
{
	CMapRoom *room = map->getCurrentRoom();
	QString command  = room->getBeforeExitCommand();
	
	if (command.stripWhiteSpace()!="")
		mudView->slotInputReturnPressed(command );
}

/** Execute the after exiting command of the current room */
void CMapWindow::executeAfterExitCommand(void)
{
	CMapRoom *room = map->getCurrentRoom();
	QString command  = room->getAfterExitCommand();
	
	if (command.stripWhiteSpace()!="")
		mudView->slotInputReturnPressed(command );
}


/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION

void CMapWindow::closeEvent(QCloseEvent *e)
{
	map->saveMap();
	emit (slotMapperClosed());
}

void CMapWindow::slotToolsToolbarClicked(int id)
{
	bool edit = (id ==0);
	if (id<10)
	{
		map->setTool(toolList.at(id));
	}
	else
	{
		commandCallback(id);
		return;
	}

	edit_menu->setItemEnabled(ID_MAPPER_EDIT_CUT,edit);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_COPY,edit);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_PASTE,edit);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DELETE,edit);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_SELECTALL,edit);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DESELECTALL,edit);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_INVERTSEL,edit);

	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_CUT,edit);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_COPY,edit);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_PASTE,edit);
}

void CMapWindow::commandCallback(int id_){
	switch (id_)
	{
		ON_CMD(ID_MAPPER_FILE_EXPORT,          slotFileExport())
		ON_CMD(ID_MAPPER_FILE_EXPORT_KCMFILE,  slotFileExportToKCMapFile())
		ON_CMD(ID_MAPPER_FILE_IMPORT,          slotFileImport())
		ON_CMD(ID_MAPPER_FILE_NEW,             slotFileNew())
		ON_CMD(ID_MAPPER_FILE_CLOSE,           slotFileCloseWindow())
		ON_CMD(ID_MAPPER_FILE_INFO,            slotFileInfo())

		ON_CMD(ID_MAPPER_OPTIONS_PREFS,        slotOptionsPrefs())
		ON_CMD(ID_MAPPER_OPTIONS_FOLLOW,       slotOptionsFollow())
		ON_CMD(ID_MAPPER_OPTIONS_CREATE,       slotOptionsCreate())

		ON_CMD(ID_MAPPER_EDIT_CUT,             slotEditCut())
		ON_CMD(ID_MAPPER_EDIT_COPY,            slotEditCopy())
		ON_CMD(ID_MAPPER_EDIT_PASTE,           slotEditPaste())
		ON_CMD(ID_MAPPER_EDIT_DELETE,          slotEditDelete())
		ON_CMD(ID_MAPPER_EDIT_SELECTALL,       slotSelectAll())
		ON_CMD(ID_MAPPER_EDIT_DESELECTALL,     slotDeselectAll())
		ON_CMD(ID_MAPPER_EDIT_INVERTSEL,       slotInvertSelection())	
		ON_CMD(ID_MAPPER_EDIT_UNDO_AUTO,       slotUndoAutoCreation())	
				
		ON_CMD(ID_MAPPER_VIEW_STATUSBAR,       slotViewStatusBar())
		ON_CMD(ID_MAPPER_VIEW_TOOLS,           slotViewTools())
		ON_CMD(ID_MAPPER_VIEW_MODES,           slotViewModesToolBar())
		ON_CMD(ID_MAPPER_VIEW_MAIN,            slotViewMainToolBar())
		ON_CMD(ID_MAPPER_VIEW_LOWER,           slotViewLowerMap())
		ON_CMD(ID_MAPPER_VIEW_HIGHER,          slotViewHigherMap())
		ON_CMD(ID_MAPPER_VIEW_OVERVIEW,        slotViewOverview())
		ON_CMD(ID_MAPPER_VIEW_ALWAYS_ON_TOP,   slotViewAlwaysOnTop())		
		ON_CMD(ID_MAPPER_VIEW_SPEEDWALK,       slotViewSpeedwalk())
	
		ON_CMD(ID_MAPPER_LEVEL_UP,             slotLevelUp())
		ON_CMD(ID_MAPPER_LEVEL_DOWN,           slotLevelDown())	
		ON_CMD(ID_MAPPER_LEVEL_DEL,            slotDelLevel())		
		ON_CMD(ID_MAPPER_TOOL_NEW_TOP,         slotCreateTopLevel())
		ON_CMD(ID_MAPPER_TOOL_NEW_BOTTOM,      slotCreateBottomLevel())

		ON_CMD(ID_MAPPER_TOOL_ZONE_UP,         slotZoneUp())
		ON_CMD(ID_MAPPER_TOOL_ZONE_PROP,       slotZoneProp())		
		
		ON_CMD(ID_MAPPER_TOGGLE_GRID,          slotShowGrid())
		ON_CMD(ID_MAPPER_TOOL_VIEW_CURENT,     slotViewCurrentPosition())
	}
}

void CMapWindow::statusCallback(int id_)
{
	switch (id_)
	{
		ON_STATUS_MSG(ID_MAPPER_FILE_NEW,        i18n("Erase current map and create a new map"))
		ON_STATUS_MSG(ID_MAPPER_FILE_EXPORT,     i18n("Export the map to a new file"))
		ON_STATUS_MSG(ID_MAPPER_FILE_EXPORT_KCMFILE,     i18n("Export the map to a text file (config style)"))
		ON_STATUS_MSG(ID_MAPPER_FILE_IMPORT,     i18n("Import a map from a file (Not yet implemented)"))
		ON_STATUS_MSG(ID_MAPPER_FILE_CLOSE,      i18n("Close the map window"))
		ON_STATUS_MSG(ID_MAPPER_FILE_INFO,       i18n("Display information about the map"))

		ON_STATUS_MSG(ID_MAPPER_OPTIONS_PREFS,   i18n("Configure mapper settings"))
		ON_STATUS_MSG(ID_MAPPER_OPTIONS_FOLLOW,  i18n("Turns the follow player mode on or off"))
		ON_STATUS_MSG(ID_MAPPER_OPTIONS_CREATE,  i18n("Create new rooms whilst following the player"))

		ON_STATUS_MSG(ID_MAPPER_EDIT_CUT,        i18n("Cut map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_COPY,       i18n("Copy map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_PASTE,      i18n("Paste map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_DELETE,     i18n("Delete map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_SELECTALL,  i18n("Select all elements"))		
		ON_STATUS_MSG(ID_MAPPER_EDIT_DESELECTALL,i18n("Deselect all elements"))		
		ON_STATUS_MSG(ID_MAPPER_EDIT_INVERTSEL,  i18n("Invert selected elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_UNDO_AUTO,  i18n("Undo automatically created changes"))
		
		ON_STATUS_MSG(ID_MAPPER_VIEW_STATUSBAR,  i18n("Display/hide the status bar"))
	  	ON_STATUS_MSG(ID_MAPPER_VIEW_TOOLS,      i18n("Display/hide tools toolbar"))
		ON_STATUS_MSG(ID_MAPPER_VIEW_MODES,      i18n("Display/hide modes toolbar"))
		ON_STATUS_MSG(ID_MAPPER_VIEW_MAIN,       i18n("Display/hide main toolbar"))
		ON_STATUS_MSG(ID_MAPPER_VIEW_ALWAYS_ON_TOP, i18n("Keep mapper window on top"))

		ON_STATUS_MSG(ID_MAPPER_TOOL_SELECT,	 i18n("The select tool"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_CREATE_ROOM,i18n("Create a new room"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_CREATE_PATH,i18n("Create a new path"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_CREATE_TEXT,i18n("Create a text lable"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_DELETE,	 i18n("Delete a map element"))
		
		ON_STATUS_MSG(ID_MAPPER_TOOL_VIEW_CURENT,i18n("Center view on current position"))
		
		ON_STATUS_MSG(ID_MAPPER_LEVEL_UP,        i18n("Up a level"))
		ON_STATUS_MSG(ID_MAPPER_LEVEL_DOWN,      i18n("Down a level"))
		
		ON_STATUS_MSG(ID_MAPPER_LEVEL_DEL,       i18n("Delete the current level"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_NEW_TOP,	 i18n("Create a new top level"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_NEW_BOTTOM, i18n("Create a new bottom level"))

		ON_STATUS_MSG(ID_MAPPER_TOOL_ZONE_UP,    i18n("Display the parent zone of the current zone"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_ZONE_PROP,  i18n("Edit the properties of the current zone"))		
				
		
		ON_STATUS_MSG(ID_MAPPER_TOGGLE_GRID,	 i18n("Show/hide the grid"))

	}
}

void CMapWindow::slotViewCurrentPosition(void)
{
	map->showLevel(map->getCurrentRoom()->getX(),map->getCurrentRoom()->getY(),map->getCurrentRoom()->getLevel(),map->getCurrentRoom()->getZone());	
	mapView->center(map->getCurrentRoom()->getLowX()+HALF_ROOM_SIZE,map->getCurrentRoom()->getLowY()+HALF_ROOM_SIZE);

}

void CMapWindow::slotOptionsCreate(void)
{
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		mudPro->setCreateMode(!mudPro->getCreateMode());
		menuBar()->setItemChecked(ID_MAPPER_OPTIONS_CREATE, mudPro->getCreateMode());
		modesToolbar->setButton(ID_MAPPER_OPTIONS_CREATE, mudPro->getCreateMode());
		mudPro->writeData();
	}
}

void CMapWindow::slotOptionsFollow(void)
{
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		mudPro->setFollowMode(!mudPro->getFollowMode());
		menuBar()->setItemChecked(ID_MAPPER_OPTIONS_FOLLOW, mudPro->getFollowMode());
		modesToolbar->setButton(ID_MAPPER_OPTIONS_FOLLOW, mudPro->getFollowMode());
		mudPro->writeData();
	}
}

void CMapWindow::slotDelLevel(void)
{
	if ( QMessageBox::information (this,i18n("Kmud Mapper"),
                                      i18n("This will erase the current level\n"
                                      "Do you wish to erase the level?"),
                                      i18n("&Erase"),i18n("&Cancel"),0,1)==0)
	{

		// Delete the current level
		map->deleteLevel(map->getCurrentLevel(),map->getCurrentZone(),true);
		map->showLevel(map->getCurrentRoom()->getX(),map->getCurrentRoom()->getY(),map->getCurrentRoom()->getLevel(),map->getCurrentZone());	
	}
}

void CMapWindow::slotZoneUp(void)
{
	map->slotZoneUp();
}

void CMapWindow::slotZoneProp(void)
{
	map->zoneProperties(map->getCurrentZone());
}

void CMapWindow::slotLevelUp(void)
{
	map->slotLevelUp();
}

void CMapWindow::slotLevelDown(void)
{
	map->slotLevelDown();
}

/** This slot is called when the copy menu option is selected */
void CMapWindow::slotEditCopy(void)
{
	map->slotCopy();
}

/** This slot is called when the paste menu option is selected */
void CMapWindow::slotEditCut(void)
{
	map->slotCut();
}

/** This slot is called when the cut menu option is selected */
void CMapWindow::slotEditPaste(void)
{
	map->slotPaste();
}

/** This slot is called when the delete menu option is selected */
void CMapWindow::slotEditDelete(void)
{
	map->slotDelete();
}

void CMapWindow::slotViewStatusBar(void)
{
	///////////////////////////////////////////////////////////////////
	// turn the Statusbar on or off
	bViewStatusbar=!bViewStatusbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_STATUSBAR, bViewStatusbar);
	statusBar()->clear();
	enableStatusBar();
	slotStatusMsg(i18n("Ready."));
}

void CMapWindow::slotViewSpeedwalk(void)
{
	///////////////////////////////////////////////////////////////////
	// trun speed walk on or off
	bViewSpeedwalk=!bViewSpeedwalk;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_SPEEDWALK, bViewSpeedwalk);
	if (bViewSpeedwalk)
	{
		if (speedwalk_width == 0)
			speedwalk_width = 30;
		split->setSeparator(speedwalk_width);
	}
	else
	{
		speedwalk_width = split->getSeparator();
		split->setSeparator(0);
	}
	slotStatusMsg(i18n("Ready."));	
}

void CMapWindow::slotViewOverview(void)
{
	bViewOverview = !bViewOverview;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_OVERVIEW, bViewOverview);	
	map->setViewOverview(bViewOverview);
}

void CMapWindow::slotViewTools(void)
{
	bViewToolsToolbar=!bViewToolsToolbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_TOOLS, bViewToolsToolbar);	
	if (bViewToolsToolbar) enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_TOOLS); else enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_TOOLS);
}

void CMapWindow::slotStatusMsg(const char *text)
{
	///////////////////////////////////////////////////////////////////
	// change status message permanently
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG );
}


void CMapWindow::slotStatusHelpMsg(const char *text)
{
	///////////////////////////////////////////////////////////////////
	// change status message of whole statusbar temporary (text, msec)
	statusBar()->message(text, 2000);
}


void CMapWindow::slotFileCloseWindow(void)
{
	emit (slotMapperClosed());
}

void CMapWindow::slotFileExport(void)
{
	int pos;
	QString filename = KFileDialog::getSaveFileName (""	, "*.map", this, "export");
	if (filename !="" && filename != NULL)
	{
	  pos = filename.findRev(".map");
	  if (pos == -1 || pos != ((int)filename.length()-4))
	  {
		  filename.append(".map");
		}
		map->exportMap(filename);
  }
}

/** export the map to a KConfig style */
void CMapWindow::slotFileExportToKCMapFile()
{
  int pos;
	QString filename = KFileDialog::getSaveFileName (""	, "*.cmap", this, "export");
	if (filename !="" && filename != NULL)
	{
	  pos = filename.findRev(".cmap");
	  if (pos == -1 || pos != ((int)filename.length()-5))
	  {
		  filename.append(".cmap");
		}
		map->exportKCMap(filename);
  }
}

void CMapWindow::slotFileImport(void)
{
	QString filename = KFileDialog::getOpenFileName ("", "*.map |native file (*.map)\n*.cmap |config style file (*.cmap)", this, "export");
	if (filename !="" && filename != NULL)
	{
	  int pos = filename.findRev(".cmap");
	  if (pos>0 && pos == ((int)filename.length()-5))
	  {
		  map->importKCMap(filename);
		}
		else
		{
			map->importMap(filename);
		}
	}

	CMapRoom *room=map->getCurrentRoom();									

	if (room)
	  mapView->center(room->getLowX()+HALF_ROOM_SIZE,room->getLowY()+HALF_ROOM_SIZE);

}

void CMapWindow::slotFileNew(void)
{
	if ( QMessageBox::information (this,i18n("Kmud Mapper"),
                                      i18n("This will erase the current map\n"
                                      "Do you wish to erase the map?"),
                                      i18n("&Erase"),i18n("&Cancel"),0,1)==0)
	{
		map->eraseMap();
		map->createNewMap();
		map->redraw();
	}
}

void CMapWindow::slotOptionsPrefs(void)
{
	directions *directions;

	DlgMapOptionsPreferences d(this, "Preferences");

	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);
	 	directions = mudPro->getDirections();
		d.setNorth(directions->north);
		d.setNorthEast(directions->northeast);
		d.setEast(directions->east);
		d.setSouthEast(directions->southeast);
		d.setSouth(directions->south);
		d.setSouthWest(directions->southwest);
		d.setWest(directions->west);
		d.setNorthWest(directions->northwest);
		d.setUp(directions->up);
		d.setDown(directions->down);
		d.setDoValidMoveChecking(mudPro->getDoValidMoveChecking());
		d.setFailedMoveMsg1(mudPro->getFailedMoveMsg1());
		d.setFailedMoveMsg2(mudPro->getFailedMoveMsg2());
		d.setFailedMoveMsg3(mudPro->getFailedMoveMsg3());
	}

	d.setBackgroundColour(mapperData->backgroundColour);
	d.setDefaultPathColour(mapperData->defaultPathColour);
	d.setDefaultRoomColour(mapperData->defaultRoomColour);
	d.setLowerPathColour(mapperData->lowerPathColour);
	d.setLowerRoomColour(mapperData->lowerRoomColour);
	d.setHigherPathColour(mapperData->higherPathColour);
	d.setHigherRoomColour(mapperData->higherRoomColour);
	d.setGridColour(mapperData->gridColour);
	d.setLoginColour(mapperData->loginColour);
	d.setSelectedColour(mapperData->selectedColour);
	d.setSpecialColour(mapperData->specialColour);
	d.setCurrentColour(mapperData->currentColour);
	d.setEditColour(mapperData->editColour);		
	d.setDefaultTextColour(mapperData->defaultTextColour);
	d.setDefaultTwoWay(map->getDefaultPathTwoWay());
	d.setSpeedWalkLimit(map->getSpeedWalkLimit());
	d.setSpeedwalkDelay(mapperData->speedwalkDelay);
	d.setDoLimitSpeedWalk(map->getSpeedWalkAbortActive());
	
	if (d.exec())
	{
		directions->north = d.getNorth();
		directions->northeast = d.getNorthEast();
		directions->east = d.getEast();
		directions->southeast = d.getSouthEast();
		directions->south = d.getSouth();
		directions->southwest = d.getSouthWest();
		directions->west = d.getWest();
		directions->northwest = d.getNorthWest();
		directions->up = d.getUp();
		directions->down = d.getDown();		
		
		mapperData->speedwalkDelay = d.getSpeedwalkDelay();
		mapperData->backgroundColour = d.getBackgroundColour();
		mapperData->gridColour = d.getGridColour();	
		map->setElementsColour(d.getLowerPathColour(),d.getDefaultPathColour(),d.getHigherPathColour(),
                           d.getLowerRoomColour(),d.getDefaultRoomColour(),d.getHigherRoomColour(),d.getLoginColour(),d.getSelectedColour(),
                           d.getDefaultTextColour(),d.getSpecialColour(),d.getEditColour(),d.getCurrentColour());
		map->setDefaultPathTwoWay(d.getDefaultTwoWay());
		
		map->setSpeedWalkLimit(d.getSpeedWalkLimit());
		map->setSpeedWalkAbortActive(d.getDoLimitSpeedWalk());
  		
		if (entry!=NULL)
		{
			CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);
			mudPro->setDoValidMoveChecking(d.getDoValidMoveChecking());
			mudPro->setFailedMoveMsg1(d.getFailedMoveMsg1());
			mudPro->setFailedMoveMsg2(d.getFailedMoveMsg2());
			mudPro->setFailedMoveMsg3(d.getFailedMoveMsg3());
		}
				
		map->redraw();
	}


	saveOptions();
}

void CMapWindow::slotSelectAll(void)
{
	map->slotSelectAll();
}

void CMapWindow::slotDeselectAll(void)
{
	map->slotDeselectAll();
}

void CMapWindow::slotInvertSelection(void)
{
	map->slotInvertSelection();
}

/** This slot is used to create a new top level */
void CMapWindow::slotCreateTopLevel(void)
{
	map->createLevel(UP,map->getCurrentZone());
	slotEnableLevelUp(true);
}

/** This slot is used to create a new bottom level */
void CMapWindow::slotCreateBottomLevel(void)
{
	map->createLevel(DOWN,map->getCurrentZone());
	slotLevelUp();
	slotEnableLevelDown(true);	
}


/** This slot is used to show/hide the grid */
void CMapWindow::slotShowGrid(void)
{
	bGridShow = !bGridShow;
	map->setGridVisable(bGridShow);
	toolsToolbar->setButton(ID_MAPPER_TOGGLE_GRID,bGridShow);
	view_menu->setItemChecked(ID_MAPPER_TOGGLE_GRID,bGridShow);
}

/** Turn on or off the higher map */
void CMapWindow::slotViewHigherMap(void)
{
	bViewHigherMap= !bViewHigherMap;	
	view_menu->setItemChecked(ID_MAPPER_VIEW_HIGHER,bViewHigherMap);
	map->setViewHigherMap(bViewHigherMap);
}

/** Turn on or off the lower map */
void CMapWindow::slotViewLowerMap(void)
{
	bViewLowerMap= !bViewLowerMap;
	view_menu->setItemChecked(ID_MAPPER_VIEW_LOWER,bViewLowerMap);
	map->setViewLowerMap(bViewLowerMap);
}

/** turn on or off the main toolbar */
void CMapWindow::slotViewMainToolBar(void)
{
	bViewMainToolbar=!bViewMainToolbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_MAIN, bViewMainToolbar);	
	if (bViewMainToolbar) enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MAIN); else enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MAIN);
}

/** turn on or off the modes toolbar */
void CMapWindow::slotViewModesToolBar(void)
{
	bViewModesToolbar=!bViewModesToolbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_MODES, bViewModesToolbar);	
	if (bViewModesToolbar) enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MODES); else enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MODES);
}

void CMapWindow::slotMovePlayer(QString dirCmd)
{
	emit movePlayer(dirCmd);
}

void CMapWindow::slotLevelChanged(signed int level)
{
	QString lvlStr;
	
	lvlStr.sprintf(i18n("Level : %d"),level);
	statusBar()->changeItem(lvlStr, ID_MAPPER_STATUS_LEVEL );
}

void CMapWindow::slotZoneChanged(QString zoneName)
{
	statusBar()->changeItem("Zone : " + zoneName, ID_MAPPER_STATUS_ZONE );
}


void CMapWindow::slotFileInfo(void)
{
        DlgMapInfo d(this, "Information");
        CMapRoom *loginRoom = map->getLoginRoom();
        int lvl,room,text,path,zone;

        map->getTotals(&lvl,&room,&text,&path,&zone);

        d.setStart(loginRoom->getX(),loginRoom->getY(),loginRoom->getLevel(),mapperData->zoneList.find(loginRoom->getZone()));
        d.setGeneral(mud_Title,mud_Login);
        d.setTotals(lvl,room,path,text,zone);

        d.exec();
}       

void CMapWindow::slotViewAlwaysOnTop(void)
{
  bStayOnTop = !bStayOnTop;
  view_menu->setItemChecked(ID_MAPPER_VIEW_ALWAYS_ON_TOP, bStayOnTop);
  if (bStayOnTop)
		KWM::setDecoration(winId(), KWM::normalDecoration | KWM::staysOnTop);		
	else
		KWM::setDecoration(winId(), KWM::normalDecoration);		
}


void CMapWindow::slotSplitMove(void)
{
/*
	if (!bViewSpeedwalk)
	{
//		bViewSpeedwalk = true;
//		menuBar()->setItemChecked(ID_MAPPER_VIEW_SPEEDWALK,bViewSpeedwalk);
	}		
	else
	{
		// speedwalk_width = split->getSeparator();
	}
*/
}

void CMapWindow::slotUndoAutoCreation()
{
  mapUndo->slotUndo();
}

/** Used to enable/disable level up button */
void CMapWindow::slotEnableLevelUp(bool enable)
{
	level_menu->setItemEnabled(ID_MAPPER_LEVEL_UP,enable);
	toolsToolbar->setItemEnabled(ID_MAPPER_LEVEL_UP,enable);
}

/** Used to enable/disable level down button */
void CMapWindow::slotEnableLevelDown(bool enable)
{
	level_menu->setItemEnabled(ID_MAPPER_LEVEL_DOWN,enable);
	toolsToolbar->setItemEnabled(ID_MAPPER_LEVEL_DOWN,enable);
}

/** Used to enable/disable zone up button */
void CMapWindow::slotEnableZoneUp(bool enable)
{
	zone_menu->setItemEnabled(ID_MAPPER_TOOL_ZONE_UP,enable);
	toolsToolbar->setItemEnabled(ID_MAPPER_TOOL_ZONE_UP,enable);
}

/** initialiizes keyboard accelerators */
void CMapWindow::initAccelerators()
{
	//define defaults, then load current values form config file
	
	// Boss Key, default is ESCAPE
	if (accelerators->defaultKey("boss_key") == 0)
	{
	  accelerators->insertItem(i18n("Boss key"), "boss_key", Key_Escape);
	}
	
	// Undo map change, default is CTRL-BACKSPACE
	if (accelerators->defaultKey("map_undo") == 0)
	{
	  accelerators->insertItem(i18n("Mapper Undo"), "map_undo", "CTRL+Backspace");
	}
	// read values from kmudrc
	accelerators->readSettings(kapp->getConfig());
	
	accelerators->connectItem("map_undo", this, SLOT(slotUndoAutoCreation()));
  accelerators->changeMenuAccel(edit_menu, ID_MAPPER_EDIT_UNDO_AUTO, "map_undo");

}

/** clear the undo list */
void CMapWindow::slotClearUndo()
{
  mapUndo->clear();
  edit_menu->setItemEnabled(ID_MAPPER_EDIT_UNDO_AUTO, false);
}

/**  */
void CMapWindow::slotUndoPossible()
{
  edit_menu->setItemEnabled(ID_MAPPER_EDIT_UNDO_AUTO, true);
}

/** returns pointer to the mapper data class */
CMapData* CMapWindow::getMapperData()
{
	return mapperData;
}
