/***************************************************************************
                          cmaptext.cpp
                      -------------------
    description          : Map text element
    begin                : Wed Oct 20 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmaptext.h"

/** The constructor for text elements
  * @param str The text displayed by the text element
  * @param x The x position of the text on the map
  * @param y The y position of the text on the map
  * @param font The font used to display the text
  */
CMapText::CMapText (QString str,int x,int y,QFont f,CMapZone *mapZone)
{
	text = str;
	font = f;
 	col = black;
 	zone = mapZone;
 	QFontMetrics fm(f);
	setCords(x,y,x+fm.width(str),y-fm.height());	
}

/** The destrutor for text elements */
CMapText::~CMapText()
{

}

void CMapText::setEditMode(bool edit)
{
	editMode = false;
	
	if (edit)
		setSelected(false);
}
	
/** Over ridden CMapElement::getElementType to return the type TEXT
  * @return The element type of the element
  */	
elementTyp CMapText::getElementType(void)	
{ return TEXT; }	

/** This method is over ridden from CMapElement::paint. It is used
  * draw the text on the map.
  * @param p A pointer to the paint device the text is to be drawn on
  */
void CMapText::setColour(QColor colour)
{ col = colour; }

/** This is used to return the colour of the text
  * @return The colour of the text
  */
QColor CMapText::getColour(void)
{ return col; }

/** This method is over ridden from CMapElement::paint. It is used
  * draw the text on the map.
  * @param p A pointer to the paint device the text is to be drawn on
  * @param currentZone The current zone
  */
void CMapText::paint(QPainter *p,CMapZone *currentZone)
{
	p->setPen(getColour());
	p->setBrush(getColour());
//	QColor col = getColour();
	p->setFont(font);
	p->drawText(getLowX(),getLowY(),text,-1);

	if (getSelected())
	{
		p->setPen(getSelectColour());
		p->setBrush(getSelectColour());
		p->drawRect(getHiX()-3,getLowY()-3,6,6);
	}

}

/** This method is over ridden from CMapElement::dragPaint. It is used
  * draw the text on the map at a given offset.
  * @param p A pointer to the paint device the text is to be drawn on
  * @param offsetx The x offset of from the text's posotion where the text is to be drawn
  * @param offsetx The y offset of from the text's posotion where the text is to be drawn	
  * @param currentZone The current zone
  */	
void CMapText::dragPaint(QPainter *p,int offsetx,int offsety,CMapZone *currentZone)
{
	p->setPen(black);
	p->setFont(font);
	p->drawText(getLowX()+offsetx,getLowY()+offsety,text,-1);
}

/** This method is over ridden from CMapElement::lowerpaint. It is used
  * draw the text on the map as a lower level.
  * @param p A pointer to the paint device the text is to be drawn on
  * @param currentZone The current zone
  */
void CMapText::lowerPaint(QPainter *p,CMapZone *currentZone)
{
}

/** This method is over ridden from CMapElement::higherpaint. It is used
  * draw the text on the map as a higher level.
  * @param p A pointer to the paint device the text is to be drawn on
  * @param currentZone The current zone
  */
void CMapText::higherPaint(QPainter *p,CMapZone *currentZone)
{
}

/** This is used to create  a new copy of the text element and return
  * a pointer to the new text element
  * @return A pointer to the copy of the text element.
  */
CMapElement *CMapText::copy(void)
{
	CMapText *newText =  new CMapText (text,getLowX(),getLowY(),font,getZone());
	newText->setLevel(getLevel());
	newText->setColour(getColour());
	newText->setSelectColour(getSelectColour());
	return newText;
}

/** Used to set the font of the text. This font is used when drawing the text.
  * @param f The required font of the text
  */
void CMapText::setFont(QFont f)
{ font = f; }

/** This menthod will return the font used to display the text
  * @return The font used to display text
  */
QFont CMapText::getFont(void)
{ return font; }

/** Gets the text of the text element
  * @return The text
  */
QString CMapText::getText(void)
{ return text; }

/** Used to find out which level the text element is on
  * @return The level number
  */
int CMapText::getLevel(void)
{ return lvl; }

/** Sets the text of the text element
  * @param str The new string that the text element should be set to
  */
void CMapText::setText(QString str)
{ text = str; }

/** Used to tell the text element which level it is on
  * @param level The level number
  */
void CMapText::setLevel(int level)
{ lvl = level; }

/** Set the zone that the text element belongs to
  * @param mapZone The zone
  */
void CMapText::setZone(CMapZone *mapZone)
{ zone = mapZone; }

/** Get the zone that the text element belongs to
  * @return the zone
  */
CMapZone *CMapText::getZone(void)
{ return zone; }

void CMapText::read(KCMapFile* kcmfile)
{
  int x, y;
  QColor* defCol = new QColor(getDefaultColour());
  lvl = kcmfile->readNumEntry("level", 0);
  x = kcmfile->readNumEntry("X", -1);
  y = kcmfile->readNumEntry("Y", -1);
  font = kcmfile->readFontEntry("font");
  col = kcmfile->readColorEntry("color", defCol);
  delete defCol;
  text = kcmfile->readEntry("text");

  QFontMetrics fm(font);
  setLowHiX(x, x+fm.width(text));
  setLowHiY(y, y-fm.height());
}

void CMapText::write(KCMapFile* kcmfile)
{
  kcmfile->writeEntry("level", lvl);
  kcmfile->writeEntry("X", getLowX());
  kcmfile->writeEntry("Y", getLowY());

  kcmfile->writeEntry("text", text);
  kcmfile->writeEntry("font", font);
  kcmfile->writeEntry("color", col);
}
